// -*- C++ -*- (c) 2005-2008 Peter Rockai <me@mornfall.net>

#include <adept/dpkgpm.h>
#include <adept/pkgsystem.h>
#include <wibble/string.h>

#ifndef RPM
#include <apt-pkg/configuration.h>
#include <apt-pkg/error.h>
#include <apt-pkg/strutl.h>

#include <signal.h>
#include <sys/wait.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>

#include <iostream>
#include <sstream>

using namespace std;
using namespace ept;
using namespace ept::core;
using namespace adept;
using wibble::str::fmt;
namespace wexcept = wibble::exception;

DpkgHarness::DpkgHarness( package::Source &p )
    : m_seenOpCount( 0 ), m_totalOpCount( 0 ), pkgs( p ), m_recover( false )
{
}

void DpkgHarness::recover()
{
    m_recover = true;
    List empty;
    List::iterator I = empty.begin();

    StringArray args = setupArgs( I );
    {
        PkgSystem::TemporaryUnlock unlock( _system );
        forkDpkg( args );
    }
    m_recover = false;
}

bool DpkgHarness::go( const List &l )
{
    cerr << "DpkgHarness::go()" << endl;

    m_list.clear();
    std::copy( l.begin(), l.end(), std::back_inserter( m_list ) );

    computeTotals();

    runScripts( "DPkg::Pre-Invoke", false );
    runScripts( "DPkg::Pre-Install-Pkgs", true );

    for (vector<Item>::iterator I = m_list.begin(); I != m_list.end();) {
        StringArray args = setupArgs( I );

        // TODO if m_cancel cleanup and break out of the loop

        if ( _config->FindB("Debug::pkgDPkgPM",false) == true )
            continue;

        forkDpkg( args );
        runScripts( "DPkg::Post-Invoke", false );
    }
    return true;
}

void DpkgHarness::computeTotals()
{
    m_seenOpCount = m_totalOpCount = 0;
    m_seenOps.clear();
    for (vector<Item>::iterator I = m_list.begin(); I != m_list.end();I++)
    {
        PackageState s = pkgs.getInternal< package::State >( I->Pkg );
        int x = 0;
        switch ( I->Op ) {
            // installed, half-configured, half-installed, config-files
            case Item::Remove: x = 4; break;
            // install: ?
            // upgrade: half-configured, unpacked, half-installed
            case Item::Install: x = (s.upgrade() ? 3 : 2); break;
            // config-files, not-installed
            case Item::Purge: x = 2; break;
            // upgrade: unpacked, half-configured, installed
            // install: 
            case Item::Configure: x = (s.upgrade() ? 3 : 3); break;
        }
        m_totalOpCount += x;
    }
}

bool DpkgHarness::forkDpkg( StringArray args )
{
    wexcept::AddContext _ctx( "Running dpkg");
    wexcept::AddContext _ctx2( fmt( args ) );
    cerr << "DPkgPM::forkDpkg:" << fmt( args ) << endl;

    // TODO do we leak filedescriptors here?
    socketpair( PF_UNIX, SOCK_STREAM, 0, m_dpkgFds );
    socketpair( PF_UNIX, SOCK_STREAM, 0, m_dpkgStdin );
    m_dpkgPipe = Pipe( m_dpkgFds[ 0 ] );

    cout << flush;
    clog << flush;
    cerr << flush;

    /* Mask off sig int/quit. We do this because dpkg also does when
       it forks scripts. What happens is that when you hit ctrl-c it sends
       it to all processes in the group. Since dpkg ignores the signal
       it doesn't die but we do! So we must also ignore it */
    sighandler_t old_SIGQUIT = signal(SIGQUIT,SIG_IGN);
    sighandler_t old_SIGINT = signal(SIGINT,SIG_IGN);

    // Fork dpkg
    pid_t Child = ExecFork();
    if (Child == 0) {
        if (! setupChild ()) {
            cerr << "Error in dpkg post-fork setup!" << endl;
            _exit (100);
        }
        const char * argv[ args.size() + 1 ];
        std::transform( args.begin(), args.end(), argv,
                        std::mem_fun_ref( &string::c_str ) );
        argv[ args.size() ] = 0;
        execvp( argv[0], const_cast< char * const * >( argv ) );
        cerr << "Error executing dpkg!" << endl;
        _exit (100);
    }

    close( m_dpkgFds[1] );

    int Status = 0;
    int ret = 0;
    while ((ret = waitpid (Child, &Status, WNOHANG)) != Child) {
        if (errno == EINTR || ret == 0) {
            dpkgMonitor();
            usleep( 50000 ); // 0.05 second hang
            // TODO if m_cancel, kill( Child, 15 )
            continue;
        }
        runScripts( "DPkg::Post-Invoke", false );

        signal(SIGQUIT,old_SIGQUIT);
        signal(SIGINT,old_SIGINT);
        throw wexcept::System( "Waiting for subprocess." );
    }
    dpkgMonitor();

    signal(SIGQUIT,old_SIGQUIT);
    signal(SIGINT,old_SIGINT);

    // do we also run post-invoke when successful? why not?
    // Check for an error code.
    if (WIFEXITED(Status) == 0 || WEXITSTATUS(Status) != 0)
    {
        runScripts( "DPkg::Post-Invoke", false );
        if (WIFSIGNALED(Status) != 0 && WTERMSIG(Status) == SIGSEGV)
            throw AptException("Sub-process caught SIGSEGV.");

        if (WIFEXITED(Status) != 0)
            throw AptException(
                fmt( "Sup-process returned error code %u",
                     WEXITSTATUS( Status ) ) );

        throw AptException( "Sub-process exited unexpectedly" );
    }

    return true;
}

static int argsSize( const DpkgHarness::StringArray arr )
{
    int ret = 0;
    DpkgHarness::StringArray::const_iterator i;
    for ( i = arr.begin(); i != arr.end(); ++i )
        ret += i->size();
    return ret;
}

DpkgHarness::StringArray DpkgHarness::setupArgs( std::vector<Item>::iterator &I )
{
    cerr << "DPkgPM::setupArgs()" << endl;
    unsigned int MaxArgs = _config->FindI( "Dpkg::MaxArgs", 350 );
    unsigned int MaxArgBytes = _config->FindI( "Dpkg::MaxArgBytes", 8192 );

    vector<Item>::iterator J = I;
    for (; J != m_list.end() && J->Op == I->Op; J++)
        ;

    // Generate the argument list
    StringArray args;
        
    if (J - I > (signed) MaxArgs)
        J = I + MaxArgs;

    unsigned int n = 0;
    string Tmp = _config->Find("Dir::Bin::dpkg","dpkg");
    args.push_back( Tmp );

    // Stick in any custom dpkg options
    Configuration::Item const *Opts = _config->Tree("DPkg::Options");
    if (Opts != 0)
    {
        Opts = Opts->Child;
        for (; Opts != 0; Opts = Opts->Next)
        {
            if (Opts->Value.empty() == true)
                continue;
            args.push_back( Opts->Value );
        }
    }

    args.push_back( "--status-fd" );
    // we hardcode 3 here, since dpkg seems to screw up when the fd is
    // more than a single digit; in setupChild, we dup2 the real fd to 3...
    args.push_back( "3" );

    if ( m_recover ) {
        args.push_back( "--configure" );
        args.push_back( "-a" );
    } else {
        switch (I->Op)
        {
            case Item::Remove:
                args.push_back( "--force-depends" );
                args.push_back( "--force-remove-essential" );
                args.push_back( "--remove" );
                m_currentOp = ORemove;
                break;
                
            case Item::Purge:
                args.push_back( "--force-depends" );
                args.push_back( "--force-remove-essential" );
                args.push_back( "--purge" );
                m_currentOp = OPurge;
                break;

            case Item::Configure:
                args.push_back( "--configure" );
                m_currentOp = OConfigure;
                break;

            case Item::Install:
                args.push_back( "--unpack" );
                m_currentOp = OInstall;
                break;
        }

        // Write in the file or package names
        if (I->Op == Item::Install)
        {
            for (;I != J && argsSize( args ) < MaxArgBytes; I++)
            {
                if (I->File[0] != '/')
                    throw AptException(
                        fmt( "Internal Error: path '%s' not absolute",
                             I->File.c_str() ) );
                args.push_back( I->File );
            }
        }
        else
        {
            for (;I != J && argsSize( args ) < MaxArgBytes; I++)
            {
                args.push_back( I->Pkg.Name() );
            }
        }
    }

    J = I;
    return args;
}

bool DpkgHarness::setupChild ()
{
    cerr << "DpkgHarness::setupChild ()" << endl;
    std::string dir = _config->FindDir("DPkg::Run-Directory","/");

    if ( chdir( dir.c_str() ) != 0 ) {
        throw wexcept::System( fmt( "Could not chdir to %s.", dir.c_str() ) );
    }

    if (_config->FindB("DPkg::FlushSTDIN",true) == true && isatty(STDIN_FILENO))
    {
        int Flags,dummy;
        if ( (Flags = fcntl(STDIN_FILENO,F_GETFL,dummy)) < 0 ||
             fcntl(STDIN_FILENO,F_SETFL,Flags | O_NONBLOCK) < 0 )
            throw wexcept::System( "Error flushing stdin." );

        while (read(STDIN_FILENO,&dummy,1) == 1)
            ;

        if ( fcntl(STDIN_FILENO,F_SETFL,Flags & (~(long)O_NONBLOCK)) < 0 )
            throw wexcept::System( "Error flushing stdin." );
    }

    /* No Job Control Stop Env is a magic dpkg var that prevents it
       from using sigstop */
    putenv( const_cast< char * >( "DPKG_NO_TSTP=yes" ) );
    cerr << "writing end of the pipe: " << m_dpkgFds[1] << endl;
    close( m_dpkgFds[0] );
    close( m_dpkgStdin[0] );
    dup2( m_dpkgFds[1], 3 );
    dup2( m_dpkgStdin[1], STDIN_FILENO );

    return true;
}

/* TODO we need to get some sort of conffile handling here. Dpkg
 * should notify us about the conffile issue through the status
 * pipe. */
void DpkgHarness::dpkgMonitor()
{
    if ( !m_dpkgPipe.active() )
        return;

    std::string line = m_dpkgPipe.nextLine();
    if ( line.empty() )
        return;

    typedef std::pair< std::string, std::string > SP;
    SP st = split( line, ": " );

    if ( st.first == "status" ) {
        SP s1 = split( st.second, ": " );
        SP s2 = split( s1.second, ": " );
        // TODO about that space.....
        if ( s2.first == "conffile-prompt " ) { // event
            // TODO we assume there are no ' in filenames...
            // s2.second ~ ['/etc/hello-foo' '/etc/hello-foo.dpkg-new' 1 1 ]
            SP files = split( s2.second, "' '" );
            std::string user( files.first, 1, std::string::npos );
            std::string system( files.second, 0, files.second.find( '\'' ) );
            handleConffile( s1.first, user, system );
        } else
            updateStatus( s1.first, s2.first, s2.second );
    }

}

void DpkgHarness::handleConffile( std::string conffile,
                             std::string user, std::string system )
{
    std::cerr << "ADEPT: conffile prompt: "
              << conffile << " [" << user << ", " << system << "]" << std::endl;
}

void DpkgHarness::updateStatus( std::string pkg, std::string ev, std::string /* r */ )
{
    OpAndStatus os = std::make_pair( m_currentOp, ev );
    if ( m_seenOps[ std::make_pair( os, pkg ) ] == 0 ) {
        m_seenOpCount++;
        m_seenOps[ std::make_pair( os, pkg ) ] = 1;
    }
}

void DpkgHarness::runScripts( const char *Cnf, bool sP )
{
    cerr << "DpkgHarness::runScripts ('" << Cnf << "', " << sP << ")" << endl;
    Configuration::Item const *Opts = _config->Tree(Cnf);
    if (Opts == 0 || Opts->Child == 0)
        return;
    Opts = Opts->Child;

    unsigned int Count = 1;
    for (; Opts != 0; Opts = Opts->Next, Count++)
    {
        if (Opts->Value.empty() == true)
            continue;

        // Determine the protocol version
        string OptSec = Opts->Value;
        string::size_type Pos;
        if ((Pos = OptSec.find(' ')) == string::npos || Pos == 0)
            Pos = OptSec.length();
        OptSec = "DPkg::Tools::Options::" + string(Opts->Value.c_str(),Pos);

        m_version = _config->FindI (OptSec + "::Version", 1);

        // Purified Fork for running the script
        // pid_t Process = ExecFork();
        forkScript (Opts->Value.c_str(), sP);
   }
}

bool DpkgHarness::sendV1Pkgs( FILE *F )
{
    bool Die = false;
    for (vector<Item>::iterator I = m_list.begin(); I != m_list.end(); ++I)
    {
        // Only deal with packages to be installed from .deb
        if (I->Op != Item::Install)
            continue;

        // No errors here..
        if (I->File[0] != '/')
            continue;

        /* Feed the filename of each package that is pending install
           into the pipe. */
        fprintf(F,"%s\n",I->File.c_str());
        if (ferror(F) != 0)
        {
            Die = true;
            break;
        }
    }
    return ! Die;
}

bool DpkgHarness::sendV2Pkgs( FILE *F )
{
    fprintf(F,"VERSION 2\n");
   
    /* Write out all of the configuration directives by walking the 
       configuration tree */
    const Configuration::Item *Top = _config->Tree(0);
    for (; Top != 0;)
    {
        if (Top->Value.empty() == false)
        {
            fprintf(F,"%s=%s\n",
                    QuoteString(Top->FullTag(),"=\"\n").c_str(),
                    QuoteString(Top->Value,"\n").c_str());
        }
        
        if (Top->Child != 0)
        {
            Top = Top->Child;
            continue;
        }
        
        while (Top != 0 && Top->Next == 0)
            Top = Top->Parent;
        if (Top != 0)
            Top = Top->Next;
    }   
    fprintf(F,"\n");
    
    // Write out the package actions in order.
    for (vector<Item>::iterator I = m_list.begin(); I != m_list.end(); I++)
    {
        pkgDepCache::StateCache &S = pkgs.db().state()[I->Pkg];
        
        fprintf(F,"%s ",I->Pkg.Name());
        // Current version
        if (I->Pkg->CurrentVer == 0)
            fprintf(F,"- ");
        else
            fprintf(F,"%s ",I->Pkg.CurrentVer().VerStr());

        // Show the compare operator
        // Target version
        if (S.InstallVer != 0)
        {
            int Comp = 2;
            if (I->Pkg->CurrentVer != 0)
                Comp = S.InstVerIter(pkgs.db().state()).CompareVer(
                    I->Pkg.CurrentVer());
            if (Comp < 0)
                fprintf(F,"> ");
            if (Comp == 0)
                fprintf(F,"= ");
            if (Comp > 0)
                fprintf(F,"< ");
            fprintf(F,"%s ",S.InstVerIter(pkgs.db().state()).VerStr());
        }
        else
            fprintf(F,"> - ");
      
        // Show the filename/operation
        if (I->Op == Item::Install)
        {
            // No errors here..
            if (I->File[0] != '/')
                fprintf(F,"**ERROR**\n");
            else
                fprintf(F,"%s\n",I->File.c_str());
        }      
        if (I->Op == Item::Configure)
            fprintf(F,"**CONFIGURE**\n");
        if (I->Op == Item::Remove ||
            I->Op == Item::Purge)
            fprintf(F,"**REMOVE**\n");
        
        if (ferror(F) != 0) {
            throw AptException( "Sending package data to script." );
        }
    }
    return true;
}

void DpkgHarness::forkScript( const char *cmd, bool fP )
{
    wexcept::AddContext _ctx( fmt( "Running '%s'", cmd ) );
    cerr << "DpkgHarness::forkScript: " << cmd << endl;
    if (fP) {
        if (pipe( m_scriptPipe ) != 0)
            throw wexcept::System( "Failed to create IPC pipe to subprocess" );
        SetCloseExec( m_scriptPipe[ 0 ], true );
        SetCloseExec( m_scriptPipe[ 1 ], true );
    }
    pid_t Child = ExecFork ();
    if (Child == 0)
    {
        setupScript (cmd, fP);

        const char *Args[4];
        Args[0] = "/bin/sh";
        Args[1] = "-c";
        Args[2] = cmd;
        Args[3] = 0;
        execv(Args[0],(char **)Args);
        _exit(100);
    }

    if (fP) {
        if (! feedPackages ())
            throw AptException( "Failed feeding packages to script" );
    }

    int Status = 0;
    int ret;
    while ((ret = waitpid (Child, &Status, WNOHANG)) != Child) {
        if (errno == EINTR || ret == 0) {
            dpkgMonitor();
            usleep (50000); // 0.05 second hang
            continue;
        }
    }

    if (WIFEXITED(Status) == 0 || WEXITSTATUS(Status) != 0)
        throw AptException( "Script died unexpectedly." );
}

void DpkgHarness::setupScript( const char * /*cmd*/, bool fP )
{
    cerr << "DpkgHarness::setupScript()" << endl;
    if (fP) {
        dup2 (m_scriptPipe [0], STDIN_FILENO);
        SetCloseExec(STDOUT_FILENO, false);
        SetCloseExec(STDIN_FILENO, false);
        SetCloseExec(STDERR_FILENO, false);
    }
}

bool DpkgHarness::feedPackages()
{
    close(m_scriptPipe[ 0 ]);

    FILE *F = fdopen(m_scriptPipe[ 1 ], "w");
    if (F == 0)
        throw wexcept::System( "Failed to open new FD" );

    // Feed it the filenames.
    bool Die = false;
    if (m_version <= 1)
        Die = !sendV1Pkgs(F);
    else
        Die = !sendV2Pkgs(F);

    fclose(F);
    return ! Die;
}

void DpkgHarness::tellDpkg( std::string s )
{
    ::write( m_dpkgStdin[ 0 ], s.c_str(), s.size() );
}

#endif
