/*
 * Comments in this program are written in Japanese,
 * because this program is a Japanese input method.
 * (many Japanese gramatical terms will appear.)
 *
 * Kana-Kanji conversion engine Anthy.
 * ̾Ѵ󥸥Anthy(󥷡)
 *
 * Funded by IPA̤Ƨեȥ¤ 2001 9/22
 * Funded by IPA̤Ƨեȥ¤ 2005
 * Copyright (C) 2000-2006 TABATA Yusuke, UGAWA Tomoharu
 * Copyright (C) 2004-2006 YOSHIDA Yuichi
 * Copyright (C) 2000-2006 KMC(Kyoto University Micro Computer Club)
 * Copyright (C) 2001-2002 TAKAI Kosuke, Nobuoka Takahiro
 *
 */
/*
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */
/*
 * AnthyѴǽϥ饤֥ȤƹƤꡢ
 * եˤϥ饤֥󶡤ؿ(API)ҤƤޤ
 *
 * 饤֥󶡤ؿϲΤ褦ʤΤޤ
 * (1)饤֥Τνλ
 * (2)ѴƥȤκ
 * (3)ѴƥȤФʸꡢʸĹѹμ
 *
 * 󥿡ե˴ؤƤ doc/LIB򻲾ȤƤ
 * AnthyΥɤ򤷤褦Ȥ
 * doc/GLOSSARY Ѹİ뤳Ȥ򴫤ޤ
 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <dic.h>
#include <splitter.h>
#include <conf.h>
#include <ordering.h>
#include <logger.h>
#include <record.h>
#include <anthy.h>
#include <record.h>
#include <xchar.h> /* for KK_VU */
#include "main.h"
#include "config.h"


/** AnthyνλɤΥե饰 */
static int is_init_ok;

static int default_encoding;

/** (API) Τν */
int
anthy_init(void)
{
  if (is_init_ok) {
    /* 2ٽʤ褦 */
    return 0;
  }

  /* ƥ֥ƥ˽ */
  if (anthy_init_dic()) {
    anthy_log(0, "Failed to initialize dictionary.\n");
    return -1;
  }

  if (anthy_init_splitter()) {
    anthy_log(0, "Failed to init splitter.\n");
    return -1;
  }
  anthy_init_contexts();
  anthy_init_personality();

  default_encoding = ANTHY_EUC_JP_ENCODING;
  is_init_ok = 1;
  return 0;
}

/** (API) ǡβ */
void
anthy_quit(void)
{
  if (!is_init_ok) {
    return ;
  }
  anthy_quit_contexts();
  anthy_quit_personality();
  anthy_quit_splitter();
  /* ¿Υǡ¤ϤallocatorˤäƲ */
  anthy_quit_dic();

  is_init_ok = 0;
}

/** (API) ܤξ */
void
anthy_conf_override(const char *var, const char *val)
{
  anthy_do_conf_override(var, val);
}

/** (API) personality */
int
anthy_set_personality(const char *id)
{
  return anthy_do_set_personality(id);
}

/** (API) Ѵcontextκ */
struct anthy_context *
anthy_create_context(void)
{
  if (!is_init_ok) {
    return 0;
  }
  return anthy_do_create_context(default_encoding);
}

/** (API) ѴcontextΥꥻå */
void
anthy_reset_context(struct anthy_context *ac)
{
  anthy_do_reset_context(ac);
}

/** (API) Ѵcontextβ */
void
anthy_release_context(struct anthy_context *ac)
{
  anthy_do_release_context(ac);
}

/** 
 * ѴɬפɤȽ
 */
static int
need_reconvert(xstr *xs)
{
  int i;
  
  for (i = 0; i < xs->len; ++i) {
    xchar xc = xs->str[i];
    int type = anthy_get_xchar_type(xc);

    /* ʸξϵѴ
     * ֥פϥեȥɤʿ̾⡼ɤʸȤäƤΤǡ
     * ѴоݤȤϤʤ
     */
    if (!(type & (XCT_HIRA | XCT_SYMBOL | XCT_NUM |
		  XCT_WIDENUM | XCT_OPEN | XCT_CLOSE)) &&
	xc != KK_VU) {
      return 1;
    }
  }
  return 0;
}


/** (API) Ѵʸ */
int
anthy_set_string(struct anthy_context *ac, const char *s)
{
  xstr *xs;
  int retval;

  anthy_dic_activate_session(ac->dic_session);
  /* Ѵ򳫻Ϥ˸Ŀͼreload */
  anthy_reload_record();

  xs = anthy_cstr_to_xstr(s, ac->encoding);
  /**/
  if (!need_reconvert(xs)) {
    /* ̤Ѵ */
    retval = anthy_do_context_set_str(ac, xs, 0);
  } else {
    /* 䥫ʤäƤѴƤߤ */
    struct anthy_conv_stat stat;
    struct seg_ent *seg;
    int i;
    xstr* hira_xs;
    /* 줿ʸѴ򤫤 */
    retval = anthy_do_context_set_str(ac, xs, 1);

    /* ʸʿ̾ */
    anthy_get_stat(ac, &stat);
    hira_xs = NULL;
    for (i = 0; i < stat.nr_segment; ++i) {
      seg = anthy_get_nth_segment(&ac->seg_list, i);
      hira_xs = anthy_xstrcat(hira_xs, &seg->cands[0]->str);
    }
    /* ѴԤʤ */
    retval = anthy_do_context_set_str(ac, hira_xs, 0);
    anthy_free_xstr(hira_xs);
  }

  anthy_free_xstr(xs);
  return retval;
}

/** (API) ʸĹѹ */
void
anthy_resize_segment(struct anthy_context *ac, int nth, int resize)
{
  anthy_dic_activate_session(ac->dic_session);
  anthy_do_resize_segment(ac, nth, resize);
}

/** (API) Ѵξ֤μ */
int
anthy_get_stat(struct anthy_context *ac, struct anthy_conv_stat *s)
{
  s->nr_segment = ac->seg_list.nr_segments;
  return 0;
}

/** (API) ʸξ֤μ */
int
anthy_get_segment_stat(struct anthy_context *ac, int n,
		       struct anthy_segment_stat *s)
{
  struct seg_ent *seg;
  seg = anthy_get_nth_segment(&ac->seg_list, n);
  if (seg) {
    s->nr_candidate = seg->nr_cands;
    s->seg_len = seg->str.len;
    return 0;
  }
  return -1;
}

static int
get_special_candidate_index(int nth, struct seg_ent *seg)
{
  int i;
  int mask = XCT_NONE;
  if (nth >= 0) {
    return nth;
  }
  if (nth == NTH_UNCONVERTED_CANDIDATE ||
      nth == NTH_HALFKANA_CANDIDATE) {
    return nth;
  }
  if (nth == NTH_KATAKANA_CANDIDATE) {
    mask = XCT_KATA;
  } else if (nth == NTH_HIRAGANA_CANDIDATE) {
    mask = XCT_HIRA;
  }
  for (i = 0; i < seg->nr_cands; i++) {
    if (anthy_get_xstr_type(&seg->cands[i]->str) & mask) {
      return i;
    }
  }
  return NTH_UNCONVERTED_CANDIDATE;
}

/** (API) ʸμ */
int
anthy_get_segment(struct anthy_context *ac, int nth_seg,
		  int nth_cand, char *buf, int buflen)
{
  struct seg_ent *seg;
  char *p;
  int len;

  /* ʸФ */
  if (nth_seg < 0 || nth_seg >= ac->seg_list.nr_segments) {
    return -1;
  }
  seg = anthy_get_nth_segment(&ac->seg_list, nth_seg);

  /* ʸᤫФ */
  p = NULL;
  if (nth_cand < 0) {
    nth_cand = get_special_candidate_index(nth_cand, seg);
  }
  if (nth_cand == NTH_HALFKANA_CANDIDATE) {
    xstr *xs = anthy_xstr_hira_to_half_kata(&seg->str);
    p = anthy_xstr_to_cstr(xs, ac->encoding);
    anthy_free_xstr(xs);
  } else if (nth_cand == NTH_UNCONVERTED_CANDIDATE) {
    /* Ѵʸ */
    p = anthy_xstr_to_cstr(&seg->str, ac->encoding);
  } else if (nth_cand >= 0 && nth_cand < seg->nr_cands) {
    p = anthy_xstr_to_cstr(&seg->cands[nth_cand]->str, ac->encoding);
  }
  if (!p) {
    return -1;
  }

  /* Хåե˽񤭹 */
  len = strlen(p);
  if (!buf) {
    free(p);
    return len;
  }
  if (len + 1 > buflen) {
    /* Хåե­ޤ */
    free(p);
    return -1;
  }
  strcpy(buf, p);
  free(p);
  return len;
}

/* ٤Ƥʸ᤬ߥåȤ줿check */
static int
commit_all_segment_p(struct anthy_context *ac)
{
  int i;
  struct seg_ent *se;
  for (i = 0; i < ac->seg_list.nr_segments; i++) {
    se = anthy_get_nth_segment(&ac->seg_list, i);
    if (se->committed < 0) {
      return 0;
    }
  }
  return 1;
}

/** (API) ʸγ */
int
anthy_commit_segment(struct anthy_context *ac, int s, int c)
{
  struct seg_ent *seg;
  if (!ac->str.str) {
    return -1;
  }
  if (s < 0 || s >= ac->seg_list.nr_segments) {
    return -1;
  }
  if (commit_all_segment_p(ac)) {
    /* ǤƤΥȤߥåȤƤ */
    return -1;
  }

  anthy_dic_activate_session(ac->dic_session);
  seg = anthy_get_nth_segment(&ac->seg_list, s);
  if (c < 0) {
    c = get_special_candidate_index(c, seg);
  }
  if (c == NTH_UNCONVERTED_CANDIDATE) {
    /*
     * Ѵʸ󤬥ߥåȤ줿Τǡбֹõ
     */
    int i;
    for (i = 0; i < seg->nr_cands; i++) {
      if (!anthy_xstrcmp(&seg->str, &seg->cands[i]->str)) {
	c = i;
      }
    }
  }
  if (c < 0 || c >= seg->nr_cands) {
    return -1;
  }
  seg->committed = c;

  if (commit_all_segment_p(ac)) {
    /* ٤ƤΥȤߥåȤ줿 */
    anthy_proc_commit(&ac->seg_list, &ac->split_info);
  }
  return 0;
}

/** (API) ͽ¬Ƥۤʸ */
int
anthy_set_prediction_string(struct anthy_context *ac, const char* s)
{
  int retval;
  xstr *xs;

  anthy_dic_activate_session(ac->dic_session);
  /* ͽ¬򳫻Ϥ˸Ŀͼreload */
  anthy_reload_record();


  xs = anthy_cstr_to_xstr(s, ac->encoding);

  retval = anthy_do_set_prediction_str(ac, xs);

  anthy_free_xstr(xs);

  return retval;
}

/** (API) ͽ¬Ѵξ֤μ */
int 
anthy_get_prediction_stat(struct anthy_context *ac, struct anthy_prediction_stat * ps)
{
  ps->nr_prediction = ac->prediction.nr_prediction;
  return 0;
}

/** (API) ͽ¬Ѵθμ */
int
anthy_get_prediction(struct anthy_context *ac, int nth, char* buf, int buflen)
{
  struct prediction_cache* prediction = &ac->prediction;
  int nr_prediction = prediction->nr_prediction;
  char* p;
  int len;

  if (nth < 0 || nr_prediction <= nth) {
    return -1;
  }

  p = anthy_xstr_to_cstr(prediction->predictions[nth].str, ac->encoding);

  /* Хåե˽񤭹 */
  len = strlen(p);
  if (!buf) {
    free(p);
    return len;
  }
  if (len + 1 > buflen) {
    free(p);
    return -1;
  } else {
    strcpy(buf, p);
    free(p);
    return len;
  }
}

/** (API) ȯ */
void
anthy_print_context(struct anthy_context *ac)
{
  anthy_do_print_context(ac, default_encoding);
}

/** (API) Anthy 饤֥ΥСɽʸ֤
 * ͭ饤֥ǤϳѿΥݡȤϹޤʤΤǴؿˤƤ
 */
const char *
anthy_get_version_string (void)
{
#ifdef VERSION
  return VERSION;
#else  /* just in case */
  return "(unknown)";
#endif
}

/** (API) */
int
anthy_context_set_encoding(struct anthy_context *ac, int encoding)
{
#ifdef USE_UCS4
  if (!ac) {
    default_encoding = encoding;
  } else {
    ac->encoding = encoding;
  }
  return encoding;
#else
  (void)ac;
  (void)encoding;
  return ANTHY_EUC_JP_ENCODING;
#endif
}
