#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Provides AptWorker which processes transactions."""
# Copyright (C) 2008-2009 Sebastian Heinlein <devel@glatzor.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

__author__  = "Sebastian Heinlein <devel@glatzor.de>"

import logging
import os
import sys
import time
import traceback

import apt
import apt.cache
import apt.debfile
import apt_pkg
import gobject
from softwareproperties.AptAuth import AptAuth

from enums import *
from errors import *
from progress import DaemonOpenProgress, \
                     DaemonInstallProgress, \
                     DaemonFetchProgress, \
                     DaemonDpkgInstallProgress, \
                     DaemonDpkgRecoverProgress

log = logging.getLogger("AptDamon.Worker")

class AptWorker(gobject.GObject):

    """Worker which processes transactions from the queue."""

    __gsignals__ = {"transaction-done":(gobject.SIGNAL_RUN_FIRST,
                                        gobject.TYPE_NONE,
                                        (gobject.TYPE_STRING,))}

    def __init__(self):
        """Initialize a new AptWorker instance."""
        gobject.GObject.__init__(self)
        self.trans = None
        self.last_action_timestamp = time.time()
        self._cache = None

    def run(self, transaction):
        """Process the given transaction in the background.

        Keyword argument:
        transaction -- core.Transcation instance to run
        """
        if self.trans:
            raise Exception("There is already a running transacion")
        self.trans = transaction
        gobject.idle_add(self._process_transaction)

    def _emit_transaction_done(self, tid):
        """Emit the transaction-done signal.

        Keyword argument:
        tid -- the id of the finished transaction
        """
        log.debug("Emitting transaction-done: %s", tid)
        self.emit("transaction-done", tid)

    def _process_transaction(self):
        """Run the worker"""
        self.last_action_timestamp = time.time()
        self.trans.status = STATUS_RUNNING
        self.trans.progress = 0
        try:
            # Prepare the package cache
            self._lock_cache()
            self._open_cache()
            # Process the transaction method
            if self.trans.role == ROLE_INSTALL_PACKAGES:
                self.install_packages(**self.trans.kwargs)
            elif self.trans.role == ROLE_INSTALL_FILE:
                self.install_file(**self.trans.kwargs)
            elif self.trans.role == ROLE_REMOVE_PACKAGES:
                self.remove_packages(**self.trans.kwargs)
            elif self.trans.role == ROLE_UPGRADE_SYSTEM:
                self.upgrade_system(**self.trans.kwargs)
            elif self.trans.role == ROLE_UPDATE_CACHE:
                self.update_cache()
            elif self.trans.role == ROLE_UPGRADE_PACKAGES:
                self.upgrade_packages(**self.trans.kwargs)
            elif self.trans.role == ROLE_COMMIT_PACKAGES:
                self.commit_packages(**self.trans.kwargs)
            elif self.trans.role == ROLE_ADD_VENDOR_KEY_FILE:
                self.add_vendor_key_from_file(**self.trans.kwargs)
            elif self.trans.role == ROLE_REMOVE_VENDOR_KEY:
                self.remove_vendor_key(**self.trans.kwargs)
        except TransactionCancelled:
            self.trans.exit = EXIT_CANCELLED
        except TransactionFailed, excep:
            self.trans.error = excep
            self.trans.exit = EXIT_FAILED
        except (KeyboardInterrupt, SystemExit):
            self.trans.exit = EXIT_CANCELLED
        except Exception, excep:
            self.trans.error = TransactionFailed(ERROR_UNKNOWN,
                                                 traceback.format_exc())
            self.trans.exit = EXIT_FAILED
        else:
            self.trans.exit = EXIT_SUCCESS
        finally:
            self.trans.progress = 100
            self.last_action_timestamp = time.time()
            tid = self.trans.tid[:]
            self.trans = None
            self._emit_transaction_done(tid)
            self._unlock_cache()
        return False

    def commit_packages(self, install, reinstall, remove, purge, upgrade):
        """Perform a complex package operation.

        Keyword arguments:
        install - list of package names to install
        reinstall - list of package names to reinstall
        remove - list of package names to remove
        purge - list of package names to purge including configuration files
        upgrade - list of package names to upgrade
        """
        #FIXME python-apt 0.8 introduced a with statement
        ac = self._cache.actiongroup()
        resolver = apt.cache.ProblemResolver(self._cache)
        self._mark_packages_for_installation(install, resolver)
        self._mark_packages_for_installation(reinstall, resolver,
                                             reinstall=True)
        self._mark_packages_for_removal(remove, resolver)
        self._mark_packages_for_removal(purge, resolver, purge=True)
        self._mark_packages_for_upgrade(upgrade, resolver)
        self._resolve_depends(resolver)
        ac.release()
        self._commit_changes()

    def _resolve_depends(self, resolver):
        """Resolve the dependencies using the given ProblemResolver."""
        resolver.install_protect()
        try:
            resolver.resolve()
        except SystemError:
            broken = [pkg.name for pkg in self._cache if pkg.is_inst_broken]
            raise TransactionFailed(ERROR_DEP_RESOLUTION_FAILED,
                                    " ".join(broken))

    def install_packages(self, package_names):
        """Install packages.

        Keyword argument:
        package_names -- list of package name which should be installed
        """
        ac = self._cache.actiongroup()
        resolver = apt.cache.ProblemResolver(self._cache)
        self._mark_packages_for_installation(package_names, resolver)
        self._resolve_depends(resolver)
        ac.release()
        self._commit_changes()

    def _check_unauthenticated(self):
        """Check if any of the cache changes get installed from an
        unauthenticated repository"""
        if self.trans.allow_unauthenticated:
            return
        unauthenticated = []
        for pkg in self._cache:
            if (pkg.markedInstall or
                pkg.markedDowngrade or
                pkg.markedUpgrade or
                pkg.markedReinstall):
                trusted = False
                for origin in pkg.candidate.origins:
                    trusted |= origin.trusted
                if not trusted:
                    unauthenticated.append(pkg.name)
        if unauthenticated:
            raise TransactionFailed(ERROR_PACKAGE_UNAUTHENTICATED,
                                    " ".join(sorted(unauthenticated)))

    def _mark_packages_for_installation(self, package_names, resolver,
                                        reinstall=False):
        """Mark packages for installation."""
        for pkg_name in package_names:
            if self._cache.has_key(pkg_name):
                pkg = self._cache[pkg_name]
            else:
                raise TransactionFailed(ERROR_NO_PACKAGE,
                                        "Package %s isn't available" % pkg_name)
            if reinstall:
                if not pkg.isInstalled:
                    raise TransactionFailed(ERROR_PACKAGE_NOT_INSTALLED,
                                            "Package %s isn't installed" % \
                                            pkg.name)
            else:
                #FIXME: Turn this into a non-critical message
                if pkg.isInstalled:
                    raise TransactionFailed(ERROR_PACKAGE_ALREADY_INSTALLED,
                                            "Package %s is already installed" %\
                                            pkg_name)
            pkg.markInstall(False, True, True)
            resolver.clear(pkg)
            resolver.protect(pkg)
 
    def add_vendor_key_from_file(self, path):
        """Add the signing key from the given file to the trusted vendors.

        Keyword argument:
        path -- absolute path to the key file
        """
        try:
            #FIXME: use gobject.spawn_async or reactor.spawn
            #FIXME: use --dry-run before?
            auth = AptAuth()
            auth.add(os.path.expanduser(path))
        except Exception, error:
            raise TransactionFailed(ERROR_KEY_NOT_INSTALLED,
                                    "Key file %s couldn't be installed: %s" % \
                                    (path, error))

    def remove_vendor_key(self, fingerprint):
        """Remove repository key.

        Keyword argument:
        fingerprint -- fingerprint of the key to remove
        """
        try:
            #FIXME: use gobject.spawn_async or reactor.spawn
            #FIXME: use --dry-run before?
            auth = AptAuth()
            auth.rm(fingerprint)
        except Exception, error:
            raise TransactionFailed(ERROR_KEY_NOT_REMOVED,
                                    "Key with fingerprint %s couldn't be "
                                    "removed: %s" % (fingerprint, error))

    def install_file(self, path):
        """Install local package file.

        Keyword argument:
        path -- absolute path to the package file
        """
        # Check if the dpkg can be installed at all
        deb = apt.debfile.DebPackage(path, self._cache)
        if not deb.check():
            raise TransactionFailed(ERROR_DEP_RESOLUTION_FAILED,
                                    deb._failure_string)
        # Check for required changes and apply them before
        (install, remove, unauth) = deb.required_changes
        if len(install) > 0 or len(remove) > 0:
            dpkg_range = (64, 99)
            self._commit_changes(fetch_range=(5, 33),
                                 install_range=(34, 63))
            self._lock_cache()
        # Install the dpkg file
        if deb.install(DaemonDpkgInstallProgress(self.trans,
                                                 begin=64, end=95)):
            raise TransactionFailed(ERROR_UNKNOWN, deb._failure_string)

    def remove_packages(self, package_names):
        """Remove packages.

        Keyword argument:
        package_names -- list of package name which should be installed
        """
        ac = self._cache.actiongroup()
        resolver = apt.cache.ProblemResolver(self._cache)
        self._mark_packages_for_removal(package_names, resolver)
        self._resolve_depends(resolver)
        ac.release()
        self._commit_changes(fetch_range=(10, 10),
                             install_range=(10, 90))
        #FIXME: should we use a persistant cache? make a check?
        #self._open_cache(prange=(90,99))
        #for p in pkgs:
        #    if self._cache.has_key(p) and self._cache[p].isInstalled:
        #        self.ErrorCode(ERROR_UNKNOWN, "%s is still installed" % p)
        #        self.Finished(EXIT_FAILED)
        #        return

    def _mark_packages_for_removal(self, package_names, resolver, purge=False):
        """Mark packages for installation."""
        for pkg_name in package_names:
            if not self._cache.has_key(pkg_name):
                raise TransactionFailed(ERROR_NO_PACKAGE,
                                        "Package %s isn't available" % pkg_name)
            pkg = self._cache[pkg_name]
            if not pkg.isInstalled:
                raise TransactionFailed(ERROR_PACKAGE_NOT_INSTALLED,
                                        "Package %s isn't installed" % pkg_name)
            if pkg._pkg.Essential == True:
                raise TransactionFailed(ERROR_NOT_REMOVE_ESSENTIAL_PACKAGE,
                                        "Package %s cannot be removed." % \
                                        pkg_name)
            pkg.markDelete(False, purge=purge)
            resolver.clear(pkg)
            resolver.protect(pkg)
            resolver.remove(pkg)

    def upgrade_packages(self, package_names):
        """Upgrade packages.

        Keyword argument:
        package_names -- list of package name which should be upgraded
        """
        ac = self._cache.actiongroup()
        resolver = apt.cache.ProblemResolver(self._cache)
        self._mark_packages_for_upgrade(package_names, resolver)
        self._resolve_depends(resolver)
        ac.release()
        self._commit_changes()

    def _mark_packages_for_upgrade(self, package_names, resolver):
        """Mark packages for upgrade."""
        for pkg_name in package_names:
            if not self._cache.has_key(pkg_name):
                raise TransactionFailed(ERROR_NO_PACKAGE,
                                        "Package %s isn't available" % pkg_name)
            pkg = self._cache[pkg_name]
            if not pkg.isInstalled:
                raise TransactionFailed(ERROR_PACKAGE_NOT_INSTALLED,
                                        "Package %s isn't installed" % pkg_name)
            auto = not self._cache._depcache.IsAutoInstalled(pkg._pkg)
            pkg.markInstall(False, True, auto)
            resolver.clear(pkg)
            resolver.protect(pkg)

    def update_cache(self):
        """Update the cache."""
        log.info("Updating cache")
        progress = DaemonFetchProgress(self.trans, begin=10, end=95)
        try:
            self._cache.update(progress)
        except SystemError, excep:
            if self.trans.cancelled:
                raise TransactionCancelled()
            else:
                raise TransactionFailed(ERROR_REPO_DOWNLOAD_FAILED,
                                        excep.message)

    def upgrade_system(self, safe_mode=True):
        """Upgrade the system.

        Keyword argument:
        safe_mode -- if additional software should be installed or removed to
                     satisfy the dependencies the an updates
        """
        log.info("Upgrade system with safe mode: %s" % safe_mode)
        # Check for available updates
        updates = filter(lambda p: p.isUpgradable,
                         self._cache)
        if len(updates) == 0:
            self.trans.send_message(MSG_SYSTEM_ALREADY_UPTODATE, "")
            return
        self._cache.upgrade(distUpgrade=not safe_mode)
        # Check for blocked updates
        outstanding = []
        changes = self._cache.getChanges()
        for pkg in updates:
            if not pkg in changes or not pkg.markedUpgrade:
                outstanding.append(pkg)
        if len(outstanding) > 0:
            self.trans.send_message(MSG_SYSTEM_NOT_UPTODATE,
                                    "".join(map(lambda p: "%s " % p.name,
                                                outstanding)))
        self._commit_changes()

    def _open_cache(self, begin=0, end=5, quiet=False):
        """Open the APT cache.

        Keyword arguments:
        start -- the begin of the progress range
        end -- the end of the the progress range
        quiet -- if True do no report any progress
        """
        self.trans.status = STATUS_LOADING_CACHE
        try:
            progress = DaemonOpenProgress(self.trans, begin=begin, end=end,
                                          quiet=quiet)
            if not isinstance(self._cache, apt.cache.Cache):
                self._cache = apt.cache.Cache(progress)
            else:
                self._cache.open(progress)
        except Exception, excep:
            raise TransactionFailed(ERROR_NO_CACHE, excep.message)
        if self._cache.broken_count:
            broken = [pkg.name for pkg in self._cache if pkg.is_now_broken]
            raise TransactionFailed(ERROR_CACHE_BROKEN, " ".join(broken))

    def _lock_cache(self):
        """Lock the APT cache."""
        try:
            # see if the lock for the download dir can be acquired
            # (work around bug in python-apt/apps that call _fetchArchives)
            lockfile = apt_pkg.Config.FindDir("Dir::Cache::Archives") + "lock"
            lock = apt_pkg.GetLock(lockfile)
            if lock < 0:
                raise SystemError("failed to lock '%s'" % lockfile)
            else:
                os.close(lock)
            # then lock the main package system
            apt_pkg.PkgSystemLock()
        except SystemError, e:
            logging.exception("_lock_cache failed, waiting")
            self.trans.paused = True
            self.trans.status = STATUS_WAITING_LOCK
            lock_watch = gobject.timeout_add_seconds(3, self._watch_lock)
            while self.trans.paused and not self.trans.cancelled:
                gobject.main_context_default().iteration()
            gobject.source_remove(lock_watch)
            if self.trans.cancelled:
                raise TransactionCancelled()

    def _watch_lock(self):
        """Unpause the transaction if the lock can be obtained."""
        # try the lock in /var/cache/apt/archive/lock first 
        # this is because apt-get install will hold it all the time
        # while the dpkg lock is briefly given up before dpkg is
        # forked off. this can cause a race (LP: #437709)
        lockfile = apt_pkg.Config.FindDir("Dir::Cache::Archives") + "lock"
        lock = apt_pkg.GetLock(lockfile)
        if lock < 0:
            return True
        os.close(lock)
        # then the dpkg lock
        try:
            apt_pkg.PkgSystemLock()
        except SystemError:
            return True
        self.trans.paused = False
        return False

    def _unlock_cache(self):
        """Unlock the APT cache."""
        try:
            apt_pkg.PkgSystemUnLock()
        except SystemError:
            pass

    def _commit_changes(self, fetch_range=(5, 50), install_range=(50, 90)):
        """Commit previously marked changes to the cache.

        Keyword arguments:
        fetch_range -- tuple containing the start and end point of the
                       download progress
        install_range -- tuple containing the start and end point of the
                         install progress
        """
        self._check_unauthenticated()
        if self.trans.cancelled:
            raise TransactionCancelled()
        self.trans.allow_cancel = False
        fetch_progress = DaemonFetchProgress(self.trans,
                                             begin=fetch_range[0],
                                             end=fetch_range[1])
        inst_progress = DaemonInstallProgress(self.trans,
                                              begin=install_range[0],
                                              end=install_range[1])
        try:
            self._cache.commit(fetch_progress, inst_progress)
        except apt.cache.FetchFailedException, excep:
            raise TransactionFailed(ERROR_PACKAGE_DOWNLOAD_FAILED, 
                                    str(excep))
        except apt.cache.FetchCancelledException:
            raise TransactionCancelled()
        except SystemError, excep:
            self._recover()
            raise TransactionFailed(ERROR_PACKAGE_MANAGER_FAILED,
                                    "%s: %s" % (excep, inst_progress.output))

    def _recover(self):
        """Run dpkg --configure -a to recover from a failed transaciont."""
        self.trans.status = STATUS_CLEANING_UP
        progress = DaemonDpkgRecoverProgress(self.trans)
        progress.startUpdate()
        progress.run()
        progress.finishUpdate()

# vim:ts=4:sw=4:et
