#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Handles the apt system lock"""
# Copyright (C) 2010 Sebastian Heinlein <devel@glatzor.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

__author__  = "Sebastian Heinlein <devel@glatzor.de>"

__all__ = ("LockFailedError", "system")

import fcntl
import os
import struct

import apt_pkg


class LockFailedError(Exception):
    """The locking of file failed."""
    def __init__(self, flock, process=None):
        """Return a new LockFailedError instance.

        Keyword arguments:
        flock -- the path of the file lock
        process -- the process which holds the lock or None
        """
        Exception.__init__(self)
        self.flock = flock
        self.process = process


class SystemLock(object):

    """Provides an interface to lock und unlock the apt pkg system."""

    def __init__(self):
        status_path = apt_pkg.config.find_file("Dir::State::status")
        self.path = os.path.join(os.path.dirname(status_path), "lock")
        self.fd = None

    def acquire(self):
        """Lock the package system and provide information if this cannot be
        done.

        This is a reemplemenataion of apt_pkg.PkgSystemLock(), since we want to
        handle an incomplete dpkg run separately.
        """
        def get_lock_fd(lock_path):
            """Return the file descriptor of the lock file or raise
            LockFailedError if the lock cannot be obtained.
            """
            fd_lock = apt_pkg.get_lock(lock_path)
            if fd_lock < 0:
                process = None
                try:
                    # Get the pid of the locking application
                    fd_lock_read = open(lock_path, "r")
                    flk = struct.pack('hhQQi', fcntl.F_WRLCK, os.SEEK_SET, 0,
                                      0, 0)
                    flk_ret = fcntl.fcntl(fd_lock_read, fcntl.F_GETLK, flk)
                    pid = struct.unpack("hhQQi", flk_ret)[4]
                    # Get the command of the pid
                    fd_status = open("/proc/%s/status" % pid, "r")
                    try:
                        for key, value in (line.split(":") for line in \
                                           fd_status.readlines()):
                            if key == "Name":
                                process = value.strip()
                                break
                    finally:
                        fd_status.close()
                except:
                    pass
                finally:
                    fd_lock_read.close()
                raise LockFailedError(lock_path, process)
            else:
                return fd_lock

        # Try the lock in /var/cache/apt/archive/lock first
        # this is because apt-get install will hold it all the time
        # while the dpkg lock is briefly given up before dpkg is
        # forked off. this can cause a race (LP: #437709)
        archive_dir = apt_pkg.config.find_dir("Dir::Cache::Archives")
        lock_archive = os.path.join(archive_dir, "lock")
        lock_fd_archive = get_lock_fd(lock_archive)
        try:
            # Then the status lock
            self.fd = get_lock_fd(self.path)
        finally:
            os.close(lock_fd_archive)

    def release(self):
        """Unlock the apt package system."""
        if self.fd is not None:
            os.close(self.fd)
            self.fd = None

system = SystemLock()

# vim:ts=4:sw=4:et
