// desc_parse.cc
//
//  Copyright 2004 Daniel Burrows
//
//  Parses a description into a fragment.

#include "desc_parse.h"
#include "ui.h"

#include <vscreen/fragment.h>

using namespace std;

/** Expands the given description string to a fragment representing the
 *  long description.
 *
 *  The basic form of the new fragment is a sequence of boxes.
 *
 *  \param desc a Description tag
 *  \return a fragment representing that description
 */
fragment *make_desc_fragment(string desc)
{
  string::size_type loc=0;
  vector<fragment*> fragments;

  // Skip the short description
  while(loc<desc.size() && desc[loc]!='\n')
    ++loc;

  if(loc<desc.size()) // Skip the '\n'
    ++loc;

  while(loc<desc.size())
    {
      // We're at the beginning of a line here.
      //
      // Find out what type of line it is first.  (be liberal -- lines
      // not starting with ' ' are formatted as part of the current
      // paragraph)
      if(desc[loc]==' ')
	++loc;

      // Now check whether it's part of a paragraph, a blank line,
      // or a preformatted line.
      switch(desc[loc])
	{
	case ' ':
	  // preformatted:
	  {
	    ++loc;

	    int amt=0;
	    while(loc+amt<desc.size() && desc[loc+amt]!='\n')
	      ++amt;

	    // For now, clip it -- although I think that there may be
	    // some merit in wrapping overlong preformatted lines.
	    fragments.push_back(clipbox(text_fragment(string(desc, loc, amt))));

	    loc+=amt;
	    if(loc<desc.size())
	      ++loc;
	    break;
	  }
	case '.':
	  // blank.
	  {
	    fragments.push_back(newline_fragment());

	    while(loc<desc.size() && desc[loc]!='\n')
	      ++loc;

	    if(loc<desc.size())
	      ++loc;
	    break;
	  }
	default:
	  // It's a paragraph.
	  {
	    bool cont=true;
	    string::size_type amt=0;
	    string par="";

	    do {
	      amt=0;
	      while(loc+amt<desc.size() && desc[loc+amt]!='\n')
		++amt;

	      par=par+string(desc, loc, amt);

	      loc+=amt;

	      // If we hit a newline and didn't just output a whitespace
	      // character, insert one.
	      if(loc<desc.size() && par.size()>0 && par[par.size()-1]!=' ')
		par+=" ";

	      // Skip the newline
	      if(loc<desc.size())
		++loc;

	      // Use 'loc+1' here because we have to preserve the
	      // leading ' ' if we're exiting the loop.  This means
	      // that packages whose descriptions end with '\nX'
	      // where X!=' ' will be displayed oddly, but such
	      // descriptions are noncompliant anyway.
	      if(!(loc+1<desc.size() && desc[loc+1]!=' ' && desc[loc+1]!='.'))
		cont=false;
	      else
		++loc;
	    } while(cont && loc+1<desc.size());

	    fragments.push_back(wrapbox(text_fragment(par)));
	  }
	}
    }

  return sequence_fragment(fragments);
}
