// vs_tree.cc
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Implementation of stuff in vs_tree.h

#include "vs_tree.h"
#include "vs_editline.h"
#include "config/keybindings.h"
#include "config/colors.h"

using namespace std;

keybindings *vs_tree::bindings=NULL;

bool vs_tree_search_string::operator()(const vs_treeitem &item)
{
  return item.matches(s);
}

vs_tree::vs_tree()
  :vscreen_widget(),
   root(NULL),
   begin(new vs_tree_root_iterator(NULL)),
   end(begin),
   top(begin),
   selected(top),
   hierarchical(true),
   prev_level(NULL)
{
  focussed.connect(SigC::slot(vscreen_update));
  unfocussed.connect(SigC::slot(vscreen_update));
}

vs_tree::vs_tree(vs_treeitem *_root, bool showroot)
  :vscreen_widget(),
   root(NULL),
   begin(new vs_tree_root_iterator(NULL)),
   end(begin),
   top(begin),
   selected(top),
   hierarchical(true),
   prev_level(NULL)
{
  set_root(_root, showroot);

  focussed.connect(SigC::slot(vscreen_update));
  unfocussed.connect(SigC::slot(vscreen_update));
}

vs_tree::~vs_tree()
{
   while(prev_level)
    {
      flat_frame *next=prev_level->next;
      delete prev_level;
      prev_level=next;
    }

  delete root; root=NULL;
}

void vs_tree::do_shown()
{
  if(selected!=end)
    selected->highlighted(this);
}

size vs_tree::size_request()
{
  return size(1, 1);
}

void vs_tree::set_root(vs_treeitem *_root, bool showroot)
{
  // Clear out the "history list"
  while(prev_level)
    {
      flat_frame *next=prev_level->next;
      delete prev_level;
      prev_level=next;
    }

  if(selected!=end)
    selected->unhighlighted(this);

  if(root)
    delete root;

  root=_root;

  if(root)
    {
      if(showroot)
	{
	  vs_tree_root_iterator *realbegin=new vs_tree_root_iterator(_root);
	  // NOTE: realbegin will be DELETED when it is assigned to begin,
	  // because the temporary that wraps it will be destroyed.
	  // This is Just Plain Evil (probably conversion from vs_levelrefs
	  // to vs_treeiterators shouldn't be allowed) but the workaround is
	  // here: reference its end() routine *before* we assign it.

	  end=realbegin->end();
	  begin=realbegin; // now realbegin is INVALID!!
	}
      else
	{
	  begin=_root->begin();
	  end=_root->end();
	}

      top=begin;
    }
  else
    {
      top=begin=end=new vs_tree_root_iterator(NULL);
    }

  selected=top;
  while(selected!=end && !selected->get_selectable())
    selected++;
  if(selected!=end)
    selected->highlighted(this);
}

void vs_tree::sync_bounds()
  // As I said: yuck!
{
  begin=root->begin();
  if(top==end)
    top=begin;
  if(selected==end)
    selected=begin;
  end=root->end();
}

int vs_tree::line_of(vs_treeiterator item)
  // Returns the Y coordinate of the given item.  (so we have to count
  // from 1)
{
  int j;
  vs_treeiterator i=top;
  if(item==top)
    return 1;

  j=1;
  do {
    if(hierarchical)
      ++i;
    else
      i.move_forward_level();

    ++j;

    if(i==item)
      return j;
  } while(i!=end);

  i=top;
  j=1;
  do {
    if(hierarchical)
      --i;
    else
      i.move_backward_level();

    --j;

    if(i==item)
      return j;
  } while(i!=begin);

  assert(0);
}

bool vs_tree::item_visible(vs_treeiterator pkg)
{
  int width,height;
  vs_treeiterator i=top;

  getmaxyx(height,width);

  if(!hierarchical)
    --height;

  while(height>0 && i!=pkg && i!=end)
    {
      --height;
      ++i;
    }

  return height>0 && i!=end;
}

void vs_tree::set_selection(vs_treeiterator to)
{
  if(item_visible(to))
    {
      if(selected!=end)
	selected->unhighlighted(this);
      selected=to;
      if(selected!=end)
	selected->highlighted(this);

      vscreen_update();
    }
  else
    {
      if(selected!=end)
	selected->unhighlighted(this);
      selected=top=to;
      if(selected!=end)
	selected->highlighted(this);

      vscreen_update();
    }
}

bool vs_tree::get_cursorvisible()
{
  return (selected!=end && selected->get_selectable());
}

point vs_tree::get_cursorloc()
{
  if(selected==end || !selected->get_selectable())
    return point(0,0);
  else
    return point(0, hierarchical?line_of(selected)-1:line_of(selected));
}

void vs_tree::line_down()
{
  int width,height;
  getmaxyx(height,width);

  if(!hierarchical)
    --height;

  if(selected!=end)
    {
      vs_treeiterator orig=selected,prevtop=top;
      selected->unhighlighted(this);

      int newline=line_of(selected);
      bool movedonce=false;

      while(newline<height &&
	    selected!=end &&
	    (!movedonce || !selected->get_selectable()))
	{
	  ++selected;
	  ++newline;
	  movedonce=true;
	}

      if(newline==height &&
	 selected!=end &&
	 (!movedonce || !selected->get_selectable()))
	// We ran off the screen and couldn't find anything to select,
	// or we started at the bottom of the screen.
	{
	  ++selected;

	  if(selected!=end)
	    {
	      // Shift the top forward.
	      ++top;

	      if(!selected->get_selectable())
		// Ok, well, we still can't select this.  Too bad.
		{
		  selected=orig;

		  if(selected==prevtop)
		    // We moved forward by one, so the previous top
		    // needs to be moved forward.  (saves an
		    // expensive line_of call)
		    {
		      selected=top;
		      int l=0;

		      while(selected!=end &&
			    !selected->get_selectable() &&
			    l<height)
			{
			  ++selected;
			  ++l;
			}
		    }
		}
	    }
	}

      if(selected==end)
	--selected;

      if(selected!=end)
	selected->highlighted(this);

      vscreen_update();
    }
}

void vs_tree::set_hierarchical(bool _hierarchical)
{
  if(_hierarchical!=hierarchical)
    {
      hierarchical=_hierarchical;

      if(_hierarchical)
	{
	  while(prev_level && prev_level->next)
	    {
	      flat_frame *next=prev_level->next;
	      delete prev_level;
	      prev_level=next;
	    }

	  if(prev_level)
	    {
	      top=prev_level->top;
	      begin=prev_level->begin;
	      end=prev_level->end;
	      selected=prev_level->selected;

	      delete prev_level;
	      prev_level=NULL;
	    }
	}

      vscreen_update();
    }
}

void vs_tree::line_up()
{
  int width,height;
  getmaxyx(height,width);

  if(!hierarchical)
    --height;

  vs_treeiterator orig=selected;
  if(selected!=end)
    selected->unhighlighted(this);

  if(selected!=begin)
    {
      bool movedonce=false;

      while(selected!=begin &&
	    selected!=top &&
	    (!movedonce || !selected->get_selectable()))
	{
	  if(hierarchical)
	    --selected;
	  else
	    selected.move_backward_level();
	  movedonce=true;
	}

      if(selected==top &&
	 selected!=begin &&
	 (!movedonce || !selected->get_selectable()))
	// We ran off the screen and couldn't find anything to select,
	// or we started at the top of the screen.
	{
	  if(hierarchical)
	    --selected;
	  else
	    selected.move_backward_level();

	  // The top always moves.  If it is the beginning, something
	  // is seriously messed up.
	  assert(top!=begin);
	  if(hierarchical)
	    --top;
	  else
	    top.move_backward_level();
	  if(!selected->get_selectable())
	    // Oops.  Even moving one more doesn't give us anything
	    // to select.  Restore the selection.
	    {
	      selected=orig;
	      if(line_of(selected)>height)
		// If the selection was pushed off the screen,
		// bring it back on.
		{
		  if(hierarchical)
		    --selected;
		  else
		    selected.move_backward_level();

		  while(selected!=top && !selected->get_selectable())
		    if(hierarchical)
		      --selected;
		    else
		      selected.move_backward_level();
		}
	    }
	}
    }

  if(selected!=end)
    selected->highlighted(this);

  vscreen_update();
}

void vs_tree::page_down()
{
  int width,height;
  getmaxyx(height,width);

  if(!hierarchical)
    --height;

  int count=height;
  vs_treeiterator newtop=top;
  while(count>0 && newtop!=end)
    {
      if(hierarchical)
	++newtop;
      else
	newtop.move_forward_level();
      count--;
    }

  if(count==0 && newtop!=end)
    {
      int l=0;
      (*selected).unhighlighted(this);
      selected=top=newtop;
      while(l<height && selected!=end && !selected->get_selectable())
	if(hierarchical)
	  ++selected;
	else
	  selected.move_forward_level();
      if(l==height || selected==end)
	selected=top;
      (*selected).highlighted(this);
      vscreen_update();
    }
}

void vs_tree::page_up()
{
  int width,height;
  getmaxyx(height,width);

  if(!hierarchical)
    --height;

  int count=height;
  vs_treeiterator newtop=top;
  while(count>0 && newtop!=begin)
    {
      if(hierarchical)
	--newtop;
      else
	newtop.move_backward_level();
      count--;
    }

  if(newtop!=top)
    {
      int l=0;
      if(selected!=end)
	(*selected).unhighlighted(this);
      selected=top=newtop;
      while(l<height && selected!=end && !selected->get_selectable())
	if(hierarchical)
	  ++selected;
	else
	  selected.move_forward_level();
      if(l==height || selected==end)
	selected=top;

      if(selected!=end)
	(*selected).unhighlighted(this);
      vscreen_update();
    }
}

void vs_tree::jump_to_begin()
{
  int width,height;
  getmaxyx(height,width);

  if(!hierarchical)
    --height;

  int l=0;
  vs_treeiterator prev=selected;

  if(selected!=end)
    selected->unhighlighted(this);

  selected=begin;
  while(l<height && selected!=end && !selected->get_selectable())
    if(hierarchical)
      ++selected;
    else
      selected.move_forward_level();
  if(l==height || selected==end)
    selected=begin;

  if(selected!=end)
    selected->highlighted(this);

  if(top!=begin)
    top=begin;

  vscreen_update();
}

void vs_tree::jump_to_end()
{
  int width,height;
  getmaxyx(height,width);

  if(!hierarchical)
    --height;

  int l=-1;
  vs_treeiterator last=end,newtop=end,prev=selected;
  if(hierarchical)
    --last;
  else
    last.move_backward_level();
  while(newtop!=begin && newtop!=top && height>0)
    {
      if(hierarchical)
	--newtop;
      else
	newtop.move_backward_level();
      --height;
      l++;
    }

  if(selected!=end)
    selected->unhighlighted(this);

  selected=last;
  while(l>=0 && selected!=end && !selected->get_selectable())
    {
      if(hierarchical)
	--selected;
      else
	selected.move_backward_level();
      l--;
    }
  if(selected==end && l<0)
    selected=last;

  if(selected!=end)
    selected->highlighted(this);

  if(newtop!=top)
    top=newtop;

  vscreen_update();
}

void vs_tree::level_line_up()
{
  vs_treeiterator tmp=selected;
  tmp.move_backward_level();
  if(tmp!=end)
    set_selection(tmp);
}

void vs_tree::level_line_down()
{
  vs_treeiterator tmp=selected;
  tmp.move_forward_level();
  if(tmp!=end)
    set_selection(tmp);
}

bool vs_tree::handle_char(chtype ch)
{
  // umm...
  //width++;
  //height++;

  if(selected!=vs_treeiterator(NULL))
    {
      if(hierarchical && bindings->key_matches(ch, "Parent"))
	{
	  if(!selected.is_root())
	    set_selection(selected.get_up());
	}
      else if(!hierarchical && prev_level && bindings->key_matches(ch, "Left"))
	{
	  selected->unhighlighted(this);

	  top=prev_level->top;
	  begin=prev_level->begin;
	  end=prev_level->end;
	  selected=prev_level->selected;

	  flat_frame *next=prev_level->next;
	  delete prev_level;
	  prev_level=next;

	  selected->highlighted(this);

	  vscreen_update();
	}
      else if(!hierarchical &&
	      selected!=end && selected->get_selectable() &&
	      selected->begin()!=selected->end() &&
	      (bindings->key_matches(ch, "Right") ||
	       bindings->key_matches(ch, "Confirm")))
	{
	  selected->unhighlighted(this);
	  prev_level=new flat_frame(begin, end, top, selected, prev_level);

	  begin=selected->begin();
	  end=selected->end();
	  top=begin;
	  selected=begin;

	  selected->highlighted(this);

	  vscreen_update();
	}
      else if(bindings->key_matches(ch, "Down"))
	line_down();
      else if(bindings->key_matches(ch, "Up"))
	line_up();
      else if(bindings->key_matches(ch, "NextPage"))
	page_down();
      else if(bindings->key_matches(ch, "PrevPage"))
	page_up();
      else if(bindings->key_matches(ch, "Begin"))
	jump_to_begin();
      else if(bindings->key_matches(ch, "End"))
	jump_to_end();
      else if(bindings->key_matches(ch, "LevelUp"))
	level_line_up();
      else if(bindings->key_matches(ch, "LevelDown"))
	level_line_down();
      /*else if(bindings->key_matches(ch, "Search"))
	{
	  vs_statusedit *ed=new vs_statusedit("Search for: ");
	  ed->entered.connect(slot(this, &vs_tree::search_for));
	  add_widget(ed);
	  vscreen_update();
	}
      else if(bindings->key_matches(ch, "ReSearch"))
      search_for("");*/
      else
	{
	  if(selected!=end && selected->get_selectable() && selected->dispatch_char(ch, this))
	    vscreen_update();
	  else
	    return vscreen_widget::handle_char(ch);
	}
      return true;
    }
  return false;
}

void vs_tree::search_for(vs_tree_search_func &matches)
{
  vs_treeiterator curr((selected==vs_treeiterator(NULL))?begin:selected, hierarchical),
    start(curr);
  // Make an iterator that ignores all the rules >=)

  if(curr!=end)
    {
      if(hierarchical)
	++curr;
      else
	curr.move_forward_level();

      // Don't forget this case!
      if(curr==end)
	curr=begin;
    }

  while(curr!=start && !matches(*curr))
    {
      if(hierarchical)
	++curr;
      else
	curr.move_forward_level();

      if(curr==end)
	curr=begin;
    }

  if(curr==start)
    beep();
  else
    {
      selected->unhighlighted(this);
      set_selection(curr);
      selected->highlighted(this);
      while(!curr.is_root())
	{
	  curr=curr.get_up();
	  curr.expand();
	}
      vscreen_update();
    }
}

void vs_tree::paint()
{
  int width,height;
  int selectedln=line_of(selected);

  getmaxyx(height,width);

  if(selectedln>height)
    {
      while(selected!=top && selectedln>height)
	{
	  if(hierarchical)
	    ++top;
	  else
	    top.move_forward_level();
	  selectedln--;
	}
    }
  else
    {
      while(selected!=top && selectedln<0)
	{
	  if(hierarchical)
	    --top;
	  else
	    top.move_backward_level();
	  selectedln++;
	}
    }

  if(selected!=end && selected->get_selectable())
    selected->highlighted(this);
  // Some classes need this to update display stuff properly.  For instance,
  // when a new pkg_tree is created, its 'update the status line' signal
  // won't be properly called without this.

  vs_treeiterator i=top;
  int y=0;

  if(!hierarchical && y<height)
    {
      string todisp="";

      // Uh...I'd rather use the iterators to do this..
      flat_frame *curr=prev_level;
      while(curr)
	{
	  if(todisp.empty())
	    todisp=curr->selected->label()+todisp;
	  else
	    todisp=curr->selected->label()+("::"+todisp);
	  curr=curr->next;
	}

      if(todisp.empty())
	todisp="TOP LEVEL";

      while(todisp.size()<(unsigned) width)
	todisp+=" ";

      attrset(get_color("ScreenHeaderColor"));
      mvaddnstr(y, 0, todisp.c_str(), width);

      ++y;
    }

  // FIXME: this is a hack around nasty edge cases.  All the tree code needs
  //       a rewrite.
  vs_treeiterator prev=i;
  while(y<height && i!=end)
    {
      vs_treeitem *curr=&*i;

      if(get_isfocussed() && i==selected && i->get_selectable())
	attrset(curr->get_highlight_attr());
      else
	attrset(curr->get_normal_attr());

      curr->paint(this, y, hierarchical);

      if(hierarchical)
	++i;
      else
	i.move_forward_level();
      y++;

      // FIXME: this is a hack.
      if(i==prev) // If we hit the end, it will refuse to advance.
	break;
      prev=i;
    }
}

void vs_tree::dispatch_mouse(short id, int x, int y, int z, mmask_t bstate)
{
  if(!hierarchical)
    --y;

  vs_treeiterator i=top;
  while(y>0 && i!=end)
    {
      if(hierarchical)
	++i;
      else
	i.move_forward_level();

      --y;
    }

  if(y==0 && i!=end)
    {
      set_selection(i);

      i->dispatch_mouse(id, x, bstate, this);
    }
}

void vs_tree::init_bindings()
{
  bindings=new keybindings(&global_bindings);
}
