#!/bin/sh
# -*- mode: sh; coding: utf-8 -*-
# Test archive mirror creation
# Copyright © 2003 Colin Walters <walters@verbum.org>
#
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.

set -e

arg0="$0"
srcdir=`dirname "$arg0"`

. ${srcdir}/test-framework

setup_with_trivial_archives

test_class "Archive mirror"

begin_test "Archive mirror creation 1"
cat ~/.arch-params/archives/jane@example.com--2003 >&2
tla make-archive --mirror jane@example.com--2003 ${CACHEDARCHIVEDIR}/jane-archive-mirror || test_fail "failed to create the mirror"
tla whereis-archive --all-locations jane@example.com--2003 > ,,locations || test_fail "jane@example.com--2003 not registered"
file_matches "${CACHEDARCHIVEDIR}/jane-archive-mirror" ,,locations ||  test_fail "mirror not registered"
test -f ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror || test_fail "mirror flag not set"
test $(cat ${ARCHIVEDIR}/jane-archive-mirror/=meta-info/mirror) = "jane@example.com--2003" || test_fail "mirror file has incorrect contents"
end_test

begin_test "Archive mirror creation 2 (with listing)"
tla make-archive --listing --mirror foo@example.org--2003 ${CACHEDARCHIVEDIR}/foo-archive-mirror || test_fail "failed to create the mirror"
tla whereis-archive --all-locations foo@example.org--2003 > ,,locations || test_fail "foo@example.org--2003 not registered"
file_matches "${CACHEDARCHIVEDIR}/foo-archive-mirror" ,,locations ||  test_fail "mirror not registered"
test -f ${ARCHIVEDIR}/foo-archive-mirror/=meta-info/mirror || test_fail "mirror flag not set"
test $(cat ${ARCHIVEDIR}/foo-archive-mirror/=meta-info/mirror) = "foo@example.org--2003" || test_fail "mirror file has incorrect contents"
end_test 

begin_test "Archive-mirror shouldn't update cached revisions"
tla archive-mirror jane@example.com--2003 || test_fail "mirror run failed"
tla cacherev jane@example.com--2003/hello-world--mainline--1.0--base-0 || test_fail "cacherev failed"
test -f ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/hello-world--mainline--1.0--base-0.tar.gz || test_fail "no cacherev created"
tla archive-mirror jane@example.com--2003 || test_fail "mirror run failed"
test ! -f ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/hello-world--mainline--1.0--base-0.tar.gz
end_test 

begin_test "Archive-mirror with revision spec should add cached revisions"
tla archive-mirror jane@example.com--2003 hello-world--mainline--1.0--base-0
test -f ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/hello-world--mainline--1.0--base-0.tar.gz
end_test

begin_test "Archive-mirror with revision spec should remove cached revisions"
tla uncacherev jane@example.com--2003/hello-world--mainline--1.0--base-0
test ! -f ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/hello-world--mainline--1.0--base-0.tar.gz
tla archive-mirror jane@example.com--2003 hello-world--mainline--1.0--base-0
test ! -f ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/hello-world--mainline--1.0--base-0.tar.gz
end_test

begin_test "Archive-mirror in a project tree should mirror the same archive"
tla get jane@example.com--2003/hello-world--mainline--1.0 mainline
cd mainline
tla my-default-archive -d
tla archive-mirror
cd ..
end_test

begin_test "Archive-mirror pushing with no my-id"
rm -rf ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0
mv ${HOME}/.arch-params/\=id{,.old}
tla archive-mirror jane@example.com--2003
test -d ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0
mv ${HOME}/.arch-params/\=id{.old,}
end_test

begin_test "Archive-mirror pushing copies ancestry"
rm -rf ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0
tla archive-mirror jane@example.com--2003
test -f ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/ancestry.gz || test_fail "no ancestry copied across"
test -f ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0/base-0/ancestry.gz.checksum || test_fail "missing ancestry checksum"
end_test

begin_test "Archive-mirror pushing does not fail without ancestry"
rm -rf ${ARCHIVEDIR}/jane-archive-mirror/hello-world--mainline--1.0
mv ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/ancestry.gz{,.old}
mv ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/ancestry.gz.checksum{,.old}
test ! -f ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/ancestry.gz
tla archive-mirror jane@example.com--2003
mv ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/ancestry.gz{.old,}
mv ${ARCHIVEDIR}/jane-archive/hello-world--mainline--1.0/base-0/ancestry.gz.checksum{.old,}
end_test

begin_test "Mirroring does not use the cache"

tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 base0
cd base0
tla commit -s "make patch-1"
cd ..
CACHED_DELTA=$(cat $HOME/.arch-params/=arch-cache)/archives/jane@example.com--2003/hello-world--mainline--1.0--patch-1/delta.tar.gz
test -f $CACHED_DELTA
echo garbage > $CACHED_DELTA
tla archive-mirror jane@example.com--2003
tla cache-config -d
tla register-archive -d jane@example.com--2003
tla register-archive ${CACHEDARCHIVEDIR}/jane-archive-mirror
test "$(tla archive-meta-info jane@example.com--2003/mirror)" ==\
  "jane@example.com--2003"
tla get jane@example.com--2003/hello-world--mainline--1.0--patch-1
tla register-archive -d jane@example.com--2003
tla register-archive ${CACHEDARCHIVEDIR}/jane-archive
rm -rf $CACHEDIR
tla cache-config $CACHEDIR
tla register-archive  -f -d  ${CACHEDARCHIVEDIR}/jane-archive-mirror
# until all commands support the registry:
tla register-archive -d jane@example.com--2003
tla register-archive ${CACHEDARCHIVEDIR}/jane-archive
rm -rf ${ARCHIVEDIR}/jane-archive-mirror
tla make-archive --mirror jane@example.com--2003 \
  ${CACHEDARCHIVEDIR}/jane-archive-mirror
PATCHTAR="hello-world--mainline--1.0/patch-1/hello-world--mainline--1.0--patch-1.patches.tar.gz"
#seed the cache
tla get-changeset jane@example.com--2003/hello-world--mainline--1.0--patch-1
echo "bad" > $CACHED_DELTA
test -f ${ARCHIVEDIR}/jane-archive/$PATCHTAR
test ! -f ${ARCHIVEDIR}/jane-archive-mirror/$PATCHTAR 
tla archive-mirror jane@example.com--2003
diff ${ARCHIVEDIR}/jane-archive/$PATCHTAR ${ARCHIVEDIR}/jane-archive-mirror/$PATCHTAR || test_fail "tarballs differed"

end_test

clean_workdir

# tag: Colin Walters Tue, 16 Sep 2003 23:03:55 -0400 (test-mirror-creation.sh)
#
