/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fcntl.h>
#include <netdb.h>
#include <unistd.h>
#include <string.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include <gtk/gtk.h>

#include "import.h"
#include "tracer.h"
#include "baseband.h"
#include "hci.h"
#include "l2cap.h"

static void create_packet(GtkTreeStore *store, guint layer, guint type,
				guint size, gpointer buffer, guint64 timestamp)
{
	GtkTreeIter iter;
	gpointer data;
	gchar *packet;

	data = g_memdup(buffer, size);
	if (data == NULL)
		return;

	switch (layer) {
	case LAYER_BASEBAND:
		packet = decode_baseband(type, data, size);
		break;
	case LAYER_HCI:
		packet = decode_hci(type, data, size);
		break;
	default:
		packet = create_hex(NULL, data, size);
		break;
	}

	gtk_tree_store_insert_with_values(store, &iter, NULL, -1,
					COLUMN_LAYER, layer,
					COLUMN_TYPE, type,
					COLUMN_SIZE, size,
					COLUMN_DATA, data,
					COLUMN_TIMESTAMP, timestamp,
					COLUMN_PACKET, packet, -1);

	g_free(packet);
}

#define DATA_SIZE 512

struct import_data {
	GtkTreeStore *store;

	guint8 format;
	guint32 version;
	guint32 type;

	gpointer buffer;
	gsize size;
	gsize offset;
};

static gboolean process_unknown(struct import_data *data)
{
	if (data->offset < BTSNOOP_HDR_SIZE)
		return FALSE;

	if (memcmp(data->buffer, btsnoop_id, sizeof(btsnoop_id)) == 0) {
		struct btsnoop_hdr *hdr = data->buffer;

		data->format = FORMAT_BTSNOOP;
		data->version = GUINT32_FROM_BE(hdr->version);
		data->type = GUINT32_FROM_BE(hdr->type);

		g_memmove(data->buffer, data->buffer + BTSNOOP_HDR_SIZE,
						data->size - BTSNOOP_HDR_SIZE);

		data->offset -= BTSNOOP_HDR_SIZE;
	} else {
		struct pktlog_hdr *hdr = data->buffer;

		if ((GUINT32_FROM_BE(hdr->len) & 0xffff0000) == 0 &&
				(hdr->type < 0x04 || hdr->type == 0xff))
			data->format = FORMAT_PKTLOG;
		else
			data->format = FORMAT_HCIDUMP;
	}

	return TRUE;
}

static gboolean process_hcidump(struct import_data *data)
{
	struct hcidump_hdr *pkt = data->buffer;
	guint16 size;
	guint64 ts;
	guint type;

	if (data->offset < HCIDUMP_HDR_SIZE)
		return FALSE;

	size = GUINT16_FROM_LE(pkt->len);

	if (data->offset < size + HCIDUMP_HDR_SIZE)
		return FALSE;

	ts = GUINT32_FROM_LE(pkt->ts_sec);
	ts = (ts << 32) + GUINT32_FROM_LE(pkt->ts_usec);

	switch (((guint8 *) (data->buffer + HCIDUMP_HDR_SIZE))[0]) {
	case 0x01:
		type = 0x00;
		break;
	case 0x02:
		type = (pkt->in) ? 0x02 : 0x03;
		break;
	case 0x04:
		type = 0x01;
		break;
	default:
		type = 0xee;
		break;
	}

	create_packet(data->store, LAYER_HCI, type, size - 1,
				data->buffer + HCIDUMP_HDR_SIZE + 1, ts);

	g_memmove(data->buffer, data->buffer + HCIDUMP_HDR_SIZE + size,
					data->size - HCIDUMP_HDR_SIZE - size);

	data->offset -= HCIDUMP_HDR_SIZE + size;

	return TRUE;
}

static gboolean process_btsnoop(struct import_data *data)
{
	struct btsnoop_pkt *pkt = data->buffer;
	guint32 size;
	guint64 ts;
	guint type;

	if (data->offset < BTSNOOP_PKT_SIZE)
		return FALSE;

	size = GUINT32_FROM_BE(pkt->size);

	if (data->offset < size + BTSNOOP_PKT_SIZE)
		return FALSE;

	ts = GUINT64_FROM_BE(pkt->ts) - 0x00E03AB44A676000ll;
	ts = (((ts / 1000000ll) + 946684800ll) << 32) + (ts % 1000000ll);

	switch (data->type) {
	case 1001:
		if (GUINT32_FROM_BE(pkt->flags) & 0x02) {
			if (GUINT32_FROM_BE(pkt->flags) & 0x01)
				type = 0x01;
			else
				type = 0x00;
		} else {
			if (GUINT32_FROM_BE(pkt->flags) & 0x01)
				type = 0x03;
			else
				type = 0x02;
		}

		create_packet(data->store, LAYER_HCI, type, size,
					data->buffer + BTSNOOP_PKT_SIZE, ts);
		break;

	case 1002:
		switch (((guint8 *) (data->buffer + BTSNOOP_PKT_SIZE))[0]) {
		case 0x01:
			type = 0x00;
			break;
		case 0x02:
			if (GUINT32_FROM_BE(pkt->flags) & 0x01)
				type = 0x03;
			else
				type = 0x02;
			break;
		case 0x04:
			type = 0x01;
			break;
		default:
			type = 0xee;
			break;
		}

		create_packet(data->store, LAYER_HCI, type, size - 1,
				data->buffer + BTSNOOP_PKT_SIZE + 1, ts);
		break;

	case 4202:
		if (GUINT32_FROM_BE(pkt->flags) & 0x01)
			type = 0xff;
		else
			type = 0x02;

		create_packet(data->store, LAYER_BASEBAND, type, size,
					data->buffer + BTSNOOP_PKT_SIZE, ts);
		break;
	}

	g_memmove(data->buffer, data->buffer + BTSNOOP_PKT_SIZE + size,
					data->size - BTSNOOP_PKT_SIZE - size);

	data->offset -= BTSNOOP_PKT_SIZE + size;

	return TRUE;
}

static gboolean process_pktlog(struct import_data *data)
{
	struct pktlog_hdr *pkt = data->buffer;
	guint32 size;

	if (data->offset < PKTLOG_HDR_SIZE)
		return FALSE;

	size = GUINT32_FROM_BE(pkt->len);

	if (data->offset < size + 4)
		return FALSE;

	create_packet(data->store, LAYER_HCI, pkt->type,
					size - PKTLOG_HDR_SIZE + 4,
					data->buffer + PKTLOG_HDR_SIZE,
						GUINT64_FROM_BE(pkt->ts));

	g_memmove(data->buffer, data->buffer + size + 4,
						data->size - size - 4);

	data->offset -= size + 4;

	return TRUE;
}

static void process_data(struct import_data *data)
{
	gboolean cont = TRUE;

	while (cont == TRUE) {
		switch (data->format) {
		case FORMAT_UNKNOWN:
			cont = process_unknown(data);
			break;

		case FORMAT_HCIDUMP:
			cont = process_hcidump(data);
			break;

		case FORMAT_BTSNOOP:
			cont = process_btsnoop(data);
			break;

		case FORMAT_PKTLOG:
			cont = process_pktlog(data);
			break;

		default:
			cont = FALSE;
			break;
		}
	}
}

static void import_destroy(gpointer user_data)
{
	struct import_data *data = user_data;

	g_object_set_data(G_OBJECT(data->store), "io", NULL);

	g_signal_emit_by_name(data->store, "notify", NULL);

	g_object_unref(data->store);

	g_free(data->buffer);
	g_free(data);
}

static gboolean import_callback(GIOChannel *chan,
					GIOCondition cond, gpointer user_data)
{
	struct import_data *data = user_data;
	GIOStatus status;
	gsize bytes_read;

	if (cond & G_IO_NVAL) {
		g_io_channel_unref(chan);
		return FALSE;
	}

	if (cond & (G_IO_ERR | G_IO_HUP)) {
		g_io_channel_unref(chan);
		return FALSE;
	}

	if (data->size - data->offset < DATA_SIZE) {
		gpointer temp;
		temp = g_realloc(data->buffer, data->size + DATA_SIZE);
		if (temp == NULL)
			return TRUE;

		data->buffer = temp;
		data->size += DATA_SIZE;
	}

	status = g_io_channel_read_chars(chan, data->buffer + data->offset,
						data->size - data->offset,
							&bytes_read, NULL);

	if (status == G_IO_STATUS_EOF || status == G_IO_STATUS_ERROR) {
		g_io_channel_unref(chan);
		return FALSE;
	}

	if (status != G_IO_STATUS_NORMAL)
		return TRUE;

	data->offset += bytes_read;

	process_data(data);

	return TRUE;
}

GtkTreeModel *create_model(void)
{
	GtkTreeStore *store;

	store = gtk_tree_store_new(8, G_TYPE_UINT, G_TYPE_UINT,
				G_TYPE_UINT, G_TYPE_POINTER, G_TYPE_UINT64,
				G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING);

	return GTK_TREE_MODEL(store);
}

static struct import_data *create_import_data(GtkTreeModel *model)
{
	struct import_data *data;

	data = g_try_malloc(sizeof(struct import_data));
	if (data == NULL)
		return NULL;

	data->format = FORMAT_UNKNOWN;
	data->version = 0;
	data->type = 0;

	data->buffer = g_try_malloc(DATA_SIZE);
	if (data->buffer == NULL) {
		g_free(data);
		return NULL;
	}

	data->size = DATA_SIZE;
	data->offset = 0;

	data->store = GTK_TREE_STORE(model);

	return data;
}

int import_from_file(GtkTreeModel *model, const char *filename)
{
	struct import_data *data;
	GIOChannel *io;

	data = create_import_data(model);
	if (data == NULL)
		return -1;

	io = g_io_channel_new_file(filename, "r", NULL);
	if (io == NULL) {
		g_free(data->buffer);
		g_free(data);
		return -1;
	}

	g_io_channel_set_close_on_unref(io, TRUE);
	g_io_channel_set_encoding(io, NULL, NULL);
	g_io_channel_set_buffered(io, FALSE);

	g_object_ref(model);

	g_object_set_data(G_OBJECT(model), "io", io);

	g_io_add_watch_full(io, G_PRIORITY_DEFAULT_IDLE,
				G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
				import_callback, data, import_destroy);

	return 0;
}

static int do_connect(const char *hostname)
{
	struct addrinfo *ai, *runp;
	struct addrinfo hints;
	int sk, err;

	memset(&hints, 0, sizeof(hints));
	hints.ai_flags = AI_ADDRCONFIG;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = IPPROTO_TCP;

	err = getaddrinfo(hostname, "10839", &hints, &ai);
	if (err < 0)
		return -1;

	runp = ai;

	while (runp != NULL) {
		sk = socket(runp->ai_family, runp->ai_socktype,
							runp->ai_protocol);
		if (sk < 0)
			continue;

		if (connect(sk, runp->ai_addr, runp->ai_addrlen) == 0) {
			freeaddrinfo(ai);
			return sk;
		}

		runp = runp->ai_next;
	}

	freeaddrinfo(ai);

	return -1;
}

int import_from_socket(GtkTreeModel *model, const char *hostname)
{
	struct import_data *data;
	GIOChannel *io;
	int sk;

	data = create_import_data(model);
	if (data == NULL)
		return -1;

	sk = do_connect(hostname);
	if (sk < 0) {
		g_free(data->buffer);
		g_free(data);
		return -1;
	}

	io = g_io_channel_unix_new(sk);
	if (io == NULL) {
		close(sk);
		g_free(data->buffer);
		g_free(data);
		return -1;
	}

	g_io_channel_set_close_on_unref(io, TRUE);
	g_io_channel_set_encoding(io, NULL, NULL);
	g_io_channel_set_buffered(io, FALSE);
	g_io_channel_set_flags(io, G_IO_FLAG_NONBLOCK, NULL);

	g_object_ref(model);

	g_object_set_data(G_OBJECT(model), "io", io);

	g_io_add_watch_full(io, G_PRIORITY_DEFAULT,
				G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
				import_callback, data, import_destroy);

	return 0;
}

void finish_import(GtkTreeModel *model)
{
	GIOChannel *io;

	io = g_object_get_data(G_OBJECT(model), "io");
	if (io == NULL)
		return;

	g_io_channel_shutdown(io, TRUE, NULL);
}

static gboolean create_packet_column(GtkTreeModel *model, GtkTreePath *path,
					GtkTreeIter *iter, gpointer user_data)
{
	guint layer, type, size, handle, psm;
	gpointer data;
	gchar *packet;

	gtk_tree_model_get(model, iter,
				COLUMN_LAYER, &layer, COLUMN_TYPE, &type,
				COLUMN_SIZE, &size, COLUMN_DATA, &data,
				COLUMN_HANDLE, &handle, COLUMN_PSM, &psm, -1);

	if (layer != LAYER_L2CAP)
		return FALSE;

	packet = decode_l2cap(handle, psm, data, size);

	gtk_tree_store_set(GTK_TREE_STORE(model), iter,
						COLUMN_PACKET, packet, -1);

	g_free(packet);

	return FALSE;
}

void analyze_import(GtkTreeModel *model)
{
	analyze_l2cap(GTK_TREE_STORE(model));

	gtk_tree_model_foreach(model, create_packet_column, NULL);
}
