/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <gtk/gtk.h>

#include "l2cap.h"
#include "hci.h"
#include "baseband.h"
#include "tracer.h"

static void add_reference(GtkTreeModel *model, GtkTreeIter *iter)
{
	GtkTreeIter item;

	gtk_tree_store_append(GTK_TREE_STORE(model), &item, iter);

	gtk_tree_store_set(GTK_TREE_STORE(model), &item,
					COLUMN_LAYER, LAYER_DATA, -1);
}

static void add_to_existing(GtkTreeModel *model, GtkTreePath *path,
		GtkTreeIter *iter, guint matchtype, guint matchhandle,
		gpointer newdata, guint newsize, guint64 timestamp)
{
	GtkTreePath *temp_path = gtk_tree_path_copy(path);
	GtkTreeIter item;
	guint layer, type, size, handle;
	gpointer data, temp;

	while (gtk_tree_path_prev(temp_path) == TRUE) {
		if (gtk_tree_model_get_iter(model, &item, temp_path) == FALSE)
			break;

		gtk_tree_model_get(model, &item, COLUMN_LAYER, &layer,
						COLUMN_TYPE, &type,
						COLUMN_SIZE, &size,
						COLUMN_DATA, &data,
						COLUMN_HANDLE, &handle, -1);

		if (layer != LAYER_L2CAP ||
				type != matchtype || handle != matchhandle)
			continue;

		temp = g_try_realloc(data, size + newsize);
		if (temp == NULL)
			break;

		g_memmove(temp + size, newdata, newsize);

		gtk_tree_store_set(GTK_TREE_STORE(model), &item,
					COLUMN_SIZE, size + newsize,
					COLUMN_DATA, temp,
					COLUMN_TIMESTAMP, timestamp, -1);

		gtk_tree_store_move_after(GTK_TREE_STORE(model), &item, iter);

		add_reference(model, &item);
		break;
	}

	gtk_tree_path_free(temp_path);
}

static gboolean create_l2cap(GtkTreeModel *model, GtkTreePath *path,
					GtkTreeIter *iter, gpointer user_data)
{
	GtkTreeIter item;
	guint layer, type, size;
	gpointer data, temp;
	guint64 timestamp;
	struct hci_acldata_hdr *hdr;
	guint16 handle, dlen;
	guint8 pb;

	gtk_tree_model_get(model, iter, COLUMN_LAYER, &layer,
					COLUMN_TYPE, &type,
					COLUMN_SIZE, &size,
					COLUMN_DATA, &data,
					COLUMN_TIMESTAMP, &timestamp, -1);

	if (layer != LAYER_HCI)
		return FALSE;

	if (type != 0x02 && type != 0x03)
		return FALSE;

	hdr = data;

	handle = GUINT16_FROM_LE(hdr->handle);
	dlen = GUINT16_FROM_LE(hdr->dlen);
	pb = (handle & 0x3000) >> 12;
	handle &= 0x0fff;

	switch (pb & 0x03) {
	case 0x00:
	case 0x02:
		gtk_tree_store_insert_after(GTK_TREE_STORE(model),
							&item, NULL, iter);

		temp = g_memdup(data + HCI_ACLDATA_HDR_SIZE, dlen);

		gtk_tree_store_set(GTK_TREE_STORE(model), &item,
					COLUMN_LAYER, LAYER_L2CAP,
					COLUMN_TYPE, type,
					COLUMN_SIZE, dlen,
					COLUMN_DATA, temp,
					COLUMN_TIMESTAMP, timestamp,
					COLUMN_HANDLE, handle, -1);

		add_reference(model, &item);
		break;

	case 0x01:
		add_to_existing(model, path, iter, type, handle,
				data + HCI_ACLDATA_HDR_SIZE, dlen, timestamp);
		break;
	}

	return FALSE;
}

static gboolean create_psm(GtkTreeModel *model, GtkTreePath *path,
					GtkTreeIter *iter, gpointer user_data)
{
	GHashTable *hash = user_data;
	struct l2cap_packet_hdr *hdr;
	struct l2cap_command_hdr *cmd;
	guint layer, type, size, handle, psm, key;
	gpointer data, value;
	guint16 cid;

	gtk_tree_model_get(model, iter, COLUMN_LAYER, &layer,
					COLUMN_TYPE, &type,
					COLUMN_SIZE, &size,
					COLUMN_DATA, &data,
					COLUMN_HANDLE, &handle, -1);

	if (layer != LAYER_L2CAP)
		return FALSE;

	hdr = data;
	cid = GUINT16_FROM_LE(hdr->cid);

	switch (cid) {
	case 0x0001:
		if (type == 0x02)
			type = 0x03;
		else if (type == 0x03)
			type = 0x02;

		cmd = data + L2CAP_PACKET_HDR_SIZE;
		switch (cmd->code) {
		case 0x02:
			{
				struct l2cap_connect_req *req = data +
							L2CAP_PACKET_HDR_SIZE +
							L2CAP_COMMAND_HDR_SIZE;
				key = (type << 28) | (handle << 16) |
						GUINT16_FROM_LE(req->scid);
				psm = GUINT16_FROM_LE(req->psm);
				g_hash_table_replace(hash,
							GUINT_TO_POINTER(key),
							GUINT_TO_POINTER(psm));
			}
			break;
		}
		break;

	default:
		if (cid > 0x0039) {
			key = (type << 28) | (handle << 16) | cid;

			value = g_hash_table_lookup(hash, GUINT_TO_POINTER(key));
			if (value != NULL) {
				psm = GPOINTER_TO_UINT(value);
				gtk_tree_store_set(GTK_TREE_STORE(model), iter,
							COLUMN_PSM, psm, -1);
			}
		}
		break;
	}

	return FALSE;
}

void analyze_l2cap(GtkTreeStore *store)
{
	GHashTable *hash;

	gtk_tree_model_foreach(GTK_TREE_MODEL(store), create_l2cap, NULL);

	hash = g_hash_table_new(NULL, NULL);

	gtk_tree_model_foreach(GTK_TREE_MODEL(store), create_psm, hash);

	g_hash_table_destroy(hash);
}

static gchar *cid2str(guint16 cid)
{
	switch (cid) {
	case 0x0000:
		return "Null identifier";
	case 0x0001:
		return "Signal channel";
	case 0x0002:
		return "Connectionless reception channel";
	default:
		return (cid < 0x0040) ? "Reserved" : "Data channel";
	}
}

static gchar *code2str(guint8 code)
{
	switch (code) {
	case 0x01:
		return "Command Reject";
	case 0x02:
		return "Connection Request";
	case 0x03:
		return "Connection Response";
	case 0x04:
		return "Configuration Request";
	case 0x05:
		return "Configuration Response";
	case 0x06:
		return "Disconnection Request";
	case 0x07:
		return "Disconnection Response";
	case 0x08:
		return "Echo Request";
	case 0x09:
		return "Echo Response";
	case 0x0a:
		return "Information Request";
	case 0x0b:
		return "Information Response";
	default:
		return "Reserved";
	}
}

static gchar *signal_channel(gchar *text, gpointer data, guint size)
{
	gboolean first = TRUE;

	if (data == NULL || size < L2CAP_PACKET_HDR_SIZE)
		return create_hex(text, data, size);

	while (size > 0) {
		struct l2cap_command_hdr *hdr = data;
		gchar *linebreak, *temp = text;
		guint16 len;

		linebreak = (first == TRUE) ? "\n" :
					"\n<span size=\"2048\">\n</span>";

		len = GUINT16_FROM_LE(hdr->len);

		text = g_strdup_printf("%s%s%s (0x%02x) ident %d length %d",
					temp, linebreak, code2str(hdr->code),
						hdr->code, hdr->ident, len);

		g_free(temp);

		text = create_hex(text, data + L2CAP_COMMAND_HDR_SIZE, len);

		data += L2CAP_COMMAND_HDR_SIZE + len;
		size -= L2CAP_COMMAND_HDR_SIZE + len;

		first = FALSE;
	}

	return text;
}

gchar *decode_l2cap(guint handle, guint psm, gpointer data, guint size)
{
	struct l2cap_packet_hdr *hdr = data;
	guint16 cid;
	guint16 len;
	gchar *text, *str;

	if (data == NULL || size < L2CAP_PACKET_HDR_SIZE)
		return NULL;

	cid = GUINT16_FROM_LE(hdr->cid);
	len = GUINT16_FROM_LE(hdr->len);

	if (psm > 0)
		str = g_strdup_printf("[handle %d psm %d]", handle, psm);
	else
		str = g_strdup_printf("[handle %d]", handle);

	text = g_strdup_printf("%s (0x%02x) length %d %s",
						cid2str(cid), cid, len, str);

	g_free(str);

	text = check_packet_size(text, len, size - L2CAP_PACKET_HDR_SIZE);

	data += L2CAP_PACKET_HDR_SIZE;
	size -= L2CAP_PACKET_HDR_SIZE;

	switch (cid) {
	case 0x0001:
		return signal_channel(text, data, size);

	case 0x0002:
		data += 2;
		size -= 2;
		break;
	}

	return create_hex(text, data, size);
}
