/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "tracer.h"
#include "import.h"

guint8 btsnoop_id[8] = { 0x62, 0x74, 0x73, 0x6e, 0x6f, 0x6f, 0x70, 0x00 };

static void type_to_text(GtkTreeViewColumn *column,
				GtkCellRenderer *cell, GtkTreeModel *model,
					GtkTreeIter *iter, gpointer user_data)
{
	guint layer, type;
	gchar *text = NULL;

	gtk_tree_model_get(model, iter, COLUMN_LAYER, &layer,
						COLUMN_TYPE, &type, -1);

	switch (layer) {
	case LAYER_DATA:
		text = g_strdup_printf("Data ");
		break;

	case LAYER_BASEBAND:
		if (type == 0xff)
			text = g_strdup_printf("Info ");
		else
			text = g_strdup_printf("Baseband ");
		break;

	case LAYER_HCI:
		switch (type) {
		case 0x00:
			text = g_strdup_printf("HCI Command ");
			break;
		case 0x01:
			text = g_strdup_printf("HCI Event ");
			break;
		case 0x02:
			text = g_strdup_printf("ACL data TX ");
			break;
		case 0x03:
			text = g_strdup_printf("ACL data RX ");
			break;
		case 0xff:
			text = g_strdup_printf("Info ");
			break;
		default:
			text = g_strdup_printf("<tt>%02x</tt> ", type);
			break;
		}
		break;

	case LAYER_LMP:
		text = g_strdup_printf("LMP ");
		break;

	case LAYER_L2CAP:
		switch (type) {
		case 0x02:
			text = g_strdup_printf("L2CAP frame TX ");
			break;
		case 0x03:
			text = g_strdup_printf("L2CAP frame RX ");
			break;
		default:
			text = g_strdup_printf("L2CAP <i>%d</i> ", type);
			break;
		}
		break;

	default:
		text = NULL;
		break;
	}

	g_object_set(cell, "markup", text, NULL);

	g_free(text);
}

static void timestamp_to_text(GtkTreeViewColumn *column,
				GtkCellRenderer *cell, GtkTreeModel *model,
					GtkTreeIter *iter, gpointer user_data)
{
	GtkTreePath *path;
	gint depth;
	guint layer;
	guint64 timestamp;
	gchar *text = NULL;

	gtk_tree_model_get(model, iter, COLUMN_LAYER, &layer,
					COLUMN_TIMESTAMP, &timestamp, -1);

	path = gtk_tree_model_get_path(model, iter);
	depth = gtk_tree_path_get_depth(path);
	gtk_tree_path_free(path);

	if (depth < 2 && timestamp > 0) {
		struct tm tm;
		time_t time = timestamp >> 32;
		localtime_r(&time, &tm);

		text = g_strdup_printf("%04d-%02d-%02d %02d:%02d:%02d.%06d",
				tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday,
					tm.tm_hour, tm.tm_min, tm.tm_sec,
					(guint32) timestamp & 0xffffffff);
#if 0
		text = g_strdup_printf("<tt>%8lu.%06d</tt>", timestamp >> 32,
					(guint32) timestamp & 0xffffffff);
#endif
	}

	g_object_set(cell, "markup", text, NULL);

	g_free(text);
}

static gboolean filter_func(GtkTreeModel *model,
				GtkTreeIter *iter, gpointer user_data)
{
	GtkWidget *child = user_data;
	guint flags, layer, type;

	flags = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child), "flags"));

	gtk_tree_model_get(model, iter, COLUMN_LAYER, &layer,
						COLUMN_TYPE, &type, -1);

	if (layer == LAYER_DATA)
		return FALSE;

	if (layer == LAYER_BASEBAND && (flags & (1 << LAYER_BASEBAND)))
		return TRUE;

	if (layer == LAYER_HCI) {
		if ((type == 0x02 || type == 0x03) &&
				(flags & (1 << LAYER_L2CAP)))
			return FALSE;

		if (flags & (1 << LAYER_HCI))
			return TRUE;
	}

	if (layer == LAYER_L2CAP && (flags & (1 << LAYER_L2CAP)))
		return TRUE;

	return FALSE;
}

static void inserted_callback(GtkTreeModel *model, GtkTreePath  *path,
					GtkTreeIter *iter, gpointer user_data)
{
	GtkWidget *widget = user_data;
	GtkWidget *tree;
	GtkTreeSelection *selection;
	gboolean running, saved;

	saved = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(widget), "saved"));

	running = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(widget),
								"running"));

	if (running == FALSE || saved == TRUE)
		return;

	tree = g_object_get_data(G_OBJECT(widget), "tree");

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));

	if (gtk_tree_selection_get_selected(selection, NULL, NULL) == TRUE)
		return;

	gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(tree), path,
						NULL, FALSE, 0.0, 0.0);
}

static GtkWidget *create_tracer_widget(GtkTreeModel *model, gboolean saved)
{
	GtkWidget *vbox;
	GtkWidget *scrolled;
	GtkWidget *tree;
	GtkTreeModel *filter;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	gboolean running = TRUE;
	guint flags;

	vbox = gtk_vbox_new(FALSE, 0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);

	tree = gtk_tree_view_new();
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tree), TRUE);
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(tree), FALSE);
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(tree), TRUE);
	gtk_container_add(GTK_CONTAINER(scrolled), tree);

	g_object_set_data(G_OBJECT(vbox), "tree", tree);

	g_object_set_data(G_OBJECT(vbox), "model", model);

	flags = (1 << LAYER_BASEBAND) | (1 << LAYER_HCI);
	g_object_set_data(G_OBJECT(vbox), "flags", GUINT_TO_POINTER(flags));

	filter = gtk_tree_model_filter_new(model, NULL);
	gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filter),
						filter_func, vbox, NULL);
	gtk_tree_view_set_model(GTK_TREE_VIEW(tree), GTK_TREE_MODEL(filter));
	g_object_unref(filter);

	g_object_set_data(G_OBJECT(vbox), "filter", filter);

	//gtk_tree_view_expand_all(GTK_TREE_VIEW(tree));

	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "yalign", 0.0, NULL);
	gtk_tree_view_insert_column_with_data_func(GTK_TREE_VIEW(tree),
			0, _("Type"), renderer, type_to_text, NULL, NULL);

	column = gtk_tree_view_get_column(GTK_TREE_VIEW(tree), 0);
	gtk_tree_view_column_set_min_width(column, 150);

	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "yalign", 0.0, NULL);
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(tree),
			1, _("Packet"), renderer, "markup", COLUMN_PACKET, NULL);

	column = gtk_tree_view_get_column(GTK_TREE_VIEW(tree), 1);
	gtk_tree_view_column_set_min_width(column, 400);
	gtk_tree_view_column_set_expand(column, TRUE);

	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "yalign", 0.0, NULL);
	gtk_tree_view_insert_column_with_data_func(GTK_TREE_VIEW(tree),
			2, _("Timestamp"), renderer, timestamp_to_text, NULL, NULL);

	g_object_set_data(G_OBJECT(vbox), "running",
					GUINT_TO_POINTER(running));

	g_object_set_data(G_OBJECT(vbox), "saved",
					GUINT_TO_POINTER(saved));

	g_signal_connect(G_OBJECT(filter), "row-inserted",
					G_CALLBACK(inserted_callback), vbox);

	return vbox;
}

GtkWidget *create_tracer_from_file(const gchar *filename, GCallback callback)
{
	GtkWidget *widget;
	GtkTreeModel *model;

	model = create_model();
	if (model == NULL)
		return NULL;

	widget = create_tracer_widget(model, TRUE);

	g_object_set_data(G_OBJECT(widget), "filename", g_strdup(filename));

	g_object_ref(widget);

	g_signal_connect(model, "notify", callback, widget);

	if (import_from_file(model, filename) < 0) {
		gtk_widget_destroy(widget);
		return NULL;
	}

	gtk_widget_show_all(widget);

	return widget;
}

GtkWidget *create_live_import(const gchar *hostname, GCallback callback)
{
	GtkWidget *widget;
	GtkTreeModel *model;

	model = create_model();
	if (model == NULL)
		return NULL;

	widget = create_tracer_widget(model, FALSE);

	g_object_set_data(G_OBJECT(widget), "hostname", g_strdup(hostname));

	g_object_ref(widget);

	g_signal_connect(model, "notify", callback, widget);

	if (import_from_socket(model, hostname) < 0) {
		gtk_widget_destroy(widget);
		return NULL;
	}

	gtk_widget_show_all(widget);

	return widget;
}
