/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *  Copyright (C) 2006-2007  Bastien Nocera <hadess@hadess.net>
 *
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>

#include <glib/gi18n-lib.h>

#include "client.h"

#include "bluetooth-device-selection.h"

enum {
	SELECTED_DEVICE_CHANGED,
	LAST_SIGNAL
};

static int selection_table_signals[LAST_SIGNAL] = { 0 };

#define BLUETOOTH_DEVICE_SELECTION_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), \
										 BLUETOOTH_TYPE_DEVICE_SELECTION, BluetoothDeviceSelectionPrivate))

typedef struct _BluetoothDeviceSelectionPrivate BluetoothDeviceSelectionPrivate;

struct _BluetoothDeviceSelectionPrivate {
	BluetoothClient *client;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkWidget *label;

	/* Widgets/UI bits that can be shown or hidden */
	GtkCellRenderer *bonded_cell;
	GtkWidget *search_button;

	guint show_bonded : 1;
	guint show_search : 1;
};

G_DEFINE_TYPE(BluetoothDeviceSelection, bluetooth_device_selection, GTK_TYPE_VBOX)

static void
name_to_text (GtkTreeViewColumn *column, GtkCellRenderer *cell,
	      GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	gchar *address;
	gchar *name;

	gtk_tree_model_get (model, iter, COLUMN_ADDRESS, &address,
			    COLUMN_NAME, &name, -1);

	/* If we don't have a name, replace the name with the
	 * Bluetooth address, with the ":" replaced by "-" */
	if (name == NULL) {
		name = g_strdup (address);
		g_strdelimit (name, ":", '-');
	}

	g_object_set (cell, "text", name ? name : address, NULL);

	g_free (name);
	g_free (address);
}

static void
type_to_icon (GtkTreeViewColumn *column, GtkCellRenderer *cell,
	      GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	guint type;

	gtk_tree_model_get (model, iter, COLUMN_TYPE, &type, -1);

	switch (type) {
	case BLUETOOTH_TYPE_PHONE:
		g_object_set (cell, "icon-name", "stock_cell-phone", NULL);
		break;
	case BLUETOOTH_TYPE_MODEM:
		g_object_set (cell, "icon-name", "modem", NULL);
		break;
	case BLUETOOTH_TYPE_COMPUTER:
		g_object_set (cell, "icon-name", "computer", NULL);
		break;
	case BLUETOOTH_TYPE_NETWORK:
		g_object_set (cell, "icon-name", "network-wireless", NULL);
		break;
	case BLUETOOTH_TYPE_HEADSET:
		g_object_set (cell, "icon-name", "stock_headphones", NULL);
		break;
	case BLUETOOTH_TYPE_KEYBOARD:
		g_object_set (cell, "icon-name", "input-keyboard", NULL);
		break;
	case BLUETOOTH_TYPE_MOUSE:
		g_object_set (cell, "icon-name", "input-mouse", NULL);
		break;
	case BLUETOOTH_TYPE_CAMERA:
		g_object_set (cell, "icon-name", "camera-photo", NULL);
		break;
	case BLUETOOTH_TYPE_PRINTER:
		g_object_set (cell, "icon-name", "printer", NULL);
		break;
	default:
		g_object_set (cell, "icon-name", "stock_bluetooth", NULL);
		break;
	}
}

static void
bonded_to_icon (GtkTreeViewColumn *column, GtkCellRenderer *cell,
	      GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	gboolean bonded;

	gtk_tree_model_get (model, iter, COLUMN_BONDED, &bonded, -1);

	if (bonded == FALSE)
		g_object_set (cell, "stock-id", NULL, NULL);
	else
		g_object_set (cell, "stock-id", GTK_STOCK_DIALOG_AUTHENTICATION, NULL);
}

static void
type_to_text (GtkTreeViewColumn *column, GtkCellRenderer *cell,
	      GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	guint type;

	gtk_tree_model_get (model, iter, COLUMN_TYPE, &type, -1);

	g_object_set (cell, "text", bluetooth_type_to_string (type), NULL);
}

void
bluetooth_device_selection_start_discovery (BluetoothDeviceSelection *self)
{
	BluetoothDeviceSelectionPrivate *priv = BLUETOOTH_DEVICE_SELECTION_GET_PRIVATE(self);

	gtk_widget_set_sensitive (GTK_WIDGET(priv->search_button), FALSE);
	bluetooth_client_discover_devices (priv->client, NULL);
}

gchar *
bluetooth_device_selection_get_selected_device (BluetoothDeviceSelection *self)
{
	BluetoothDeviceSelectionPrivate *priv = BLUETOOTH_DEVICE_SELECTION_GET_PRIVATE(self);
	GtkTreeIter iter;
	gchar *address;
	gboolean selected;

	selected = gtk_tree_selection_get_selected (priv->selection, NULL, &iter);
	if (selected == FALSE)
		return NULL;

	gtk_tree_model_get (priv->model, &iter, COLUMN_ADDRESS, &address, -1);
	return address;
}

static void
search_button_clicked (GtkButton *button, gpointer user_data)
{
	BluetoothDeviceSelection *self = BLUETOOTH_DEVICE_SELECTION(user_data);

	bluetooth_device_selection_start_discovery (self);
}

#if 0
static void
discoveries_completed (BluetoothClient *client, gpointer user_data)
{
	BluetoothDeviceSelection *self = BLUETOOTH_DEVICE_SELECTION(user_data);
	BluetoothDeviceSelectionPrivate *priv = BLUETOOTH_DEVICE_SELECTION_GET_PRIVATE(self);

	gtk_widget_set_sensitive (GTK_WIDGET(priv->search_button), TRUE);
}
#endif

static void
select_browse_device_callback (GtkTreeSelection *selection, gpointer user_data)
{
	BluetoothDeviceSelection *self = user_data;
	gchar *address;

	g_object_notify (G_OBJECT(self), "device-selected");
	address = bluetooth_device_selection_get_selected_device (self);
	g_signal_emit (G_OBJECT (self),
		       selection_table_signals[SELECTED_DEVICE_CHANGED],
		       0, address);
	g_free (address);
}

static void
bluetooth_device_selection_init(BluetoothDeviceSelection *self)
{
	BluetoothDeviceSelectionPrivate *priv = BLUETOOTH_DEVICE_SELECTION_GET_PRIVATE(self);
	GtkWidget *tree, *scrolled, *frame, *box, *hbox;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	priv->show_bonded = FALSE;
	priv->show_search = FALSE;

	gtk_box_set_spacing (GTK_BOX(self), 6);
	gtk_box_set_homogeneous (GTK_BOX(self), FALSE);
	gtk_container_set_border_width (GTK_CONTAINER(self), 8);

	frame = gtk_frame_new ("");
	priv->label = gtk_frame_get_label_widget (GTK_FRAME(frame));
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_container_add (GTK_CONTAINER(self), frame);

	box = gtk_vbox_new (FALSE, 8);
	gtk_container_add (GTK_CONTAINER(frame), box);

	priv->client = bluetooth_client_new ();

#if 0
	g_signal_connect (G_OBJECT(priv->client), "discoveries-completed",
			  G_CALLBACK(discoveries_completed), self);
#endif

	/* Create the scrolled window */
	scrolled = gtk_scrolled_window_new (NULL, NULL);

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrolled),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scrolled),
					     GTK_SHADOW_OUT);

	/* Create the tree view */
	tree = gtk_tree_view_new ();

	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW(tree), TRUE);

	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW(tree), TRUE);

	g_object_set (tree, "show-expanders", FALSE, NULL);

	column = gtk_tree_view_column_new ();

	gtk_tree_view_column_set_title (column, _("Device"));

	/* The type icon */
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_set_spacing (column, 4);
	gtk_tree_view_column_pack_start (column, renderer, FALSE);

	gtk_tree_view_column_set_cell_data_func (column, renderer,
						 type_to_icon, NULL, NULL);

	/* The device name */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);

	gtk_tree_view_column_set_cell_data_func (column, renderer,
						 name_to_text, NULL, NULL);

	/* The bonded icon */
	priv->bonded_cell = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_end (column, priv->bonded_cell, FALSE);

	gtk_tree_view_column_set_cell_data_func (column, priv->bonded_cell,
						 bonded_to_icon, NULL, NULL);

	gtk_tree_view_append_column (GTK_TREE_VIEW(tree), column);

	gtk_tree_view_column_set_min_width (GTK_TREE_VIEW_COLUMN(column), 280);

	gtk_tree_view_insert_column_with_data_func (GTK_TREE_VIEW(tree), -1,
						    "Type", gtk_cell_renderer_text_new(),
						    type_to_text, NULL, NULL);

	priv->selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(tree));

	gtk_tree_selection_set_mode (priv->selection, GTK_SELECTION_SINGLE);

	g_signal_connect (G_OBJECT(priv->selection), "changed",
			  G_CALLBACK(select_browse_device_callback), self);

	priv->model = bluetooth_client_get_model_for_adapter (priv->client, NULL);
	if (priv->model) {
		gtk_tree_view_set_model (GTK_TREE_VIEW(tree), priv->model);
		g_object_unref (priv->model);
	}

	gtk_container_add (GTK_CONTAINER(scrolled), tree);
	gtk_box_pack_start (GTK_BOX(box), scrolled, TRUE, TRUE, 6);

	hbox = gtk_hbox_new (FALSE, 0);
	priv->search_button = gtk_button_new_with_label (_("Search"));
	g_signal_connect (G_OBJECT(priv->search_button), "clicked",
			  G_CALLBACK(search_button_clicked), self);
	gtk_box_pack_start (GTK_BOX(box), hbox, FALSE, FALSE, 0);
	if (priv->show_search == TRUE)
		gtk_box_pack_start (GTK_BOX(hbox), priv->search_button,
				    FALSE, FALSE, 0);

	gtk_widget_show_all (scrolled);

	bluetooth_device_selection_start_discovery (self);
}

static void
bluetooth_device_selection_finalize (GObject *object)
{
	BluetoothDeviceSelectionPrivate *priv = BLUETOOTH_DEVICE_SELECTION_GET_PRIVATE(object);

	bluetooth_client_cancel_discovery (priv->client, NULL);
}

enum {
	PROP_0,
	PROP_TITLE,
	PROP_DEVICE_SELECTED,
	PROP_SHOW_BONDING,
	PROP_SHOW_SEARCH
};

static void
bluetooth_device_selection_set_property (GObject *object, guint prop_id,
					 const GValue *value, GParamSpec *pspec)
{
	BluetoothDeviceSelectionPrivate *priv = BLUETOOTH_DEVICE_SELECTION_GET_PRIVATE(object);

	switch (prop_id) {
	case PROP_TITLE:
		{
			char *str;

			if (!g_value_get_string(value))
				break;

			str = g_strdup_printf ("<b>%s</b>", g_value_get_string(value));
			gtk_label_set_markup (GTK_LABEL(priv->label), str);
			g_free (str);
		}
		break;
	case PROP_SHOW_BONDING:
		priv->show_bonded = g_value_get_boolean (value);
		if (priv->bonded_cell != NULL)
			g_object_set (G_OBJECT (priv->bonded_cell), "visible", priv->show_bonded, NULL);
		break;
	case PROP_SHOW_SEARCH:
		priv->show_search = g_value_get_boolean (value);
		if (priv->search_button != NULL)
			g_object_set (G_OBJECT (priv->search_button), "visible", priv->show_search, NULL);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void
bluetooth_device_selection_get_property (GObject *object, guint prop_id,
					 GValue *value, GParamSpec *pspec)
{
	BluetoothDeviceSelection *self = BLUETOOTH_DEVICE_SELECTION(object);
	BluetoothDeviceSelectionPrivate *priv = BLUETOOTH_DEVICE_SELECTION_GET_PRIVATE(object);

	switch (prop_id) {
	case PROP_DEVICE_SELECTED:
		g_value_set_string (value, bluetooth_device_selection_get_selected_device (self));
		break;
	case PROP_SHOW_BONDING:
		g_value_set_boolean (value, priv->show_bonded);
		break;
	case PROP_SHOW_SEARCH:
		g_value_set_boolean (value, priv->show_search);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void
bluetooth_device_selection_class_init (BluetoothDeviceSelectionClass *klass)
{
	g_type_class_add_private(klass, sizeof(BluetoothDeviceSelectionPrivate));

	G_OBJECT_CLASS(klass)->finalize = bluetooth_device_selection_finalize;

	G_OBJECT_CLASS(klass)->set_property = bluetooth_device_selection_set_property;
	G_OBJECT_CLASS(klass)->get_property = bluetooth_device_selection_get_property;

	selection_table_signals[SELECTED_DEVICE_CHANGED] =
		g_signal_new ("selected-device-changed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (BluetoothDeviceSelectionClass, selected_device_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE, 1, G_TYPE_STRING);

	g_object_class_install_property (G_OBJECT_CLASS(klass),
					 PROP_TITLE, g_param_spec_string ("title",
									  NULL, NULL, NULL, G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS(klass),
					 PROP_DEVICE_SELECTED, g_param_spec_string ("device-selected",
										    NULL, NULL, NULL, G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS(klass),
					 PROP_SHOW_BONDING, g_param_spec_boolean ("show-bonding",
										  NULL, NULL, FALSE, G_PARAM_READWRITE));
	g_object_class_install_property (G_OBJECT_CLASS(klass),
					 PROP_SHOW_SEARCH, g_param_spec_boolean ("show-search",
										 NULL, NULL, FALSE, G_PARAM_READWRITE));
}

GtkWidget *
bluetooth_device_selection_new (const gchar *title)
{
	return g_object_new(BLUETOOTH_TYPE_DEVICE_SELECTION,
			    "title", title,
			    NULL);
}

