/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2008  Marcel Holtmann <marcel@holtmann.org>
 *  Copyright (C) 2006-2007  Bastien Nocera <hadess@hadess.net>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "client.h"

#include "general.h"
#include "adapter.h"
#include "dialog.h"

gboolean show_confirm_dialog(void)
{
	GtkWidget *dialog;
	gint response;
	gchar *text;

	text = g_strdup_printf("<big><b>%s</b></big>\n\n%s",
				_("Remove from list of bonded devices?"),
				_("If you delete the bonding, you have to "
					"re-pair the device before next use."));

	dialog = gtk_message_dialog_new_with_markup(NULL, GTK_DIALOG_MODAL,
				GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE, text);

	g_free(text);

	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL,
							GTK_RESPONSE_CANCEL);

	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_DELETE,
							GTK_RESPONSE_ACCEPT);

	response = gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);

	if (response == GTK_RESPONSE_ACCEPT)
		return TRUE;

	return FALSE;
}

gboolean show_remove_dialog(void)
{
	GtkWidget *dialog;
	gint response;
	gchar *text;

	text = g_strdup_printf("<big><b>%s</b></big>\n\n%s",
				_("Remove device entry from service?"),
				_("If you delete the device entry, you can't "
					"use it with this service anymore."));

	dialog = gtk_message_dialog_new_with_markup(NULL, GTK_DIALOG_MODAL,
				GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE, text);

	g_free(text);

	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL,
							GTK_RESPONSE_CANCEL);

	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_DELETE,
							GTK_RESPONSE_ACCEPT);

	response = gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);

	if (response == GTK_RESPONSE_ACCEPT)
		return TRUE;

	return FALSE;
}

static void create_connect(GtkWidget *notebook)
{
	GtkWidget *vbox;
	GtkWidget *label;

	vbox = gtk_vbox_new(FALSE, 6);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 6);

	label = create_label(_("Connecting to device"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	g_object_set_data(G_OBJECT(notebook), "label", label);

	gtk_widget_show_all(vbox);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, NULL);
}

static void connect_callback(GtkWidget *button, gpointer user_data)
{
	GtkWidget *notebook = user_data;
	GtkWidget *widget;
	gchar *address;
	DialogConnectCallbackFunc callback;

	gtk_widget_hide(button);

	widget = g_object_get_data(G_OBJECT(notebook), "cancel");
	gtk_widget_hide(widget);

	widget = g_object_get_data(G_OBJECT(notebook), "close");
	gtk_widget_set_sensitive(widget, FALSE);
	gtk_widget_show(widget);

	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), 1);

	address = g_object_get_data(G_OBJECT(button), "address");

	g_object_set_data(G_OBJECT(button), "address", NULL);

	callback = g_object_get_data(G_OBJECT(notebook), "callback");
	if (callback != NULL)
		callback(address, notebook);

	g_free(address);
}

static void showall_callback(GtkWidget *button, gpointer user_data)
{
	GtkWidget *widget = user_data;
	GtkTreeModel *model;
	gboolean active;

	active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));

	g_object_set_data(G_OBJECT(widget), "showall",
					GUINT_TO_POINTER(active));

	model = g_object_get_data(G_OBJECT(widget), "model");
	if (model == NULL)
		return;

	gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(model));
}

static void select_callback(GtkTreeSelection *selection, gpointer user_data)
{
	GtkWidget *widget = user_data;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean selected;
	gchar *temp, *address = NULL;

	selected = gtk_tree_selection_get_selected(selection, &model, &iter);

	temp = g_object_get_data(G_OBJECT(widget), "address");

	if (selected == TRUE)
		gtk_tree_model_get(model, &iter, COLUMN_ADDRESS, &address, -1);

	g_object_set_data(G_OBJECT(widget), "address", address);

	g_free(temp);

	gtk_widget_set_sensitive(widget, selected);
}

static gint device_compare(GtkTreeModel *model, GtkTreeIter *a,
					GtkTreeIter *b, gpointer user_data)
{
	gint rssi1 = 0, rssi2 = 0;

	gtk_tree_model_get(model, a, COLUMN_RSSI, &rssi1, -1);
	gtk_tree_model_get(model, b, COLUMN_RSSI, &rssi2, -1);

	return (ABS(rssi1) - ABS(rssi2));
}

static gboolean device_filter(GtkTreeModel *model,
					GtkTreeIter *iter, gpointer user_data)
{
	GtkWidget *widget = user_data;
	GtkTreeModelFilterVisibleFunc func;
	gboolean active;

	gtk_tree_model_get(model, iter, COLUMN_ACTIVE, &active, -1);
	if (active == FALSE)
		return FALSE;

	active = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(widget),
								"showall"));
	if (active == TRUE)
		return TRUE;

	func = g_object_get_data(G_OBJECT(widget), "filter");
	if (func == NULL)
		return TRUE;

	return func(model, iter, NULL);
}

static void create_selection(GtkWidget *notebook, BluetoothClient *client,
					GtkTreeModelFilterVisibleFunc func)
{
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *scrolled;
	GtkWidget *tree;
	GtkTreeModel *model, *sorted;
	GtkTreeSelection *selection;

	vbox = gtk_vbox_new(FALSE, 6);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 6);

	label = create_label(_("Select device"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled),
							GTK_SHADOW_OUT);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);

	model = bluetooth_client_get_model_with_filter(client, NULL,
							device_filter, vbox);

	g_object_set_data(G_OBJECT(vbox), "model", model);
	g_object_set_data(G_OBJECT(vbox), "filter", func);
	g_object_set_data(G_OBJECT(vbox), "showall", GUINT_TO_POINTER(FALSE));

	sorted = gtk_tree_model_sort_new_with_model(model);
	gtk_tree_sortable_set_default_sort_func(GTK_TREE_SORTABLE(sorted),
						device_compare, NULL, NULL);

	g_object_unref(model);

	tree = create_tree(sorted, TRUE);
	gtk_widget_set_size_request(tree, 350, 200);
	gtk_container_add(GTK_CONTAINER(scrolled), tree);

	g_object_unref(sorted);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

	button = g_object_get_data(G_OBJECT(notebook), "connect");

	g_signal_connect(G_OBJECT(selection), "changed",
				G_CALLBACK(select_callback), button);

	button = gtk_check_button_new_with_label(_("Show all devices"));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(button), "toggled",
				G_CALLBACK(showall_callback), vbox);

	gtk_widget_show_all(vbox);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, NULL);
}

void show_device_dialog(DialogConnectCallbackFunc callback,
				DialogCleanupCallbackFunc cleanup,
				GtkTreeModelFilterVisibleFunc filter)
{
	BluetoothClient *client;
	GtkWidget *dialog;
	GtkWidget *button;
	GtkWidget *notebook;

	client = bluetooth_client_new();

	dialog = gtk_dialog_new_with_buttons(_("Create Device"), NULL,
						GTK_DIALOG_NO_SEPARATOR, NULL);

	gtk_dialog_set_default_response(GTK_DIALOG(dialog),
						GTK_RESPONSE_APPLY);

	gtk_container_set_border_width(GTK_CONTAINER(dialog), 8);

	notebook = gtk_notebook_new();
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), FALSE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);
	gtk_widget_set_no_show_all(notebook, TRUE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), notebook);
	gtk_widget_show(notebook);

	g_object_set_data(G_OBJECT(notebook), "callback", callback);

	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);

	g_object_set_data(G_OBJECT(notebook), "cancel", button);

	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
					GTK_STOCK_CLOSE, GTK_RESPONSE_APPLY);
	gtk_widget_hide(button);

	g_object_set_data(G_OBJECT(notebook), "close", button);

	button = gtk_button_new_from_stock(GTK_STOCK_CONNECT);
	gtk_widget_set_sensitive(button, FALSE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area),
								button);
	gtk_widget_show(button);

	g_object_set_data(G_OBJECT(notebook), "connect", button);

	create_selection(notebook, client, filter);

	create_connect(notebook);

	g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(connect_callback), notebook);

	bluetooth_client_discover_devices(client, NULL);

	gtk_dialog_run(GTK_DIALOG(dialog));

	if (cleanup)
		cleanup(notebook);

	gtk_widget_destroy(dialog);

	bluetooth_client_cancel_discovery(client, NULL);

	g_object_unref(client);
}
