from shutil import rmtree

from bzrlib import (
    bzrdir,
    revision as _mod_revision,
    )
from bzrlib.branch import Branch
from bzrlib.errors import NoWorkingTree, NotBranchError
from bzrlib.workingtree import WorkingTree

from errors import (NotCheckout, UncommittedCheckout, ParentMissingRevisions,
                    NoParent)


def zap(path, remove_branch=False, allow_modified=False):
    try:
        wt = bzrdir.BzrDir.open(path).open_workingtree(path,
                                                       recommend_upgrade=False)
    except (NoWorkingTree, NotBranchError):
        raise NotCheckout(path)
    tree_base = wt.bzrdir.transport.base
    branch = wt.branch
    branch_base = branch.bzrdir.transport.base
    if tree_base == branch_base:
        raise NotCheckout(path)
    if not allow_modified:
        delta = wt.changes_from(wt.basis_tree(), want_unchanged=False)
        if delta.has_changed():
            raise UncommittedCheckout()
    if remove_branch:
        parent_loc = branch.get_parent()
        if parent_loc is None:
            raise NoParent()
        parent = Branch.open(parent_loc)
        last_revision = _mod_revision.ensure_null(parent.last_revision())
        p_ancestry = parent.repository.get_ancestry(last_revision)
        if (last_revision != _mod_revision.NULL_REVISION and
            branch.last_revision() not in p_ancestry):
            raise ParentMissingRevisions(branch.get_parent())
    rmtree(path)
    if remove_branch:
        t = branch.bzrdir.transport
        while t.base != branch_base:
            t = t.clone('..')
        t = t.clone('..')
        t.delete_tree('')


def test_suite():
    import os
    from unittest import makeSuite

    from bzrlib.tests import TestCaseInTempDir

    class TestZap(TestCaseInTempDir):

        def make_checkout(self):
            wt = bzrdir.BzrDir.create_standalone_workingtree('source')
            return wt.branch.create_checkout('checkout', lightweight=True)

        def make_checkout2(self):
            wt = self.make_checkout()
            wt2 = wt.branch.bzrdir.sprout('source2').open_workingtree()
            return wt2.branch.create_checkout('checkout2', lightweight=True)

        def test_is_checkout(self):
            self.assertRaises(NotCheckout, zap, '.')
            wt = bzrdir.BzrDir.create_standalone_workingtree('.')
            self.assertRaises(NotCheckout, zap, '.')

        def test_zap_works(self):
            self.make_checkout()
            self.assertIs(True, os.path.exists('checkout'))
            zap('checkout')
            self.assertIs(False, os.path.exists('checkout'))
            self.assertIs(True, os.path.exists('source'))

        def test_zap_branch(self):
            self.make_checkout2()
            base = WorkingTree.open('checkout').branch.base
            self.assertIs(True, os.path.exists('checkout'))
            self.assertRaises(NoParent, zap, 'checkout', remove_branch=True)
            self.assertIs(True, os.path.exists('checkout'))
            self.assertIs(True, os.path.exists('source'))
            zap('checkout2', remove_branch=True)
            self.assertIs(False, os.path.exists('checkout2'))
            self.assertIs(False, os.path.exists('source2'))

        def test_checks_modified(self):
            checkout = self.make_checkout()
            os.mkdir('checkout/foo')
            checkout.add('foo')
            self.assertRaises(UncommittedCheckout, zap, 'checkout')
            checkout.commit('commit changes to branch')
            zap('checkout')

        def test_allow_modified(self):
            checkout = self.make_checkout()
            os.mkdir('checkout/foo')
            checkout.add('foo')
            self.assertRaises(UncommittedCheckout, zap, 'checkout')
            zap('checkout', allow_modified=True)

    return makeSuite(TestZap)
