;;;; chicken-setup
;
; Copyright (c) 2000-2004, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Unter den Gleichen 1
; 37130 Gleichen
; Germany


(declare
  (run-time-macros)
  (hide unpack with-ext run-setup-script simple-install install write-info verbose install-bin-path
	repository-tree repository-hosts last-decent-host fetch-file-from-net download-data
	download-repository-tree fixpath fetch-only install-with-su copy-file compute-tmpdir
	temporary-directory chdir rmtmpdir list-installed tmpdir-created keep-stuff) 
  (foreign-declare #<<EOF
#include "c_defaults.h"
EOF

) )

(declare (uses srfi-1 regex utils posix tcp match-support format))


(include "parameters.scm")
(include "build.scm")
(include "chicken-more-macros.scm")
(include "chicken-match-macros.scm")


(define verbose #f)
(define fetch-only #f)
(define install-with-su #f)
(define temporary-directory #f)
(define tmpdir-created #f)
(define keep-stuff #f)

(define (yes-or-no? str . default)
  (let ([def (:optional default #f)])
    (let loop ()
      (printf "~%~A (yes/no) " str)
      (when def (printf "[~A] " def))
      (flush-output)
      (let ([ln (read-line)])
	(when (and def (string=? "" ln))
	  (set! ln def) )
	(cond [(string-ci=? "yes" ln) #t]
	      [(string-ci=? "no" ln) #f]
	      [else
	       (printf "~%Please enter \"yes\" or \"no\".~%")
	       (loop) ] ) ) ) ) )

(define (patch which rx subst)
  (when verbose (printf "patching ~A ...~%" which))
  (match which
    [(from to) 
     (with-output-to-file to
       (lambda ()
	 (with-input-from-file from
	   (lambda ()
	     (let loop ()
	       (let ([ln (read-line)])
		 (unless (eof-object? ln)
		   (write-line (string-substitute rx subst ln #t)) 
		   (loop) ) ) ) ) ) ) ) ]
    [both
     (let ([tmp (create-temporary-file)])
       (patch (list both tmp) rx subst)
       (system* "mv ~A ~A" tmp both) ) ] ) )

(define run-verbose (make-parameter #t))

(define (fixpath prg)
  (cond [(and (string=? prg "csc") verbose) (make-pathname install-bin-path (string-append prg " -v"))]
	[(member prg installed-executables) (make-pathname install-bin-path prg)]
	[else prg] ) )

(define (fixmaketarget file)
  (if (and (equal? "so" (pathname-extension file))
	   (not (string=? "so" ##sys#load-dynamic-extension)) )
      (pathname-replace-extension file ##sys#load-dynamic-extension)
      file) )

(define (run:execute explist)
  (define (smooth lst)
    (let ([slst (map ->string lst)])
      (string-intersperse (cons (fixpath (car slst)) (cdr slst)) " ") ) )
  (for-each
   (lambda (cmd)
     (when (run-verbose) (printf "  ~A~%~!" cmd))
     (system* cmd) )
   (map smooth explist) ) )

(define-macro (run . explist)
  `(run:execute (list ,@(map (lambda (x) (list 'quasiquote x)) explist))) )

(define (make:find-matching-line str spec)
  (let ([match? (lambda (s) (string=? s str))])
    (let loop ([lines spec])
      (cond
       [(null? lines) #f]
       [else (let* ([line (car lines)]
		    [names (if (string? (car line))
			       (list (car line))
			       (car line))])
	       (if (ormap match? names)
		   line
		   (loop (cdr lines))))]))))

(define (make:form-error s p) (error (sprintf "~a: ~s" s p)))
(define (make:line-error s p n) (error (sprintf "~a: ~s for line: ~a" s p n)))

(define (make:check-spec spec)
  (and (or (list? spec) (make:form-error "specification is not a list" spec))
       (or (pair? spec) (make:form-error "specification is an empty list" spec))
       (andmap
	(lambda (line)
	  (and (or (and (list? line) (<= 2 (length line) 3))
		   (make:form-error "list is not a list with 2 or 3 parts" line))
	       (or (or (string? (car line))
		       (and (list? (car line))
			    (andmap string? (car line))))
		   (make:form-error "line does not start with a string or list of strings" line))
	       (let ([name (car line)])
		 (or (list? (cadr line))
		     (make:line-error "second part of line is not a list" (cadr line) name)
		     (andmap (lambda (dep)
			       (or (string? dep)
				   (make:form-error "dependency item is not a string" dep)))
			     (cadr line)))
		 (or (null? (cddr line))
		     (procedure? (caddr line))
		     (make:line-error "command part of line is not a thunk" (caddr line) name)))))
	spec)))

(define (make:check-argv argv)
  (or (string? argv)
      (and (vector? argv)
	   (andmap string? (vector->list argv)))
      (error "argument is not a string or string vector" argv)))

(define (make:make/proc/helper spec argv)
  (make:check-spec spec)
  (make:check-argv argv)
  (letrec ([made '()]
	   [exn? (condition-predicate 'exn)]
	   [exn-message (condition-property-accessor 'exn 'message)]
	   [make-file
	    (lambda (s indent)
	      (let* ([line (make:find-matching-line s spec)]
		     [s2 (fixmaketarget s)] 
		     [date (and (file-exists? s2)
				(file-modification-time s2))])

		(when verbose
		  (printf "make: ~achecking ~a~%" indent s2))

		(if line
		    (let ([deps (cadr line)])
		      (for-each (let ([new-indent (string-append " " indent)])
				  (lambda (d) (make-file d new-indent)))
				deps)
		      (let ([reason
			     (or (not date)
				 (ormap (lambda (dep)
					  (let ([dep2 (fixmaketarget dep)])
					    (unless (file-exists? dep2)
					      (error (sprintf "dependancy ~a was not made~%" dep2)))
					    (and (> (file-modification-time dep2) date)
						 dep2)) )
					deps))])
			(when reason
			  (let ([l (cddr line)])
			    (unless (null? l)
			      (set! made (cons s made))
			      (when verbose
				(printf "make: ~amaking ~a~a~%"
					indent
					s2
					(cond
					 [(not date)
					  (string-append " because " s2 " does not exist")]
					 [(string? reason)
					  (string-append " because " reason " changed")]
					 [else
					  (string-append (sprintf " just because (reason: ~a date: ~a)" 
								  reason date))]) ) )
			      (handle-exceptions exn
				  (begin
				    (printf "make: Failed to make ~a: ~a~%"
					    (car line)
					    (if (exn? exn)
						(exn-message exn)
						exn))
				    (signal exn) )
				((car l))))))))
		    (unless date
		      (error (sprintf "don't know how to make ~a" s2))))))])
    (cond
     [(string? argv) (make-file argv "")]
     [(equal? argv '#()) (make-file (caar spec) "")]
     [else (for-each (lambda (f) (make-file f "")) (vector->list argv))])
    (when verbose
      (for-each (lambda (item)
		  (printf "make: made ~a~%" item))
	(reverse made)))) )

(define make/proc
  (case-lambda
   [(spec) (make:make/proc/helper spec '#())]
   [(spec argv) (make:make/proc/helper spec argv)]))

(define-macro (make spec . argv)
  (let ([make (lambda (spec argv)
		(let ([form-error (lambda (s . p) (apply error s spec p))])
		  (and (or (list? spec) (form-error "illegal specification (not a sequence)"))
		       (or (pair? spec) (form-error "empty specification"))
		       (andmap
			(lambda (line)
			  (and (or (and (list? line) (>= (length line) 2))
				   (form-error "clause does not have at least 2 parts" line))
			       (let ([name (car line)])
				 (or (list? (cadr line))
				     (make:line-error "second part of clause is not a sequence" (cadr line))))))
			spec))
		  `(make/proc (list ,@(map (lambda (line)
					     `(list ,(car line)
						    (list ,@(cadr line))
						    ,@(let ([l (cddr line)])
							(if (null? l)
							    '()
							    `((lambda ()
								,@l))))))
					   spec))
			      ,(if (vector? argv) `',argv (car argv)))))])
    (if (pair? argv)
	(make spec argv)
	(make spec '#())) ) )

(define (usage)
  (display #<<EOF
usage: chicken-setup [FILENAME | OPTION] ...

  -help                      show this text
  -version                   show version of this program
  -repository                prints the location of the extension repository
  -uninstall EXTENSION       remove extension from repository
  -host HOSTNAME[:PORT]      specify alternative host for downloading
  -list                      list installed extensions and exit
  -run FILENAME              load and execute given file
  -program-path DIRECTORY    specify path for installing executables or scripts
  -script FILENAME           execute script with remaining arguments and exit
  -fetch                     only download, don't extract, build or install
  -verbose                   be verbose

  Builds and installs extension libraries.

EOF
  )
  (exit) )

(define install-bin-path (foreign-value "C_INSTALL_BIN_HOME" c-string))

(define program-path (make-parameter install-bin-path))

(define (with-ext filename ext)
  (if (equal? (pathname-extension filename) ext) 
      filename
      (let ([f2 (pathname-replace-extension filename ext)])
	(and (file-exists? f2) f2) ) ) )

(define (run-setup-script filename)
  (when verbose (printf "executing ~A ...~%" filename))
  (load filename) )

(define (write-info id files info)
  (when verbose (printf "writing info ~A -> ~S ...~%" id info))
  (with-output-to-file (make-pathname (repository-path) (->string id) "setup")
    (lambda ()
      (pp `((files ,@files) ,@info)) ) ) )

(define (simple-install filename)
  (let ([so (pathname-replace-extension filename ##sys#load-dynamic-extension)])
    (run (csc -O2 -d0 -vs ,filename)
	 (cp ,so ,(repository-path)) )
    (write-info (pathname-strip-extension filename) (list (make-pathname (repository-path) so)) '())
    (unless keep-stuff (run (rm ,so)) ) ) )

(define (compute-tmpdir fname)
  (string-append fname ".dir") )

(define (chdir dir)
  (when verbose (printf "changing working directory to `~A'~%" dir))
  (change-directory dir) )

(define (rmtmpdir)
  (when tmpdir-created
    (chdir "..")
    (unless keep-stuff (run (rm -fr ,temporary-directory)) ) ) )

(define (unpack filename)
  (let ([tmpdir (compute-tmpdir filename)])
    (if (file-exists? tmpdir)
	(chdir tmpdir)
	(begin
	  (run (mkdir ,tmpdir))
	  (set! tmpdir-created #t)
	  (chdir tmpdir)
	  (run (gunzip -c ,(string-append "../" filename) |\|| tar xvf -)) ) )
    (set! temporary-directory tmpdir) ) )

(define (copy-file from to)
  (if install-with-su
      (run (su -c ,(sprintf "'cp ~A ~A'" from to)))
      (run (cp ,from ,to)) ) )

(define (install-extension id files #!optional (info '()))
  (let* ([files (if (list? files) files (list files))]
	 [rpath (repository-path)] 
	 [files (if (eq? (software-type) 'windows)
		    (map (lambda (f) 
			   (if (string=? (pathname-extension f) "so")
			       (pathname-replace-extension f ##sys#load-dynamic-extension)
			       f) )
			 files)
		    files) ] )
    (copy-file (string-intersperse files " ") rpath)
    (write-info id (map (cut make-pathname rpath <>) files) info) ) )

(define (install-program id files #!optional (info '()))
  (let ([files (if (list? files) files (list files))]
	[ppath (program-path)] )
    (copy-file (string-intersperse files " ") ppath)
    (write-info id (map (cut make-pathname ppath <>) files) info) ) )

(define (install-script id files #!optional (info '()))
  (let* ([files (if (list? files) files (list files))]
	 [ppath (program-path)] 
	 [pfiles (map (cut make-pathname ppath <>) files)] )
    (copy-file (string-intersperse files " ") ppath)
    (run (cmod a+x ,(string-intersperse pfiles " "))) 
    (write-info id pfiles info) ) )

(define (uninstall-extension ext)
  (let* ([info (extension-info ext)]
	 [files (and info (assq 'files info))] )
    (if files
	(begin
	  (printf "deleting ~A ...~%" ext)
	  (for-each 
	   (lambda (f)
	     (printf "  deleting ~A~%" f)
	     (delete-file* f) )
	   (cdr files) ) )
	(print "no files to uninstall") )
    (delete-file* (make-pathname (repository-path) (->string ext) "setup")) ) )

(define (test-compile code #!key (cflags "") (ldflags "") (verb verbose) (compile-only #f))
  (let* ([fname (create-temporary-file "c")]
	 [oname (pathname-replace-extension fname "o")]
	 [r (handle-exceptions ex -1
	      (with-output-to-file fname (cut display code))
	      (system 
	       (let ([cmd (sprintf "~A ~A ~A ~A ~A >/dev/null 2>&1"
				   cc
				   (if compile-only "-c" "")
				   cflags
				   fname
				   (if compile-only "" ldflags) ) ] )
		 (when verb (print cmd))
		 cmd) ) ) ] )
    (system (sprintf "rm -f ~A.*" fname))
    (zero? r) ) )

; Repository-format:
;
; ((NAME FILENAME REQUIRED-NAME ...) ...)

(define repository-hosts '(("www.call-with-current-continuation.org" "eggs" 80)))
(define repository-tree #f)
(define last-decent-host #f)

(define (download-repository-tree)
  (unless repository-tree
    (print "downloading catalog ...")
    (let loop ([hosts repository-hosts])
      (if (null? hosts)
	  (error "unable to connect")
	  (match hosts
	    [((host path port) . more)
	     (or (handle-exceptions ex
		     (begin (printf "could not connect to ~A.~%" host) #f)
		   (printf "downloading catalog from ~A ...~%" host)
		   (let-values ([(i o) (tcp-connect host port)])
		     (fprintf 
		      o
		      "GET /~A/repository HTTP/1.0\r\nHost: ~A\r\nConnection: close\r\nContent-length: 0\r\n\r\n" 
		      path host)
		     (let loop ()
		       (let ([ln (read-line i)])
			 (if (string=? "" ln)
			     (begin
			       (set! repository-tree (read i))
			       (set! last-decent-host (car hosts))
			       (close-input-port i)
			       (close-output-port o)
			       #t)
			     (loop) ) ) ) ) )
		 (loop more) ) ] ) ) ) ) )

(define (download-data hostdata item)
  (match hostdata
    [(host path port)
     (let ([fname (third (assq item repository-tree))])
       (printf "downloading ~A from ~A ...~%" fname hostdata)
       (let-values ([(i o) (tcp-connect host port)])
	 (fprintf o "GET /~A/~A HTTP/1.0\r\nHost: ~A\r\nConnection: close\r\nContent-length: 0\r\n\r\n" path fname host)
	 (let loop ()
	   (let ([ln (read-line i)])
	     (if (string=? "" ln)
		 (let ([data (read-string #f i)])
		   (close-input-port i)
		   (close-output-port o)
		   (with-output-to-file fname (cut display data)) ) 
		 (loop) ) ) ) ) ) ] ) )

(define (fetch-file-from-net ext)
  (define (requirements reqs)
    (if (null? reqs)
	'()
	(apply lset-union eq? reqs (map (lambda (r) (requirements (cdddr (assq r repository-tree)))) reqs)) ) )
  (and (yes-or-no? (sprintf "The extension ~A does not exists.\nDo you want to download it ?" ext) "yes")
       (begin (download-repository-tree) #t)
       (let ([a (assq (string->symbol ext) repository-tree)])
	 (if a
	     (let ([reqs (remove extension-info (requirements (cdddr a)))])
	       (when (pair? reqs)
		 (print "downloading required extensions ...")
		 (for-each (cut download-data last-decent-host <>) reqs)
		 (print "installing required extensions ...")
		 (for-each (cut install <>) (map ->string reqs)) )
	       (begin (download-data last-decent-host (first a)) #t) )
	     (error "extension does not exist in the repository" ext) ) ) ) )

(define (install filename)
  (let ([df (not fetch-only)])
    (let loop ()
      (cond [(and df (with-ext filename "setup")) => run-setup-script]
	    [(and df (with-ext filename "scm")) => simple-install]
	    [(and df (with-ext filename "egg")) => 
	     (lambda (f)
	       (unpack f)
	       (loop) ) ]
	    [(fetch-file-from-net filename) 
	     (when df (loop)) ] ) ) ) )

(define (list-installed)
  (for-each
   (lambda (f)
     (let ([info (extension-info f)])
       (format #t "~32A ~32@A~%" 
	       f
	       (or (and-let* ([v (assq 'version info)])
		     (sprintf "Version: ~A" (cadr v)) )
		   "") ) ) )
   (sort (delete-duplicates (grep "[^.].*\\.*" (map pathname-file (directory (repository-path)))) string=?)
	 string<?) ) )

(set! ##sys#user-read-hook
  (let ([read-char read-char]
	[read read]
	[old-hook ##sys#user-read-hook] )
    (lambda (char port)
      (if (char=? char #\!)
	  (begin
	    (read-line port)
	    (set! ##sys#she-bang-checked #t)
	    (read port) )
	  (old-hook char port) ) ) ) )

(define (main args)
  (if (null? args)
      (usage)
      (let loop ([args args])
	(match args
	  [((or "-help" "--help" "-h") . _) (usage)]
	  [("-uninstall" fname . more)
	   (uninstall-extension fname) 
	   (loop more) ]
	  [("-list" . _)
	   (list-installed)
	   (exit) ]
	  [("-run" fname . more)
	   (load fname)
	   (loop more) ]
	  [("-repository")
	   (print (repository-path))
	   (exit) ]
	  [("-repository" dir . more)
	   (repository-path dir)
	   (loop more) ]
	  [("--" . more)
	   (exit) ]
	  [("-program-path")
	   (print (program-path))
	   (exit) ]
	  [("-program-path" dir . more)
	   (program-path dir)
	   (loop more) ]
	  [("-version" . _)
	   (printf "chicken-setup - Version ~A, Build ~A~%" build-version build-number)
	   (exit) ]
	  [("-script" filename . args)
	   (command-line-arguments args)
	   (set! ##sys#she-bang-checked #f)
	   (load filename) 
	   (exit) ]
	  [("-fetch" . more)
	   (set! fetch-only #t)
	   (loop more) ]
	  [("-host" host . more)
	   (set! repository-hosts
	     (cons (match (string-match "(.+)\\:([0-9]+)" host)
		     [(_ host port) (list host "eggs" (string->number port))]
		     [_ (list host "eggs" 80)] )
		   repository-hosts) ) 
	   (loop more) ]
	  [("-keep" . more)
	   (set! keep-stuff #t)
	   (loop more) ]
	  [("-verbose" . more)
	   (set! verbose #t)
	   (loop more) ]
	  [(filename . more)
	   (when (and (> (string-length filename) 0) (char=? #\- (string-ref filename 0)))
	     (error "invalid option" filename) )
	   (install filename)
	   (loop more) ]
	  [() #f] ) ) ) )

(handle-exceptions ex (exit -1)
  (main (command-line-arguments)) 
  (rmtmpdir) )
