/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Gordon Allott <gord.allott@canonical.com>
 *
 */

#include "ctk-check-menu-item.h"
#include "glib.h"

#define CTK_CHECK_MENU_ITEM_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  CTK_TYPE_CHECK_MENU_ITEM, \
  CtkCheckMenuItemPrivate))

enum
{
  PROP_0,

  PROP_ACTIVE,
  PROP_INCONSISTENT
};

enum
{
  TOGGLED,

  LAST_SIGNAL
};

struct _CtkCheckMenuItemPrivate
{
  gboolean active;
  gboolean inconsistent;
};


void ctk_check_menu_item_on_activated (CtkCheckMenuItem *menuitem);

static guint check_menu_item_signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (CtkCheckMenuItem, ctk_check_menu_item, CTK_TYPE_MENU_ITEM);

static void
ctk_check_menu_item_init (CtkCheckMenuItem *menuitem)
{
  g_signal_connect (menuitem, "activated", G_CALLBACK (ctk_check_menu_item_on_activated), NULL);
}

static void
ctk_check_menu_item_finalize (GObject *object)
{
  /* TODO: Add deinitalization code here */

  G_OBJECT_CLASS (ctk_check_menu_item_parent_class)->finalize (object);
}

static void
ctk_check_menu_item_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  g_return_if_fail (CTK_IS_CHECK_MENU_ITEM (object));
  CtkCheckMenuItem *menuitem = CTK_CHECK_MENU_ITEM (object);

  switch (prop_id)
    {
      case PROP_ACTIVE:
        ctk_check_menu_item_set_active (menuitem, g_value_get_boolean (value));
        break;
      case PROP_INCONSISTENT:
        ctk_check_menu_item_set_inconsistent (menuitem, g_value_get_boolean (value));
        break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
ctk_check_menu_item_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  g_return_if_fail (CTK_IS_CHECK_MENU_ITEM (object));
  CtkCheckMenuItem *menuitem = CTK_CHECK_MENU_ITEM (object);
  switch (prop_id)
    {
      case PROP_ACTIVE:
        g_value_set_boolean (value, ctk_check_menu_item_get_active (menuitem));
        break;
      case PROP_INCONSISTENT:
        g_value_set_boolean (value, ctk_check_menu_item_get_inconsistent (menuitem));
        break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
ctk_check_menu_item_toggled (CtkCheckMenuItem *self)
{
  /* TODO: Add default signal handler implementation here */
}

static void
ctk_check_menu_item_class_init (CtkCheckMenuItemClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);

  object_class->finalize = ctk_check_menu_item_finalize;
  object_class->set_property = ctk_check_menu_item_set_property;
  object_class->get_property = ctk_check_menu_item_get_property;

  g_type_class_add_private (klass, sizeof (CtkCheckMenuItemPrivate));

  klass->toggled = ctk_check_menu_item_toggled;

  g_object_class_install_property (object_class,
                                   PROP_ACTIVE,
                                   g_param_spec_boolean ("active",
                                                        "active",
                                                        "Whether the menu item is checked.",
                                                        FALSE,
                                                        G_PARAM_READABLE | G_PARAM_WRITABLE));

  g_object_class_install_property (object_class,
                                   PROP_INCONSISTENT,
                                   g_param_spec_boolean ("inconsistent",
                                                        "inconsistent",
                                                        "Weather the menu item is in an \"inconsistent\" state",
                                                        FALSE,
                                                        G_PARAM_READABLE | G_PARAM_WRITABLE));

  check_menu_item_signals[TOGGLED] =
    g_signal_new ("toggled",
                  G_OBJECT_CLASS_TYPE (klass),
                  0,
                  G_STRUCT_OFFSET (CtkCheckMenuItemClass, toggled),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}

/* constructors */
CtkCheckMenuItem *
ctk_check_menu_item_new (void)
{
  CtkCheckMenuItem *item;
  item = g_object_new (CTK_TYPE_CHECK_MENU_ITEM,
                       "label", "",
                       NULL);
  return item;
}

/* constructors */
CtkCheckMenuItem *
ctk_check_menu_item_new_with_label (const gchar *label)
{
  CtkCheckMenuItem *item;
  item = g_object_new (CTK_TYPE_CHECK_MENU_ITEM,
                       "label", label,
                       NULL);
  return item;
}


gboolean
ctk_check_menu_item_get_inconsistent (CtkCheckMenuItem *menuitem)
{
  g_return_val_if_fail (CTK_IS_CHECK_MENU_ITEM (menuitem), FALSE);
  return CTK_CHECK_MENU_ITEM_GET_PRIVATE (menuitem)->inconsistent;
}

void
ctk_check_menu_item_set_inconsistent (CtkCheckMenuItem *menuitem,
                                      gboolean setting)
{
  g_return_if_fail (CTK_IS_CHECK_MENU_ITEM (menuitem));
  CtkCheckMenuItemPrivate *priv = CTK_CHECK_MENU_ITEM_GET_PRIVATE (menuitem);
  priv->inconsistent = setting;
  g_object_notify (G_OBJECT (menuitem), "inconsistent");
}

gboolean
ctk_check_menu_item_get_active (CtkCheckMenuItem *menuitem)
{
  g_return_val_if_fail (CTK_IS_CHECK_MENU_ITEM (menuitem), FALSE);

  gboolean ret = CTK_CHECK_MENU_ITEM_GET_PRIVATE (menuitem)->active;

  return ret;
}

void
ctk_check_menu_item_set_active (CtkCheckMenuItem *menuitem, gboolean is_active)
{
  g_return_if_fail (CTK_IS_CHECK_MENU_ITEM (menuitem));
  CtkCheckMenuItemPrivate *priv = CTK_CHECK_MENU_ITEM_GET_PRIVATE( menuitem);
  gboolean changed = FALSE;
  if (priv->active != is_active)
    changed = TRUE;

  priv->active = is_active;

  g_object_notify (G_OBJECT (menuitem), "active");

  if (changed)
    g_signal_emit (menuitem, check_menu_item_signals[TOGGLED], 0);
}


/* private functions */
void
ctk_check_menu_item_on_activated (CtkCheckMenuItem *menuitem)
{
  g_return_if_fail (CTK_IS_CHECK_MENU_ITEM (menuitem));
  CtkCheckMenuItemPrivate *priv = CTK_CHECK_MENU_ITEM_GET_PRIVATE (menuitem);

  ctk_check_menu_item_set_active (menuitem, !priv->active);
}
