/*
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Mirco Müller <mirco.mueller@canonical.com>
 */

#include <glib.h>
#include <stdlib.h>
#include <unistd.h>
#include <clutk/clutk.h>

static void test_layer_new (void);
static void test_layer_get_width (void);
static void test_layer_set_width (void);
static void test_layer_get_height (void);
static void test_layer_set_height (void);
static void test_layer_get_color (void);
static void test_layer_set_color (void);
static void test_layer_get_opacity (void);
static void test_layer_set_opacity (void);
static void test_layer_set_image_pixbuf (void);
static void test_layer_get_image_pixbuf (void);
static void test_layer_set_mask_pixbuf (void);
static void test_layer_get_mask_pixbuf (void);
static void test_layer_set_image_from_file (void);
static void test_layer_set_image_from_pixbuf (void);
static void test_layer_set_image_from_surface (void);
static void test_layer_get_image_id (void);
static void test_layer_set_mask_from_file (void);
static void test_layer_set_mask_from_pixbuf (void);
static void test_layer_set_mask_from_surface (void);
static void test_layer_get_image_id (void);
static void test_layer_get_mask_id (void);
static void test_layer_setget_image_repeat_mode (void);
static void test_layer_setget_mask_repeat_mode (void);
static void test_layer_is_valid (void);
static void test_layer_setget_enabled (void);
static void test_layer_actor_new (void);
static void test_layer_actor_setget_width (void);
static void test_layer_actor_setget_height (void);
static void test_layer_actor_get_num_layers (void);
static void test_layer_actor_add_layer (void);
static void test_layer_actor_get_layer (void);
static void test_layer_actor_flatten (void);
static void test_layer_actor_is_flattened (void);

void
test_layer_actor_support_create_suite (void)
{
#define DOMAIN "/LayerActor"

  g_test_add_func (DOMAIN"/layer new",
                   test_layer_new);

  g_test_add_func (DOMAIN"/layer get width",
                   test_layer_get_width);

  g_test_add_func (DOMAIN"/layer set width",
                   test_layer_set_width);

  g_test_add_func (DOMAIN"/layer get height",
                   test_layer_get_height);

  g_test_add_func (DOMAIN"/layer set height",
                   test_layer_set_height);

  g_test_add_func (DOMAIN"/layer get color",
                   test_layer_get_color);

  g_test_add_func (DOMAIN"/layer set color",
                   test_layer_set_color);

  g_test_add_func (DOMAIN"/layer get opacity",
                   test_layer_get_opacity);

  g_test_add_func (DOMAIN"/layer set opacity",
                   test_layer_set_opacity);

  g_test_add_func (DOMAIN"/layer set image-pixbuf",
                   test_layer_set_image_pixbuf);

  g_test_add_func (DOMAIN"/layer get image-pixbuf",
                   test_layer_get_image_pixbuf);

  g_test_add_func (DOMAIN"/layer set mask-pixbuf",
                   test_layer_set_mask_pixbuf);

  g_test_add_func (DOMAIN"/layer get mask-pixbuf",
                   test_layer_get_mask_pixbuf);

  g_test_add_func (DOMAIN"/layer set image from file",
                   test_layer_set_image_from_file);

  g_test_add_func (DOMAIN"/layer set image from pixbuf",
                   test_layer_set_image_from_pixbuf);

  g_test_add_func (DOMAIN"/layer set image from surface",
                   test_layer_set_image_from_surface);

  g_test_add_func (DOMAIN"/layer set mask from file",
                   test_layer_set_mask_from_file);

  g_test_add_func (DOMAIN"/layer set mask from pixbuf",
                   test_layer_set_mask_from_pixbuf);

  g_test_add_func (DOMAIN"/layer set mask from surface",
                   test_layer_set_mask_from_surface);

  g_test_add_func (DOMAIN"/layer get image-id",
                   test_layer_get_image_id);

  g_test_add_func (DOMAIN"/layer get mask-id",
                   test_layer_get_mask_id);

  g_test_add_func (DOMAIN"/layer set|get image repeat-mode",
                   test_layer_setget_image_repeat_mode);

  g_test_add_func (DOMAIN"/layer set|get mask repeat-mode",
                   test_layer_setget_mask_repeat_mode);

  g_test_add_func (DOMAIN"/layer is valid",
                   test_layer_is_valid);

  g_test_add_func (DOMAIN"/layer set|get enabled",
                   test_layer_setget_enabled);

  g_test_add_func (DOMAIN"/layer actor new",
                   test_layer_actor_new);

  g_test_add_func (DOMAIN"/layer actor set|get width",
                   test_layer_actor_setget_width);

  g_test_add_func (DOMAIN"/layer actor set|get height",
                   test_layer_actor_setget_height);

  g_test_add_func (DOMAIN"/layer actor get num-layers",
                   test_layer_actor_get_num_layers);

  g_test_add_func (DOMAIN"/layer actor add layer",
                   test_layer_actor_add_layer);

  g_test_add_func (DOMAIN"/layer actor get layer",
                   test_layer_actor_get_layer);

  g_test_add_func (DOMAIN"/layer actor flatten",
                   test_layer_actor_flatten);

  g_test_add_func (DOMAIN"/layer actor is flattened",
                   test_layer_actor_is_flattened);
}

static void
test_layer_new ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (200, 150, CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  g_assert (layer);
  g_object_unref (layer);
}

static void
test_layer_get_width ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (200, 150, CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_width (layer), ==, 200);
  g_object_unref (layer);
}

static void
test_layer_set_width ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (200, 150, CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_width (layer), ==, 200);
  ctk_layer_set_width (layer, 300);
  g_assert_cmpuint (ctk_layer_get_width (layer), ==, 300);
  g_object_unref (layer);
}

static void
test_layer_get_height ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (300, 250, CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_height (layer), ==, 250);
  g_object_unref (layer);
}

static void
test_layer_set_height ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (300, 250, CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_height (layer), ==, 250);
  ctk_layer_set_height (layer, 123);
  g_assert_cmpuint (ctk_layer_get_height (layer), ==, 123);
  g_object_unref (layer);
}

static void
test_layer_get_color ()
{
  CtkLayer*    layer = NULL;
  ClutterColor color = {123, 213, 192, 111};

  layer = ctk_layer_new (300, 250, CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  ctk_layer_get_color (layer, &color);
  g_assert_cmpint (color.red,   ==, 0);
  g_assert_cmpint (color.green, ==, 0);
  g_assert_cmpint (color.blue,  ==, 0);
  g_assert_cmpint (color.alpha, ==, 0);
  g_object_unref (layer);
}

static void
test_layer_set_color ()
{
  CtkLayer*    layer = NULL;
  ClutterColor color_a = {123, 213, 192, 111};
  ClutterColor color_b = {0, 0, 0, 0};

  layer = ctk_layer_new (300, 250, CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  ctk_layer_set_color (layer, &color_a);
  ctk_layer_get_color (layer, &color_b);
  g_assert_cmpint (color_b.red,   ==, 123);
  g_assert_cmpint (color_b.green, ==, 213);
  g_assert_cmpint (color_b.blue,  ==, 192);
  g_assert_cmpint (color_b.alpha, ==, 111);
  g_object_unref (layer);

}

static void
test_layer_get_opacity ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (300, 250, CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_opacity (layer), ==, 0);
  g_object_unref (layer);
}

static void
test_layer_set_opacity ()
{
  CtkLayer* layer   = NULL;

  layer = ctk_layer_new (300, 250, CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  ctk_layer_set_opacity (layer, 128);
  g_assert_cmpuint (ctk_layer_get_opacity (layer), ==, 128);
  g_object_unref (layer);
}

static void
test_layer_set_image_pixbuf ()
{
  CtkLayer*  layer = NULL;
  GdkPixbuf* pixbuf;
  GdkPixbuf* new_pixbuf;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  pixbuf = ctk_layer_get_image_pixbuf (layer);
  g_assert (!pixbuf);

  new_pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 20, 20);
  ctk_layer_set_image_pixbuf (layer, new_pixbuf);
  pixbuf = ctk_layer_get_image_pixbuf (layer);

  g_assert_cmpint (gdk_pixbuf_get_width (pixbuf),
                   ==,
                   gdk_pixbuf_get_width (new_pixbuf));
  g_assert_cmpint (gdk_pixbuf_get_height (pixbuf),
                   ==,
                   gdk_pixbuf_get_height (new_pixbuf));
  g_assert_cmpint (gdk_pixbuf_get_n_channels (pixbuf),
                   ==,
                   gdk_pixbuf_get_n_channels (new_pixbuf));
  g_assert_cmpint (gdk_pixbuf_get_bits_per_sample (pixbuf),
                   ==,
                   gdk_pixbuf_get_bits_per_sample (new_pixbuf));

  g_object_unref (layer);
  g_object_unref (pixbuf);
}

static void
test_layer_get_image_pixbuf ()
{
  CtkLayer*  layer = NULL;
  GdkPixbuf* pixbuf;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  pixbuf = ctk_layer_get_image_pixbuf (layer);
  g_assert (!pixbuf);
  g_object_unref (layer);
}

static void
test_layer_set_mask_pixbuf ()
{
  CtkLayer*  layer = NULL;
  GdkPixbuf* pixbuf;
  GdkPixbuf* new_pixbuf;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  pixbuf = ctk_layer_get_mask_pixbuf (layer);
  g_assert (!pixbuf);

  new_pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 20, 20);
  ctk_layer_set_mask_pixbuf (layer, new_pixbuf);
  pixbuf = ctk_layer_get_mask_pixbuf (layer);

  g_assert_cmpint (gdk_pixbuf_get_width (pixbuf),
                   ==,
                   gdk_pixbuf_get_width (new_pixbuf));
  g_assert_cmpint (gdk_pixbuf_get_height (pixbuf),
                   ==,
                   gdk_pixbuf_get_height (new_pixbuf));
  g_assert_cmpint (gdk_pixbuf_get_n_channels (pixbuf),
                   ==,
                   gdk_pixbuf_get_n_channels (new_pixbuf));
  g_assert_cmpint (gdk_pixbuf_get_bits_per_sample (pixbuf),
                   ==,
                   gdk_pixbuf_get_bits_per_sample (new_pixbuf));

  g_object_unref (layer);
  g_object_unref (pixbuf);
}

static void
test_layer_get_mask_pixbuf ()
{
  CtkLayer*  layer = NULL;
  GdkPixbuf* pixbuf;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  pixbuf = ctk_layer_get_mask_pixbuf (layer);
  g_assert (!pixbuf);
  g_object_unref (layer);
}

static void
test_layer_set_image_from_file ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_image_id (layer), ==, 0);
  ctk_layer_set_image_from_file (layer, TESTDIR"/20x20.png");
  g_assert_cmpuint (ctk_layer_get_image_id (layer), !=, 0);
  g_object_unref (layer);
}

static void
test_layer_set_image_from_pixbuf ()
{
  CtkLayer*  layer  = NULL;
  GdkPixbuf* pixbuf = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_image_id (layer), ==, 0);
  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 20, 20);
  ctk_layer_set_image_from_pixbuf (layer, pixbuf);
  g_assert_cmpuint (ctk_layer_get_image_id (layer), !=, 0);

  g_object_unref (layer);
  g_object_unref (pixbuf);
}

static void
test_layer_set_image_from_surface ()
{
  CtkLayer*        layer   = NULL;
  cairo_surface_t* surface = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_image_id (layer), ==, 0);
  surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, 20, 20);
  g_assert_cmpint (cairo_surface_status (surface), ==, CAIRO_STATUS_SUCCESS);
  ctk_layer_set_image_from_surface (layer, surface);
  g_assert_cmpuint (ctk_layer_get_image_id (layer), !=, 0);

  g_object_unref (layer);
  cairo_surface_destroy (surface);
}

static void
test_layer_set_mask_from_file ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_mask_id (layer), ==, 0);
  ctk_layer_set_mask_from_file (layer, TESTDIR"/20x20.png");
  g_assert_cmpuint (ctk_layer_get_mask_id (layer), !=, 0);
  g_object_unref (layer);
}

static void
test_layer_set_mask_from_pixbuf ()
{
  CtkLayer*  layer  = NULL;
  GdkPixbuf* pixbuf = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_mask_id (layer), ==, 0);
  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 20, 20);
  ctk_layer_set_mask_from_pixbuf (layer, pixbuf);
  g_assert_cmpuint (ctk_layer_get_mask_id (layer), !=, 0);

  g_object_unref (layer);
  g_object_unref (pixbuf);
}

static void
test_layer_set_mask_from_surface ()
{
  CtkLayer*        layer   = NULL;
  cairo_surface_t* surface = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_mask_id (layer), ==, 0);
  surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, 20, 20);
  g_assert_cmpint (cairo_surface_status (surface), ==, CAIRO_STATUS_SUCCESS);
  ctk_layer_set_mask_from_surface (layer, surface);
  g_assert_cmpuint (ctk_layer_get_mask_id (layer), !=, 0);

  g_object_unref (layer);
  cairo_surface_destroy (surface);
}

static void
test_layer_get_image_id ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_image_id (layer), ==, 0);
  g_object_unref (layer);
}

static void
test_layer_get_mask_id ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpuint (ctk_layer_get_mask_id (layer), ==, 0);
  g_object_unref (layer);
}

static void
test_layer_setget_image_repeat_mode ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  ctk_layer_set_image_repeat_mode (layer, CTK_LAYER_REPEAT_X);
  g_assert_cmpuint (ctk_layer_get_image_repeat_mode (layer),
                    ==,
                    CTK_LAYER_REPEAT_X);
  ctk_layer_set_image_repeat_mode (layer, CTK_LAYER_REPEAT_Y);
  g_assert_cmpuint (ctk_layer_get_image_repeat_mode (layer),
                    ==,
                    CTK_LAYER_REPEAT_Y);
  ctk_layer_set_image_repeat_mode (layer,CTK_LAYER_REPEAT_X|CTK_LAYER_REPEAT_Y);
  g_assert_cmpuint (ctk_layer_get_image_repeat_mode (layer),
                    ==,
                    CTK_LAYER_REPEAT_X|CTK_LAYER_REPEAT_Y);
  g_object_unref (layer);
}

static void
test_layer_setget_mask_repeat_mode ()
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  ctk_layer_set_mask_repeat_mode (layer, CTK_LAYER_REPEAT_X);
  g_assert_cmpuint (ctk_layer_get_mask_repeat_mode (layer),
                    ==,
                    CTK_LAYER_REPEAT_X);
  ctk_layer_set_mask_repeat_mode (layer, CTK_LAYER_REPEAT_Y);
  g_assert_cmpuint (ctk_layer_get_mask_repeat_mode (layer),
                    ==,
                    CTK_LAYER_REPEAT_Y);
  ctk_layer_set_mask_repeat_mode (layer,CTK_LAYER_REPEAT_X|CTK_LAYER_REPEAT_Y);
  g_assert_cmpuint (ctk_layer_get_mask_repeat_mode (layer),
                    ==,
                    CTK_LAYER_REPEAT_X|CTK_LAYER_REPEAT_Y);
  g_object_unref (layer);
}

static void
test_layer_is_valid ()
{
  CtkLayer*  layer  = NULL;
  GdkPixbuf* pixbuf = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpint (ctk_layer_is_valid (layer), ==, TRUE);
  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 22, 21);
  ctk_layer_set_mask_from_pixbuf (layer, pixbuf);
  g_assert_cmpint (ctk_layer_is_valid (layer), ==, FALSE);

  g_object_unref (layer);
  g_object_unref (pixbuf);
}

static void
test_layer_setget_enabled (void)
{
  CtkLayer* layer = NULL;

  layer = ctk_layer_new (20, 20, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  g_assert_cmpint (ctk_layer_get_enabled (layer), ==, TRUE);
  ctk_layer_set_enabled (layer, FALSE);
  g_assert_cmpint (ctk_layer_get_enabled (layer), ==, FALSE);

  g_object_unref (layer);
}

static void
test_layer_actor_new ()
{
  ClutterActor* actor = NULL;

  actor = ctk_layer_actor_new (200, 200);
  g_assert (actor);
  g_object_unref (actor);
}

static void
test_layer_actor_setget_width ()
{
  ClutterActor* actor = NULL;

  actor = ctk_layer_actor_new (75, 63);
  g_assert_cmpuint (ctk_layer_actor_get_width (CTK_LAYER_ACTOR (actor)),
                    ==,
                    75);
  ctk_layer_actor_set_width (CTK_LAYER_ACTOR (actor), 300);
  g_assert_cmpuint (ctk_layer_actor_get_width (CTK_LAYER_ACTOR (actor)),
                    ==,
                    300);
  g_object_unref (actor);
}

static void
test_layer_actor_setget_height ()
{
  ClutterActor* actor = NULL;

  actor = ctk_layer_actor_new (123, 92);
  g_assert_cmpuint (ctk_layer_actor_get_height (CTK_LAYER_ACTOR (actor)),
                    ==,
                    92);
  ctk_layer_actor_set_height (CTK_LAYER_ACTOR (actor), 12);
  g_assert_cmpuint (ctk_layer_actor_get_height (CTK_LAYER_ACTOR (actor)),
                    ==,
                    12);
  g_object_unref (actor);
}

static void
test_layer_actor_get_num_layers ()
{
  ClutterActor* actor   = NULL;

  actor = ctk_layer_actor_new (123, 92);
  g_assert_cmpuint (ctk_layer_actor_get_num_layers (CTK_LAYER_ACTOR (actor)),
                    ==,
                    0);

  g_object_unref (actor);
}

static void
test_layer_actor_add_layer ()
{
  ClutterActor* actor   = NULL;
  CtkLayer*     layer_1 = NULL;
  CtkLayer*     layer_2 = NULL;

  actor = ctk_layer_actor_new (123, 92);
  g_assert_cmpuint (ctk_layer_actor_get_num_layers (CTK_LAYER_ACTOR (actor)),
                    ==,
                    0);

  layer_1 = ctk_layer_new (123, 92,CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  layer_2 = ctk_layer_new (123, 92,CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);

  ctk_layer_actor_add_layer (CTK_LAYER_ACTOR (actor), layer_1);
  ctk_layer_actor_add_layer (CTK_LAYER_ACTOR (actor), layer_2);

  g_assert_cmpuint (ctk_layer_actor_get_num_layers (CTK_LAYER_ACTOR (actor)),
                    ==,
                    2);

  g_object_unref (actor);
}

static void
test_layer_actor_get_layer ()
{
  ClutterActor* actor     = NULL;
  CtkLayer*     layer     = NULL;
  CtkLayer*     new_layer = NULL;

  actor = ctk_layer_actor_new (123, 92);
  g_assert_cmpuint (ctk_layer_actor_get_num_layers (CTK_LAYER_ACTOR (actor)),
                    ==,
                    0);

  layer = ctk_layer_new (123, 92, CTK_LAYER_REPEAT_NONE, CTK_LAYER_REPEAT_NONE);
  ctk_layer_actor_add_layer (CTK_LAYER_ACTOR (actor), layer);
  new_layer = ctk_layer_actor_get_layer (CTK_LAYER_ACTOR (actor), 0);
  g_assert (layer == new_layer);

  g_object_unref (actor);
}

static void
test_layer_actor_flatten ()
{
  /*ClutterActor* actor   = NULL;
  CtkLayer*     layer_1 = NULL;
  CtkLayer*     layer_2 = NULL;

  actor = ctk_layer_actor_new (123, 92);
  g_assert_cmpuint (ctk_layer_actor_get_num_layers (CTK_LAYER_ACTOR (actor)),
                    ==,
                    0);

  layer_1 = ctk_layer_new (123, 92,CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  layer_2 = ctk_layer_new (123, 92,CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);

  ctk_layer_actor_add_layer (CTK_LAYER_ACTOR (actor), layer_1);
  ctk_layer_actor_add_layer (CTK_LAYER_ACTOR (actor), layer_2);

  ctk_layer_actor_flatten (CTK_LAYER_ACTOR (actor));

  g_object_unref (actor);*/
}

static void
test_layer_actor_is_flattened ()
{
  /*ClutterActor* actor   = NULL;
  CtkLayer*     layer_1 = NULL;
  CtkLayer*     layer_2 = NULL;

  actor = ctk_layer_actor_new (123, 92);
  g_assert_cmpint (ctk_layer_actor_is_flattened (CTK_LAYER_ACTOR (actor)),
                   ==,
                   FALSE);

  layer_1 = ctk_layer_new (123, 92,CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);
  layer_2 = ctk_layer_new (123, 92,CTK_LAYER_REPEAT_NONE,CTK_LAYER_REPEAT_NONE);

  ctk_layer_actor_add_layer (CTK_LAYER_ACTOR (actor), layer_1);
  ctk_layer_actor_add_layer (CTK_LAYER_ACTOR (actor), layer_2);

  ctk_layer_actor_flatten (CTK_LAYER_ACTOR (actor));
  g_assert_cmpint (ctk_layer_actor_is_flattened (CTK_LAYER_ACTOR (actor)),
                   ==,
                   TRUE);

  g_object_unref (actor);*/
}
