# Copyright (C) 2008, 2009, 2010  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Exceptions for the Computer Janitor daemon."""


from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'DuplicateCruftError',
    'LandmarkPackageError',
    'MissingLandmarkError',
    'NoSuchCruftError',
    'NonDownloadableError',
    'PermissionDeniedError',
    ]


import dbus
import computerjanitor


class DBusServiceBaseException(computerjanitor.Exception, dbus.DBusException):
    """Base class exception for the Computer Janitor DBus service."""


class PermissionDeniedError(DBusServiceBaseException):
    """Permission denied by policy."""


class CruftError(DBusServiceBaseException):
    """Cruft exceptions passed back to dbus client."""

    _errmsg = None

    def __init__(self, cruft_name):
        self.cruft_name = cruft_name

    def __str__(self):
        return self._errmsg.format(self)


class DuplicateCruftError(CruftError):
    """Duplicate cruft name with different cleanup."""

    _errmsg = 'Duplicate cruft with different cleanup: {0.cruft_name}'


class NoSuchCruftError(CruftError):
    """There is no cruft by the given name."""

    _errmsg = 'No such cruft: {0.cruft_name}'


class LandmarkPackageError(DBusServiceBaseException):
    """Base class for problems with the landmark packages."""

    _errmsg = None

    def __init__(self, package):
        self.package = package

    def __str__(self):
        # gettext translation needs to be called at run time.
        return self._errmsg.format(self)


class MissingLandmarkError(LandmarkPackageError):
    """A landmark package could not be found."""

    _errmsg = 'Landmark package {0.package} is missing'


class NonDownloadableError(LandmarkPackageError):
    """A landmark package is not downloadable."""

    _errmsg = 'Landmark package {0.package} is not downloadable'


class PackageCleanupError(DBusServiceBaseException):
    """Could not complete plugin post-cleanup."""

    _errmsg = 'Post-cleanup exception'
