# Copyright (C) 2008, 2009, 2010  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Test the cruft collector."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'test_suite',
    ]


import os
import shutil
import apt_pkg
import tempfile
import unittest
import pkg_resources
import multiprocessing

from computerjanitor.plugin import Plugin
from computerjanitord.collector import Collector
from computerjanitord.errors import DuplicateCruftError, PackageCleanupError
from computerjanitord.tests.test_application import ApplicationTestSetupMixin


LOCK_FILE = pkg_resources.resource_filename(
    'computerjanitord.tests',
    os.path.join('data', 'var', 'lib', 'dpkg', 'status'))


class MockCruft:
    """Mock cruft that supports the required `get_name()` interface."""

    def __init__(self, name):
        self.name = name

    def get_name(self):
        return self.name


class MockPlugin(Plugin):
    cruft_class = MockCruft

    def __init__(self, prefix, shortnames):
        super(MockPlugin, self).__init__()
        self.prefix = prefix
        self.shortnames = shortnames

    def get_cruft(self):
        for shortname in self.shortnames:
            yield self.cruft_class('{0}:{1}'.format(self.prefix, shortname))


class SawAppPlugin(Plugin):
    """All this plugin does is set a marker attribute on the `Application`.

    This proves that access to the application through the plugin works.
    """
    def get_cruft(self):
        self.app.saw_app_plugin = True
        return []


class LockingPlugin(Plugin):
    """All this plugin does is acquire a fake apt lock after cleanup."""

    def get_cruft(self):
        return []

    def post_cleanup(self):
        # If the lock cannot be acquired, an exception is raised.
        with apt_pkg.FileLock(LOCK_FILE):
            # Do something.
            pass


class MockPluginManager:
    def __init__(self, app, plugin_dirs):
        # Ignore plugin_dirs
        self.app = app

    def get_plugins(self):
        shortnames = ('one', 'two', 'three')
        for prefix in ('foo', 'bar', 'baz'):
            plugin = MockPlugin(prefix, shortnames)
            plugin.set_application(self.app)
            yield plugin
        for plugin_class in (SawAppPlugin, LockingPlugin):
            plugin = plugin_class()
            plugin.set_application(self.app)
            yield plugin


class MockCruftExtra(MockCruft):
    """Cruft with a different class."""


class MockPluginExtra(MockPlugin):
    """A mock plugin that returns cruft with a different class."""

    cruft_class = MockCruftExtra


class IgnoredDuplicateCruftPluginManager(MockPluginManager):
    """Add an additional piece of ignorable duplication cruft."""
    
    def __init__(self, app, plugin_dirs):
        self.app = app
        # Ignore plugin_dirs

    def get_plugins(self):
        yield MockPlugin('one', ('foo', 'bar'))
        yield MockPlugin('one', ('baz', 'foo'))


class BadDuplicateCruftPluginManager(MockPluginManager):
    """Add an additional piece of bad duplicate cruft."""
    def __init__(self, app, plugin_dirs):
        self.app = app
        # Ignore plugin_dirs

    def get_plugins(self):
        yield MockPlugin('one', ('foo', 'bar'))
        yield MockPluginExtra('one', ('baz', 'foo'))


class TestCollector(unittest.TestCase, ApplicationTestSetupMixin):
    """Test the cruft collector."""

    def setUp(self):
        # Set up the test data Application.
        ApplicationTestSetupMixin.setUp(self)
        self.tempdir = tempfile.mkdtemp()
        whitelist_dirs = (self.tempdir,)
        with open(os.path.join(self.tempdir, 'one.whitelist'), 'w') as fp:
            print >> fp, 'foo:two'
            print >> fp, 'bar:one'
            print >> fp, 'baz:three'
        self.collector = Collector(self.app, MockPluginManager, whitelist_dirs)

    def tearDown(self):
        shutil.rmtree(self.tempdir)
        ApplicationTestSetupMixin.tearDown(self)

    def test_cruft_collector(self):
        cruft_names = set(cruft.get_name() for cruft in self.collector.cruft)
        self.assertEqual(cruft_names, set(('foo:one', 'foo:three',
                                           'bar:two', 'bar:three',
                                           'baz:one', 'baz:two')))

    def test_plugin_needs_application(self):
        # SawAppPlugin returned by the MockPluginManager sets this attribute
        # on the Application.
        self.assertTrue(self.app.saw_app_plugin)

    def test_collector_name_mapping(self):
        cruft_keys = set(self.collector.cruft_by_name)
        cruft_names = set(cruft.get_name() for cruft in self.collector.cruft)
        self.assertEqual(cruft_keys, cruft_names)

    def test_cleanup_lock(self):
        # Pretend we're running synaptic at the same time.  We have to acquire
        # this fake-synaptic lock in a subprocess.
        lock_event = multiprocessing.Event()
        continue_event = multiprocessing.Event()
        class AptLockThread(multiprocessing.Process):
            def run(self):
                with apt_pkg.FileLock(LOCK_FILE):
                    lock_event.set()
                    continue_event.wait(1.0)
        apt_locker = AptLockThread()
        apt_locker.start()
        lock_event.wait(1.0)
        self.assertRaises(PackageCleanupError, self.collector.clean, [])
        continue_event.set()


class TestDuplicateCruftCollector(
    unittest.TestCase, ApplicationTestSetupMixin):

    def setUp(self):
        # Set up the test data Application.
        ApplicationTestSetupMixin.setUp(self)

    def tearDown(self):
        ApplicationTestSetupMixin.tearDown(self)

    def test_duplicate_cruft_error(self):
        self.assertRaises(DuplicateCruftError, Collector,
                          self.app, BadDuplicateCruftPluginManager, [])

    def test_duplicate_cruft_error_message(self):
        try:
            Collector(self.app, BadDuplicateCruftPluginManager, [])
        except DuplicateCruftError as error:
            self.assertEqual(
                str(error),
                'Duplicate cruft with different cleanup: one:foo')
        else:
            raise AssertionError('DuplicateCruftError expected')

    def test_ignored_duplicate_cruft(self):
        collector = Collector(self.app, IgnoredDuplicateCruftPluginManager, [])
        self.assertEqual(list(cruft.get_name() for cruft in collector.cruft),
                         ['one:foo', 'one:bar', 'one:baz'])


def test_suite():
    suite = unittest.TestSuite()
    suite.addTests(unittest.makeSuite(TestCollector))
    suite.addTests(unittest.makeSuite(TestDuplicateCruftCollector))
    return suite
