# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Test the plugin application interfaces."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'ApplicationTestSetupMixin',
    'test_suite',
    ]


import os
import apt
import apt_pkg
import unittest
import warnings
import pkg_resources

from contextlib import contextmanager

import computerjanitord.application

from computerjanitord.application import (
    Application, MissingLandmarkError, NonDownloadableError)


@contextmanager
def landmarks(*packages):
    # Hack the module global list of known landmark packages.
    old_landmarks = computerjanitord.application.LANDMARK_PACKAGES[:]
    computerjanitord.application.LANDMARK_PACKAGES[:] = packages
    yield
    computerjanitord.application.LANDMARK_PACKAGES[:] = old_landmarks


class MockCruft:
    def __init__(self, name):
        self.name = name

    def get_name(self):
        warnings.warn('.get_name() is deprecated; use .name',
                      DeprecationWarning)
        return self.name


class ApplicationTestSetupMixin:
    """Set up an `Application` instance with test data in its apt_cache."""

    def setUp(self):
        self.data_dir = os.path.abspath(
            pkg_resources.resource_filename('computerjanitord.tests', 'data'))
        # Make the test insensitive to the platform's architecture.
        apt_pkg.Config.Set('APT::Architecture', 'i386')
        self.cache = apt.Cache(rootdir=self.data_dir)
        self.app = Application(self.cache)

    def tearDown(self):
        # Clear the cache.
        cache_dir = os.path.join(self.data_dir, 'var', 'cache', 'apt')
        for filename in os.listdir(cache_dir):
            if filename.endswith('.bin'):
                os.remove(os.path.join(cache_dir, filename))


class TestApplication(unittest.TestCase, ApplicationTestSetupMixin):
    """Test the `Application` interface."""

    def setUp(self):
        ApplicationTestSetupMixin.setUp(self)

    def tearDown(self):
        ApplicationTestSetupMixin.tearDown(self)

    def test_verify_apt_cache_good_path(self):
        # All essential packages are in the cache by default.
        self.assertEqual(self.app.verify_apt_cache(), None)

    def test_verify_apt_cache_with_nondownloadable_landmark(self):
        # Test that a missing landmark file causes an exception.
        with landmarks('gzip', 'dash-nodownload'):
            self.assertRaises(NonDownloadableError, self.app.verify_apt_cache)

    def test_verify_apt_cache_with_missing_landmark(self):
        with landmarks('gzip', 'dash', 'i-am-not-here'):
            self.assertRaises(MissingLandmarkError, self.app.verify_apt_cache)


def test_suite():
    suite = unittest.TestSuite()
    suite.addTests(unittest.makeSuite(TestApplication))
    return suite
