# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Test the whitelister."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'test_suite',
    ]


import os
import shutil
import tempfile
import unittest

from computerjanitord.whitelist import Whitelist


class MockCruft:
    """Mock cruft that supports the required `get_name()` interface."""

    def __init__(self, shortname):
        self.name = 'foo:' + shortname

    def get_name(self):
        return self.name


class TestWhitelister(unittest.TestCase):
    """Test the `Whitelist` class."""

    def setUp(self):
        self.tempdir_1 = tempfile.mkdtemp()
        self.tempdir_2 = tempfile.mkdtemp()
        self.whitelist_dirs = (self.tempdir_1, self.tempdir_2)

    def tearDown(self):
        shutil.rmtree(self.tempdir_1)
        shutil.rmtree(self.tempdir_2)

    def test_read_single_whitelist_file(self):
        # Read a .whitelist file from one directory.
        with open(os.path.join(self.tempdir_1, 'one.whitelist'), 'w') as fp:
            print >> fp, 'foo:one'
        whitelist = Whitelist(self.whitelist_dirs)
        self.assertEqual(whitelist.whitelist, set(['foo:one']))

    def test_read_bigger_whitelist_file(self):
        with open(os.path.join(self.tempdir_1, 'one.whitelist'), 'w') as fp:
            print >> fp, 'foo:one'
            print >> fp
            print >> fp, 'foo:two'
            print >> fp, '# foo:three'
            print >> fp, 'foo:four'
        whitelist = Whitelist(self.whitelist_dirs)
        self.assertEqual(whitelist.whitelist,
                         set(['foo:one', 'foo:two', 'foo:four']))

    def test_read_multiple_whitelist_files(self):
        with open(os.path.join(self.tempdir_1, 'one.whitelist'), 'w') as fp:
            print >> fp, 'foo:one'
            print >> fp
            print >> fp, 'foo:two'
            print >> fp, '# foo:three'
            print >> fp, 'foo:four'
        with open(os.path.join(self.tempdir_2, 'two.whitelist'), 'w') as fp:
            print >> fp, 'foo:three'
            print >> fp
            print >> fp, 'foo:four'
        with open(os.path.join(self.tempdir_2, 'three.whitelist'), 'w') as fp:
            print >> fp, 'foo:five'
            print >> fp, 'foo:one'
        whitelist = Whitelist(self.whitelist_dirs)
        self.assertEqual(
            whitelist.whitelist,
            set(['foo:one', 'foo:two', 'foo:three', 'foo:four', 'foo:five']))

    def test_is_whitelisted(self):
        with open(os.path.join(self.tempdir_1, 'one.whitelist'), 'w') as fp:
            print >> fp, 'foo:one'
            print >> fp
            print >> fp, 'foo:two'
            print >> fp, '# foo:three'
            print >> fp, 'foo:four'
        with open(os.path.join(self.tempdir_2, 'two.whitelist'), 'w') as fp:
            print >> fp, 'foo:three'
            print >> fp
            print >> fp, 'foo:four'
        with open(os.path.join(self.tempdir_2, 'three.whitelist'), 'w') as fp:
            print >> fp, 'foo:five'
            print >> fp, 'foo:one'
        whitelist = Whitelist(self.whitelist_dirs)
        mock_cruft = [MockCruft(name) for name in ('zero', 'one', 'two',
                                                   'three', 'four', 'five',
                                                   'six', 'seven', 'eight')]
        cruft = set(cruft.get_name() for cruft in mock_cruft
                    if not whitelist.is_whitelisted(cruft))
        self.assertEqual(
            cruft, set(['foo:zero', 'foo:six', 'foo:seven', 'foo:eight']))


def test_suite():
    suite = unittest.TestSuite()
    suite.addTests(unittest.makeSuite(TestWhitelister))
    return suite
