/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by:
 *               Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *               Neil Jagdish Patel <neil.patel@canonical.com>
 */

/**
 * SECTION:dee-shared-model
 * @short_description: A #DeeModel that can synchronize with other
 *    #DeeSharedModel objects across D-Bus.
 * @include: dee.h
 *
 * #DeeSharedModel is created with a name (usually namespaced and unique to
 * your program(s)) which is used to locate other #DeeSharedModels created
 * with the same name through D-Bus, and will keep synchronized  with them.
 *
 * This allows to you build MVC programs with a sane model API, but have the
 * controller (or multiple views) in a seperate process.
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <memory.h>
#include <time.h>
#include <unistd.h>
#include <gobject/gvaluecollector.h>
#include <dbus/dbus-gtype-specialized.h>

#include "dee-peer.h"
#include "dee-model.h"
#include "dee-proxy-model.h"
#include "dee-sequence-model.h"
#include "dee-shared-model.h"
#include "dee-versioned-model.h"
#include "dee-marshal.h"
#include "trace-log.h"

//static void dee_shared_model_model_iface_init (DeeModelIface *iface);
G_DEFINE_TYPE (DeeSharedModel,
               dee_shared_model,
               DEE_TYPE_PROXY_MODEL);

#define DEE_SHARED_MODEL_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, DEE_TYPE_SHARED_MODEL, DeeSharedModelPrivate))

#define DBUS_TYPE_G_VALUE_ARRAY_ARRAY (dbus_g_type_get_collection ("GPtrArray", dbus_g_type_get_collection("GPtrArray", G_TYPE_VALUE)))

/**
 * DeeSharedModelPrivate:
 *
 * Ignore this structure.
 **/
struct _DeeSharedModelPrivate
{
  DeePeer    *swarm;
  DBusGProxy *leader;
  gchar      *model_path;
  GHashTable *peers; /* mapping unique name -> proxy */

  DBusGProxy      *swarm_leader;
  DBusGConnection *connection;
  
  /* Buffer of DeeSharedModelRevisions that we keep in order to batch
   * our DBus signals. The invariant is that all buffered revisions
   * are of the same type */
  GSList     *revision_queue;
  guint       revision_queue_timeout_id;
  
  gboolean    invalidated;
  gboolean    found_first_peer;
  gboolean    suppress_remote_signals;
};

typedef struct
{
  /* The revision type is: ROWS_ADDED, ROWS_REMOVED, or ROWS_CHANGED */
  gint        type;
  guint32     pos;
  guint64     seqnum;
  GPtrArray  *row;
} DeeSharedModelRevision;

/* Globals */
static GQuark           dee_shared_model_error_quark       = 0;
static gboolean         dee_shared_model_types_initialized = FALSE;

enum
{
  PROP_0,
  PROP_SIZE,
  PROP_SEQNUM,
  PROP_COLUMNS,
  PROP_PEER
};

enum
{
  /* Public signal */
  READY,

  /* Used for dbus-glib hooks, not public */
  ROWS_REMOVED,
  ROWS_ADDED,
  ROWS_CHANGED,

  LAST_SIGNAL
};
static guint32 _signals[LAST_SIGNAL] = { 0 };

/* Forwards */
static void     on_peer_connected                      (DeeModel     *self);
static void     on_peer_found                          (DeeModel     *self,
                                                        const gchar   *name);
static void     on_peer_lost                           (DeeModel     *self,
                                                        const gchar   *name);

static gboolean _dee_shared_model_server_clone          (DeeModel      *self,
                                                         gchar        **columns,
                                                         guint64       *last_seqnum,
                                                         GPtrArray    **rows,
                                                         GArray       **seqnums,
                                                         GError       **error);

static gboolean _dee_shared_model_server_invalidate     (DeeModel     *self,
                                                         GError       **error);

static void     disconnect_and_unref_peer        (const gchar   *key,
                                                  DBusGProxy    *proxy,
                                                  DeeSharedModel     *model);

static DeeSharedModelRevision*
                dee_shared_model_revision_new   (guint              type,
                                                  guint32            pos,
                                                  guint64            seqnum,
                                                  GPtrArray         *row);

static void     dee_shared_model_revision_free  (DeeSharedModelRevision *rev);

static gboolean flush_revision_queue             (DeeModel         *self);

static void     enqueue_revision                 (DeeModel          *self,
                                                  guint              type,
                                                  guint32            pos,
                                                  guint64            seqnum,
                                                  GPtrArray         *row);

static GPtrArray* row_to_ptr_array               (DeeModel         *self,
                                                  DeeModelIter     *iter);

static void        dee_shared_model_assume_column_types (DeeModel *self,
                                                          GPtrArray *sample_row);
                                                          
static void        on_self_row_added             (DeeModel     *self,
                                                  DeeModelIter *iter);

static void        on_self_row_removed           (DeeModel     *self,
                                                  DeeModelIter *iter);

static void        on_self_row_changed           (DeeModel     *self,
                                                  DeeModelIter *iter);

static gboolean    emit_ready_signal             (DeeModel     *model);


#include "dee-shared-model-client.h"
#include "dee-shared-model-server.h"


/* Create a new revision. The revision will own @row */
static DeeSharedModelRevision*
dee_shared_model_revision_new (guint type, guint32 pos, guint64 seqnum, GPtrArray *row)
{ 
  DeeSharedModelRevision *rev;

  g_return_val_if_fail (row != NULL, NULL);

  // FIXME: Use g_slice instead og g_new
  rev = g_new0(DeeSharedModelRevision, 1);
  rev->type = type;
  rev->pos = pos;
  rev->seqnum = seqnum;
  rev->row = row;

  return rev;
}

/* Free all resources owned by a revision, and the revision itself */
static void
dee_shared_model_revision_free (DeeSharedModelRevision *rev)
{
  g_return_if_fail (rev != NULL);
  
  g_ptr_array_unref (rev->row);
  g_free (rev);
}

/* Emit all queued revisions in one signal on the bus.
 * Clears the revision_queue_timeout  if there is one set */
static gboolean
flush_revision_queue (DeeModel *self)
{
  DeeSharedModelPrivate *priv;
  GSList           *iter;
  GArray           *positions;
  GArray           *seqnums;
  GPtrArray        *rows;
  gint              n_revisions, rev_type;

  g_return_val_if_fail (DEE_IS_SHARED_MODEL (self), FALSE);
  priv = DEE_SHARED_MODEL (self)->priv;

  /* Clear the current timeout of we have one running */
  if (priv->revision_queue_timeout_id != 0)
    {
      g_source_remove (priv->revision_queue_timeout_id);
      priv->revision_queue_timeout_id = 0;
    }  

  n_revisions = g_slist_length (priv->revision_queue);
  if (n_revisions == 0)
    return FALSE;
  
  positions = g_array_sized_new (FALSE, FALSE, sizeof(guint32), n_revisions);
  seqnums = g_array_sized_new (FALSE, FALSE, sizeof(guint64), n_revisions);
  rows = g_ptr_array_sized_new (n_revisions);

  /* Since we always prepend to the queue we need to reverse it */
  priv->revision_queue = g_slist_reverse (priv->revision_queue);

  /* Build up positions, seqnums, and rows to emit with the signal */
  rev_type = ((DeeSharedModelRevision*)priv->revision_queue->data)->type;
  DeeSharedModelRevision *rev;
  for (iter = priv->revision_queue; iter; iter = iter->next)
    {
      rev = (DeeSharedModelRevision*)iter->data;
      if (rev_type != rev->type)
        {
          g_critical ("Revision type mismatch when flushing revision queue!");
          return FALSE;
        }
      g_array_append_val (positions, rev->pos);
      g_array_append_val (seqnums, rev->seqnum);
      g_ptr_array_add (rows, rev->row);
      //trace("Emit(%u): %u %"G_GUINT64_FORMAT,
      //      rev_type,
      //      g_array_index (positions, guint32, positions->len - 1),
      //      g_array_index (seqnums, guint64, seqnums->len - 1));
    }

  /* Throw the signal to the bus. We should be able to simply do it
   * like g_signal_emit (self, rev_type, 0, rows, positions, seqnums), but
   * dbus-glib can not handle that for some reasons. Hurra. */
  switch (rev_type)
    {
      case ROWS_REMOVED:
        g_signal_emit_by_name (self, "rows-removed", positions, seqnums);
        break;
      case ROWS_ADDED:
        g_signal_emit_by_name (self, "rows-added", rows, positions, seqnums);
        break;
      case ROWS_CHANGED:
        g_signal_emit_by_name (self, "rows-changed", rows, positions, seqnums);
        break;
      default:
        g_critical ("Unknown revision type %i", rev_type);
    }

  /* Free and reset the queue */
  g_slist_foreach (priv->revision_queue,
                   (GFunc)dee_shared_model_revision_free,
                   NULL);
  g_slist_free (priv->revision_queue);
  priv->revision_queue = NULL;

  g_array_free (positions, TRUE);
  g_array_free (seqnums, TRUE);
  g_ptr_array_free (rows, TRUE);
  
  return FALSE;
}

/* Prepare a revision to be emitted as a signal on the bus. The revisions
 * are queued up so that we can emit them in batches */
static void
enqueue_revision (DeeModel *self,
                  guint      type,
                  guint32    pos,
                  guint64    seqnum,
                  GPtrArray *row)
{
  DeeSharedModelPrivate  *priv;
  DeeSharedModelRevision *rev;

  g_return_if_fail (DEE_IS_SHARED_MODEL (self));
  priv = DEE_SHARED_MODEL (self)->priv;

  rev = dee_shared_model_revision_new (type, pos, seqnum, row);

  /* If this revision is of the same type as the other ones currently queued,
   * simply add it to the queue. If it is of a new type we need to flush the
   * queue first */
  if (priv->revision_queue != NULL &&
      ((DeeSharedModelRevision*)priv->revision_queue->data)->type != rev->type)
    {
      flush_revision_queue (self);
      
    }

  priv->revision_queue = g_slist_prepend (priv->revision_queue, rev);

  /* Flush the revision queue once in idle */
  if (priv->revision_queue_timeout_id == 0)
    {
      priv->revision_queue_timeout_id = g_idle_add ((GSourceFunc)flush_revision_queue,
                                                     self);
    }
}

/* GObject stuff */
static void
dee_shared_model_finalize (GObject *object)
{
  DeeSharedModelPrivate *priv = DEE_SHARED_MODEL (object)->priv;

  /* Flush any pending revisions */
  if (priv->revision_queue != NULL)
    {
      flush_revision_queue (DEE_MODEL(object));
    }
  
  if (priv->swarm)
    {
      g_object_unref (priv->swarm);
    }
  if (priv->leader)
    {
      g_object_unref (priv->leader);
    }
  if (priv->model_path)
    {
      g_free (priv->model_path);
    }
  if (priv->peers)
    {
      g_hash_table_foreach (priv->peers,
                            (GHFunc)disconnect_and_unref_peer,
                            object);
      g_hash_table_destroy (priv->peers);
      priv->peers = NULL;
    }
  if (priv->revision_queue)
    {
      g_slist_foreach (priv->revision_queue,
                       (GFunc)dee_shared_model_revision_free,
                       NULL);
      g_slist_free (priv->revision_queue);
      priv->revision_queue = NULL;
    }
  if (priv->connection)
    {
      dbus_g_connection_unref (priv->connection);
    }

  G_OBJECT_CLASS (dee_shared_model_parent_class)->finalize (object);
}

static void
dee_shared_model_set_property (GObject      *object,
                               guint         id,
                               const GValue *value,
                               GParamSpec   *pspec)
{
  DeeSharedModelPrivate *priv;
  
  priv = DEE_SHARED_MODEL (object)->priv;

  switch (id)
    {
    case PROP_SIZE:
    case PROP_SEQNUM:
    case PROP_COLUMNS:
      g_critical ("Setting a read-only property. This should never happen!");
      break;
    case PROP_PEER:
      /* Steal ref to peer */
      if (priv->swarm != NULL)
        g_object_unref (priv->swarm);
      priv->swarm = g_value_get_object (value);
      break;  
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_shared_model_get_property (GObject    *object,
                               guint       id,
                               GValue     *value,
                               GParamSpec *pspec)
{
  DeeSharedModelPrivate *priv;
  
  priv = DEE_SHARED_MODEL (object)->priv;

  switch (id)
    {
    case PROP_SIZE:
      g_value_set_uint (value,
                        dee_model_get_n_rows (DEE_MODEL (object)));
      break;
    case PROP_SEQNUM:
      g_value_set_uint64 (value,
                          dee_versioned_model_get_last_seqnum (DEE_MODEL (object)));
      break;
    case PROP_COLUMNS:
      g_value_take_string (value,
                          dee_model_build_col_spec (DEE_MODEL (object)));
      break;
    case PROP_PEER:
      g_value_set_object (value, priv->swarm);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_shared_model_class_init (DeeSharedModelClass *klass)
{
  GParamSpec    *pspec;
  GObjectClass  *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize     = dee_shared_model_finalize;  
  obj_class->set_property = dee_shared_model_set_property;
  obj_class->get_property = dee_shared_model_get_property;
  //obj_class->constructed  = dee_shared_model_constructed;

  _signals[READY] =
    g_signal_new ("ready",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                  0,
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  /* Dbus signals for communication between instances */  
    
  /**
   * DeeSharedModel::rows-removed: (skip)
   *
   * Private signal leaked because of how dbus-glib work with signals
   */
  _signals[ROWS_REMOVED] =
    g_signal_new ("rows-removed",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                  0,
                  NULL, NULL,
                  _dee_marshal_VOID__BOXED_BOXED,
                  G_TYPE_NONE, 2,
                  DBUS_TYPE_G_UINT_ARRAY,
                  DBUS_TYPE_G_UINT64_ARRAY);
  
  /**
   * DeeSharedModel::rows-added: (skip)
   *
   * Private signal leaked because of how dbus-glib work with signals
   */
  _signals[ROWS_ADDED] =
    g_signal_new ("rows-added",
                  G_OBJECT_CLASS_TYPE (klass),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                  0,
                  NULL, NULL,
                  _dee_marshal_VOID__BOXED_BOXED_BOXED,
                  G_TYPE_NONE, 3,
                  DBUS_TYPE_G_VALUE_ARRAY_ARRAY,
                  DBUS_TYPE_G_UINT_ARRAY,
                  DBUS_TYPE_G_UINT64_ARRAY);

  /**
   * DeeSharedModel::rows-changed: (skip)
   *
   * Private signal leaked because of how dbus-glib work with signals
   */
  _signals[ROWS_CHANGED] =
    g_signal_new ("rows-changed",
                  G_OBJECT_CLASS_TYPE (klass),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                  0,
                  NULL, NULL,
                  _dee_marshal_VOID__BOXED_BOXED_BOXED,
                  G_TYPE_NONE, 3,
                  DBUS_TYPE_G_VALUE_ARRAY_ARRAY,
                  DBUS_TYPE_G_UINT_ARRAY,
                  DBUS_TYPE_G_UINT64_ARRAY);

  /**
   * DeeSharedModel:size:
   *
   * The number of rows in the model
   */
  pspec = g_param_spec_uint ("size", "Size",
                             "Number of rows in the model",
                             0, G_MAXUINT32, 0,
                             G_PARAM_READABLE);
  g_object_class_install_property (obj_class, PROP_SIZE, pspec);

  /**
   * DeeSharedModel:seqnum:
   *
   * Expert: The last known sequence number
   */
  pspec = g_param_spec_uint64 ("seqnum", "Sequence number",
                               "The last known sequence number",
                               0, G_MAXUINT64, 0,
                               G_PARAM_READABLE);
  g_object_class_install_property (obj_class, PROP_SEQNUM, pspec);

  /**
   * DeeSharedModel:columns:
   *
   * Expert: The last known sequence number
   */
  pspec = g_param_spec_string ("columns", "Column types",
                              "The type signatures used for the columns of this model",
                              "",
                              G_PARAM_READABLE);
  g_object_class_install_property (obj_class, PROP_COLUMNS, pspec);

  /**
   * DeeSharedModel:peer: (transfer-full):
   *
   * The #DeePeer that this model uses to connect to the swarm
   */
  pspec = g_param_spec_object ("peer", "Peer",
                               "The peer object that monitors the swarm",
                               DEE_TYPE_PEER,
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY);
  g_object_class_install_property (obj_class, PROP_PEER, pspec);

  /* Register as a DBus object */
  dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass),
                                   &dbus_glib__dee_shared_model_server_object_info);

  /* Add private data */
  g_type_class_add_private (obj_class, sizeof (DeeSharedModelPrivate));
}

static void
dee_shared_model_init (DeeSharedModel *self)
{
  DeeSharedModelPrivate *priv;

  priv = self->priv = DEE_SHARED_MODEL_GET_PRIVATE (self);

  priv->invalidated = FALSE;
  priv->swarm = NULL;
  priv->model_path = NULL;
  priv->peers = g_hash_table_new_full (g_str_hash, g_str_equal,
                                       g_free, NULL);

  priv->revision_queue = NULL;
  priv->revision_queue_timeout_id = 0;
  
  if (!dee_shared_model_error_quark)
    dee_shared_model_error_quark = g_quark_from_string ("dbus-model-error");

  priv->connection = NULL;
  priv->swarm_leader = NULL;
  
  priv->suppress_remote_signals = FALSE;
  
  /* Connect to our own signals so we can queue up revisions to be emitted
   * on the bus */
  g_signal_connect (self, "row-added", G_CALLBACK (on_self_row_added), NULL);
  g_signal_connect (self, "row-removed", G_CALLBACK (on_self_row_removed), NULL);
  g_signal_connect (self, "row-changed", G_CALLBACK (on_self_row_changed), NULL);
}

static gboolean
sanity_check_seqnums (DeeModel *self, DBusGProxy *sender, GArray *seqnums)
{
  DeeSharedModelPrivate *priv;
  guint64                seqnum;

  g_return_val_if_fail (DEE_IS_SHARED_MODEL (self), FALSE);
  g_return_val_if_fail (sender != NULL, FALSE);
  g_return_val_if_fail (seqnums != NULL, FALSE);
  g_return_val_if_fail (seqnums->len > 0, FALSE);

  priv = DEE_SHARED_MODEL (self)->priv;
  seqnum = dee_versioned_model_get_last_seqnum (self);

  switch (dee_versioned_model_check_seqnums (self, seqnums))
    {
      case DEE_SEQNUM_STATE_IN_THE_FUTURE:
        /* Remote model out of sync. If we are leaders we must invalidate
         * the remote model. Otherwise ignore this batch of changes */
        if (dee_peer_is_swarm_leader (priv->swarm))
        {
          dbus_g_proxy_call_no_reply (sender,
                                      "Invalidate",
                                      G_TYPE_INVALID);
          trace_object (self, "Invalidated %s: seqnum %"G_GUINT64_FORMAT
                        " is in the future. Expected seqnum %"G_GUINT64_FORMAT,
                        dbus_g_proxy_get_bus_name (sender),
                        g_array_index (seqnums, guint64, 0), seqnum + 1);
        }      
        g_warning ("Remote model '%s' out of sync. "
                   "Expected seqnum %"G_GUINT64_FORMAT". "
                   "Ignoring %i added rows",
                   dbus_g_proxy_get_bus_name (sender),
                   seqnum + 1, seqnums->len);
        return FALSE;
      case DEE_SEQNUM_STATE_IN_THE_PAST:
        /* This model appears to have lost some revisions.
         * if it's not the leader invalidate it, if it is the leader
         * we invalidate the remote model */
        if (dee_peer_is_swarm_leader (priv->swarm))
        {
          dbus_g_proxy_call_no_reply (sender,
                                      "Invalidate",
                                      G_TYPE_INVALID);
          trace_object (self, "Invalidated %s: seqnum %"G_GUINT64_FORMAT
                        " is in the past. Expected seqnum %"G_GUINT64_FORMAT,
                        dbus_g_proxy_get_bus_name (sender),
                        g_array_index (seqnums, guint64, 0), seqnum + 1);
        }
        else 
        {
          g_warning ("Model out of sync. Cloning from leader");
          _dee_shared_model_server_invalidate (self, NULL);      
        }
        return FALSE;
      case DEE_SEQNUM_STATE_NOT_CONSECUTIVE:
        /* Remote model out of sync. If we are leaders we must invalidate
         * the remote model. Otherwise ignore this batch of changes */
        if (dee_peer_is_swarm_leader (priv->swarm))
        {
          dbus_g_proxy_call_no_reply (sender,
                                      "Invalidate",
                                      G_TYPE_INVALID);
          trace_object (self, "Invalidated %s: Non consecutive seqnums",
                        dbus_g_proxy_get_bus_name (sender));
        }
        else
        {
          trace_object (self, "Ignoring RowsAdded with non-consecutive seqnums");
        }
        return FALSE;
      case DEE_SEQNUM_STATE_OK:
        return TRUE;
    }
  
  g_return_val_if_reached (FALSE);
}

/* DBus callback: A peer added some rows */
static void
on_peer_rows_added (DBusGProxy *proxy,
                    GPtrArray  *rows,
                    GArray     *positions,
                    GArray     *seqnums,
                    DeeModel   *self)
{
  DeeSharedModelPrivate *priv;
  DeeModelIter          *iter;
  guint                  n_rows, n_columns;

  g_return_if_fail (DEE_IS_SHARED_MODEL (self));
  g_return_if_fail (rows->len == seqnums->len);
  g_return_if_fail (positions->len == seqnums->len);
  
  priv = DEE_SHARED_MODEL (self)->priv;
  n_rows = dee_model_get_n_rows (self);
  n_columns = dee_model_get_n_columns (self);

  /* Make sure we have registered our column types, and if not assume those
   * of the incoming change  */
  if (n_columns == 0 && rows->len > 0)
    dee_shared_model_assume_column_types (self,
                                          g_ptr_array_index (rows, 0));

  /* Sanity check incoming seqnums */
  if (!sanity_check_seqnums (self, proxy, seqnums))
    return;
  
  /* Invariant from this point on: local seqnum + 1 == seqnums[0] */
  
  GPtrArray *row;
  guint32    pos;
  guint64    seqnum;
  gint       i, j;
  
  priv->suppress_remote_signals = TRUE;
  for (i = 0; i < rows->len; i++)
    {
      row = g_ptr_array_index (rows, i);
      pos = g_array_index (positions, guint32, i);
      seqnum = g_array_index (seqnums, guint64, i);

      /* Insert row at requested pos. and write data  to it.
       * We freeze the signals in order not to emit the row-added signal
       * before we are done setting the row data */
      dee_model_freeze_signals (self);
      if (pos == 0)
        iter = dee_model_prepend (self, -1);
      else if (pos == n_rows)
        iter = dee_model_append (self, -1);
      else if (pos < n_rows)
        iter = dee_model_insert (self, pos, -1);
      else
        {
          g_warning ("Can not add row. Index out of bounds: %i", pos);
          dee_model_thaw_signals (self);
          continue;
        }
          
      for (j = 0; j < row->len; j++)
        {
          dee_model_set_value_silently ((DeeModel *)self, (DeeModelIter *)iter,
                                        j, (GValue*)g_ptr_array_index (row, j));
        }
      
      /* Emit queued signal for the added row */
      dee_model_thaw_signals (self);
      
      n_rows++;
    }
  priv->suppress_remote_signals = FALSE;
}

/* DBus callback: A peer removed some rows */
static void
on_peer_rows_removed (DBusGProxy *proxy,
                      GArray     *positions,
                      GArray     *seqnums,
                      DeeModel   *self)
{
  DeeSharedModelPrivate *priv;
  gint                   i;
  guint                  n_rows, n_columns;
  
  g_return_if_fail (DEE_IS_SHARED_MODEL (self));
  g_return_if_fail (positions->len == seqnums->len);
  g_return_if_fail (positions->len > 0);

  priv = DEE_SHARED_MODEL (self)->priv;
  n_rows = dee_model_get_n_rows (self);
  n_columns = dee_model_get_n_columns (self);
  
  /* Make sure we have registered our column types, and if not something
   * is probably askew, but it may not be a problem... */
  if (n_columns == 0 && positions->len > 0)
    g_warning ("Row removal detected, but we don't have column types yet");
  
  /* Sanity check incoming seqnums */
  if (!sanity_check_seqnums (self, proxy, seqnums))
    return;
  
  /* Invariant from this point on: local seqnum + 1 == seqnums[0] */

  guint32        pos;
  guint64        seqnum;
  DeeModelIter  *iter;
  
  priv->suppress_remote_signals = TRUE;
  for (i = 0; i < positions->len; i++)
    {
      
      pos = g_array_index (positions, guint32, i);
      seqnum = g_array_index (seqnums, guint64, i);

      if (pos >= n_rows)
        {
          g_warning ("Can not remove row. Index out of bounds: %i", pos);
          continue;
        } 

      iter = dee_model_get_iter_at_row (self, pos);
      dee_model_remove (self, iter);
      n_rows--;        
    }
  
  priv->suppress_remote_signals = FALSE;
}

/* DBus callback: A peer changed some rows */
static void
on_peer_rows_changed (DBusGProxy *proxy,
                      GPtrArray  *rows,
                      GArray     *positions,
                      GArray     *seqnums,
                      DeeModel   *self)
{
  DeeSharedModelPrivate *priv;
  DeeModelIter     *iter;
  guint             n_rows, n_columns;
  gint              i, j;
  guint64           seqnum;

  g_return_if_fail (DEE_IS_SHARED_MODEL (self));
  g_return_if_fail (rows->len == positions->len);
  g_return_if_fail (rows->len == seqnums->len);
  
  priv = DEE_SHARED_MODEL (self)->priv;
  n_rows = dee_model_get_n_rows (self);
  n_columns = dee_model_get_n_columns (self);

  /* Make sure we have registered our column types, and if not something
   * is probably askew, but it may not be a problem... */
  if (n_columns == 0 && positions->len > 0)
    g_warning ("Row change detected, but we don't have column types yet");
  
  /* Sanity check incoming seqnums */
  if (!sanity_check_seqnums (self, proxy, seqnums))
    return;
  
  /* Invariant from this point on: local seqnum + 1 == seqnums[0] */
  
  GPtrArray *row;
  guint32    pos;
  
  priv->suppress_remote_signals = TRUE;  
  for (i = 0; i < rows->len; i++)
    {
      pos = g_array_index (positions, guint32, i);
      seqnum = g_array_index (seqnums, guint64, i);      

      /* Sanity check bounds */
      if (pos >= n_rows)
        {
          g_warning ("Can not update row. "
                     "Index out of bounds: %"G_GUINT32_FORMAT,
                     pos);
          continue;
        }

      /* Sanity check the new seqnums */
      if (seqnum < dee_versioned_model_get_seqnum (self, pos))
        {
          g_warning ("Can not update row. "
                     "Seqnum is in the past: %"G_GUINT64_FORMAT,
                     seqnum);
          continue;
        }
      
      /* Update the row. We do this with set_value_silently() in order
       * to not trigger n_column emissions of row-changed */
      iter = dee_model_get_iter_at_row (self, pos);
      row = g_ptr_array_index (rows, i);
      for (j = 0; j < n_columns; j++)
        {
          dee_model_set_value_silently (self, iter, j,
                                        (GValue*)g_ptr_array_index (row, j));
        }

      /* Update the seqnum for the row, as well as out last seen seqnum.
       * We must do this manually since we use dee_model_set_value_silently() */
      dee_versioned_model_set_seqnum (self, pos, seqnum);
      dee_versioned_model_set_last_seqnum (self, seqnum);
      
      /* Emit signal manually because we did the
       * changes with set_value_silently() */
      g_signal_emit_by_name (self, "row-changed", iter);
    }
  
  priv->suppress_remote_signals = FALSE;    
}

static void
on_self_row_added (DeeModel *self, DeeModelIter *iter)
{
  DeeSharedModelPrivate *priv;
  guint32 pos;
  
  priv = DEE_SHARED_MODEL (self)->priv;

  if (!priv->suppress_remote_signals)
    {
      pos = dee_model_get_position (self, iter);
      enqueue_revision (self,
                        ROWS_ADDED,
                        pos,
                        dee_versioned_model_get_seqnum (self, pos),
                        row_to_ptr_array (self, iter));
    }
}

static void
on_self_row_removed (DeeModel *self, DeeModelIter *iter)
{
  DeeSharedModelPrivate *priv;
  guint32 pos;
  
  priv = DEE_SHARED_MODEL (self)->priv;

  if (!priv->suppress_remote_signals)
    {
      pos = dee_model_get_position (self, iter);
      enqueue_revision (self,
                        ROWS_REMOVED,
                        pos,
                        dee_versioned_model_get_seqnum (self, pos),
                        row_to_ptr_array (self, iter));
    }
}

static void
on_self_row_changed (DeeModel *self, DeeModelIter *iter)
{
  DeeSharedModelPrivate *priv;
  guint32 pos;
  
  priv = DEE_SHARED_MODEL (self)->priv;

  if (!priv->suppress_remote_signals)
    {
      pos = dee_model_get_position (self, iter);
      enqueue_revision (self,
                        ROWS_CHANGED,
                        pos,
                        dee_versioned_model_get_seqnum (self, pos),
                        row_to_ptr_array (self, iter));
    }
}

static gboolean
emit_ready_signal (DeeModel *model)
{
  g_signal_emit (model, _signals[READY], 0);
  return FALSE;
}

/* Clears all data in the model and resets it to start from scratch */
static void
reset_model (DeeModel *self)
{
  g_return_if_fail (DEE_IS_SHARED_MODEL (self));

  /* Make sure we don't have any buffered signals awaiting emission */
  flush_revision_queue (self);
  
  /* Emit 'removed' on all rows and free old row data */
  dee_model_clear (self);
  
  dee_versioned_model_set_last_seqnum (self, 0);
}

static void
import_model (DeeModel        *self,
              GPtrArray       *rows,
              GArray          *seqnums,
              gboolean         free_data)
{
  DeeSharedModelPrivate *priv;
  DeeModelIter          *iter;
  GPtrArray              *row;
  gint                    i, j, n_columns;
  guint64                 seqnum;
  
  priv = DEE_SHARED_MODEL (self)->priv;
  
  reset_model (self);

  for (i = 0; i < rows->len; i++)
    {
      /* Store row data. We freeze the signals in order not to emit the
       * addition before we are done setting the data on that row */
      dee_model_freeze_signals (self);      
      iter = dee_model_append (self, -1);
      row = g_ptr_array_index (rows, i);
      n_columns = MIN(dee_model_get_n_columns (self), row->len);
      
      /* We use set_values_silently() in order not to trigger
       * row-changed signals */
      for (j = 0; j < n_columns; j++)
        {
          GValue *value = g_ptr_array_index (row, j);
          dee_model_set_value_silently (self, iter, j, value);

          if (free_data)
            {
              g_value_unset (value);
              g_free (value);
            }
        }    

      if (free_data)
        g_ptr_array_free (row, TRUE);
        
      /* Set seqnum for new row */
      seqnum = g_array_index (seqnums, guint64, i);
      dee_versioned_model_set_seqnum (self, i, seqnum);
      dee_versioned_model_set_last_seqnum (self,
                      MAX (seqnum, dee_versioned_model_get_last_seqnum (self)));

      /* This will emit the queued row-added signal */
      dee_model_thaw_signals (self);
    }

  trace_object (self, "%u rows after import. Last seqnum is %u",
                dee_model_get_n_rows (self),
                dee_versioned_model_get_last_seqnum (self));

  if (free_data)
    {
      g_ptr_array_free (rows, TRUE);
      g_array_free (seqnums, TRUE);
    }
}

static void
clone_callback (DBusGProxy *proxy,
                gchar      *columns,
                guint64     last_seqnum,
                GPtrArray  *rows,
                GArray     *seqnums,
                GError     *error,
                gpointer    userdata)
{
  DeeModel              *self;
  DeeSharedModelPrivate *priv;
  gint              i;
  gboolean          column_types_added;

  g_return_if_fail (DEE_IS_SHARED_MODEL (userdata));
  g_return_if_fail (rows->len == seqnums->len);

  self = DEE_MODEL (userdata);
  priv = DEE_SHARED_MODEL (self)->priv;

  trace_object (self, "Model '%s' received %i rows in Clone()",
         dee_shared_model_get_swarm_name (DEE_SHARED_MODEL (self)), rows->len);
  
  /* If we don't have column types set yet, adopt those of the clone */
  if (dee_model_get_n_columns (self) < 1)
    {      
      dee_model_set_n_columns (self, strlen(columns));
      for (i = 0; columns[i]; i++)
        {
          dee_model_set_column_type (self, i,
                                     dee_model_get_type_for_signature (columns[i]));
        }
      column_types_added = TRUE;
    }
  else
    {
      column_types_added = FALSE;
    }
  
  /* If we are not leaders and this is our first contact, emit 'ready'.
   * Also if we are swarm leaders, but had no column type info, then
   * we are ready now */
  if (!dee_peer_is_swarm_leader (priv->swarm))
    {
      if (!priv->found_first_peer)
        emit_ready_signal (self);
    }
  else
    {
      if (column_types_added)
        emit_ready_signal (self);
    }
  
  priv->found_first_peer = TRUE;
  priv->invalidated = FALSE;
  
  priv->suppress_remote_signals = TRUE;
  import_model (self, rows, seqnums, TRUE);
  priv->suppress_remote_signals = FALSE;
  
  dee_versioned_model_set_last_seqnum (self, last_seqnum);
    
  g_free (columns);  
}

static void
on_peer_connected (DeeModel     *self)
{
  DeeSharedModelPrivate *priv;

  g_return_if_fail (DEE_IS_SHARED_MODEL (self));

  priv = DEE_SHARED_MODEL (self)->priv;
  
  /* If we are swarm leaders and we have column type info we are ready by now.
   * Otherwise we will be ready when we receive the model clone from the leader
   */
  if (dee_peer_is_swarm_leader (priv->swarm))
    {
      if (dee_model_get_n_columns (self) > 0)
        g_idle_add ((GSourceFunc)emit_ready_signal, self);
    }
  else
    {
      com_canonical_Dee_Model_clone_async (priv->leader,
                                           clone_callback,
                                           self);
    }
  
}

static void
on_peer_found (DeeModel      *self,
               const gchar   *name)
{
  DeeSharedModelPrivate *priv;
  DBusGProxy       *proxy;
  GError           *error = NULL;

  g_return_if_fail (DEE_IS_SHARED_MODEL (self));
  
  priv = DEE_SHARED_MODEL (self)->priv;

  proxy = dbus_g_proxy_new_for_name_owner (priv->connection,
                                           name,
                                           priv->model_path,
                                           DEE_SHARED_MODEL_DBUS_IFACE,
                                           &error);
  if (proxy == NULL)
    {
      g_warning ("Unable to connect to peer (%s): %s",
                 name,
                 error ? error->message: "Unknown");
      if (error)
        g_error_free (error);

      return;
    }

  /* The proxy is valid, so let's connect to all it's signals and add it to
   * the collection
   */
  g_hash_table_insert (priv->peers, g_strdup (name), proxy);

  dbus_g_proxy_add_signal (proxy,
                           "RowsAdded",
                           DBUS_TYPE_G_VALUE_ARRAY_ARRAY,
                           DBUS_TYPE_G_UINT_ARRAY,
                           DBUS_TYPE_G_UINT64_ARRAY,
                           G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (proxy, "RowsAdded",
                               G_CALLBACK (on_peer_rows_added), self, NULL);

  dbus_g_proxy_add_signal (proxy,
                           "RowsRemoved",
                           DBUS_TYPE_G_UINT_ARRAY,
                           DBUS_TYPE_G_UINT64_ARRAY,
                           G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (proxy, "RowsRemoved",
                               G_CALLBACK (on_peer_rows_removed), self, NULL);

  dbus_g_proxy_add_signal (proxy,
                           "RowsChanged",
                           DBUS_TYPE_G_VALUE_ARRAY_ARRAY,
                           DBUS_TYPE_G_UINT_ARRAY,
                           DBUS_TYPE_G_UINT64_ARRAY,
                           G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (proxy, "RowsChanged",
                               G_CALLBACK (on_peer_rows_changed), self, NULL);

  trace ("Peer model found: %s", name);
}

static void
disconnect_and_unref_peer (const gchar *key,
                           DBusGProxy  *proxy,
                           DeeSharedModel   *self)
{
  if (proxy)
    {      
      dbus_g_proxy_disconnect_signal (proxy,
                                      "RowsAdded",
                                      G_CALLBACK (on_peer_rows_added),
                                      self);
      dbus_g_proxy_disconnect_signal (proxy,
                                      "RowsRemoved",
                                      G_CALLBACK (on_peer_rows_removed),
                                      self);
      dbus_g_proxy_disconnect_signal (proxy,
                                      "RowsChanged",
                                      G_CALLBACK (on_peer_rows_changed),
                                      self);
      g_object_unref (proxy);
    }
}

static void
on_peer_lost (DeeModel     *self,
              const gchar   *name)
{
  DeeSharedModelPrivate *priv;
  DBusGProxy       *proxy;

  g_return_if_fail (DEE_IS_SHARED_MODEL (self));
  priv = DEE_SHARED_MODEL (self)->priv;

  proxy = g_hash_table_lookup (priv->peers, name);
  if (proxy)
    {
      /* We don't need to disconnect_and_unref_peer(name, proxy, self)
       * since dbus-glib (silently and undocumented) destroys the proxy
       * when it's created with dbus_g_proxy_new_for_name_owner() and the
       * name owner leaves the bus.
       * So we just unref it and hope that dbus-glib maintains said behaviour */
      g_object_unref (proxy);
      g_hash_table_remove (priv->peers, name);
      
      trace_object (self, "Peer model lost: %s", name);
    }
  else
    trace_object (self, "Unknown peer lost");
}


/* If we don't have any column types assigned assume those of @sample_row.
 * Only call this method if we don't already have column types */
static void
dee_shared_model_assume_column_types (DeeModel *self,
                                      GPtrArray *sample_row)
{
  DeeSharedModelPrivate *priv;
  gint                   i;

  priv = DEE_SHARED_MODEL (self)->priv;

  g_return_if_fail (dee_model_get_n_columns (self) == 0);
  g_return_if_fail (sample_row != NULL);

  dee_model_set_n_columns (self, sample_row->len);
  for (i = 0; i < sample_row->len; i++)
    {
      dee_model_set_column_type (self, i,
                              G_VALUE_TYPE (g_ptr_array_index (sample_row, i)));
    }
#ifdef ENABLE_TRACE_LOG
  gchar *colspec = dee_model_build_col_spec (self);
  trace_object (self, "Assumed column types '%s'", colspec);
  g_free (colspec);
#endif
}



static void
unset_and_free_gvalue (gpointer data)
{
  GValue *value = (GValue*)data;

  g_value_unset (value);
  g_free (value);
}

static GPtrArray *
row_to_ptr_array (DeeModel *self, DeeModelIter *iter)
{
  GPtrArray *array;
  gint       i, n_columns;

  g_return_val_if_fail (DEE_IS_SHARED_MODEL (self), NULL);

  n_columns = dee_model_get_n_columns (self);
  array = g_ptr_array_sized_new (n_columns);
  g_ptr_array_set_free_func (array, unset_and_free_gvalue);  
  
  for (i = 0; i < n_columns; i++)
    {
      GValue *value = g_new0 (GValue, 1);

      dee_model_get_value (self, iter, i, value);

      g_ptr_array_add (array, (gpointer)value);
    }

  return array;
}

static DeeModel*
dee_shared_model_new_real (const gchar *name, DeeModel* back_end)
{
  DeeModel              *self;
  DeePeer               *swarm;
  DeeSharedModelPrivate *priv;
  gchar                 *dummy;

  swarm = g_object_new (DEE_TYPE_PEER,
                        "swarm-name", name,
                        NULL);

  self = g_object_new (DEE_TYPE_SHARED_MODEL,
                       "back-end", back_end,
                       "peer", swarm,
                       NULL);
                       
  priv = DEE_SHARED_MODEL(self)->priv;
  
  /* Create a canonical object path from the well known swarm name */
  dummy = g_strdup (name);
  priv->model_path = g_strconcat ("/com/canonical/dee/model/",
                                  g_strdelimit (dummy, ".", '/'),
                                  NULL);
   
  g_free (dummy);
   
  return self;
}

/* Public Methods */

/**
 * dee_shared_model_new:
 * @name: A well known name to publish this model under. Models sharing this name
 *        will synchronize with each other
 * @n_columns: number of columns in the model
 * @VarArgs: @n_columns number of #GTypes
 *
 * Creates a new #DeeSharedModel with the specified @name, and with @n_columns
 * columns of the types passed in.
 *
 * A shared model created with this constructor will store row data in a
 * suitably picked memory backed model.
 *
 * In order to start synchronizing the new model with peer models you must call
 * dee_shared_model_connect() on it.
 *
 * Return value: a new #DeeSharedModel
 */
DeeModel*
dee_shared_model_new (const gchar *name, guint n_columns, ...)
{
  DeeModel  *back_end;
  va_list    args;
  gint       i;

  g_return_val_if_fail (name != NULL, NULL);
  g_return_val_if_fail (n_columns > 0, NULL);

  back_end = dee_sequence_model_new (0);
  dee_model_set_n_columns (back_end, n_columns);

  va_start (args, n_columns);
  for (i = 0; i < n_columns; i++)
    {
      GType type = va_arg (args, GType);

      if (!dee_model_check_type (type))
        {
          g_warning ("DeeSharedModel: %s is an invalid type", g_type_name (type));
          g_object_unref (back_end);
          return NULL;
        }

      dee_model_set_column_type (back_end, i, type);
    }
  va_end (args);

  return dee_shared_model_new_real (name, back_end);

  ;
}

/**
 * dee_shared_model_new_with_name:
 * @name: A well known name to publish this model under. Models sharing this name
 *        will synchronize with each other
 *
 * Create a new empty shared model without any column schema associated.
 * The column schema will be set in one of two ways: Firstly you may set it
 * manually with dee_model_set_n_columns() and dee_model_set_column_type(),
 * or secondly it will be set once the first rows are exchanged with a
 * peer model.
 * In the second case the
 * column schema will constructed exlicitly from the Clone metadata or
 * implicitly from the row value types which ever arrives first.
 *
 * A shared model created with this constructor will store row data in a
 * suitably picked memory backed model.
 * 
 * In order to start synchronizing the new model with peer models you must call
 * dee_shared_model_connect() on it.
 *
 * Return value: a new #DeeSharedModel
 */
DeeModel*
dee_shared_model_new_with_name (const gchar *name)
{
  DeeModel *self;

  g_return_val_if_fail (name != NULL, NULL);

  self = dee_shared_model_new_real (name,
                                    dee_sequence_model_new (0));

  return self;
}

/**
 * dee_shared_model_new_with_back_end:
 * @name: (transfer none): A well known name to publish this model under.
 *        Models sharing this name will synchronize with each other
 * @back_end: (transfer full): The #DeeModel that will actually store
 *            the model data. Ownership of the ref to @back_end is transfered to
 *            the shared model.
 *
 * Create a new shared model storing all data in @back_end.
 *
 * In order to start synchronizing the new model with peer models you must call
 * dee_shared_model_connect() on it.
 */
DeeModel*
dee_shared_model_new_with_back_end (const gchar *name, DeeModel *back_end)
{
  DeeModel *self;

  g_return_val_if_fail (name != NULL, NULL);

  self = dee_shared_model_new_real (name, back_end);

  return self;
}

/**
 * dee_shared_model_connect
 * @self: The model to export on the bus
 *
 * Connect to the swarm of peer models and start listening for updates on
 * peer models. Updates to this model will not be broadcast to the swarm
 * until after this method has been called.
 */
void
dee_shared_model_connect (DeeSharedModel *self)
{
  DeeSharedModelPrivate *priv;
  
  g_return_if_fail (DEE_IS_SHARED_MODEL (self));

  priv = self->priv;

  /* We need to call dbus_g_bus_get at least once to initialize the
   * specialized types
   */
  if (dee_shared_model_types_initialized == FALSE)
    {
      dee_shared_model_types_initialized = TRUE;      

      /* Marshaller for RowsAdded and RowsChanged */
      dbus_g_object_register_marshaller (_dee_marshal_VOID__BOXED_BOXED_BOXED,
                                         G_TYPE_NONE,
                                         DBUS_TYPE_G_VALUE_ARRAY_ARRAY,
                                         DBUS_TYPE_G_UINT_ARRAY,
                                         DBUS_TYPE_G_UINT64_ARRAY,
                                         G_TYPE_INVALID);

      /* Marshaller for RowsRemoved */
      dbus_g_object_register_marshaller (_dee_marshal_VOID__BOXED_BOXED,
                                         G_TYPE_NONE,
                                         DBUS_TYPE_G_UINT_ARRAY,
                                         DBUS_TYPE_G_UINT64_ARRAY,
                                         G_TYPE_INVALID);      
    }  

  /* Connect to the swarm */
  priv->connection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
  g_signal_connect_swapped (priv->swarm,
                            "connected", G_CALLBACK (on_peer_connected),
                            self);
  g_signal_connect_swapped (priv->swarm,
                            "peer-found", G_CALLBACK (on_peer_found),
                            self);
  g_signal_connect_swapped (priv->swarm,
                            "peer-lost", G_CALLBACK (on_peer_lost),
                            self);  

  /* Export this model on the bus */
  dbus_g_connection_register_g_object (priv->connection,
                                       priv->model_path,
                                       G_OBJECT (self));


  /* Connect to the swarm leader, who will be the "authoritative model",
   * in case of conflicts. Note that this proxy _always_ point at the
   * swarm leader, even if the leader changes */
  priv->leader = dbus_g_proxy_new_for_name (priv->connection,
                                            dee_peer_get_swarm_name (priv->swarm),
                                            priv->model_path,
                                            DEE_SHARED_MODEL_DBUS_IFACE);

  /* Connect to the swarm */
  dee_peer_connect (priv->swarm);
}

/**
 * dee_shared_model_get_swarm_name:
 * @self: The model to get the name for
 *
 * Convenience function for accessing the #DeePeer:swarm-name property of the
 * #DeePeer defined in the #DeeSharedModel:peer property.
 *
 * Returns: The name of the swarm this model synchrnonizes with
 */
const gchar*
dee_shared_model_get_swarm_name (DeeSharedModel *self)
{
  DeeSharedModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_SHARED_MODEL (self), NULL);

  priv = self->priv;
  return dee_peer_get_swarm_name (priv->swarm);
}

/**
 * dee_shared_model_get_peer:
 * @self: The model to get the #DeePeer for
 *
 * Convenience function for accessing the #DeeSharedModel:peer property
 *
 * Returns: (transfer-none): The #DeePeer used to interact with the peer models
 */
DeePeer*
dee_shared_model_get_peer (DeeSharedModel *self)
{
  DeeSharedModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_SHARED_MODEL (self), NULL);

  priv = self->priv;
  return priv->swarm;
}


/*
 * Dbus Methods
 */


static gboolean
_dee_shared_model_server_clone (DeeModel          *self,
                                gchar            **columns,
                                guint64           *last_seqnum,
                                GPtrArray        **rows,
                                GArray           **seqnums,
                                GError           **error)
{
  DeeSharedModelPrivate *priv;
  GPtrArray        *ret;
  gchar            *col_spec;
  DeeModelIter     *iter;
  gint              i;
  guint             n_rows, n_columns;

  trace ("Commence Clone()");
  
  g_return_val_if_fail (DEE_IS_SHARED_MODEL (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);  
  
  priv = DEE_SHARED_MODEL (self)->priv;
  n_rows = dee_model_get_n_rows (self);
  n_columns = dee_model_get_n_columns (self);
  ret = g_ptr_array_sized_new (n_rows);

  /* Clone the rows */  
  iter = dee_model_get_first_iter (self);
  while (!dee_model_is_last (self, iter))
    {      
      GPtrArray *row = g_ptr_array_sized_new (n_rows);
      
      for (i = 0; i < n_columns; i++)
        {
          GValue *value = g_new0 (GValue, 1);
          dee_model_get_value (self, iter, i, value);          
          g_ptr_array_add (row, value);
        }

      g_ptr_array_add (ret, row);
      
      iter = dee_model_next (self, iter);
    }

  /* Generate column spec */
  col_spec = dee_model_build_col_spec (self);
  
  /* Return last known seqnum */
  *last_seqnum = dee_versioned_model_get_last_seqnum (self);
  
  /* Note: DBus-GLib will not just unref the *seqnums array, but also clear
   *       the memory! So passing back *seqnums = g_array_ref (priv->seqnums)
   *       is a very bad idea (and would also be subtly racy, but leave that
   *       for now ) */ ;
  *seqnums = g_array_sized_new (FALSE, FALSE,
                                sizeof(guint64), n_rows);
  guint64 dum;
  for (i = 0; i < n_rows; i++)
    {
      dum = dee_versioned_model_get_seqnum (self, i);
      g_array_append_val (*seqnums, dum);
    }
  
  *columns = col_spec;
  *rows = ret;  

  trace ("Cloned %i rows", ret->len);
  g_assert (n_rows == ret->len);
  
  return TRUE;
}

/* Handle and Invalidate() message */
static gboolean
_dee_shared_model_server_invalidate (DeeModel   *self,
                                     GError    **error)
{
  DeeSharedModelPrivate *priv;

  g_return_val_if_fail (DEE_IS_SHARED_MODEL (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);
  
  priv = DEE_SHARED_MODEL (self)->priv;
  
  if (dee_peer_is_swarm_leader (priv->swarm))
    {
      g_warning ("Refusing to invalidate swarm leader");
      *error = g_error_new (dee_shared_model_error_quark,
                            DEE_SHARED_MODEL_ERROR_LEADER_INVALIDATED,
                            "Can not invalidate swarm leader");
      return FALSE;
    }
  
  priv->invalidated = TRUE;
  com_canonical_Dee_Model_clone_async (priv->leader,
                                       clone_callback,
                                       self);
  trace ("Model invalidated");

  return TRUE;
}
