# Copyright 2011 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Manuel de la Pena <manuel.delapena@canonical.com>
"""Test base dirs on windows."""

from mocker import MockerTestCase

from desktopcouch.application.platform.windows.base_dirs import (
    _get_special_folders,
    SHELL_FOLDERS_KEY)


#ignore typical mocker lint errors
# pylint: disable=W0104
class BaseDirsTestCase(MockerTestCase):
    """Test the base dirs implementation."""

    def setUp(self):
        super(BaseDirsTestCase, self).setUp()
        # replace _winreg so that we do not open the actual registry
        self._winreg = self.mocker.mock()

    def test_get_special_folders_cannot_open_hive(self):
        """Get special folders and get issue opening root."""
        self._winreg.HKEY_CURRENT_USER
        self.mocker.result(0)
        self._winreg.ConnectRegistry(None, 0)
        self.mocker.throw(Exception('Could not open hive'))
        self.mocker.replay()
        self.assertEqual({}, _get_special_folders(self._winreg))

    def test_get_special_folders_cannot_open_key(self):
        """Get special folders ang get issues opening key."""
        self._winreg.HKEY_CURRENT_USER
        self.mocker.result(0)
        self._winreg.ConnectRegistry(None, 0)
        self.mocker.result('hive')
        self._winreg.OpenKey('hive', SHELL_FOLDERS_KEY)
        self.mocker.throw(Exception('Could not open key'))
        self._winreg.CloseKey('hive')
        self.mocker.replay()
        self.assertEqual({}, _get_special_folders(self._winreg))

    def test_get_special_folders_cannot_get_data(self):
        """Get special folders and get issues when reading the data."""
        self._winreg.HKEY_CURRENT_USER
        self.mocker.result(0)
        self._winreg.ConnectRegistry(None, 0)
        self.mocker.result('hive')
        self._winreg.OpenKey('hive', SHELL_FOLDERS_KEY)
        self.mocker.result('key')
        self._winreg.QueryInfoKey('key')[1]  # pylint: disable=W0106
        self.mocker.throw(Exception('Cannot get info.'))
        self._winreg.CloseKey('hive')
        self._winreg.CloseKey('key')
        self.mocker.replay()
        self.assertEqual({}, _get_special_folders(self._winreg))

    def test_get_special_folders(self):
        """Get the special folders with no issues."""
        self._winreg.HKEY_CURRENT_USER
        self.mocker.result(0)
        self._winreg.ConnectRegistry(None, 0)
        self.mocker.result('hive')
        self._winreg.OpenKey('hive', SHELL_FOLDERS_KEY)
        self.mocker.result('key')
        self._winreg.QueryInfoKey('key')[1]  # pylint: disable=W0106
        self.mocker.result(1)
        self._winreg.EnumValue('key', 0)
        self.mocker.result(('AppData', 'path', 1))
        self._winreg.CloseKey('hive')
        self._winreg.CloseKey('key')
        self.mocker.replay()
        self.assertEqual({'AppData': 'path'},
            _get_special_folders(self._winreg))
