# Copyright 2009-2010 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>
#          Mark G. Saye <mark.saye@canonical.com>
#          Stuart Langridge <stuart.langridge@canonical.com>
#          Chad Miller <chad.miller@canonical.com>
#          Vincenzo Di Somma <vincenzo.di.somma@canonical.com>

"""The Desktop Couch Records API."""

import copy
# pylint: disable=W0402
import string
# pylint: enable=W0402
import warnings

from time import time
from uuid import uuid4

# please keep desktopcouch python 2.5 compatible for now
# pylint can't deal with failing imports even when they're handled
# pylint: disable=F0401
try:
    # Python 2.5
    import simplejson as json
except ImportError:
    # Python 2.6+
    import json
# pylint: enable=F0401

from couchdb import Server
from couchdb.http import ResourceNotFound, ResourceConflict, PreconditionFailed
from couchdb.design import ViewDefinition

from desktopcouch.records import Record


DEFAULT_DESIGN_DOCUMENT = None  # each view in its own eponymous design doc.
DCTRASH = 'dc_trash'


def base_n(num, base, numerals=string.printable):
    """Take an integer and return a string representation of the number encoded
    into a given number base.
    >>> base_n(0, 10)
    '0'
    >>> base_n(42, 10)
    '42'
    >>> base_n(10, 42)
    'a'
    >>> base_n(142813190999624924427737229010582846569L, 62)
    '3gJJKymTqPPK8FSHHj2UkN'
    """
    if num == 0:
        return numerals[0]
    div, mod = divmod(num, base)
    return base_n(div, base).lstrip("0") + numerals[mod]


def transform_to_records(view_results):
    """Transform view resulst into Record objects."""
    for result in view_results:
        yield Record(result.value)


class FieldsConflict(Exception):
    """Raised in case of an unrecoverable couchdb conflict."""

    #pylint: disable=W0231
    def __init__(self, conflicts):
        self.conflicts = conflicts
    #pylint: enable=W0231

    def __str__(self):
        return "<CouchDB Conflict Error: %s>" % self.conflicts


class NoSuchDatabase(Exception):
    "Exception for trying to use a non-existent database"

    def __init__(self, dbname):
        self.database = dbname
        super(NoSuchDatabase, self).__init__()

    def __str__(self):
        return ("Database %s does not exist on this server. (Create it by "
                "passing create=True)") % self.database


class Database(object):
    """A desktopcouch.records specific CouchDb database."""

    def __init__(self, database, uri, record_factory=None, create=False,
                 views_factory=None, server_class=Server,
                 **server_class_extras):
        if views_factory is None:
            self._views_factory = ViewDefinition
        else:
            self._views_factory = views_factory
        self.server_uri = uri
        self._database_name = database
        self.record_factory = record_factory or Record
        self._create = create
        self._server_class = server_class
        self._server_class_extras = server_class_extras
        self._server = None
        self.db = None
        self._dctrash = None
        self._reconnect()
        self._changes_since = self.db.info()["update_seq"]
        self._changes_last_used = 0  # Immediate run works.

    @staticmethod
    def _is_reconnection_fail(ex):
        """Check whether this is the bug in httplib."""
        return isinstance(ex, AttributeError) and \
                ex.args == ("'NoneType' object has no attribute 'makefile'",)

    def _reconnect(self, uri=None):
        """Reconnect after losing connection."""
        self._server = self._server_class(uri or self.server_uri,
                **self._server_class_extras)
        if self._create:
            try:
                self._server.create(self._database_name)
            except PreconditionFailed:
                pass  # Not an error to want it created and DB exists.

        try:
            if self.db is None:
                self.db = self._server[self._database_name]
            else:
                # Monkey-patch the object the user already uses.  Oook!
                new_db = self._server[self._database_name]
                self.db.resource = new_db.resource
        except ResourceNotFound:
            raise NoSuchDatabase(self._database_name)

    def _temporary_query(self, map_fun, reduce_fun=None, language='javascript',
            wrapper=None, **options):
        """Pass-through to CouchDB library.  Deprecated."""
        warnings.warn(
            "._temporary_query is deprecated.", DeprecationWarning,
            stacklevel=2)
        return self.db.query(map_fun, reduce_fun, language,
                             wrapper, **options)

    @property
    def server(self):
        """Return the server attribute."""
        return self._server

    def get_record(self, record_id):
        """Get a record from back end storage."""

        def make_getter(source_db, document_id, attachment_name, content_type):
            """Closure storing the database for lower levels to use
            when needed.

            """
            def getter():
                """Get the attachment and content type."""
                return source_db.get_attachment(
                    document_id, attachment_name), content_type
            return getter

        try:
            couch_record = self.db[record_id]
        except ResourceNotFound:
            return None
        data = {}

        data.update(couch_record)
        record = self.record_factory(data=data)
        if "_attachments" in data:
            for att_name, att_attributes in data["_attachments"].iteritems():
                record.attach_by_reference(att_name,
                        make_getter(self.db, record_id, att_name,
                                att_attributes["content_type"]))

        return record

    def put_record(self, record):
        """Put a record in back end storage."""
        record_id = record.record_id
        record_data = record._data  # pylint: disable=W0212
        if not record_id:
            # Do not rely on couchdb to create an ID for us.
            # pylint: disable=E1101
            record_id = base_n(uuid4().int, 62)
            record.record_id = record_id
            # pylint: enable=E1101
        self.db[record_id] = record_data

        # pylint: disable=W0212
        for attachment_name in getattr(record, "_detached", []):
            self.db.delete_attachment(record_data, attachment_name)

        for attachment_name in record.list_attachments():
            data, content_type = record.attachment_data(attachment_name)
            self.db.put_attachment(record_data,
                                   data,
                                   attachment_name,
                                   content_type)
        # pylint: enable=W0212

        return record_id

    def put_records_batch(self, batch):
        """Put a batch of records in back end storage."""
        # used to access fast the record
        records_hash = {}
        for record in batch:
            record_id = record.record_id
            if not record_id:
                # Do not rely on couchdb to create an ID for us.
                # pylint: disable=E1101
                record_id = base_n(uuid4().int, 62)
                record.record_id = record_id
                # pylint: enable=E1101
            records_hash[record_id] = record
        # although with a single record we need to test for the
        # revision, with a batch we do not, but we have to make sure
        # that we did not get an error
        # pylint: disable=W0212
        batch_put_result = self.db.update([record._data for record in batch])
        for current_tuple in batch_put_result:
            success, docid, rev_or_exc = current_tuple
            if success:
                record = records_hash[docid]
                record_data = record._data
                # set the new rev
                record_data["_rev"] = rev_or_exc
                for attachment_name in getattr(record, "_detached", []):
                    self.db.delete_attachment(record_data, attachment_name)
                for attachment_name in record.list_attachments():
                    data, content_type = record.attachment_data(
                        attachment_name)
                    self.db.put_attachment(
                        {"_id": docid, "_rev": rev_or_exc},
                        data, attachment_name, content_type)
        # pylint: enable=W0212
        # all success record have the blobs added we return result of
        # update
        return batch_put_result

    def update_fields(self, record_id, fields, cached_record=None):
        """Safely update a number of fields. 'fields' being a
        dictionary with path_tuple: new_value for only the fields we
        want to change the value of, where path_tuple is a tuple of
        fieldnames indicating the path to the possibly nested field
        we're interested in. old_record a the copy of the record we
        most recently read from the database.

        In the case the underlying document was changed, we try to
        merge, but only if none of the old values have changed. (i.e.,
        do not overwrite changes originating elsewhere.)

        This is slightly hairy, so that other code won't have to be.
        """
        # Initially, the record in memory and in the db are the same
        # as far as we know. (If they're not, we'll get a
        # ResourceConflict later on, from which we can recover.)
        if cached_record is None:
            cached_record = self.db[record_id]
        if isinstance(cached_record, Record):
            cached_record = cached_record._data  # pylint: disable=W0212
        record = copy.deepcopy(cached_record)
        # Loop until either failure or success has been determined
        while True:
            modified = False
            conflicts = {}
            # loop through all the fields that need to be modified
            for path, new_value in fields.items():
                if not isinstance(path, tuple):
                    path = (path,)
                # Walk down in both copies of the record to the leaf
                # node that represents the field, creating the path in
                # the in memory record if necessary.
                db_parent = record
                cached_parent = cached_record
                for field in path[:-1]:
                    db_parent = db_parent.setdefault(field, {})
                    cached_parent = cached_parent.get(field, {})
                # Get the values of the fields in the two copies.
                db_value = db_parent.get(path[-1])
                cached_value = cached_parent.get(path[-1])
                # If the value we intend to store is already in the
                # database, we need do nothing, which is our favorite.
                if db_value == new_value:
                    continue
                # If the value in the db is different than the value
                # our copy holds, we have a conflict. We could bail
                # here, but we can give better feedback if we gather
                # all the conflicts, so we continue the for loop
                if db_value != cached_value:
                    conflicts[path] = (db_value, new_value)
                    continue
                # Otherwise, it is safe to update the field with the
                # new value.
                modified = True
                db_parent[path[-1]] = new_value
            # If we had conflicts, we can bail.
            if conflicts:
                raise FieldsConflict(conflicts)
            # If we made changes to the document, we'll need to save
            # it.
            if modified:
                try:
                    self.db[record_id] = record
                except ResourceConflict:
                    # We got a conflict, meaning the record has
                    # changed in the database since we last loaded it
                    # into memory. Let's get a fresh copy and try
                    # again.
                    record = self.db[record_id]
                    continue
            # If we get here, nothing remains to be done, and we can
            # take a well deserved break.
            break

    def record_exists(self, record_id):
        """Check if record with given id exists."""
        return record_id in self.db

    # pylint: disable=W0212
    #Access to a protected member
    def delete_record(self, record_id):
        """Delete record with given id"""
        record = self.get_record(record_id)
        if record is None:
            return None
        new_record = copy.deepcopy(record)
        if not self._dctrash:
            self._dctrash = self.__class__(
                database=DCTRASH,
                uri=self.server_uri,
                create=True,
                **self._server_class_extras)
        # pylint: disable=E1101
        new_record.record_id = base_n(uuid4().int, 62)
        # pylint: enable=E1101
        del new_record._data['_rev']
        try:
            del new_record._data['_attachments']
        except KeyError:
            pass
        new_record.application_annotations['desktopcouch'] = {
            'private_application_annotations':
            {'original_database_name': self._database_name,
             'original_id': record_id}}
        del self.db[record_id]
        return self._dctrash.put_record(new_record)
    # pylint: enable=W0212

    def delete_view(self, view_name, design_doc=DEFAULT_DESIGN_DOCUMENT):
        """Remove a view, given its name.  Raises a KeyError on a unknown
        view.  Returns a dict of functions the deleted view defined."""
        if design_doc is None:
            design_doc = view_name

        doc_id = "_design/%(design_doc)s" % locals()

        # No atomic updates.  Only read & mutate & write.  Le sigh.
        # First, get current contents.
        try:
            view_container = self.db[doc_id]["views"]
        except (KeyError, ResourceNotFound):
            raise KeyError

        deleted_data = view_container.pop(view_name)  # Remove target

        if len(view_container) > 0:
            # Construct a new list of objects representing all views to have.
            views = [
                    self._views_factory(design_doc, k, v.get("map"),
                                   v.get("reduce"))
                    for k, v in view_container.iteritems()]
            # Push back a new batch of view.  Pray to Eris that this doesn't
            # clobber anything we want.

            # sync_many does nothing if we pass an empty list.  It even gets
            # its design-document from the ViewDefinition items, and if there
            # are no items, then it has no idea of a design document to
            # update.  This is a serious flaw.  Thus, the "else" to follow.
            self._views_factory.sync_many(self.db, views, remove_missing=True)
        else:
            # There are no views left in this design document.

            # Remove design document.  This assumes there are only views in
            # design documents.  :(
            del self.db[doc_id]

        assert not self.view_exists(view_name, design_doc)

        return deleted_data

    def execute_view(self, view_name, design_doc=DEFAULT_DESIGN_DOCUMENT,
            **params):
        """Execute view and return results."""

        if design_doc is None:
            design_doc = view_name

        view_id_fmt = "_design/%(design_doc)s/_view/%(view_name)s"
        return self.db.view(view_id_fmt % locals(), **params)

    def add_view(self, view_name, map_js, reduce_js=None,
                 design_doc=DEFAULT_DESIGN_DOCUMENT):
        """Create a view, given a name and the two parts (map and reduce).
        Return the document id."""
        if design_doc is None:
            design_doc = view_name

        view = self._views_factory(design_doc, view_name, map_js, reduce_js)
        view.sync(self.db)
        assert self.view_exists(view_name, design_doc)

    def view_exists(self, view_name, design_doc=DEFAULT_DESIGN_DOCUMENT):
        """Does a view with a given name, in a optional design document
        exist?"""
        if design_doc is None:
            design_doc = view_name

        doc_id = "_design/%(design_doc)s" % locals()

        try:
            view_container = self.db[doc_id]["views"]
            return view_name in view_container
        except (KeyError, ResourceNotFound):
            return False

    def list_views(self, design_doc):
        """Return a list of view names for a given design document.  There is
        no error if the design document does not exist or if there are no views
        in it."""
        doc_id = "_design/%s" % design_doc
        try:
            return list(self.db[doc_id]["views"])
        except (KeyError, ResourceNotFound):
            return []

    def get_records(self, record_type=None, create_view=False,
            design_doc=DEFAULT_DESIGN_DOCUMENT):
        """A convenience function to get records from a view named
        C{get_records_and_type}.  We optionally create a view in the design
        document.  C{create_view} may be True or False, and a special value,
        None, is analogous to  O_EXCL|O_CREAT .

        Set record_type to a string to retrieve records of only that
        specified type. Otherwise, usse the view to return *all* records.
        If there is no view to use or we insist on creating a new view
        and cannot, raise KeyError .

        You can use index notation on the result to get rows with a
        particular record type.
        =>> results = get_records()
        =>> for foo_document in results["foo"]:
        ...    print foo_document

        Use slice notation to apply start-key and end-key options to the view.
        =>> results = get_records()
        =>> people = results[['Person']:['Person','ZZZZ']]
        """
        warnings.warn(
            ".get_records is deprecated, use .get_all_records instead",
            DeprecationWarning, stacklevel=2)
        view_name = "get_records_and_type"
        view_map_js = """
            function(doc) {
                emit(doc.record_type, doc);
            }"""

        if design_doc is None:
            design_doc = view_name

        exists = self.view_exists(view_name, design_doc)

        if exists:
            if create_view is None:
                raise KeyError("Exclusive creation failed.")
        else:
            if create_view == False:
                raise KeyError("View doesn't already exist.")
        if not exists:
            self.add_view(view_name, view_map_js, design_doc=design_doc)
        viewdata = self.execute_view(view_name, design_doc)
        if record_type is None:
            return viewdata
        else:
            return viewdata[record_type]

    def get_view_results_as_records(self, view_name, record_type=None):
        """Get the results of a view as a list of Record objects."""
        view_results = self.execute_view(view_name, view_name)
        if record_type:
            view_results = view_results[record_type]
        return [record for record in transform_to_records(view_results)]

    def get_all_records(self, record_type=None):
        """Get all records from the database, optionally by record type."""
        view_name = "get_records_and_type"
        try:
            return self.get_view_results_as_records(
                view_name, record_type=record_type)
        except ResourceNotFound:
            view_map_js = """
            function(doc) {
                emit(doc.record_type, doc);
            }"""
            self.add_view(view_name, view_map_js, design_doc=view_name)
            return self.get_view_results_as_records(
                view_name, record_type=record_type)

    def get_changes(self, niceness=10):
        """Get a list of database changes.  This is the sister function of
        report_changes that returns a list instead of calling a function for
        each."""
        changes = []
        self.report_changes(lambda **k: changes.append(k), niceness=niceness)
        return changes

    def _get_changes(self, changes_since=None):
        """Get changes feed from couchdb."""
        # Can't use self._server.resource.get() directly because
        # it encodes "/".
        if not changes_since is None:
            return self.db.resource.get('_changes', since=changes_since)
        return self.db.resource.get('_changes')

    def report_changes(self, function_to_send_changes_to, niceness=10):
        """Check to see if there are any changes on this database since last
        call (or since this object was instantiated), call a function for each,
        and return the number of changes reported.

        The callback function is called for every single change, with the
        keyword parameters of the dictionary of values returned from couchdb.

        =>> def f(seq=None, id=None, changes=None):
        ...    pass

        =>> db_foo.report_changes(f)
        =>> time.sleep(30)
        =>> db_foo.report_changes(f)

        or

        =>> # Make function that returns true, to signal never to remove.
        =>> report_forever = lambda **kw: db_foo.report_changes(**kw) or True
        =>> cb_id = glib.mainloop.idle_add(report_forever, f)

        or

        =>> task_id = twisted.task.looping_call(db_foo.report_changes, f)
        """
        now = time()
        call_count = 0
        if not niceness or now > self._changes_last_used + niceness:
            status, _resp, respbody = self.db.resource.get(
                "_changes", since=self._changes_since)
            data = respbody.read()
            if status != 200:
                raise IOError(
                    "HTTP response code %s.\n%s" % (status, data))
            structure = json.loads(data)
            for change in structure.get("results"):  # pylint: disable=E1103
                # kw-args can't have unicode keys
                change_encoded_keys = dict(
                    (k.encode("utf8"), v) for k, v in change.iteritems())
                function_to_send_changes_to(**change_encoded_keys)
                # Not s.last_seq.  Exceptions!
                self._changes_since = change["seq"]
                call_count += 1
            # If exception in cb, we never update governor.
            self._changes_last_used = now
        return call_count

    def ensure_full_commit(self):
        """
        Make sure that CouchDb flushes all writes to the database,
        flushing all delayed commits, before going on.
        """
        self.db.resource.post(
            path='_ensure_full_commit',
            headers={'Content-Type': 'application/json'})


class CouchDatabaseBase(Database):
    """Deprecated."""

    def __init__(self, database, uri, record_factory=None, create=False,
                 server_class=Server, **server_class_extras):
        warnings.warn(
            "CouchDatabaseBase is deprecated, use Database instead.",
            DeprecationWarning, stacklevel=2)
        super(CouchDatabaseBase, self).__init__(
            database, uri, record_factory=record_factory, create=create,
            server_class=server_class, **server_class_extras)
