# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python tools
#
# Copyright © 2006-2008 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

import pgm
from pgm.graph.group import Group
from pgm.graph.image import Image
from pgm.timing import implicit


class ProgressBarOsd(Group):

    def __init__(self):
        super(ProgressBarOsd, self).__init__()

        self._cursor_position = 0

        self.background = Image()
        self.background.layout = pgm.IMAGE_FILLED
        self.background.alignment = pgm.IMAGE_LEFT
        self.background.height = 1.0
        self.background.width = 4.0
        self.background.bg_color = (0, 0, 0, 0)
        self.background.visible = True

        self.foreground = Image()
        self.foreground.layout = pgm.IMAGE_FILLED
        self.foreground.alignment = pgm.IMAGE_LEFT
        self.foreground.height = 1.0
        self.foreground.width = 4.0
        self.foreground.size = (0, 0)
        self.foreground.bg_color = (0, 0, 0, 0)
        self.foreground.visible = True
        self._animated_foreground = implicit.AnimatedObject(self.foreground)
        self._animated_foreground.mode = implicit.REPLACE
        self._animated_foreground.setup_next_animations(duration = 200)

        self.add(self.background)
        self.add(self.foreground)

        self.length = 0

    def size__set(self, size):
        self.background.size = size
        # Needed for nicer animations
        self.foreground.height = size[1]
        self._paint_foreground(animated = False)

    def length__set(self, length):
        """
        Set the length. The bar is automatically new painted
        @param length: the length of the process
        @type length: as above: something to do mathematic stuff
        """
        self._length = length
        if self._length <= 0:
            self._length = 1
        if self._cursor_position > self._length:
             self._cursor_position = self._length
        self._paint_foreground()

    def length__get(self):
        """
        to get the length:
        @return: returns the length
        """
        return self._length

    def _paint_foreground(self, animated=True):
        """
        Here the position is calculated and the bar is painted
        """
        next_pos = self.background.size[0] / self.length * \
                                    self._cursor_position
        if animated:
            self._animated_foreground.width = next_pos
        else:
            self._animated_foreground.stop_animations()
            self.foreground.width = next_pos

    def cursor_position__set(self, position):
        """
        Set the position of the cursor. The bar is automatically new painted
        (with animation)
        @param position: the position to set to
        @type position: something to do mathematic stuff with
        """
        self._cursor_position = position
        self._cursor_position = max(0, min(self._cursor_position, self._length))
        self._paint_foreground()

    def cursor_position__get(self):
        """
        returns the position of the cursor
        @return : the position of the cursor
        """
        return self._cursor_position

    def jump_to_position(self, position):
        """
        Jump to this position. This means that the cursor doesn't go there
        with animationm but that it is there immediatly
        @param position: the position to jump to
        @type position: something to do mathematic stuff with
        """
        self._cursor_position = position
        self._cursor_position = max(0, min(self._cursor_position, self.length))
        self._paint_foreground(animated=False)
