// -*- Mode: C++; c-basic-offset: 4; -*-
#include <kpopupmenu.h>
#include <qtimer.h>
#include <kdebug.h>
#include <klineedit.h>
#include <qlabel.h>
#include <qpainter.h>
#include <qpushbutton.h>
#include <klocale.h>
#include <kapplication.h>
#include <qthread.h>
#include <qobjectlist.h>
#include <qregexp.h>
#include <qtextbrowser.h>
#include <qlayout.h>
#include <kglobal.h>
#include <kiconloader.h>

#include <iostream>
#include <ext/algorithm>

#include "lister.h"
#include <apt-front/cache/cache.h>
#include <apt-front/actor.h>
#include <apt-front/cache/component/packages.h>
#include <apt-front/cache/component/state.h>
#include <apt-front/predicate/factory.h>

using namespace aptFront;
using namespace aptFront::predicate;
using namespace aptFront::cache;
using namespace aptFront::utils;
using namespace ept;

Lister::Lister( QWidget *parent, const char *name )
    : ExtendableList( parent, name ),
      m_baseF( adapt< entity::Entity >(
                   predicate::Package::member(
                       &entity::Package::hasVersion ) ) ),
      m_interactiveF( True<entity::Entity>() ), m_itemCount( -1 ),
      m_rebuildScheduled( false ), m_inRebuild( false ),
      m_inDtor( false )
{
    observeComponent< component::State >();
    observeComponent< component::Packages >();
    setRootIsDecorated( false );
    setSelectionModeExt( Extended );
    setAllColumnsShowFocus (true);

    m_icons["package-install"] = "adept_install";
    m_icons["package-remove"] = "adept_remove";
    m_icons["package-upgrade"] = "adept_upgrade";
    m_icons["package-keep"] = "adept_keep";

    m_baseIcon = SmallIcon( "extender_closed" );
    m_extendedIcon = SmallIcon( "extender_opened" );

    setSorting( -1 );
    addColumn(" ", 18);
    addColumn("Package", 180);
    addColumn("Status", 110);
    addColumn("Action", 90);
    addColumn("Description", 300);
    setResizeMode (LastColumn);
    connect( this, SIGNAL( selectionChanged() ), SLOT( updateActions() ) );
    connect( this,
             SIGNAL( contextMenuRequested( QListViewItem *,
                                           const QPoint &, int ) ),
             this, SLOT(
                 contextMenu( QListViewItem *, const QPoint &, int) ) );
    m_tip = 0;
    // m_tip = new ListerTooltip( viewport(), this );
}

Lister::~Lister()
{
    m_inDtor = true;
    delete m_tip;
}

void Lister::scheduleRebuild() {
    kdDebug() << "Lister::scheduleRebuild" << endl;
    if (!m_rebuildScheduled)
        QTimer::singleShot( 0, this, SLOT( rebuild() ) );
    m_rebuildScheduled = true;
}

void Lister::updateActions() {
    emit actionsChanged( this );
}

void Lister::notifyPostChange( component::Base * )
{
    kdDebug() << "notifyRefresh()" << endl;
    updateActions();
    triggerUpdate();
}

void Lister::notifyPreRebuild( component::Base *b )
{
    kdDebug() << "notifyPreRebuild()" << endl;
    Cache &c = cache::Global::get( m_cache );
    setEnabled( false );
    /* if (b == &c.packages()) {
        clear();
        } */
}

void Lister::notifyPostRebuild( component::Base *b )
{
    kdDebug() << "notifyPostRebuild()" << endl;
    Cache &c = cache::Global::get( m_cache );
    setEnabled( true );
    if (b == &c.packages()) { // our stuff is invalidated
        c.openState(); // ensure state is opened
        c.openRecords(); // ensure records are opened
        c.openDebtags(); // ensure debtags are opened
        cleanRebuild();
    } else { // just state rebuilding, feeling free to ignore ;-)
        triggerUpdate();
        // scheduleRebuild();
    }
}

bool lessByName( const entity::Entity &e1, const entity::Entity &e2 ) {
    entity::Package p1 = e1, p2 = e2;
    return p1.name() < p2.name();
}

struct AsyncRebuild : public QThread {
    component::Packages &p;
    Lister::Vector &v;
    AsyncRebuild( component::Packages &_p,
                  Lister::Vector &_v ) : p( _p ), v( _v ) {}
    virtual void run() {
        v.clear();
        v.insert( v.begin(),
                      p.packagesBegin(), p.packagesEnd() );
        std::sort( v.begin(), v.end(), lessByName );
    }
};

namespace ept {

struct Timer: public QThread {
    Timer() : timer( 0 ) {}
    void run() {
        while (true) {
            msleep( 10 );
            timer ++;
        }
    }
    int timer;
};

}

void Lister::cleanRebuild()
{
    kdDebug() << "cleanRebuild running" << endl;
    clock_t _c = clock(), C;
    Cache &c = cache::Global::get( m_cache );

    setEnabled( false );
    AsyncRebuild async( cache::Global::get( m_cache ).packages(), m_all );
    async.start();
    c.progress().OverallProgress( 0, 10000, 100, "Sorting" );
    int p = 0;
    while( async.running() ) {
        kapp->processEvents();
        usleep( 50000 );
        c.progress().Progress( ++p );
    }
    C = (clock() - _c) / 1000; _c = clock();
    kdDebug() << m_all.size() << " sorted, time = " << C << endl;
    setEnabled( true );
    scheduleRebuild();
}

Lister::CreateItem::CreateItem( Lister *_l )
    : l( _l ), time( 0 ), items( 0 ), last( 0 )
{
}

Lister::CreateItem::~CreateItem()
{
    // delete timer;
}

Lister::Map::value_type Lister::CreateItem::operator()( entity::Entity e ) {
    items ++;
    if ((l->creationTimer->timer - time > 9) ||
        (items < 50 && items % 8 == 0)) {
        kdDebug() << "time = " << time << " timer = " <<
                  l->creationTimer->timer << endl;
        l->reallyUpdate();
        kapp->processEvents();
        time = l->creationTimer->timer;
        cache::Global::get( l->m_cache ).progress().Progress( time );
    }
    last = last ?
           new ListerItem( l, last, e ) : new ListerItem( l, e );
    return std::make_pair( e, last );
}

void Lister::reallyUpdate()
{
    bool en = isUpdatesEnabled();
    setUpdatesEnabled( true );
    triggerUpdate();
    setUpdatesEnabled( en );
}

void Lister::rebuild()
{
    m_rebuildScheduled = false;
    if (m_inRebuild) {
        scheduleRebuild();
        return;
    }
    m_inRebuild = true;
    Cache &c = cache::Global::get( m_cache );
    c.progress().OverallProgress( 0, 0, 0, "Filtering" );
    kdDebug() << "rebuild running" << endl;
    clock_t _c = clock(), C;
    kdDebug() << m_all.size() << " in cache" << endl;
    // we may need to ensure cleanRebuild does not hit us here...
    Range< entity::Entity > r =
        filteredRange( range( m_all.begin(), m_all.end() ), m_baseF );
    C = (clock() - _c) / 1000; _c = clock();

    setUpdatesEnabled( false );
    clear();
    m_items.clear();
    creationTimer = new Timer;
    creationTimer->start();
    std::transform( r, r.last(),
                    inserter( m_items, m_items.begin() ),
                    CreateItem( this ) );
    creationTimer->terminate();
    delete creationTimer;
    m_itemCount = m_items.size();
    C = (clock() - _c) / 1000; _c = clock();
    kdDebug() << m_items.size() << " entities synced, time = " << C << endl;
    setUpdatesEnabled( true );
    c.progress().Done();
    triggerUpdate();
    m_inRebuild = false;
}

void Lister::baseAnd( Predicate o )
{
    m_baseF = predicate::predicate( m_baseF and o );
    // emit filterChanged( m_baseF );
    scheduleRebuild();
}

void Lister::baseSet( Predicate o )
{
    m_baseF = o;
    // emit filterChanged( m_baseF );
    scheduleRebuild();
}

void Lister::interactiveAnd( Predicate o )
{
    m_interactiveF = predicate::predicate( m_interactiveF and o );
    scheduleRebuild();
}

void Lister::interactiveDrop( Predicate o )
{
    m_interactiveF = predicate::remove( m_interactiveF, o );
    scheduleRebuild();
}

bool Lister::itemSelected( Map::value_type i )
{
    return not i.second->isSelected();
}

entity::Entity Lister::extractKey( Map::value_type i )
{
    return i.first;
}

Lister::VectorRange Lister::selection()
{
    VectorRange ret;
    Map m;
    std::remove_copy_if( m_items.begin(), m_items.end(),
                         inserter( m, m.begin() ),
                         itemSelected );
    std::transform( m.begin(), m.end(),
                    consumer( ret ),
                    extractKey );
    return ret;
}

Lister::VectorRange Lister::content()
{
    VectorRange ret;
    std::transform( m_items.begin(), m_items.end(),
                    consumer( ret ),
                    extractKey );
    return ret;
}

QString ListerItem::text( int column ) const
{
    if (column == 0) return ""; // until we redo paintcell for the col
    if (entity().isType<entity::Package>()) {
        entity::Package p = entity();
        switch (column) {
        case 1: return p.name( "n/a" );
        case 2: return p.statusString( "n/a" );
        case 3: return p.actionString( "n/a" );
        case 4: return p.shortDescription( "n/a" );
            // case 2: return p.candidateVersion().versionString();
        }
    }
    return "N/A";
}

void ListerItem::paintCell (QPainter *p, const QColorGroup &cg,
                            int column, int width, int alignment )
{
    QColorGroup _cg( cg );
    QColor c = _cg.text();
    QPixmap pm( width, height() );
    QPainter _p( &pm );
    if (entity().isType<entity::Package>()) {
        entity::Package p = entity();
        if (column == 2) {
            c = Qt::blue;
            if (p.isInstalled())
                c = Qt::darkGreen;
            if (p.isUpgradable())
                c = Qt::darkYellow;
            if (p.isBroken())
                c = Qt::red;
        }
        if (column == 3) {
            c = Qt::blue;
            if (p.markedNewInstall())
                c = Qt::darkGreen;
            if (p.markedUpgrade())
                c = Qt::darkYellow;
            if (p.markedRemove())
                c = Qt::darkRed;
            if (p.willBreak())
                c = Qt::red;
        }
    }
    _cg.setColor( QColorGroup::Text, c );
    KListViewItem::paintCell( &_p, _cg, column, width, AlignTop );
    p->drawPixmap( 0, 0, pm );
}

void Lister::contextMenu( QListViewItem *it, const QPoint &pt, int /*c*/ )
{
    if (! it) // check for actor when we have one...
        return;
    m_context = dynamic_cast< ListerItem * >( it );
    VectorRange sel = selection();
    // entity::Package p = (dynamic_cast<ListerItem *>(it)->entity());
    KPopupMenu *m = new KPopupMenu (this);
    utils::Range< Actor > r = actor::Global< entity::Package >::list();
    int id = 8;
    for (; r != r.last(); ++r) {
        m->insertItem( SmallIconSet( m_icons[r->name()] ),
                       r->prettyName(), id );
        m->setItemEnabled(
            id, r->possible( utils::upcastRange< entity::Package >( sel ) ) );
        ++id;
    }
    bool open = m_context->extender();
    m->insertItem( open ? "Hide details" : "Show details", open ? 1 : 0 );
    connect(m, SIGNAL(activated(int)), this, SLOT(contextActivated(int)));
    m->exec(pt);
    delete m;
}

void Lister::contextActivated( int id )
{
    VectorRange sel = selection();
    if (id >= 8) {
        utils::Range< Actor > r = actor::Global< entity::Package >::list();
        std::advance( r, id - 8 );
        (*r)( utils::upcastRange< entity::Package >( sel ) );
        updateActions();
    }
    if (id < 8) {
        VectorRange i = sel.begin();
        while (i != i.last()) {
            if (id == 0)
                m_items[*i]->showExtender();
            if (id == 1)
                m_items[*i]->hideExtender();
            ++ i;
        }
    }
}

QString ListerItemExtender::format( const QString &what,
                                    const QString &txt, bool nobr )
{
    QString ret = "<b><nobr>" + what + "</nobr></b>&nbsp;" + (nobr ? "<nobr>" : "")
                  + txt + (nobr ? "</nobr>" : "");
    return ret;
}

ListerItemExtender::~ListerItemExtender() {
}

static void updateFont( QWidget *w, const QFont &f ) {
    w->setFont( f );
}

ListerItemExtender::ListerItemExtender( QWidget *parent, const char * n)
    : ListerItemExtenderUi( parent, n )
{
    observeComponent< component::State >();

    QFont f = m_description->font();
    f.setPointSize( f.pointSize() - 1 ); // a bit smaller font...
    updateFont( m_description, f );
    updateFont( m_status, f );
    updateFont( m_action, f );
    updateFont( m_section, f );
    updateFont( m_installedSize, f );
    updateFont( m_maintainer, f );
    updateFont( m_candidateVer, f );
    updateFont( m_installedVer, f );
}

ListerItem *ListerItemExtender::item()
{
    return dynamic_cast< ListerItem * >( m_item );
}

void ListerItemExtender::setItem( ExtendableItem *i )
{
    ItemExtender::setItem( i );
    m_entity = item()->entity();
    // setupColors();

    entity::Package pkg = m_entity;

    m_name->setText( QString( "<b>" ) + pkg.name( "oops" ) + "</b>" );
    m_section->setText(
        format( "Section:", pkg.section( "Unknown" ) ) );
    m_maintainer->setText(
        format( "Maintainer:", pkg.maintainer( "Unknown" ), false ) );
    QString l2, l = pkg.longDescription( "No long description available" );

    QRegExp rx( "^(.*)\n" );
    rx.setMinimal( true );
    l.replace( rx, "\\1</p><p>" );
    rx = QRegExp( "\\n[ ]*\\.\\n" );
    l.replace( rx, "</p><p>" );
    rx = QRegExp( "\n   " );
    l.replace( rx, " " );
    rx = QRegExp( "\n  - (.*)(\n|$)" );
    rx.setMinimal( true );
    l.replace( rx, "\n<li>\\1</li>\n" );
    l.replace( rx, "\n<li>\\1</li>\n" );
    m_description->setText( QString( "<qt><p>" ) + l + "</p></qt>" );
    m_description->adjustSize();
    kdDebug() << l << endl;
    m_description->installEventFilter( this );

    notifyPostChange( 0 );
}

void ListerItemExtender::notifyPostChange( component::Base * )
{
    entity::Package pkg = entity();
    EntityActor *a = 0;

    // XXX move to libapt-front
    m_logical->setEnabled( true );
    if (pkg.canUpgrade()) {
        a = new EntityActor( pkg.upgrade() );
    } else if (pkg.canInstall()) {
        a = new EntityActor( pkg.install() );
    } else if (pkg.canKeep()) {
        a = new EntityActor( pkg.keep() );
    } else if (pkg.canRemove()) {
        a = new EntityActor( pkg.remove() );
    }

    if (a) {
        m_logical->setText( a->actor().prettyName() );
        connect( m_logical, SIGNAL( clicked() ),
                 a, SLOT( destructiveAct() ) );
    } else {
        m_logical->setText( "Immutable" );
        m_logical->setEnabled( false );
    }

    QString cv = "n/a", iv = "n/a";
    double is = 0;

    try {
        cv = pkg.candidateVersion().versionString();
    } catch (...) {}

    try {
        iv = pkg.installedVersion().versionString();
    } catch (...) {}

    if (pkg.hasVersion())
        is = static_cast< pkgCache::VerIterator >( pkg.anyVersion() )->InstalledSize;

    const char *post = "BKMG";
    while (is > 1024 && post[0] != 'G')
        (is /= 1024), (++post);

    m_status->setText(
        format( "Status:", pkg.statusString( "Unknown" ) ) );
    m_action->setText(
        format( "Action:", pkg.actionString( "Unknown" ) ) );
    m_candidateVer->setText(
        format( "Candidate Version:", cv ) );
    m_installedVer->setText(
        format( "Installed Version:", iv ) );
    m_installedSize->setText(
        format( "Installed Size:", QString("%1%2").arg(
                    is, 0, 'f', (post[0] == 'K' || post[0] == 'B') ?
                    0 : 1 ).arg( post[0] ) ) );
}

bool ListerItemExtender::eventFilter( QObject *o, QEvent *e )
{
    if (o == m_description && e->type() == QEvent::Wheel) {
        kdDebug() << "discarding wheel event..." << endl;
        QApplication::sendEvent( this, e );
        return true;
    }
    return false;
}

void ListerItemExtender::resize( int w, int h ) {
    // setUpdatesEnabled( false );
    m_leftHeight = m_status->height() + m_action->height()
                   + m_section->height() + m_maintainer->height()
                   + m_candidateVer->height() + m_installedVer->height()
                   + m_name->height() + m_logical->height()
                   + 20;
    QWidget::resize( w, 500 );
    // setUpdatesEnabled( true );
    QWidget::resize(
        w,
        QMAX( m_description->contentsHeight() + 16,
              m_leftHeight /*leftColumn->layout()->minimumSize().height()*/ ) );
}

bool ListerItem::less( const ExtendableItem *b ) const
{
    return entity() < dynamic_cast< const ListerItem * >( b )->entity();
}

QString ListerTooltip::format( const QString &what,
                               const QString &txt, bool nobr )
{
    QString ret = "<b>" + what + "</b>&nbsp;" + (nobr ? "<nobr>" : "")
                  + txt + (nobr ? "</nobr>" : "") + "<br>";
    return ret;
}

void ListerTooltip::maybeTip( const QPoint &pt )
{
    if (! m_parent)
        return;
    kdDebug() << "ListTreeWidgetTooltip::maybeTip ()" << endl;
    ListerItem *x = dynamic_cast<ListerItem *>( m_parent->itemAt( pt ) );
    if (! x)
        return;
    if (x->extender())
        return; // no tips for extended items, thank you
    QString str = "<qt>";
    QString descr, cand, cur;
    descr = cand = cur = "<i>Not available</i>";
    entity::Package p( x->entity() );
    try {
        descr = p.shortDescription();
    } catch (...) {}
    try {
        cand = p.candidateVersion().versionString();
    } catch (...) {}
    try {
        cur = p.installedVersion().versionString();
    } catch (...) {}
    str += format( "Package:", p.name() );
    str += format( "Description:", descr );
    str += format( "Current&nbsp;Version:", cur );
    str += format( "Candidate&nbsp;Version:", cand );
    str.append ("</qt>");
    tip( m_parent->itemRect( x ), str );
}

#include "lister.moc"
