/** -*- C++ -*-
    @file libept/packageinfo.cpp
    @author Peter Rockai <me@mornfall.net>
*/

#include <qcolor.h>
#include <qlabel.h>
#include <qregexp.h>
#include <kdebug.h>
#include <klocale.h>

#include <libept/packageinfo.h>
#include <libept/utils.h>

using namespace ept;
using namespace cache;

PackageInfo::PackageInfo( QWidget *p, const char *n )
    : PackageInfoUi( p, n ), m_specificVersion( false )
{
    observeComponent< component::State >();
}

void PackageInfo::adjustFontSize( int s ) {
    ept::adjustFontSize( m_status, s );
    ept::adjustFontSize( m_change, s );
    ept::adjustFontSize( m_section, s );
    ept::adjustFontSize( m_installedSize, s );
    ept::adjustFontSize( m_maintainer, s );
    ept::adjustFontSize( m_candidateVer, s );
    ept::adjustFontSize( m_installedVer, s );
}

void PackageInfo::hideStatus() {
    m_status->hide();
    m_change->hide();
}

namespace ept {

QColor statusColor( entity::Package p )
{
    if ( !p.valid() )
        return Qt::black;

    QColor c = Qt::blue;
    if (p.isInstalled())
        c = Qt::darkGreen;
    if (p.isUpgradable())
        c = Qt::darkYellow;
    if (p.isBroken())
        c = Qt::red;
    return c;
}

QColor actionColor( entity::Package p )
{
    if ( !p.valid() )
        return Qt::black;

    QColor c = Qt::blue;
    if (p.markedNewInstall())
        c = Qt::darkGreen;
    if (p.markedUpgrade())
        c = Qt::darkYellow;
    if (p.markedReInstall())
        c = Qt::darkYellow;
    if (p.markedRemove())
        c = Qt::darkRed;
    if (p.markedPurge())
        c = Qt::red;
    if (p.willBreak())
        c = Qt::red;
    return c;
}

/* QString hexColor( QColor c )
{
    QString r( "#%1%2%3" );
    return r.arg( c.red(), -2, 16 ).arg( c.green(), -2, 16 ).arg( c.blue(), -2, 16 );
    } */

QString colorify( QColor c, QString s )
{
    return QString( "<font color=\"" ) + c.name()
            + "\">" + s + "</font>";
}

QString formatLongDescription( QString l )
{
    QRegExp rx( u8( "^(.*)\n" ) );
    rx.setMinimal( true );
    l.replace( rx, u8( "\\1</p><p>" ) );
    rx = QRegExp( u8( "\\n[ ]*\\.\\n" ) );
    l.replace( rx, u8( "</p><p>" ) );
    rx = QRegExp( u8( "\n   " ) );
    l.replace( rx, u8( " " ) );
    rx = QRegExp( u8( "\n  - (.*)(\n|$)" ) );
    rx.setMinimal( true );
    l.replace( rx, u8( "\n<li>\\1</li>\n" ) );
    l.replace( rx, u8( "\n<li>\\1</li>\n" ) );
    return QString( "<p>" ) + l + u8( "</p>" );
}

void PackageInfo::setPackage( entity::Package p )
{
    kdDebug() << "PackageInfo::setPackage()" << endl;
    // ho hum, probably XXX fix libapt-front?
    setVersion( p.valid() ? p.anyVersion() : entity::Version(), false );
}

void PackageInfo::setVersion( entity::Version v, bool specific )
{
    m_specificVersion = specific;
    m_version = v.stable();
    if ( !m_version.valid() ) return;
    kdDebug() << "PackageInfo::setVersion() (valid)" << endl;
    m_section->setText(
        labelFormat( i18n( "Section:" ), u8( v.section( u8( i18n( "Unknown" ) ) ) ) ) );
    m_maintainer->setText(
        labelFormat( i18n( "Maintainer:" ), u8( v.maintainer( u8( i18n( "Unknown" ) ) ) ),
                     false ) );

    notifyPostChange( 0 );
}

void PackageInfo::notifyPostRebuild( component::Base *b ) {
    return notifyPostChange( b );
}

void PackageInfo::notifyPostChange( component::Base * )
{
    if ( !m_version.valid() ) return;
    entity::Version v = m_version;
    entity::Package p = v.package();
    QString cv = i18n( "n/a" ), iv = i18n( "n/a" ), is = i18n( "n/a" ), status, action;

    if (p.valid()) {
        entity::Version _cv = p.candidateVersion();
        entity::Version _iv = p.installedVersion();
        
        if (_cv.valid()) {
            cv = u8( _cv.versionString() );
            is = u8( _cv.installedSizeString() );
        }
        
        if (_iv.valid()) {
            iv = _iv.versionString();
        }
    }

    std::string unk = u8( i18n( "unknown" ) );
    /* m_status->setText( i18n( "<nobr>Currently " ) + colorify(
                           statusColor( p ),
                           u8( p.statusString( unk ) ) )
                       + i18n( ", " ) + colorify(
                           actionColor( p ),
                           u8( p.actionString( unk ) ) ) + i18n( "
                           requested</nobr>" ) ); */
    m_status->setText(
        labelFormat( i18n( "Status:" ), colorify(
                         statusColor( p ),
                         u8( p.statusString( u8( i18n( "Unknown" ) ) ) ) ) ) );
    m_change->setText(
        labelFormat( i18n( "Requested change:" ), colorify(
                         actionColor( p ),
                         u8( p.actionString( u8( i18n( "Unknown" ) ) )
                         ) ) ) );

    m_candidateVer->setText(
        m_specificVersion ?
        labelFormat( i18n( "Version:" ),
                     v.versionString() + "(" + i18n( "candidate" ) + " " + cv + ")" )
        : labelFormat( i18n( "Candidate Version:" ), cv ) );
    m_installedVer->setText(
        labelFormat( i18n( "Installed Version:" ), iv ) );
    m_installedSize->setText(
        labelFormat( i18n( "Installed Size:" ), is ) );
}

}
