/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: common.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "fullengine.h"
#include "common.h"
#include "lists.h"
#include "engine.h"
#include "discover.h"
#include "handlemgr.h"
#include "volume.h"
#include "memman.h"
#include "internalAPI.h"
#include "message.h"
#include "cluster.h"
#include "remote.h"


boolean is_top_object(storage_object_t * obj) {

	boolean result = FALSE;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Examining object %s.\n", obj->name);

	/* The object must not be part of a volume. */
	if (obj->volume == NULL) {

		/* The object must not be part of a container. */
		if (obj->consuming_container == NULL) {

			/* The object must have no parent objects. */
			if (list_empty(obj->parent_objects)) {
				result = TRUE;

			} else {
				LOG_DEBUG("Object %s has parent object(s).\n", obj->name);
			}

		} else {
			LOG_DEBUG("Object %s is part of container %s.\n", obj->name, obj->consuming_container->name);
		}

	} else {
		LOG_DEBUG("Object %s is part of volume %s.\n", obj->name, obj->volume->name);
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}


int evms_can_delete(object_handle_t thing) {
	int rc = 0;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_delete(thing);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc == HANDLE_MANAGER_NO_ERROR) {
		switch (type) {
		case VOLUME:
			{
				logical_volume_t * volume = (logical_volume_t *) object;

				if (volume->flags & VOLFLAG_COMPATIBILITY) {
					if (is_kernel_volume_mounted(volume, DETAILS)) {
						rc = EBUSY;
                                        }
				}
				
				if (rc == 0) {
					/* Check if the volume is opened. */
					if (!is_volume_opened(volume)) {

						if ((volume->file_system_manager != NULL) &&
						    !(volume->flags & VOLFLAG_UNMKFS)) {
							rc = volume->file_system_manager->functions.fsim->can_unmkfs(volume);

							if (rc != 0) {
								LOG_DETAILS("Volume %s cannot be deleted because the file system %s cannot be removed.  Reason code is %d.\n", volume->name, volume->file_system_manager->short_name, rc);
							}
						}

						if (rc == 0) {
							rc = volume->object->plugin->functions.plugin->can_set_volume(volume->object, FALSE);
						}

					} else {
						LOG_DETAILS("Volume %s cannot be deleted because it is opened.\n", volume->name);
						if (volume->mount_point) {
							LOG_DETAILS("Volume %s is mounted on %s.\n", volume->name, volume->mount_point);
						}
						rc = EBUSY;
					}
				}
			}
			break;

		case CONTAINER:
			{
				storage_container_t * container = (storage_container_t *) object;

				rc = container->plugin->container_functions->can_delete_container(container);
			}
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			{
				storage_object_t * storage_object = (storage_object_t *) object;

				/*
				 * Make sure the object is a top level object.  We
				 * can't delete objects in the middle of a feature
				 * stack.  A top level object has no parent.
				 */
				if (is_top_object(storage_object)) {

					/*
					 * Ask the feature if the object can be
					 * deleted.
					 */
					rc = storage_object->plugin->functions.plugin->can_delete(storage_object);

				} else {
					/* The object is not a top level object. */
					LOG_DETAILS("Object %s cannot be deleted because it is not a top level object.\n", storage_object->name);
					rc = EINVAL;
				}
			}
			break;

		default:
			/* We can't delete a thing of this type. */
			LOG_DETAILS("A thing of type %#x cannot be deleted.\n", type);
			rc = EINVAL;
			break;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_destroy_object(storage_object_t * obj) {

	int rc = 0;

	LOG_PROC_ENTRY();

	/* Disks cannot be destroyed, they must be deleted.  Return 0 if this is a
	 * DISK so that the can_destroy_object() will not fail if this is a DISK.
	 * The disk will not be destroyed on a evms_destroy(), but anything stacked
	 * on top of the disk will.
	 */

	if (obj->object_type != DISK) {
		rc = obj->plugin->functions.plugin->can_delete(obj);

		if (rc == 0) {
			/*
			 * If the object is not produced by a container
			 * do a recursive check to see if all
			 * of its children can be deleted.
			 */
			if (obj->producing_container == NULL) {
				list_element_t iter;
				storage_object_t * child;

				LIST_FOR_EACH(obj->child_objects, iter, child) {
					rc = can_destroy_object(child);
					if (rc != 0) {
						break;
					}
				}
			}
		}
	}

	/* If the object is a DISK, we return 0 so that the recursive check doesn't
	 * fail and the recursion ends.
	 * can_destroy_object() might be called initially on a DISK (e.g., an EVMS
	 * volume is made from a DISK and the volume is being destroyed).  In that
	 * case, can_destroy_object() will return 0, but a subsequent call to
	 * evms_destroy() will destroy the volume but leave the disk.
	 */

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_can_destroy(object_handle_t thing) {

	int rc;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_destroy(thing);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc == HANDLE_MANAGER_NO_ERROR) {
		switch (type) {
			case VOLUME:
				{
					logical_volume_t * volume = (logical_volume_t *) object;

					if (volume->flags & VOLFLAG_COMPATIBILITY) {
						if (is_kernel_volume_mounted(volume, DETAILS)) {
							rc = EBUSY;
						}
					}
					
					if (rc == 0) {
						/* Check if the volume is opened. */
						if (!is_volume_opened(volume)) {

							if ((volume->file_system_manager != NULL) &&
							    !(volume->flags & VOLFLAG_UNMKFS)) {
								rc = volume->file_system_manager->functions.fsim->can_unmkfs(volume);

								if (rc != 0) {
									LOG_DETAILS("Volume %s cannot be destroyed because the file system %s cannot be removed.  Reason code is %d.\n", volume->name, volume->file_system_manager->short_name, rc);
								}
							}

							if (rc == 0) {
								rc = can_destroy_object(volume->object);
							}

						} else {
							LOG_DETAILS("Volume %s cannot be destroyed because it is opened.\n", volume->name);
							if (volume->mount_point != NULL) {
								LOG_DETAILS("Volume %s is mounted on %s.\n", volume->name, volume->mount_point);
							}
							rc = EBUSY;
						}
					}
				}
				break;

			case CONTAINER:
				{
					storage_container_t * container = (storage_container_t *) object;
					list_element_t iter;
					storage_object_t * obj;

					rc = container->plugin->container_functions->can_delete_container(container);

					if (rc == 0) {
						LIST_FOR_EACH(container->objects_consumed, iter, obj) {
							rc = can_destroy_object(obj);
							if (rc != 0) {
								break;
							}
						}
					}
				}
				break;

			case EVMS_OBJECT:
			case REGION:
			case SEGMENT:
				{
					storage_object_t * obj = object;

					if (is_top_object(obj)) {

						rc = can_destroy_object(obj);

					} else {
						LOG_DETAILS("Object %s cannot be destroyed because it is not a top level object.\n", obj->name);
						rc = EINVAL;
					}
				}
				break;

			case DISK:
				{
					storage_object_t * obj = object;

					LOG_DETAILS("Disk %s cannot be destroyed because disks cannot be destroyed.  Disks must be deleted.\n", obj->name);
					rc = EINVAL;
				}
				break;

			default:
				rc = EINVAL;
				break;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int find_top_objects(storage_object_t * obj,
			    list_anchor_t      top_object_list) {

	int rc = 0;

	if (list_empty(obj->parent_objects)) {
		list_element_t el;

		el = insert_thing(top_object_list,
				  obj,
				  INSERT_AFTER,
				  NULL);

		if (el == NULL) {
			LOG_CRITICAL("Error inserting object %s into the top object list.\n", obj->name);
			rc = ENOMEM;
		}

	} else {
		list_element_t iter;
		storage_object_t * parent;

		LIST_FOR_EACH(obj->parent_objects, iter, parent) {
			rc = find_top_objects(parent, top_object_list);

			if (rc != 0) {
				break;
			}
		}
	}

	return rc;
}


static int can_replace(object_handle_t source_handle,
		       object_handle_t target_handle,
		       debug_level_t   log_level) {

	int rc = 0;
	int src_rc = 0;
	int trg_rc = 0;
	object_type_t type;
	void  * thing = NULL;
	list_anchor_t top_objects = NULL;
	storage_object_t * source = NULL;
	storage_object_t * target = NULL;

	LOG_PROC_ENTRY();

	/* Check for valid source and target. */
	src_rc = translate_handle(source_handle, &thing, &type);
	if (src_rc == HANDLE_MANAGER_NO_ERROR) {
		switch (type) {
		case DISK:
		case SEGMENT:
		case REGION:
		case EVMS_OBJECT:
			source = (storage_object_t *) thing;
			break;

		case CONTAINER:
			LOG(log_level, "Containers cannot be replaced.\n");
			src_rc = EINVAL;
			break;

		case VOLUME:
			LOG(log_level, "Volumes cannot be replaced.\n");
			src_rc = EINVAL;
			break;

		case PLUGIN:
			LOG(log_level, "Plug-ins cannot be replaced.\n");
			src_rc = EINVAL;
			break;

		default:
			LOG(log_level, "Source handle %d maps to an unknown thing of type %#x.\n", source_handle, type);
			src_rc = EINVAL;
		}
	}

	if (src_rc == 0) {
		int tmp_rc;
		logical_volume_t * vol;

		LOG_DEBUG("Can object %s be replaced?\n", source->name);

		/*
		 * Top objects cannot be replaced.  What is the use of replacing
		 * an object that is not in use with another object that is not
		 * in use?  This also prevents the use of any target objects
		 * created by plug-ins for their own internal moves as the target
		 * will be a top level object.
		 */
		if (is_top_object(source)) {
			LOG(log_level, "Object %s is a top level object.  Top level objects cannot be replaced.\n",
				       source->name);
			src_rc = EINVAL;
		}

		/*
		 * Is the object a volume's object?  Certain rules apply to
		 * volume objects.
		 */
		if (list_empty(source->parent_objects) && (source->volume != NULL)) {
			if (source->volume->flags & VOLFLAG_COMPATIBILITY) {
				LOG(log_level, "%s is the object for compatiblity volume %s.  Compatibility volume objects cannot be replaced.\n",
					       source->name, source->volume->name);
				src_rc = EINVAL;

			} else {
				/* It's an EVMS volume.
				 * A "raw" EVMS volume (one made on top of a
				 * nonEVMS object) has its own device-mapper
				 * mapping.  The volume can have its object
				 * replaced since we can change the mapping of
				 * the volume.
				 * A non-raw EVMS volume simply uses the
				 * device-mapper device of the EVMS object.
				 * Non-raw EVSM volumes can only be replaced if
				 * the volume is not mounted, since the
				 * major:minor of the volume will change due to
				 * it using a differnt object.
				 */
				if (source->object_type == EVMS_OBJECT) {
					if (is_volume_mounted(source->volume)) {
						LOG(log_level, "%s is the object for volume %s.  The object can only be replaced if the volume is not mounted.\n",
							       source->name, source->volume->name);
						src_rc = EBUSY;
					}
				}
			}
		}

		/* Can't move non-data objects. */
		if (source->data_type != DATA_TYPE) {
			LOG(log_level, "Object %s is not a data object.  Only data objects can be replaced.\n",
				       source->name);
			src_rc = EINVAL;
		}

		if (!engine_can_online_copy()) {
			if (!engine_is_offline(source, &vol)) {
				LOG(log_level, "Object %s is part of volume %s which is currently opened.  Replace must be done off-line.\n",
					       source->name, vol->name);
				src_rc = EINVAL;
			}
		}

		/* Can't replace if there are no available targets. */
		tmp_rc = engine_get_object_list(0, DATA_TYPE, NULL, NULL, VALID_INPUT_OBJECT, &top_objects);
		if (tmp_rc == 0) {
			if (list_empty(top_objects)) {
				LOG(log_level, "There are no available target objects for the replace.\n");
				src_rc = EINVAL;
			}

			destroy_list(top_objects);

		} else {
			src_rc = tmp_rc;
		}
	}

	if (target_handle != 0) {
		trg_rc = translate_handle(target_handle, &thing, &type);
		if (trg_rc == HANDLE_MANAGER_NO_ERROR) {
			switch (type) {
			case DISK:
			case SEGMENT:
			case REGION:
			case EVMS_OBJECT:
				target = (storage_object_t *) thing;
				if (!is_top_object(target)) {
					LOG_DEBUG("Target %s is not a top level object.\n", target->name);
					trg_rc = EINVAL;
				}
				break;

			case CONTAINER:
				LOG(log_level, "Containers cannot be the target of a replace.\n");
				trg_rc = EINVAL;
				break;

			case VOLUME:
				LOG(log_level, "Volumes cannot be the target of a replace.\n");
				trg_rc = EINVAL;
				break;

			case PLUGIN:
				LOG(log_level, "Plug-ins cannot be the target of a replace.\n");
				trg_rc = EINVAL;
				break;

			default:
				LOG(log_level, "Target handle %d maps to an unknown thing of type %#x.\n",
					       target_handle, type);
				trg_rc = EINVAL;
			}

			if (trg_rc == 0) {
				/* Can't move to non-data objects. */
				if (target->data_type != DATA_TYPE) {
					LOG(log_level, "Object %s is not a data object.  Only data objects can be the target of a replace.\n",
						       target->name);
					trg_rc = EINVAL;
				}

			}
		}
	}

	rc = (src_rc != 0) ? src_rc : trg_rc;

	if ((rc == 0) && (target != NULL)) {

		LOG_DEBUG("Can object %s be replaced with object %s?\n", source->name, target->name);

		/* An object cannot replace itself. */
		if (source == target) {
			LOG(log_level, "Object %s cannot replace itself.\n", source->name);
			rc = EINVAL;

		} else {
			/* The source and target must be in the same disk group. */
			if (source->disk_group == target->disk_group) {

				/*
				 * The source object cannot be one of the
				 * objects that comprises the target object.
				 */
				STATIC_LIST_DECL(top_objects);
				list_element_t iter;
				storage_object_t * obj;

				LIST_FOR_EACH(source->parent_objects, iter, obj) {
					rc = find_top_objects(obj, &top_objects);

					if (rc != 0) {
						break;
					}
				}

				if (rc == 0) {
					LIST_FOR_EACH(&top_objects, iter, obj) {
						if (obj == target) {
							LOG(log_level, "Target object %s comprises source object %s.\n",
								       target->name, source->name);
							rc = EINVAL;
							break;
						}
					}
				}

				if (list_empty(source->parent_objects)) {
					if ((source->object_type == EVMS_OBJECT) &&
					    (target->object_type != EVMS_OBJECT)) {
						if (target->size < source->size + EVMS_FEATURE_HEADER_SECTORS * 2) {
							LOG(log_level, "Target %s is smaller than source %s.\n",
								       target->name, source->name);
							rc = EINVAL;
						}
					} else {
						if (target->size < source->size) {
							LOG(log_level, "Target %s is smaller than source %s.\n",
								       target->name, source->name);
							rc = EINVAL;
						}
					}

					/*
					 * We can't do an online move of a raw
					 * EVMS volume object to an EVMS object
					 * because that would cause the volume's
					 * major:minor to change to use that of
					 * the EVMS object instead of the
					 * mapping used to make the raw EVMS
					 * volume.  Swapping of a major:minor
					 * can't be done while the volume is
					 * mounted.
					 */
					if ((source->volume != NULL) &&
					    is_volume_mounted(source->volume) &&
					    (source->object_type != EVMS_OBJECT) &&
					    (target->object_type == EVMS_OBJECT)) {
						LOG(log_level, "non-EVMS object %s cannot be replaced with EVMS object %s while volume %s is mounted.\n",
							       source->name, target->name, source->volume->name);
						rc = EBUSY;
					}
				}

			} else {
				LOG(log_level, "Source %s and target %s are not in the same disk group.\n",
					       source->name, target->name);
				LOG(log_level, "%s is in disk group %s.\n", source->name,
					       (source->disk_group == NULL) ? "(local)" : source->disk_group->name);
				LOG(log_level, "%s is in disk group %s.\n", target->name,
					       (target->disk_group == NULL) ? "(local)" : target->disk_group->name);
				rc = EINVAL;
			}
		}
	}

	if (rc == 0) {
		if (!list_empty(source->parent_objects)) {
			storage_object_t * parent;

			/*
			 * Ask the parent if it can replace its child.  Any
			 * parent will do.  If there are multiple, parents they
			 * should all be managed by the same plug-in.
			 */
			parent = first_thing(source->parent_objects, NULL);
			rc = parent->plugin->functions.plugin->can_replace_child(parent, source, target);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_can_replace(object_handle_t source_handle, object_handle_t target_handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_replace(source_handle, target_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (replace_plugin == NULL) {
		LOG_PROC_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	rc = can_replace(source_handle, target_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * evms_can_set_info will determine if a thing's information can be set.
 */
int evms_can_set_info(object_handle_t thing_handle) {

	int rc = 0;
	object_type_t type;
	void  * thing = NULL;
	int count = -1;
	task_context_t * task;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_set_info(thing_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	task = engine_alloc(sizeof(task_context_t));

	if (task != NULL) {
		rc = translate_handle(thing_handle, &thing, &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {

			/*
			 * Check if the plug-in supports setting of object information
			 * by getting the option count for EVMS_Task_Set_Info.  If the
			 * option count is 0 or less, then the plug-in does not support
			 * the setting of object information.
			 */
			switch (type) {
				case DISK:
				case SEGMENT:
				case REGION:
				case EVMS_OBJECT:
					{
						storage_object_t * obj = (storage_object_t *) thing;
						task->object = obj;
						task->action = EVMS_Task_Set_Info;

						count = obj->plugin->functions.plugin->get_option_count(task);

						if (count <= 0) {
							LOG_DETAILS("Plug-in %s does not have any descriptors for setting object info.\n", obj->plugin->short_name);
							rc = ENOSYS;
						}
					}
					break;

				case CONTAINER:
					{
						storage_container_t * con = (storage_container_t *) thing;
						task->container = con;
						task->action = EVMS_Task_Set_Container_Info;

						count = con->plugin->functions.plugin->get_option_count(task);

						if (count <= 0) {
							LOG_DETAILS("Plug-in %s does not have any descriptors for setting container info.\n", con->plugin->short_name);
							rc = ENOSYS;
						}
					}
					break;

				case VOLUME:
					{
						logical_volume_t * vol = (logical_volume_t *) thing;

						if (vol->flags & VOLFLAG_COMPATIBILITY) {
							if (is_kernel_volume_mounted(vol, DETAILS)) {
								rc = EBUSY;
							}
						}
						
						if (rc == 0) {
							if (vol->file_system_manager != NULL) {
								task->volume = vol;
								task->action = EVMS_Task_Set_Info;

								rc = vol->file_system_manager->functions.fsim->get_option_count(task);

								if (count <= 0) {
									LOG_DETAILS("Plug-in %s does not have any descriptors for setting volume info.\n", vol->file_system_manager->short_name);
									rc = ENOSYS;
								}

							} else {
								LOG_DETAILS("Volume %s does not have a file system interface module.\n", vol->name);
								rc = ENOSYS;
							}
						}
					}
					break;

				default:
					LOG_ERROR("Cannot set info on a thing of type %#x.\n", type);
					rc = EINVAL;
					break;
			}
		}

		engine_free(task);

	} else {
		LOG_CRITICAL("Error allocating memory for a task context.\n");
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static inline int get_plugin_extended_info(plugin_record_t         * plugin,
					   char                    * descriptor_name,
					   extended_info_array_t * * info) {
	int rc = 0;

	LOG_PROC_ENTRY();

	switch (GetPluginType(plugin->id)) {
	case EVMS_DEVICE_MANAGER:
	case EVMS_SEGMENT_MANAGER:
	case EVMS_REGION_MANAGER:
	case EVMS_FEATURE:
	case EVMS_ASSOCIATIVE_FEATURE:

		rc = plugin->functions.plugin->get_plugin_info(descriptor_name, info);
		break;

	case EVMS_FILESYSTEM_INTERFACE_MODULE:
		rc = plugin->functions.fsim->get_plugin_info(descriptor_name, info);
		break;

	case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
		rc = plugin->functions.cluster->get_plugin_info(descriptor_name, info);
		break;

	default:
		LOG_ERROR("We don't get info for %d plug-in types.\n", GetPluginType(plugin->id) );
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void free_extended_info_object_contents(void * object) {

	int i, j;
	extended_info_array_t * eia = (extended_info_array_t *) object;
	extended_info_t       * ei;

	LOG_PROC_ENTRY();

	for (i = 0; i < eia->count; i++) {
		ei = &eia->info[i];

		if (ei->name != NULL) {
			engine_free(ei->name);
		}
		if (ei->title != NULL) {
			engine_free(ei->title);
		}
		if (ei->desc != NULL) {
			engine_free(ei->desc);
		}

		switch (ei->collection_type) {
		case EVMS_Collection_List:
			if (ei->collection.list != NULL) {
				if (ei->type == EVMS_Type_String) {
					for (j = 0; j < ei->collection.list->count; j++) {
						if (ei->collection.list->value[j].s != NULL) {
							engine_free(ei->collection.list->value[j].s);
						}
					}
				}
				engine_free(ei->collection.list);

			} else {
				LOG_WARNING("Collection says it has a list but the list pointer is NULL.\n");
			}
			break;

		case EVMS_Collection_Range:
			if (ei->collection.range != NULL) {
				engine_free((void *) ei->collection.range);

			} else {
				LOG_WARNING("Collection says it has a range but the range pointer is NULL.\n");
			}

		default:
			break;
		}

		if ((ei->group.group_number != 0) && (ei->group.group_name != NULL)) {
			engine_free(ei->group.group_name);
		}
	}

	LOG_PROC_EXIT_VOID();
}


int evms_get_extended_info(object_handle_t thing, char * descriptor_name, extended_info_array_t * * user_info) {
	int rc = 0;
	void * object;
	object_type_t type;
	extended_info_array_t * info = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_extended_info(thing, descriptor_name, user_info);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LOG_DEBUG("Get extended info for handle %d.\n", thing);
	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc == HANDLE_MANAGER_NO_ERROR) {
		switch (type) {
			case PLUGIN:{
					plugin_record_t * plugin = (plugin_record_t *) object;

					LOG_DEBUG("Handle %d maps to plug-in %s.\n", thing, plugin->short_name);
					rc = get_plugin_extended_info(plugin, descriptor_name, &info);
				}
				break;

			case DISK:
			case SEGMENT:
			case REGION:
			case EVMS_OBJECT:{
					storage_object_t * obj = (storage_object_t *) object;

					LOG_DEBUG("Handle %d maps to storage object %s.\n", thing, obj->name);
					rc = obj->plugin->functions.plugin->get_info(obj, descriptor_name, &info);
				}
				break;


			case CONTAINER:{
					storage_container_t * container = (storage_container_t *) object;

					LOG_DEBUG("Handle %d maps to container %s.\n", thing, container->name);
					rc = container->plugin->container_functions->get_container_info(container, descriptor_name, &info);
				}
				break;

			case VOLUME:{
					logical_volume_t * volume = (logical_volume_t *) object;

					LOG_DEBUG("Handle %d maps to volume %s.\n", thing, volume->name);
					if (volume->file_system_manager != NULL) {
						rc = volume->file_system_manager->functions.fsim->get_volume_info(volume, descriptor_name, &info);

					} else {
						/*
						 * The volume doesn't have an FSIM.
						 * Return an empty info array.
						 */
						info = engine_alloc(sizeof(extended_info_array_t));

						if (info == NULL) {
							rc = ENOMEM;
						}
					}
				}
				break;

			default:
				LOG_DEBUG("Handle %d maps to unknown object type %d.\n", thing, type);
				rc = EINVAL;
				break;
		}

		if ((rc == 0) && (info != NULL)) {

			/*
			 * Copy the extended info that was returned from the plug-in
			 * into an application memory object suitable for returning
			 * to the API caller.
			 */
			if (info->count == 0) {

				*user_info = alloc_app_struct(sizeof(info->count), NULL);
				if (*user_info != NULL) {
					/* Fill in the structure returned to the app. */
					(*user_info)->count = 0;

					/*
					 * Free the structure that was returned by the plug-in.
					 */
					engine_free(info);

				} else {
					rc = ENOMEM;
				}

			} else {
				/* The extended info array has at least one entry. */
				uint size = sizeof(extended_info_array_t) + (sizeof(extended_info_t) * (info->count - 1));

				*user_info = alloc_app_struct(size, free_extended_info_object_contents);

				if (*user_info != NULL) {
					/* Fill in the structure returned to the app. */
					memcpy(*user_info, info, size);

					/*
					 * Free the structure that was returned by the plug-in.
					 */
					engine_free(info);

				} else {
					rc = ENOMEM;
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


void update_all_stop_data_states() {

	list_element_t iter;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	/*
	 * Objects that are no longer top level objects cannot be guaranteed
	 * to have stop data on them.  Once they are consumed by another
	 * object or container, the consumer can write anything to the object
	 * anywhere on the object, including the stop data area.
	 */
	LIST_FOR_EACH(&disks_list, iter, obj) {
		if (!is_top_object(obj)) {
			obj->flags &= ~SOFLAG_HAS_STOP_DATA;
		}
	}
	LIST_FOR_EACH(&segments_list, iter, obj) {
		if (!is_top_object(obj)) {
			obj->flags &= ~SOFLAG_HAS_STOP_DATA;
		}
	}
	LIST_FOR_EACH(&regions_list, iter, obj) {
		if (!is_top_object(obj)) {
			obj->flags &= ~SOFLAG_HAS_STOP_DATA;
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


/*
 * evms_set_info will set information, specified in options, for a thing.
 */
int evms_set_info(engine_handle_t  thing_handle,
		  option_array_t * options) {

	int rc = 0;
	object_type_t type;
	void *        thing;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_set_info(thing_handle, options);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

		rc = translate_handle(thing_handle, &thing, &type);

		if (rc != HANDLE_MANAGER_NO_ERROR) {
			LOG_PROC_EXIT_INT(rc);
			return rc;
		}

	switch (type) {
		case DISK:
		case SEGMENT:
		case REGION:
		case EVMS_OBJECT:
			{
				storage_object_t * obj = (storage_object_t *) thing;
				char * old_object_name[sizeof(obj->name)] = {'\0'};

				rc = engine_can_rename(obj);

				if (rc == 0) {
					/* Check if it's part of a compatibility volume. */
					if ((obj->volume != NULL) &&
					    (obj->object_type != EVMS_OBJECT) &&
					    (obj->feature_header == NULL)) {

						/*
						 * Check if it's the top object of the
						 * compatibility volume.
						 */
						if (list_empty(obj->parent_objects)) {
							/*
							 * This is the top object of a compatibility
							 * volume.  If the object's name changes,
							 * the volume name must change as well.
							 * save the current volume name so that
							 * we can check if it changed during the
							 * call to the plug-in's set_info().
							 */
							memcpy(old_object_name, obj->name, sizeof(obj->name));
						}
					}

					rc = obj->plugin->functions.plugin->set_info(obj, options);

					/*
					 * Check if we need to handle a possible name
					 * change.
					 */
					if (old_object_name[0] != '\0') {
						if (memcmp(old_object_name, obj->name, sizeof(obj->name)) != 0) {

							char * pVolName = obj->name;
							list_element_t el;

							/*
							 * The object is a top level object of a
							 * compatibility volume and its name has
							 * changed.  Delete the old volume and
							 * create a new one with the new name.
							 */

							/* Unregister the current volume name. */
							engine_unregister_name(obj->volume->name);

							/* Delete the volume from the volumes_list. */
							remove_thing(&volumes_list, obj->volume);

							if (!(obj->volume->flags & VOLFLAG_NEW)) {
								/* Put it on the volume_delete_list. */
								el = insert_thing(&volume_delete_list,
										  obj->volume,
										  INSERT_AFTER,
										  NULL);

								if (el == NULL) {
									LOG_CRITICAL("Error putting volume %s on the volume_delete_list.\n", obj->volume->name);
									rc = 0;
								}
							}

							/*
							 * Build a new volume name for the object.
							 */
							if (strncmp(pVolName, EVMS_DEV_NODE_PATH, strlen(EVMS_DEV_NODE_PATH)) != 0) {
								pVolName = engine_alloc(strlen(pVolName) + strlen(EVMS_DEV_NODE_PATH) + 1);
								if (pVolName != NULL) {
									strcpy(pVolName, EVMS_DEV_NODE_PATH);
									strcat(pVolName, obj->name);
								} else {
									LOG_CRITICAL("Could not get memory for building a volume name for object %s.\n", obj->name);
									rc = ENOMEM;
								}
							}

							if (rc == 0) {

								/*
								 * Make a new volume patterned after the
								 * old one.
								 */
								logical_volume_t * old_vol = obj->volume;
								rc = make_volume(obj, pVolName);

								if (rc == 0) {
									obj->volume->dev_major     = old_vol->dev_major;
									obj->volume->dev_minor     = old_vol->dev_minor;
									obj->volume->serial_number = old_vol->serial_number;
									obj->volume->flags        |= old_vol->flags;
									obj->volume->flags        |= (VOLFLAG_NEW | VOLFLAG_DIRTY | VOLFLAG_NEEDS_ACTIVATE);
								}

								engine_free(pVolName);
							}
						}
					}
				}
			}
			break;

		case CONTAINER:
			{
				storage_container_t * con = (storage_container_t *) thing;
				rc = con->plugin->container_functions->set_container_info(con, options);
			}
			break;

		case VOLUME:
			{
				logical_volume_t * vol = (logical_volume_t *) thing;

				if (vol->flags & VOLFLAG_COMPATIBILITY) {
					if (is_kernel_volume_mounted(vol, DETAILS)) {
						rc = EBUSY;
                                        }
				}
				
				if (rc == 0) {
					if (vol->file_system_manager != NULL) {
						rc = vol->file_system_manager->functions.fsim->set_volume_info(vol, options);

					} else {
						LOG_DEBUG("Volume %s does not have a file system interface module.\n", vol->name);
						rc = ENOSYS;
					}
				}
			}
			break;

		default:
			LOG_ERROR("Cannot set info on a thing of type %#x.\n", type);
			rc = EINVAL;
			break;
	}

	if (rc == 0) {
		/*
		 * A plug-in can (ab)use set_info() to add or remove child
		 * objects to/from the given object.  Since we don't know
		 * what went on inside the set_info() and how it affected
		 * other objects, update the stop data state on all the
		 * objects that can have stop data.
		 */
		update_all_stop_data_states();
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Remove this object's feature headers and add the feature header sectors to
 * the kill list.
 */
int remove_feature_headers(storage_object_t * obj) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (obj->feature_header != NULL) {
		LOG_DEBUG("Removing feature headers from object %s.\n", obj->name);

		engine_free(obj->feature_header);
		obj->feature_header = NULL;

		/* Wipe out both feature headers. */
		obj->plugin->functions.plugin->add_sectors_to_kill_list(obj, obj->size - (EVMS_FEATURE_HEADER_SECTORS * 2), EVMS_FEATURE_HEADER_SECTORS * 2);

		/*
		 * If the object had stop data on it, it will be wiped out by
		 * the kill sectors.
		 */
		obj->flags &= ~SOFLAG_HAS_STOP_DATA;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Mark this object's feature header(s) dirty.  The feature header(s) for this
 * object is/are in this object's child objects.
 */
int set_feature_header_dirty(storage_object_t * obj) {

	int rc = 0;

	LOG_PROC_ENTRY();

	/* Only EVMS objects have feature headers. */
	if (obj->object_type == EVMS_OBJECT) {
		list_element_t iter;
		storage_object_t * child;

		LIST_FOR_EACH(obj->child_objects, iter, child) {
			if (child->feature_header != NULL) {
				child->flags |= SOFLAG_FEATURE_HEADER_DIRTY;
				set_feature_header_dirty(child);
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int do_volume_delete_warnings(logical_volume_t * vol) {

	int rc = 0;

	LOG_PROC_ENTRY();

	/*
	 * If the volume is new then it really doesn't have an FSIM associated with
	 * it and it's not already a compatibility volume so there are no warnings
	 * to post.
	 */
	if (vol->flags & VOLFLAG_NEW) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * The volume has an FSIM, notify the user that unmkfs will be run on the
	 * volume.
	 */
	if (vol->file_system_manager != NULL) {

		/* Check if the FSIM can unmkfs. */
		rc = vol->file_system_manager->functions.fsim->can_unmkfs(vol);
		if (rc == 0) {
			/*
			 * Only need to ask the user to run unmkfs if the current FSIM is
			 * the original FSIM, it isn't already marked for unmkfs, and the
			 * volume is not read-only.
			 */
			if ((vol->file_system_manager == vol->original_fsim) &&
			    !(vol->flags & (VOLFLAG_UNMKFS | VOLFLAG_READ_ONLY))) {

				char * choices[] = {"Continue", "Cancel", NULL};
				int answer = 0;	    /* Default is "Continue". */

				engine_user_message(&answer, choices,
						    "Volume \"%s\" has the %s file system on it.  "
						    "The file system will be removed along with the volume.\n",
						    vol->name, vol->file_system_manager->short_name);
				switch (answer) {
				case 0:
					break;

				default:
					/* Any other answer is a cancel. */
					rc = E_CANCELED;
					break;
				}
			}

		} else {
			LOG_ERROR("FSIM %s will not allow the file system to be removed from volume %s.  Error code is %d.\n", vol->file_system_manager->short_name, vol->name, rc);
		}

	} else {
		/*
		 * The volume does not have an FSIM assigned to it.  If it did not
		 * originally have and FSIM assigned to it, it may have a file system on
		 * it for which we have no FSIM.  If the volume is not marked read-only
		 * ask the user if we should run a crude "unmkfs" that wipes out the
		 * first 1MB of the disk.
		 */
		if ((vol->original_fsim == NULL) &&
		    !(vol->flags & VOLFLAG_READ_ONLY)) {
			char * choices[] = {"Write zeros", "Do not write zeros", "Cancel", NULL};
			int answer = 0;	    /* Default is "Write zeroes". */

			sector_count_t one_meg_in_sectors = (1024 * 1024) >> EVMS_VSECTOR_SIZE_SHIFT;

			engine_user_message(&answer, choices,
					    "Volume \"%s\" does not have a File System Interface Module (FSIM) assigned to it.  "
					    "The volume may have a file system on it, but none of the installed FSIM plug-ins recognizes it.  "
					    "Do you want to write zeros to the %sdisable any file system that may be on the volume?\n",
					    vol->name,
					    (vol->object->size > one_meg_in_sectors) ? "first 1MB of the volume to potentially " : "volume to ");
			switch (answer) {
			case 0:
				rc = vol->object->plugin->functions.plugin->add_sectors_to_kill_list(vol->object, 0, min(vol->vol_size, (1024 * 1024) >> EVMS_VSECTOR_SIZE_SHIFT));
				break;

			case 1:
				/* Nothing to do. */
				break;

			case 2:
				rc = E_CANCELED;
			default:
				break;

			}
		}
	}

	if (rc == 0) {
		/*
		 * If this is a compatibility volume and we did not schedule an unmkfs,
		 * warn the user that the last few sectors will have metadata written
		 * to them.
		 */
		if ((vol->flags & VOLFLAG_COMPATIBILITY) &&
		    !(vol->flags & VOLFLAG_UNMKFS)) {
			char * choices[] = {"Continue", "Cancel", NULL};
			int answer = 0;	    /* Default is "Continue". */

			engine_user_message(&answer, choices,
					    "Volume \"%s\" is not an EVMS volume.  "
					    "Removing a non-EVMS volume requires writing %zd bytes of metadata at the end of the volume. "
					    "The metadata will overwrite any data that may be at the end of the volume.  "
					    "Do you want to continue with the delete?\n",
					    vol->name, (EVMS_FEATURE_HEADER_SECTORS * 2) << EVMS_VSECTOR_SIZE_SHIFT);

			if (answer != 0) {
				/* An answer other than "Continue" is considered a cancel. */
				rc = E_CANCELED;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int delete_volume(logical_volume_t * vol) {

	int rc = 0;
	storage_object_t * obj = vol->object;

	LOG_PROC_ENTRY();

	/* The volume must not be opened. */
	if (is_volume_opened(vol)) {
		LOG_ERROR("Volume %s cannot be deleted because it is opened.\n", vol->name);
		if (vol->mount_point != NULL) {
			LOG_ERROR("Volume %s is mounted on %s.\n", vol->name, vol->mount_point);
		}
		LOG_PROC_EXIT_INT(EBUSY);
		return EBUSY;
	}

	rc = obj->plugin->functions.plugin->can_set_volume(obj, FALSE);

	if (rc != 0) {
		LOG_ERROR("Object %s does not allow volume %s to be deleted.  Reason code is %d.\n", obj->name, vol->name, rc);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * Display any warning message(s) for the removal of this volume.
	 * do_volume_delete_warning() can return E_CANCELED if the user selects
	 * to cancel the removal.
	 */
	rc = do_volume_delete_warnings(vol);

	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	destroy_handle(vol->app_handle);

	engine_unregister_name(vol->name);
	if (vol->flags & VOLFLAG_HAS_OWN_DEVICE) {
		engine_unregister_name(vol->name + EVMS_DEV_NODE_PATH_LEN);
	}

	/*
	 * Clear out the volume pointer on this object and on all the child
	 * objects in the tree.  Notify all the objects that the volume has been
	 * deleted.
	 */
	set_volume_in_object(obj,
			     NULL);    /* volume pointer */

	remove_thing(&volumes_list, vol);

	if (vol->flags & VOLFLAG_NEW) {

		/*
		 * This volume hasn't been committed yet.  Just throw away the
		 * volume structure.
		 */
		LOG_DEBUG("Volume is new, so just toss it out.\n");
		engine_free(vol);

	} else {
		list_element_t el;

		/*
		 * If the volume is not read-only, the FSIM is the original
		 * FSIM for the volume, do the setup for unmkfs.
		 */
		if (!(vol->flags & VOLFLAG_READ_ONLY) &&
		    (vol->file_system_manager == vol->original_fsim) &&
		    vol->original_fsim != NULL) {
			rc = vol->original_fsim->functions.fsim->unmkfs_setup(vol);

			if (rc != 0) {
				LOG_WARNING("Call to the unmkfs_setup() function of the %s FSIM returned arror code %d: %s\n",
					    vol->original_fsim->short_name, rc, evms_strerror(rc));
				LOG_PROC_EXIT_INT(rc);
				return rc;
			}

			/*
			 * Turn on the VOLFLAG_UNMKFS flag so that unmkfs will
			 * be run at commit time.  If the volume's VOLFLAG_MKFS
			 * is set we don't care.  The volume will be deleted and
			 * so won't have the chance to run mkfs.
			 */
			vol->flags |= VOLFLAG_UNMKFS;
		}

		/* Mark that this volume has no FSIM. */
		vol->file_system_manager = NULL;

		/*
		 * If the volume's object is associated with another object, put
		 * the other object's volume into this volume's
		 * associated_volume field.
		 */
		if (obj->associated_object != NULL) {
			vol->associated_volume = obj->associated_object->volume;
		}

		/* If the volume is active, it needs to be deactivated. */
		if (vol->flags & VOLFLAG_ACTIVE) {
			vol->flags |= VOLFLAG_NEEDS_DEACTIVATE;
		}

		/*
		 * If the object has a feature header attached, it is one for making the
		 * object into an EVMS volume.  Remove the feature header and kill the
		 * sectors so that it won't be found again.  Mark that the volume needs
		 * to be deactivated.
		 */
		if (obj->feature_header != NULL) {
			LOG_DEBUG("Removing feature headers from object %s.\n", obj->name);
			rc = obj->plugin->functions.plugin->add_sectors_to_kill_list(obj, obj->size - (EVMS_FEATURE_HEADER_SECTORS * 2), EVMS_FEATURE_HEADER_SECTORS * 2);

			if (rc == 0) {
				engine_free(obj->feature_header);
				obj->feature_header = NULL;

			} else {
				LOG_SERIOUS("add_sectors_to_kill_list() returned error code %d when called to wipe out volume feature header on volume %s.\n", rc, vol->name);
				LOG_PROC_EXIT_INT(rc);
				return rc;
			}
		}

		LOG_DEBUG("Volume exists.  Put it on the delete list.\n");
		el = insert_thing(&volume_delete_list,
				  vol,
				  INSERT_AFTER,
                                  NULL);

		if (el == NULL) {
			LOG_CRITICAL("Error putting volume %s on the volume delete list.\n", vol->name);
			LOG_PROC_EXIT_INT(ENOMEM);
			return ENOMEM;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


storage_object_t * get_working_top_object(storage_object_t * obj) {

	storage_object_t * working_top_obj = obj;

	LOG_PROC_ENTRY();

	if (debug_level <= DEBUG) {
		if (obj->volume != NULL) {
			LOG_DEBUG("Find working top object for volume %s.\n", obj->volume->name);

		} else {
			LOG_DEBUG("Find working top object for obj %s.\n", obj->name);
		}
	}

	while ((working_top_obj != NULL) &&
	       (GetPluginType(working_top_obj->plugin->id) == EVMS_ASSOCIATIVE_FEATURE) &&
	       (working_top_obj->associated_object == NULL)) {

		working_top_obj = first_thing(working_top_obj->child_objects, NULL);
	}

	LOG_DEBUG("Top object is %s.\n", working_top_obj->name);

	LOG_PROC_EXIT_PTR(working_top_obj);
	return working_top_obj;
}


/*
 * sync_volumes() will make sure that compatibility volumes have the
 * correct name, i.e., the EVMS_DEV_NODE_PATH prefix followed by the object
 * name.  (In the case where a segment is deleted or added to the disk, the
 * compatibility names of the objects may shift and hence the name of the
 * volume may change.)  It also makes sure that non-native EVMS volumes have
 * the same major:minor as the volume's object.
 */
void sync_volumes(void) {
	
	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		storage_object_t * working_top_obj;

		working_top_obj = get_working_top_object(vol->object);

		if (working_top_obj != NULL) {
			if (vol->flags & VOLFLAG_COMPATIBILITY) {
				/*
				 * Compatibility volumes must have their name
				 * derived from the volume's working object.
				 */
				if ((strcmp(vol->name + EVMS_DEV_NODE_PATH_LEN, working_top_obj->name) != 0)) {
					engine_user_message(NULL, NULL,
							    "Volume name %s has shifted to %s%s.\n",
							    vol->name, EVMS_DEV_NODE_PATH, working_top_obj->name);

					engine_unregister_name(vol->name);

					/* Give the volume its new name. */
					memset(vol->name, 0, sizeof(vol->name));
					strcpy(vol->name, EVMS_DEV_NODE_PATH);
					strcat(vol->name, working_top_obj->name);
					engine_register_name(vol->name);

					if (!(vol->flags & VOLFLAG_NEW)) {
						vol->flags |= VOLFLAG_NEEDS_ACTIVATE;
					}
				}
			}

			/*
			 * For volumes that are not featureless EVMS volumes, make sure
			 * the volume has the same major:minor as its object.  (Its
			 * immediate object, not the first object that is not an
			 * associative feature.)
			 */
			if (working_top_obj->feature_header == NULL) {
				if ((vol->dev_major != vol->object->dev_major) ||
				    (vol->dev_minor != vol->object->dev_minor)) {
					vol->dev_major = vol->object->dev_major;
					vol->dev_minor = vol->object->dev_minor;

					if (ensure_dev_node(vol->name, vol->dev_major, vol->dev_minor) == 0) {
						vol->flags |= VOLFLAG_ACTIVE;
						vol->flags &= ~VOLFLAG_NEEDS_ACTIVATE;
					}
				}
			}

		} else {
			LOG_SERIOUS("Failed to get the working top object.\n");
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


static int delete_object(storage_object_t * obj) {

	int rc = 0;
	STATIC_LIST_DECL(child_list);

	LOG_PROC_ENTRY();

	/*
	 * Make sure the object is a top level object.  We can't delete objects
	 * in the middle of a feature stack or under a volume has no parent.
	 */
	if (!is_top_object(obj)) {
		LOG_ERROR("Object %s is not a top level object.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* If the object has stop data on it, delete it. */
	if (obj->flags & SOFLAG_HAS_STOP_DATA) {
		LOG_DEBUG("Removing stop data from object %s.\n", obj->name);
		rc = obj->plugin->functions.plugin->add_sectors_to_kill_list(obj, obj->size - (EVMS_FEATURE_HEADER_SECTORS * 2), EVMS_FEATURE_HEADER_SECTORS * 2);

		if (rc == 0) {
			obj->flags &= ~SOFLAG_HAS_STOP_DATA;

		} else {
			LOG_SERIOUS("add_sectors_to_kill_list() returned error code %d when called to wipe out stop data on object %s.\n", rc, obj->name);
		}
	}

	rc = obj->plugin->functions.plugin->delete(obj, &child_list);

	if (rc == 0) {
		list_element_t iter;
		storage_object_t * child_obj;

		LIST_FOR_EACH(&child_list, iter, child_obj) {

			/*
			 * Now that the child objects are top objects, they will
			 * not contain feature headers.  Remove the feature
			 * headers from the child objects.
			 */
			remove_feature_headers(child_obj);

			/*
			 * Mark the children's child objects' feature headers
			 * dirty so that the volume complete flag will be set
			 * correctly on commit.
			 */
			set_feature_header_dirty(child_obj);

			/*
			 * The parent object may have written over the stop data
			 * when it owned the child.  Forget that the child had
			 * stop data.
			 */
			child_obj->flags &= ~SOFLAG_HAS_STOP_DATA;
		}

		/* Make sure the compatibility volume names are in sync. */
		sync_volumes();
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_delete(object_handle_t thing_handle) {
	int rc = 0;
	void * thing;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();

	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_delete(thing_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	switch (type) {
		case VOLUME:
			{
				logical_volume_t * volume = (logical_volume_t *) thing;

				LOG_DEBUG("Request to delete volume %s.\n", volume->name);

				if (volume->flags & VOLFLAG_COMPATIBILITY) {
					if (is_kernel_volume_mounted(volume, DETAILS)) {
						rc = EBUSY;
                                        }
				}
				
				if (rc == 0) {
					rc = delete_volume(volume);
				}
			}
			break;

		case CONTAINER:
			{
				storage_container_t * container = (storage_container_t *) thing;
				STATIC_LIST_DECL(objects_consumed);

				LOG_DEBUG("Request to destroy container %s.\n", container->name);

				rc = container->plugin->container_functions->delete_container(container, &objects_consumed);
				LOG_DEBUG("Return code from delete_container() is %d: %s.\n",  rc, evms_strerror(rc));
			}
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			{
				storage_object_t * obj = (storage_object_t *) thing;

				LOG_DEBUG("Request to delete object %s.\n", obj->name);

				rc = delete_object(obj);
			}
			break;

		default:
			LOG_ERROR("A thing with a type of %#x cannot be deleted.\n", type);
			rc = EINVAL;
			break;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Delete an object and all of its children.
 * This function is a service function for evms_destroy() below.
 * This function calls itself recursively to delete its children.
 */
static int destroy_object(storage_object_t * obj) {

	int rc = 0;
	STATIC_LIST_DECL(child_list);

	LOG_PROC_ENTRY();

	/* Disks are not deleted on a destroy. */

	if (obj->object_type == DISK) {
		/*
		 * When the disk was consumed the stop data may have been
		 * trashed.  Forget that it has stop data.  It will be
		 * rewritten on commit.
		 */
		obj->flags &= ~SOFLAG_HAS_STOP_DATA;

	} else {
		/*
		 * Save the producing container so we can check it after the
		 * object is deleted.
		 */
		storage_container_t * producing_container = obj->producing_container;

		/* If the object has stop data on it, delete it. */
		if (obj->flags & SOFLAG_HAS_STOP_DATA) {
			LOG_DEBUG("Removing stop data from object %s.\n", obj->name);
			rc = obj->plugin->functions.plugin->add_sectors_to_kill_list(obj, obj->size - (EVMS_FEATURE_HEADER_SECTORS * 2), EVMS_FEATURE_HEADER_SECTORS * 2);

			if (rc == 0) {
				obj->flags &= ~SOFLAG_HAS_STOP_DATA;

			} else {
				LOG_SERIOUS("add_sectors_to_kill_list() returned error code %d when called to wipe out stop data on object %s.\n", rc, obj->name);
			}
		}

		rc = obj->plugin->functions.plugin->delete(obj, &child_list);

		if (rc == 0) {
			list_element_t iter;
			storage_object_t * child;

			/*
			 * Now that the child objects are top objects, they will
			 * not contain feature headers.  Remove the feature
			 * headers from the child objects.
			 */
			LIST_FOR_EACH(&child_list, iter, child) {
				remove_feature_headers(child);
			}

			/*
			 * If the object was not produced by a container, do a recursive
			 * deletion of its children.
			 */
			if (producing_container == NULL) {
				LIST_FOR_EACH(&child_list, iter, child) {
					destroy_object(child);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int destroy_volume(logical_volume_t * volume) {

	int rc = 0;
	storage_object_t * obj = volume->object;

	LOG_PROC_ENTRY();

	/*
	 * Ask the object below the volume if it can handle
	 * being deleted.
	 */
	rc = can_destroy_object(obj);

	if (rc == 0) {

		rc = delete_volume(volume);

		if (rc == 0) {
			rc = destroy_object(obj);
		}

	} else {
		LOG_ERROR("Object %s does not allow volume %s to be deleted.  Reason code is %d.\n", obj->name, volume->name, rc);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * evms_destroy() deletes an object and all the child objects in the tree
 * beneath the object.
 */
int evms_destroy(object_handle_t thing) {

	int rc = 0;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_destroy(thing);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	switch (type) {
		case VOLUME:
			{
				logical_volume_t * volume = (logical_volume_t *) object;

				LOG_DEBUG("Request to destroy volume %s.\n", volume->name);

				if (volume->flags & VOLFLAG_COMPATIBILITY) {
					if (is_kernel_volume_mounted(volume, DETAILS)) {
						rc = EBUSY;
                                        }
				}
				
				if (rc == 0) {
					rc = destroy_volume(volume);

					if (rc == 0) {
						/*
						 * Make sure the compatibility volume names
						 * are in sync.
						 */
						sync_volumes();
					}
				}
			}
			break;

		case CONTAINER:
			{
				storage_container_t * container = (storage_container_t *) object;
				STATIC_LIST_DECL(objects_consumed);

				LOG_DEBUG("Request to destroy container %s.\n", container->name);

			       	rc = container->plugin->container_functions->delete_container(container, &objects_consumed);
			       	LOG_DEBUG("Return code from call to %s plug-in's delete_container() is %d.\n", container->plugin->short_name, rc);

			       	if (rc == 0) {
			       		/* Destroy the objects that were in the container. */
			       		list_element_t iter;
			       		storage_object_t * child;
			       		LIST_FOR_EACH(&objects_consumed, iter, child) {
			       			destroy_object(child);
			       		}
			       	}
			}
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
			{
				storage_object_t * obj = (storage_object_t *) object;

				LOG_DEBUG("Request to destroy storage object %s.\n", obj->name);

				/*
				 * Make sure the object is a top level object.  We can't
				 * destroy objects in the middle of a feature stack.  A
				 * top level object has no parent.
				 */
				if (is_top_object(obj)) {

					rc = destroy_object(obj);
					if (rc == 0) {
						/*
						 * Make sure the compatibility volume names
						 * are in sync.
						 */
						sync_volumes();
					}

				} else {
					/* The object is not a top level object. */

					LOG_ERROR("Object %s is not a top level object.\n", obj->name);
					rc = EINVAL;
				}
			}
			break;

		case DISK:
			{
				storage_object_t * obj = object;

				LOG_ERROR("Disk %s cannot be destroyed because disks cannot be destroyed.  Disks must be deleted.\n", obj->name);
				rc = EINVAL;
			}
			break;

		default:
			LOG_ERROR("A thing with a type of %#x cannot be destroyed.\n", type);
			rc = EINVAL;
			break;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int create_replace_object(storage_object_t   * source,
				 storage_object_t   * target,
				 storage_object_t * * replace_object) {
	int rc = 0;
	STATIC_LIST_DECL(input_list);
	STATIC_LIST_DECL(output_list);
	list_element_t el;

	LOG_PROC_ENTRY();

	/*
	 * Order is important here.  The Replace feature takes
	 * the first object as the source and the second object
	 * as the target.
	 */
	el = insert_thing(&input_list,
			  source,
			  INSERT_BEFORE,
			  NULL);

	if (el != NULL) {
		el = insert_thing(&input_list,
				  target,
				  INSERT_AFTER,
				  NULL);

		if (el != NULL) {
			rc = replace_plugin->functions.plugin->create(&input_list, NULL, &output_list);

			if (rc == 0) {

				/*
				 * Any stop data on the target may get
				 * trashed by a parent object.  Forget
				 * that it is there.
				 */
				target->flags &= ~SOFLAG_HAS_STOP_DATA;

				/*
				 * Return the new replace object to the
				 * caller.
				 */
				*replace_object = first_thing(&output_list, NULL);
			}

		} else {
			LOG_CRITICAL("Error when putting target object %s into the input list.\n", target->name);
			rc = ENOMEM;
		}

	} else {
		LOG_WARNING("Error when putting source object %s into the input list.\n", source->name);
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_replace(object_handle_t source_handle, object_handle_t target_handle) {

	int rc = 0;
	storage_object_t * source;
	storage_object_t * target;
	storage_object_t * replace_obj;
	object_type_t type;
	list_anchor_t source_parents = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_replace(source_handle, target_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_replace(source_handle, target_handle, ERROR);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * No need to check the return codes from translate_handle()
	 * since can_replace() translated the handles and got good
	 * return codes.
	 */
	translate_handle(source_handle,
			 (void *) &source,
			 &type);

	translate_handle(target_handle,
			 (void *) &target,
			 &type);

	if (!list_empty(source->parent_objects)) {

		/* Get a copy of the source's parent list so
		 * we can insert the replace object after it
		 * has been created.  When the replace object
		 * is created it changes the source's parent
		 * list to point to the new replace object.
		 */

		source_parents = copy_list(source->parent_objects);
		if (source_parents == NULL) {
			LOG_SERIOUS("Error making a copy of the parent list.\n");
			LOG_PROC_EXIT_INT(ENOMEM);
			return ENOMEM;
		}
	}

	rc = create_replace_object(source, target, &replace_obj);

	if (rc == 0) {
		if (source_parents != NULL) {
			list_element_t iter;
			storage_object_t * parent;

			LIST_FOR_EACH(source_parents, iter, parent) {
				rc = parent->plugin->functions.plugin->replace_child(parent, source, replace_obj);

				if (rc != 0) {
					list_element_t tmp_iter;
					LOG_WARNING("Parent object %s failed to replace child object %s with new child object %s.  Error code was %d: %s\n",
						    parent->name, source->name, replace_obj->name, rc, evms_strerror(rc));

					/* Undo any replaces that succeeded. */
					LIST_FOR_EACH(source_parents, tmp_iter, parent) {
						parent->plugin->functions.plugin->replace_child(parent, replace_obj, source);
					}
					break;
				}
			}

		} else {
			source->volume->object = replace_obj;
			source->volume->flags |= VOLFLAG_NEEDS_ACTIVATE;
		}
	}

	if (source_parents != NULL) {
		destroy_list(source_parents);
	}

	if (rc == 0) {
		/*
		 * Some plug-ins NULL out the source's volume pointer
		 * when they abandon the source as a child.  However, in this
		 * case, the source has not really been abandoned; it is still
		 * part of the volume.
		 */
		source->volume = replace_obj->volume;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int make_list(handle_array_t * ha, list_anchor_t list) {

	int rc = 0;
	int i;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	/* If handle array pointer is NULL, just return the empty list_anchor_t. */
	if (ha != NULL) {
		for (i = 0; (rc == 0) && (i < ha->count); i++) {
			rc = translate_handle(ha->handle[i],
					      &object,
					      &type);

			if (rc == HANDLE_MANAGER_NO_ERROR) {
				list_element_t el;

				el = insert_thing(list,
						  object,
						  INSERT_AFTER,
						  NULL);

				if (el == NULL) {
					LOG_CRITICAL("Error inserting thing into the list.\n");
					rc = ENOMEM;
				}
			}
		}
	}

	if (rc != 0) {
		delete_all_elements(list);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


void free_info_object_contents(void * object) {

	handle_object_info_t * info = (handle_object_info_t *) object;

	LOG_PROC_ENTRY();

	switch (info->type) {
	case PLUGIN:
		break;

	case DISK:
	case SEGMENT:
	case REGION:
	case EVMS_OBJECT:
		if (info->info.object.parent_objects != NULL) {
			engine_free(info->info.object.parent_objects);
		}

		if (info->info.object.child_objects != NULL) {
			engine_free(info->info.object.child_objects);
		}

		break;

	case CONTAINER:
		if (info->info.container.objects_consumed != NULL) {
			engine_free(info->info.container.objects_consumed);
		}
		if (info->info.container.objects_produced != NULL) {
			engine_free(info->info.container.objects_produced);
		}

		break;

	case VOLUME:
		if (info->info.volume.mount_point != NULL) {
			engine_free(info->info.volume.mount_point);
		}

		break;

	default:
		break;
	}

	LOG_PROC_EXIT_VOID();
}


int evms_get_handle_object_type(object_handle_t handle,
				object_type_t * type) {
	int rc = 0;
	void * object;

	LOG_PROC_ENTRY();

	*type = 0;

	rc = check_engine_read_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_handle_object_type(handle, type);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(handle,
			      &object,
			      type);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Make an array of handles (handle_array_t) for the objects in list.
 */
int make_handle_array(list_anchor_t      list,
		      handle_array_t * * ha) {

	int rc = 0;
	uint count;
	uint size;

	LOG_PROC_ENTRY();

	count = list_count(list);

		LOG_DEBUG("Number of objects in the list:  %d\n", count);
		if (count > 1) {
			size = sizeof(handle_array_t) + ((count -1) * sizeof(object_handle_t));
		} else {
			size = sizeof(handle_array_t);
		}

		*ha = engine_alloc(size);
		if (*ha != NULL) {
			list_element_t iter;
			void * thing;

			LIST_FOR_EACH(list, iter, thing) {
				make_handle_entry(thing, *ha);
			}

		} else {
			LOG_CRITICAL("Error allocating memory for a handle array.\n");
			rc = ENOMEM;
		}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_info(object_handle_t thing, handle_object_info_t * * user_info) {
	
	int rc = 0;
	void * object;
	object_type_t type;
	handle_object_info_t * info = NULL;

	LOG_PROC_ENTRY();

	*user_info = NULL;

	rc = check_engine_read_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_info(thing, user_info);

		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	info = alloc_app_struct(sizeof(handle_object_info_t), free_info_object_contents);

	if (info != NULL) {
		LOG_DEBUG("Get info for handle %d.\n", thing);
		rc = translate_handle(thing,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {
			switch (type) {
				case PLUGIN:
					{
						plugin_record_t * plugin = (plugin_record_t *) object;

						LOG_DEBUG("Handle %d maps to plug-in %s.\n", thing, plugin->short_name);
						info->type = PLUGIN;

						rc = ensure_app_handle(plugin);
						if (rc == 0) {
							info->info.plugin.handle                             = plugin->app_handle;
							info->info.plugin.id                                 = plugin->id;
							info->info.plugin.version.major                      = plugin->version.major;
							info->info.plugin.version.minor                      = plugin->version.minor;
							info->info.plugin.version.patchlevel                 = plugin->version.patchlevel;
							info->info.plugin.required_engine_api_version        = plugin->required_engine_api_version;
							info->info.plugin.required_plugin_api_version.plugin = plugin->required_plugin_api_version.plugin;
							info->info.plugin.required_container_api_version     = plugin->required_container_api_version;
							info->info.plugin.short_name                         = plugin->short_name;
							info->info.plugin.long_name                          = plugin->long_name;
							info->info.plugin.oem_name                           = plugin->oem_name;
							info->info.plugin.supports_containers                = (plugin->container_functions != NULL);
						}
					}
					break;

				case DISK:
				case SEGMENT:
				case REGION:
				case EVMS_OBJECT:
					{
						storage_object_t * obj = (storage_object_t *) object;

						LOG_DEBUG("Handle %d maps to storage object %s.\n", thing, obj->name);
						info->type = obj->object_type;

						rc = make_handle_array(obj->parent_objects, &info->info.object.parent_objects);
						if (rc == 0) {

							rc = make_handle_array(obj->child_objects, &info->info.object.child_objects);
							if (rc == 0) {

								rc = ensure_app_handle(obj);
								if (rc == 0) {
									info->info.object.handle = obj->app_handle;
									info->info.object.object_type = obj->object_type;
									info->info.object.data_type = obj->data_type;
								}

								if (rc == 0) {
									if (obj->plugin != NULL) {
										rc = ensure_app_handle(obj->plugin);
										if (rc == 0) {
											info->info.object.plugin = obj->plugin->app_handle;
										}

									} else {
										info->info.object.plugin = 0;
									}
								}

								if (rc == 0) {
									if (obj->producing_container != NULL) {

										rc = ensure_app_handle(obj->producing_container);
										if (rc == 0) {
											info->info.object.producing_container = obj->producing_container->app_handle;
										}
									} else {
										info->info.object.producing_container = 0;
									}
								}

								if (rc == 0) {
									if (obj->consuming_container != NULL) {

										rc = ensure_app_handle(obj->consuming_container);
										if (rc == 0) {
											info->info.object.consuming_container = obj->consuming_container->app_handle;
										}
									} else {
										info->info.object.consuming_container = 0;
									}
								}

								if (rc == 0) {
									if (obj->volume != NULL) {

										rc = ensure_app_handle(obj->volume);
										if (rc == 0) {
											info->info.object.volume = obj->volume->app_handle;
										}
									} else {
										info->info.object.volume = 0;
									}
								}

								if (rc == 0) {
									if (obj->disk_group != NULL) {

										rc = ensure_app_handle(obj->disk_group);
										if (rc == 0) {
											info->info.object.disk_group = obj->disk_group->app_handle;
										}
									} else {
										info->info.object.disk_group = 0;
									}
								}

								if (rc == 0) {
									info->info.object.dev_major = obj->dev_major;
									info->info.object.dev_minor = obj->dev_minor;
									info->info.object.flags = obj->flags;
									memcpy(info->info.object.name, obj->name, sizeof(obj->name));
									info->info.object.start = obj->start;
									info->info.object.size = obj->size;
									info->info.object.geometry = obj->geometry;
								}
							}
						}
					}
					break;

				case CONTAINER:
					{
						storage_container_t * con = (storage_container_t *) object;

						LOG_DEBUG("Handle %d maps to container %s.\n", thing, con->name);
						info->type = CONTAINER;

						rc = make_handle_array(con->objects_consumed, &info->info.container.objects_consumed);
						if (rc == 0) {
							rc = make_handle_array(con->objects_produced, &info->info.container.objects_produced);
							if (rc == 0) {

								rc = ensure_app_handle(con);
								if (rc == 0) {
									info->info.container.handle = con->app_handle;
								}

								if (rc == 0) {
									if (con->plugin != NULL) {
										rc = ensure_app_handle(con->plugin);
										if (rc == 0) {
											info->info.container.plugin = con->plugin->app_handle;
										}
									} else {
										info->info.container.plugin = 0;
									}
								}

								if (rc == 0) {
									if (con->disk_group != NULL) {

										rc = ensure_app_handle(con->disk_group);
										if (rc == 0) {
											info->info.container.disk_group = con->disk_group->app_handle;
										}
									} else {
										info->info.container.disk_group = 0;
									}
								}

								if (rc == 0) {
									info->info.container.flags = con->flags;
									memcpy(info->info.container.name, con->name, sizeof(con->name));
									info->info.container.size = con->size;
								}
							}
						}
					}
					break;

				case VOLUME:
					{
						logical_volume_t * vol = (logical_volume_t *) object;

						LOG_DEBUG("Handle %d maps to volume %s.\n", thing, vol->name);
						info->type = VOLUME;

						rc = ensure_app_handle(vol);
						if (rc == 0) {
							info->info.volume.handle = vol->app_handle;
						}

						if (rc == 0) {
							if (vol->file_system_manager != NULL) {
								rc = ensure_app_handle(vol->file_system_manager);
								if (rc == 0) {
									info->info.volume.file_system_manager = vol->file_system_manager->app_handle;
								}
							} else {
								info->info.volume.file_system_manager = 0;
							}
						}

						if (rc == 0) {
							if (vol->object != NULL) {
								rc = ensure_app_handle(vol->object);
								if (rc == 0) {
									info->info.volume.object = vol->object->app_handle;
								}
							} else {
								info->info.volume.object = 0;
							}
						}

						if (rc == 0) {
							if (vol->associated_volume != NULL) {
								rc = ensure_app_handle(vol->associated_volume);
								if (rc == 0) {
									info->info.volume.associated_volume = vol->associated_volume->app_handle;
								}
							} else {
								info->info.volume.associated_volume = 0;
							}
						}

						if (rc == 0) {
							if (vol->disk_group != NULL) {

								rc = ensure_app_handle(vol->disk_group);
								if (rc == 0) {
									info->info.volume.disk_group = vol->disk_group->app_handle;
								}
							} else {
								info->info.volume.disk_group = 0;
							}
						}

						if (rc == 0) {
							if (vol->mount_point != NULL) {
								info->info.volume.mount_point = engine_alloc(strlen(vol->mount_point) + 1);
								strcpy(info->info.volume.mount_point, vol->mount_point);
							} else {
								info->info.volume.mount_point = NULL;
							}

							/*
							 * Reget volume limits since things like
							 * min_fs_size can change if the volume is live.
							 */
							get_volume_sizes_and_limits(vol);

							info->info.volume.fs_size       = vol->fs_size;
							info->info.volume.min_fs_size   = vol->min_fs_size;
							info->info.volume.max_fs_size   = vol->max_fs_size;
							info->info.volume.vol_size      = vol->vol_size;
							info->info.volume.max_vol_size  = vol->max_vol_size;
							info->info.volume.dev_major     = vol->dev_major;
							info->info.volume.dev_minor     = vol->dev_minor;
							info->info.volume.serial_number = vol->serial_number;
							info->info.volume.flags         = vol->flags;
							memcpy(info->info.volume.name, vol->name, sizeof(vol->name));
							memcpy(info->info.volume.dev_node, vol->dev_node, sizeof(vol->dev_node));
						}
					}
					break;

				default:
					LOG_DEBUG("Handle %d maps to unknown object type %d.\n", thing, type);
					rc = EINVAL;
					break;
			}
		}

		if ((rc != 0) && (info != NULL)) {
			evms_free(info);
			info = NULL;
		}


	} else {
		rc = ENOMEM;
	}

	*user_info = info;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int add_plugin_to_list(storage_object_t * obj,
			      list_anchor_t      plugin_list) {

	int rc = 0;
	list_element_t iter;
	plugin_record_t * pPlugRec;

	LOG_PROC_ENTRY();

	/* Only get the plug-in (feature) if this is an EVMS storage object. */
	if (obj->object_type == EVMS_OBJECT) {

		/* Make sure the plug-in isn't already on the list. */
		LIST_FOR_EACH(&plugins_list, iter, pPlugRec) {
			if (pPlugRec == obj->plugin) {
				break;
			}
		}

		if (pPlugRec == NULL) {
			list_element_t el;
			/*
			 * This plug-in record is not in the list.  Insert it into the
			 * list.
			 */
			el = insert_thing(plugin_list,
					  obj->plugin,
					  INSERT_AFTER,
					  NULL);

			if (el == NULL) {
				LOG_CRITICAL("Error inserting plugin %s into the plugin list.\n", obj->plugin->short_name);
				rc = ENOMEM;
			}
		}

		if (rc == 0) {
			/* Get the feature(s) for the child object(s). */
			storage_object_t * child;

			LIST_FOR_EACH(obj->child_objects, iter, child) {
				rc = add_plugin_to_list(child, plugin_list);
				if (rc != 0) {
					break;
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_feature_list(object_handle_t    thing,
			  handle_array_t * * plugin_handles) {
	int rc = 0;
	void * object;
	object_type_t type;
	handle_array_t * plugins;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_feature_list(thing, plugin_handles);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if ((type == VOLUME) ||
	    (type == EVMS_OBJECT)) {
		STATIC_LIST_DECL(feature_list);
		storage_object_t * storage_object;

		if (type == VOLUME) {
			logical_volume_t * volume = (logical_volume_t *) object;

			storage_object = volume->object;
		} else {
			storage_object = (storage_object_t *) object;
		}

		rc = add_plugin_to_list(storage_object,
					&feature_list);

		if (rc == 0) {
			rc = make_user_handle_array(&feature_list, &plugins);
		}

	} else {
		/* The thing is not a volume or object. */
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int find_object_handle_by_name(list_anchor_t list, char * name, object_handle_t * handle) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(list, iter, obj) {
		if (strcmp(obj->name, name) == 0) {
			break;
		}
	}

	if (obj != NULL) {
		rc = ensure_app_handle(obj);

		if (rc == 0) {
			*handle = obj->app_handle;
		}

	} else {
		rc = ENOENT;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int find_container_handle_by_name(char * name, object_handle_t * handle) {

	int rc = 0;
	list_element_t iter;
	storage_container_t * con;

	LIST_FOR_EACH(&containers_list, iter, con) {
		if (strcmp(con->name, name) == 0) {
			break;
		}
	}

	if (con != NULL) {
		rc = ensure_app_handle(con);
		if (rc == 0) {
			*handle = con->app_handle;
		}

	} else {
		rc = ENOENT;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int find_volume_handle_by_name(char * name, object_handle_t * handle) {

	int rc = 0;
	list_element_t iter;
	logical_volume_t * vol;

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (strcmp(vol->name, name) == 0) {
			break;
		}
	}

	if (vol != NULL) {
		rc = ensure_app_handle(vol);
		if (rc == 0) {
			*handle = vol->app_handle;
		}

	} else {
		rc = ENOENT;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_object_handle_for_name(object_type_t type, char * name, object_handle_t * object_handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_object_handle_for_name(type, name, object_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = ENOENT;

	if (type & DISK) {
		rc = find_object_handle_by_name(&disks_list, name, object_handle);
	}

	if ((rc == ENOENT) && (type & SEGMENT)) {
		rc = find_object_handle_by_name(&segments_list, name, object_handle);
	}

	if ((rc == ENOENT) && (type & REGION)) {
		rc = find_object_handle_by_name(&regions_list, name, object_handle);
	}

	if ((rc == ENOENT) && (type & EVMS_OBJECT)) {
		rc = find_object_handle_by_name(&EVMS_objects_list, name, object_handle);
	}

	if ((rc == ENOENT) && (type & CONTAINER)) {
		rc = find_container_handle_by_name(name, object_handle);
	}

	if ((rc == ENOENT) && (type & VOLUME)) {
		rc = find_volume_handle_by_name(name, object_handle);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

