/*
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dm-ioctl4.h
 *
 * Definitions for version-4 of the Device-Mapper ioctl interface. Only
 * dm-ioctl4.c should include this header file.
 */

/*
 * A traditional ioctl interface for the device mapper.
 *
 * Each device can have two tables associated with it, an
 * 'active' table which is the one currently used by io passing
 * through the device, and an 'inactive' one which is a table
 * that is being prepared as a replacement for the 'active' one.
 *
 * DM_VERSION:
 * Just get the version information for the ioctl interface.
 *
 * DM_REMOVE_ALL:
 * Remove all dm devices, destroy all tables.  Only really used
 * for debug.
 *
 * DM_LIST_DEVICES:
 * Get a list of all the dm device names.
 *
 * DM_DEV_CREATE:
 * Create a new device, neither the 'active' or 'inactive' table
 * slots will be filled.  The device will be in suspended state
 * after creation, however any io to the device will get errored
 * since it will be out-of-bounds.
 *
 * DM_DEV_REMOVE:
 * Remove a device, destroy any tables.
 *
 * DM_DEV_RENAME:
 * Rename a device.
 *
 * DM_SUSPEND:
 * This performs both suspend and resume, depending which flag is
 * passed in.
 * Suspend: This command will not return until all pending io to
 * the device has completed.  Further io will be deferred until
 * the device is resumed.
 * Resume: It is no longer an error to issue this command on an
 * unsuspended device.  If a table is present in the 'inactive'
 * slot, it will be moved to the active slot, then the old table
 * from the active slot will be _destroyed_.  Finally the device
 * is resumed.
 *
 * DM_DEV_STATUS:
 * Retrieves the status for the table in the 'active' slot.
 *
 * DM_DEV_WAIT:
 * Wait for a significant event to occur to the device.  This
 * could either be caused by an event triggered by one of the
 * targets of the table in the 'active' slot, or a table change.
 *
 * DM_TABLE_LOAD:
 * Load a table into the 'inactive' slot for the device.  The
 * device does _not_ need to be suspended prior to this command.
 *
 * DM_TABLE_CLEAR:
 * Destroy any table in the 'inactive' slot (ie. abort).
 *
 * DM_TABLE_DEPS:
 * Return a set of device dependencies for the 'active' table.
 *
 * DM_TABLE_STATUS:
 * Return the targets status for the 'active' table.
 */

#ifndef _DM_IOCTL4_H_INCLUDED_
#define _DM_IOCTL4_H_INCLUDED_ 1

#define DM_MAX_TYPE_NAME 16
#define DM_NAME_LEN 128
#define DM_UUID_LEN 129

/**
 * struct dm_ioctl
 * @version:		Three parts: major - no backwards or forwards compatibility.
 *				     minor - only backwards compatibility.
 *				     patch - both backwards and forwards compatibility.
 *			This field will be filled in based on the version of the
 *			interface that the engine is compiled for. The kernel
 *			will then fill in this field before completing each
 *			ioctl to let the engine know the current version.
 * @data_size:		Total size of data passed to the ioctl (including
 *			this structure).
 * @data_start:		Offset to start of remaining data, relative to start of
 *			this structure.
 * @target_count:	Number of targets specified in this ioctl, or number
 *			of targets returned by this ioctl.
 * @open_count:		Number of openers of this device in the kernel.
 * @flags:
 * @event_nr:
 * @padding:
 * @dev:		Device major:minor.
 * @name:		Device name.
 * @uuid:		Device unique identifier.
 *
 * All ioctl arguments consist of a single chunk of memory, with this structure
 * at the start.  If a uuid is specified any lookup (e.g..  for a DM_INFO) will
 * be done on that, *not* the name.  Also, except for create ioctls, only the
 * name or the uuid may be specified, but not both.
 */
typedef struct dm_ioctl {
	u_int32_t version[3];
	u_int32_t data_size;
	u_int32_t data_start;
	u_int32_t target_count;
	int32_t open_count;
	u_int32_t flags;
	u_int32_t event_nr;
	u_int32_t padding;
	u_int64_t dev;
	char name[DM_NAME_LEN];
	char uuid[DM_UUID_LEN];
} dm_ioctl_t;

/**
 * struct dm_ioctl_target
 * @start:	Starting LBA of this target within the table.
 * @length:	Length of this target, in sector.
 * @status:
 * @next:	Location of the next dm_target_spec.
 *		- When specifying targets on a DM_TABLE_LOAD command, this
 *		  value is the number of bytes from the start of the "current"
 *		  dm_target_spec to the start of the "next" dm_target_spec.
 *		- When retrieving targets on a DM_TABLE_STATUS command, this
 *		  value is the number of bytes from the start of the first
 *		  dm_target_spec (that follows the dm_ioctl struct) to the
 *		  start of the "next" dm_target_spec.
 * @target_type:Name of target module.
 *
 * Used to specify tables. These structures appear after the dm_ioctl. The
 * parameter string starts immediately after this structure. Be sure to
 * add correct padding after the string to ensure the correct alignment
 * of the next dm_target_spec.
 */
typedef struct dm_ioctl_target {
	u_int64_t start;
	u_int64_t length;
	int32_t status;
	u_int32_t next;
	char target_type[DM_MAX_TYPE_NAME];
} dm_ioctl_target_t;

/**
 * struct dm_name_list
 * @dev:	Device major:minor
 * @next:	Offset, in bytes from the start of this struct, to the start
 *		of the next dm_name_list.
 * @name:	Device name.
 *
 * Used by the kernel to return a list of names and device-numbers for all
 * active devices. Just like dm_ioctl_target, these structure appear after the
 * dm_ioctl.
 **/
typedef struct dm_name_list {
	u_int64_t dev;
	u_int32_t next;
	char name[0];
} dm_name_list_t;

/*
 * Definition of the ioctl commands.
 */

typedef enum {
	/* Top level commands */
	DM_VERSION_CMD = 0,
	DM_REMOVE_ALL_CMD,
	DM_LIST_DEVICES_CMD,

	/* Device level commands */
	DM_DEV_CREATE_CMD,
	DM_DEV_REMOVE_CMD,
	DM_DEV_RENAME_CMD,
	DM_DEV_SUSPEND_CMD,
	DM_DEV_STATUS_CMD,
	DM_DEV_WAIT_CMD,

	/* Table level commands */
	DM_TABLE_LOAD_CMD,
	DM_TABLE_CLEAR_CMD,
	DM_TABLE_DEPS_CMD,
	DM_TABLE_STATUS_CMD
} dm_ioctl_cmd;

#define DM_IOCTL 0xfd

/* Top level commands. */
#define DM_VERSION	_IOWR(DM_IOCTL, DM_VERSION_CMD, struct dm_ioctl)
#define DM_REMOVE_ALL	_IOWR(DM_IOCTL, DM_REMOVE_ALL_CMD, struct dm_ioctl)
#define DM_LIST_DEVICES	_IOWR(DM_IOCTL, DM_LIST_DEVICES_CMD, struct dm_ioctl)

/* Device level commands. */
#define DM_DEV_CREATE	_IOWR(DM_IOCTL, DM_DEV_CREATE_CMD, struct dm_ioctl)
#define DM_DEV_REMOVE	_IOWR(DM_IOCTL, DM_DEV_REMOVE_CMD, struct dm_ioctl)
#define DM_DEV_RENAME	_IOWR(DM_IOCTL, DM_DEV_RENAME_CMD, struct dm_ioctl)
#define DM_DEV_SUSPEND	_IOWR(DM_IOCTL, DM_DEV_SUSPEND_CMD, struct dm_ioctl)
#define DM_DEV_STATUS	_IOWR(DM_IOCTL, DM_DEV_STATUS_CMD, struct dm_ioctl)
#define DM_DEV_WAIT	_IOWR(DM_IOCTL, DM_DEV_WAIT_CMD, struct dm_ioctl)

/* Table level commands. */
#define DM_TABLE_LOAD	_IOWR(DM_IOCTL, DM_TABLE_LOAD_CMD, struct dm_ioctl)
#define DM_TABLE_CLEAR	_IOWR(DM_IOCTL, DM_TABLE_CLEAR_CMD, struct dm_ioctl)
#define DM_TABLE_DEPS	_IOWR(DM_IOCTL, DM_TABLE_DEPS_CMD, struct dm_ioctl)
#define DM_TABLE_STATUS	_IOWR(DM_IOCTL, DM_TABLE_STATUS_CMD, struct dm_ioctl)

#define DM_VERSION_MAJOR	4
#define DM_VERSION_MINOR	0
#define DM_VERSION_PATCHLEVEL	0
#define DM_VERSION_EXTRA	"-ioctl (2003-06-04)"

/* Status bits */
#define DM_READONLY_FLAG		(1 << 0)
#define DM_SUSPEND_FLAG			(1 << 1)
#define DM_PERSISTENT_DEV_FLAG		(1 << 3)

/* Flag passed into ioctl STATUS command to get table information
 * rather than current status.
 */
#define DM_STATUS_TABLE_FLAG		(1 << 4)

/* Flags that indicate whether a table is present in either of
 * the two table slots that a device has.
 */
#define DM_ACTIVE_PRESENT_FLAG		(1 << 5)
#define DM_INACTIVE_PRESENT_FLAG	(1 << 6)

/* Indicates that the buffer passed in wasn't big enough for the results.
 */
#define DM_BUFFER_FULL_FLAG		(1 << 8)

#endif

