/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dmonmsg.h
 */

#include <byteswap.h>


#ifndef EVMS_DMONMSG2_H_INCLUDED
#define EVMS_DMONMSG2_H_INCLUDED 1

#define EVMS_DMONMSG_MAJOR_VERSION	3
#define EVMS_DMONMSG_MINOR_VERSION	0
#define EVMS_DMONMSG_PATCH_LEVEL	0
				
/*
 * Macros for putting information on the network.
 * Network protocol is big endian format.  These macros can be used to access
 * the fields of stuff on the network regardless of the endianness of the host
 * architecture.
 */
#if __BYTE_ORDER == __BIG_ENDIAN
#define NET_TO_HOST16(x)    (x)
#define NET_TO_HOST32(x)    (x)
#define NET_TO_HOST64(x)    (x)

#define HOST_TO_NET16(x)    (x)
#define HOST_TO_NET32(x)    (x)
#define HOST_TO_NET64(x)    (x)

#elif __BYTE_ORDER == __LITTLE_ENDIAN
#define NET_TO_HOST16(x)    (bswap_16(x))
#define NET_TO_HOST32(x)    (bswap_32(x))
#define NET_TO_HOST64(x)    (bswap_64(x))

#define HOST_TO_NET16(x)    (bswap_16(x))
#define HOST_TO_NET32(x)    (bswap_32(x))
#define HOST_TO_NET64(x)    (bswap_64(x))

#else
#error "__BYTE_ORDER must be defined as __LITTLE_ENDIAN or __BIG_ENDIAN"

#endif

typedef enum {
	MSG_GET_VERSION = 1,                            //  1  0x01

	EVMS_GET_API_VERSION,                           //  2  0x02
	EVMS_OPEN_ENGINE,                               //  3  0x03
	EVMS_GET_PLUGIN_LIST,                           //  4  0x04
	EVMS_GET_PLUGIN_BY_ID,                          //  5  0x05
	EVMS_GET_PLUGIN_BY_NAME,                        //  6  0x06
	EVMS_CHANGES_PENDING,                           //  7  0x07
	EVMS_WRITE_LOG_ENTRY,                           //  8  0x08
	EVMS_GET_DEBUG_LEVEL,                           //  9  0x09
	EVMS_SET_DEBUG_LEVEL,                           // 10  0x0a
	EVMS_COMMIT_CHANGES,                            // 11  0x0b
	EVMS_CLOSE_ENGINE,                              // 12  0x0c
	EVMS_CAN_DELETE,                                // 13  0x0d
	EVMS_CAN_DESTROY,                               // 14  0x0e
	EVMS_CAN_EXPAND,                                // 15  0x0f
	EVMS_CAN_SHRINK,                                // 16  0x10
	EVMS_CAN_REPLACE,                               // 17  0x11
	EVMS_CAN_SET_INFO,                              // 18  0x12
	EVMS_DELETE,                                    // 19  0x13
	EVMS_DESTROY,                                   // 20  0x14
	EVMS_GET_EXPAND_POINTS,                         // 21  0x15
	EVMS_EXPAND,                                    // 22  0x16
	EVMS_GET_SHRINK_POINTS,                         // 23  0x17
	EVMS_SHRINK,                                    // 24  0x18
	EVMS_REPLACE,                                   // 25  0x19
	EVMS_GET_HANDLE_OBJECT_TYPE,                    // 26  0x1a
	EVMS_GET_INFO,                                  // 27  0x1b
	EVMS_GET_EXTENDED_INFO,                         // 28  0x1c
	EVMS_SET_INFO,                                  // 29  0x1d
	EVMS_GET_OBJECT_HANDLE_FOR_NAME,                // 30  0x1e
	EVMS_GET_FEATURE_LIST,                          // 31  0x1f
	EVMS_CREATE,                                    // 32  0x20
	EVMS_TRANSFER,                                  // 33  0x21
	EVMS_ASSIGN,                                    // 34  0x22
	EVMS_CAN_UNASSIGN,                              // 35  0x23
	EVMS_UNASSIGN,                                  // 36  0x24
	EVMS_GET_OBJECT_LIST,                           // 37  0x25
	EVMS_GET_PLUGIN_FUNCTIONS,                      // 38  0x26
	EVMS_DO_PLUGIN_FUNCTION,                        // 49  0x27
	EVMS_CAN_CREATE_VOLUME,                         // 40  0x28
	EVMS_CAN_CREATE_COMPATIBILITY_VOLUME,           // 41  0x29
	EVMS_CAN_SET_VOLUME_NAME,                       // 42  0x2a
	EVMS_CAN_CONVERT_TO_EVMS_VOLUME,                // 43  0x2b
	EVMS_CAN_CONVERT_TO_COMPATIBILITY_VOLUME,	// 44  0x2c
	EVMS_CAN_ADD_FEATURE_TO_VOLUME,                 // 45  0x2d
	EVMS_CREATE_VOLUME,                             // 46  0x2e
	EVMS_SET_VOLUME_NAME,                           // 47  0x3f
	EVMS_CREATE_COMPATIBILITY_VOLUME,               // 48  0x30
	EVMS_CONVERT_TO_EVMS_VOLUME,                    // 59  0x31
	EVMS_CONVERT_TO_COMPATIBILITY_VOLUME,           // 50  0x32
	EVMS_ADD_FEATURE_TO_VOLUME,                     // 51  0x33
	EVMS_CAN_MKFS,                                  // 52  0x34
	EVMS_CAN_UNMKFS,                                // 53  0x35
	EVMS_CAN_FSCK,                                  // 54  0x36
	EVMS_CAN_DEFRAG,                                // 55  0x37
	EVMS_MKFS,                                      // 56  0x38
	EVMS_UNMKFS,                                    // 57  0x39
	EVMS_FSCK,                                      // 58  0x3a
	EVMS_DEFRAG,                                    // 69  0x3b
	EVMS_GET_VOLUME_LIST,                           // 60  0x3c
	EVMS_CAN_MOUNT,                                 // 61  0x3d
	EVMS_MOUNT,                                     // 62  0x3e
	EVMS_CAN_UNMOUNT,                               // 63  0x4f
	EVMS_UNMOUNT,                                   // 64  0x40
	EVMS_CAN_REMOUNT,                               // 65  0x41
	EVMS_REMOUNT,                                   // 66  0x42
	EVMS_CAN_ADD_TO_CONTAINER,                      // 67  0x43
	EVMS_CAN_REMOVE_FROM_CONTAINER,                 // 68  0x44
	EVMS_CREATE_CONTAINER,                          // 79  0x45
	EVMS_GET_CONTAINER_LIST,                        // 70  0x46
	EVMS_CREATE_TASK,                               // 71  0x47
	EVMS_INVOKE_TASK,                               // 72  0x48
	EVMS_DESTROY_TASK,                              // 73  0x49
	EVMS_GET_TASK_ACTION,                           // 74  0x4a
	EVMS_GET_ACCEPTABLE_OBJECTS,                    // 75  0x4b
	EVMS_GET_SELECTED_OBJECTS,                      // 76  0x4c
	EVMS_GET_SELECTED_OBJECT_LIMITS,                // 77  0x4d
	EVMS_SET_SELECTED_OBJECTS,                      // 78  0x4e
	EVMS_GET_OPTION_COUNT,                          // 89  0x5f
	EVMS_GET_OPTION_DESCRIPTOR,                     // 80  0x50
	EVMS_SET_OPTION_VALUE,                          // 81  0x51
	EVMS_GET_OPTION_DESCRIPTOR_BY_NAME,             // 82  0x52
	EVMS_SET_OPTION_VALUE_BY_NAME,                  // 83  0x53
	EVMS_MARK_FOR_REDISCOVER,                       // 84  0x54
	EVMS_REDISCOVER,                                // 85  0x55
	USER_MESSAGE,                                   // 86  0x56
	PROGRESS,                                       // 87  0x57
	STATUS,                                         // 88  0x58
	SHUTDOWN,                                       // 99  0x59
	INVALID_COMMAND,                                // 90  0x5a

	MSG_INVALID_CMD         = 0x7fffffff
} msg_cmd_t;

#define COMMAND_RESPONSE	0x80000000
#define COMMAND_MASK		~COMMAND_RESPONSE


inline static char * msg_cmd_name(msg_cmd_t cmd) {
	switch (cmd & COMMAND_MASK) {
		case MSG_GET_VERSION:
			return "MSG_GET_VERSION";
		case EVMS_GET_API_VERSION:
			return "EVMS_GET_API_VERSION";
		case EVMS_OPEN_ENGINE:
			return "EVMS_OPEN_ENGINE";
		case EVMS_GET_PLUGIN_LIST:
			return "EVMS_GET_PLUGIN_LIST";
		case EVMS_GET_PLUGIN_BY_ID:
			return "EVMS_GET_PLUGIN_BY_ID";
		case EVMS_GET_PLUGIN_BY_NAME:
			return "EVMS_GET_PLUGIN_BY_NAME";
		case EVMS_CHANGES_PENDING:
			return "EVMS_CHANGES_PENDING";
		case EVMS_WRITE_LOG_ENTRY:
			return "EVMS_WRITE_LOG_ENTRY";
		case EVMS_GET_DEBUG_LEVEL:
			return "EVMS_GET_DEBUG_LEVEL";
		case EVMS_SET_DEBUG_LEVEL:
			return "EVMS_SET_DEBUG_LEVEL";
		case EVMS_COMMIT_CHANGES:
			return "EVMS_COMMIT_CHANGES";
		case EVMS_CLOSE_ENGINE:
			return "EVMS_CLOSE_ENGINE";
		case EVMS_CAN_DELETE:
			return "EVMS_CAN_DELETE";
		case EVMS_CAN_DESTROY:
			return "EVMS_CAN_DESTROY";
		case EVMS_CAN_EXPAND:
			return "EVMS_CAN_EXPAND";
		case EVMS_CAN_SHRINK:
			return "EVMS_CAN_SHRINK";
		case EVMS_CAN_REPLACE:
			return "EVMS_CAN_REPLACE";
		case EVMS_CAN_SET_INFO:
			return "EVMS_CAN_SET_INFO";
		case EVMS_DELETE:
			return "EVMS_DELETE";
		case EVMS_DESTROY:
			return "EVMS_DESTROY";
		case EVMS_GET_EXPAND_POINTS:
			return "EVMS_GET_EXPAND_POINTS";
		case EVMS_EXPAND:
			return "EVMS_EXPAND";
		case EVMS_GET_SHRINK_POINTS:
			return "EVMS_GET_SHRINK_POINTS";
		case EVMS_SHRINK:
			return "EVMS_SHRINK";
		case EVMS_REPLACE:
			return "EVMS_REPLACE";
		case EVMS_GET_HANDLE_OBJECT_TYPE:
			return "EVMS_GET_HANDLE_OBJECT_TYPE";
		case EVMS_GET_INFO:
			return "EVMS_GET_INFO";
		case EVMS_GET_EXTENDED_INFO:
			return "EVMS_GET_EXTENDED_INFO";
		case EVMS_SET_INFO:
			return "EVMS_SET_INFO";
		case EVMS_GET_OBJECT_HANDLE_FOR_NAME:
			return "EVMS_GET_OBJECT_HANDLE_FOR_NAME";
		case EVMS_GET_FEATURE_LIST:
			return "EVMS_GET_FEATURE_LIST";
		case EVMS_CREATE:
			return "EVMS_CREATE";
		case EVMS_TRANSFER:
			return "EVMS_TRANSFER";
		case EVMS_ASSIGN:
			return "EVMS_ASSIGN";
		case EVMS_CAN_UNASSIGN:
			return "EVMS_CAN_UNASSIGN";
		case EVMS_UNASSIGN:
			return "EVMS_UNASSIGN";
		case EVMS_GET_OBJECT_LIST:
			return "EVMS_GET_OBJECT_LIST";
		case EVMS_GET_PLUGIN_FUNCTIONS:
			return "EVMS_GET_PLUGIN_FUNCTIONS";
		case EVMS_DO_PLUGIN_FUNCTION:
			return "EVMS_DO_PLUGIN_FUNCTION";
		case EVMS_CAN_CREATE_VOLUME:
			return "EVMS_CAN_CREATE_VOLUME";
		case EVMS_CAN_CREATE_COMPATIBILITY_VOLUME:
			return "EVMS_CAN_CREATE_COMPATIBILITY_VOLUME";
		case EVMS_CAN_SET_VOLUME_NAME:
			return "EVMS_CAN_SET_VOLUME_NAME";
		case EVMS_CAN_CONVERT_TO_EVMS_VOLUME:
			return "EVMS_CAN_CONVERT_TO_EVMS_VOLUME";
		case EVMS_CAN_CONVERT_TO_COMPATIBILITY_VOLUME:
			return "EVMS_CAN_CONVERT_TO_COMPATIBILITY_VOLUME";
		case EVMS_CAN_ADD_FEATURE_TO_VOLUME:
			return "EVMS_CAN_ADD_FEATURE_TO_VOLUME";
		case EVMS_CREATE_VOLUME:
			return "EVMS_CREATE_VOLUME";
		case EVMS_SET_VOLUME_NAME:
			return "EVMS_SET_VOLUME_NAME";
		case EVMS_CREATE_COMPATIBILITY_VOLUME:
			return "EVMS_CREATE_COMPATIBILITY_VOLUME";
		case EVMS_CONVERT_TO_EVMS_VOLUME:
			return "EVMS_CONVERT_TO_EVMS_VOLUME";
		case EVMS_CONVERT_TO_COMPATIBILITY_VOLUME:
			return "EVMS_CONVERT_TO_COMPATIBILITY_VOLUME";
		case EVMS_ADD_FEATURE_TO_VOLUME:
			return "EVMS_ADD_FEATURE_TO_VOLUME";
		case EVMS_CAN_MKFS:
			return "EVMS_CAN_MKFS";
		case EVMS_CAN_UNMKFS:
			return "EVMS_CAN_UNMKFS";
		case EVMS_CAN_FSCK:
			return "EVMS_CAN_FSCK";
		case EVMS_CAN_DEFRAG:
			return "EVMS_CAN_DEFRAG";
		case EVMS_MKFS:
			return "EVMS_MKFS";
		case EVMS_UNMKFS:
			return "EVMS_UNMKFS";
		case EVMS_FSCK:
			return "EVMS_FSCK";
		case EVMS_DEFRAG:
			return "EVMS_DEFRAG";
		case EVMS_GET_VOLUME_LIST:
			return "EVMS_GET_VOLUME_LIST";
		case EVMS_CAN_MOUNT:
			return "EVMS_CAN_MOUNT";
		case EVMS_MOUNT:
			return "EVMS_MOUNT";
		case EVMS_CAN_UNMOUNT:
			return "EVMS_CAN_UNMOUNT";
		case EVMS_UNMOUNT:
			return "EVMS_UNMOUNT";
		case EVMS_CAN_REMOUNT:
			return "EVMS_CAN_REMOUNT";
		case EVMS_REMOUNT:
			return "EVMS_REMOUNT";
		case EVMS_CAN_ADD_TO_CONTAINER:
			return "EVMS_CAN_ADD_TO_CONTAINER";
		case EVMS_CAN_REMOVE_FROM_CONTAINER:
			return "EVMS_CAN_REMOVE_FROM_CONTAINER";
		case EVMS_CREATE_CONTAINER:
			return "EVMS_CREATE_CONTAINER";
		case EVMS_GET_CONTAINER_LIST:
			return "EVMS_GET_CONTAINER_LIST";
		case EVMS_CREATE_TASK:
			return "EVMS_CREATE_TASK";
		case EVMS_INVOKE_TASK:
			return "EVMS_INVOKE_TASK";
		case EVMS_DESTROY_TASK:
			return "EVMS_DESTROY_TASK";
		case EVMS_GET_TASK_ACTION:
			return "EVMS_GET_TASK_ACTION";
		case EVMS_GET_ACCEPTABLE_OBJECTS:
			return "EVMS_GET_ACCEPTABLE_OBJECTS";
		case EVMS_GET_SELECTED_OBJECTS:
			return "EVMS_GET_SELECTED_OBJECTS";
		case EVMS_GET_SELECTED_OBJECT_LIMITS:
			return "EVMS_GET_SELECTED_OBJECT_LIMITS";
		case EVMS_SET_SELECTED_OBJECTS:
			return "EVMS_SET_SELECTED_OBJECTS";
		case EVMS_GET_OPTION_COUNT:
			return "EVMS_GET_OPTION_COUNT";
		case EVMS_GET_OPTION_DESCRIPTOR:
			return "EVMS_GET_OPTION_DESCRIPTOR";
		case EVMS_SET_OPTION_VALUE:
			return "EVMS_SET_OPTION_VALUE";
		case EVMS_GET_OPTION_DESCRIPTOR_BY_NAME:
			return "EVMS_GET_OPTION_DESCRIPTOR_BY_NAME";
		case EVMS_SET_OPTION_VALUE_BY_NAME:
			return "EVMS_SET_OPTION_VALUE_BY_NAME";
		case EVMS_MARK_FOR_REDISCOVER:
			return "EVMS_MARK_FOR_REDISCOVER";
		case EVMS_REDISCOVER:
			return "EVMS_REDISCOVER";
		case USER_MESSAGE:
			return "USER_MESSAGE";
		case PROGRESS:
			return "PROGRESS";
		case STATUS:
			return "STATUS";
		case SHUTDOWN:
			return "SHUTDOWN";
		case MSG_INVALID_CMD:
			return "MSG_INVALID_CMD ";
		default:
			return "invalid command";
	}
}


typedef struct msg_hdr_s {
	u_int32_t signature;
	u_int32_t cmd;
	u_int32_t size;
} msg_hdr_t;

#define MSG_HDR_SIG	0x534D5645	/* "evms" */

/*
 * A flat_object_handle_t looks just like an object_handle_t except with the bytes
 * set to network byte order.
 */
typedef object_handle_t flat_object_handle_t;

/*
 * A flat_handle_array_t looks just like a handle_array_t except with the fields
 * set to network byte order.
 */
typedef handle_array_t flat_handle_array_t;

/*
 * A flat_evms_version_t looks just like a evms_version_t except with the fields
 * set to network byte order.
 */
typedef evms_version_t flat_evms_version_t;

typedef struct flat_plugin_info_s {
	object_handle_t handle;				/* Handle for this plug-in record */
	plugin_id_t     id;				/* Plug-in's ID */
	evms_version_t  version;			/* Plug-in's version */
	evms_version_t  required_engine_api_version;	/* Version of the Engine services API */
							/* that the plug-in requires */
	evms_version_t	required_plugin_api_version;    /* Version of the Engine plug-in API */
							/* that the plug-in requires */
	evms_version_t  required_container_api_version; /* Version of the Engine container API */
							/* that the plug-in requires */
	u_int32_t       short_name_offset;
	u_int32_t       long_name_offset;
	u_int32_t       oem_name_offset;
	boolean         supports_containers;
} flat_plugin_info_t;

typedef struct flat_storage_object_info_s {
	object_handle_t         handle;			/* Handle of this object */
	u_int32_t               object_type;		/* SEGMENT, REGION, DISK ,... */
	u_int32_t               data_type;		/* DATA_TYPE, META_DATA_TYPE, FREE_SPACE_TYPE */
	u_int32_t               dev_major;		/* Device major number */
	u_int32_t               dev_minor;		/* Device minor number */
	object_handle_t         disk_group;		/* Disk group in which this object resides */
	plugin_handle_t         plugin;			/* Handle of the plug-in that manages */
							/* this storage object */
	object_handle_t         producing_container;	/* storage_container that produced this object */
	object_handle_t         consuming_container;	/* storage_container that consumed this object */
	u_int32_t               parent_objects_offset;	/* Offset to flat list of parent objects, filled in by parent   */
	u_int32_t               child_objects_offset;	/* Offset to flat list of objects that comprise this object */
	u_int32_t               flags;			/* Defined by SOFLAG_???? defines */
	char                    name[EVMS_NAME_SIZE+1];	/* Object's name */
	lba_t                   start;			/* Relative starting block of this object */
	sector_count_t          size;			/* Size of the storage object in sectors */
	geometry_t              geometry;		/* Optional object geometry */
	object_handle_t         volume;			/* Volume to which this object belongs */
} flat_storage_object_info_t;

typedef struct flat_storage_container_info_s {
	object_handle_t         handle;			/* Handle of this container */
	object_handle_t         disk_group;		/* Disk group in which this container resides */
	object_handle_t         plugin;			/* Handle of the plug-in that */
							/* manages this container */
	u_int32_t               flags;			/* Defined by SCFLAG_???? */
	char                    name[EVMS_NAME_SIZE+1];	/* Container's name */
	u_int32_t               objects_consumed_offset;/* Offset of handles of objects consumed */
							/* by this container */
	u_int32_t               objects_produced_offset;/* Offset of handles of objects produced */
							/* by this container */
	sector_count_t          size;			/* Size of the container */
} flat_storage_container_info_t;

typedef struct flat_logical_volume_info_s {
	object_handle_t         handle;			/* Handle of this volume */
	u_int32_t               dev_major;		/* Device major number */
	u_int32_t               dev_minor;		/* Device minor number */
	object_handle_t         disk_group;		/* Disk group in which this volume resides */
	object_handle_t         file_system_manager;	/* Handle of the File System Interface */
							/* Module that handles this volume */
	u_int32_t               mount_point_offset;	/* Dir where the volume is mounted, NULL if not mounted */
	sector_count_t          fs_size;		/* Size of the file system */
	sector_count_t          min_fs_size;		/* Minimum size of the file system */
	sector_count_t          max_fs_size;		/* Maximum size of the file system */
	sector_count_t          vol_size;		/* Size of the volume */
	sector_count_t          max_vol_size;		/* Maximum size for the volume */
	object_handle_t         associated_volume;	/* Handle of the volume to which this volume is */
							/* associated by an associative feature */
	object_handle_t         object;			/* Handle of the top level object for */
							/* this volume */
	u_int64_t               serial_number;		/* Volume's serial number */
	u_int32_t               flags;			/* Defined by VOLFLAG_???? defines */
	char                    name[EVMS_VOLUME_NAME_SIZE+1];
	char                    dev_node[EVMS_VOLUME_NAME_SIZE+1];
} flat_logical_volume_info_t;

typedef union {
	flat_plugin_info_t            plugin;
	flat_storage_object_info_t    disk;
	flat_storage_object_info_t    segment;
	flat_storage_container_info_t container;
	flat_storage_object_info_t    region;
	flat_storage_object_info_t    object;
	flat_logical_volume_info_t    volume;
} flat_object_info_t;


typedef struct flat_handle_object_info_s {
	object_type_t        type;
	flat_object_info_t   info;
} flat_handle_object_info_t;

/*
 * A flat_expand_handle_array_t looks just like a expand_handle_array_t except
 * with the fields set to network byte order.
 */
typedef expand_handle_array_t flat_expand_handle_array_t;

/*
 * A flat_shrink_handle_array_t looks just like a shrink_handle_array_t except
 * with the fields set to network byte order.
 */
typedef shrink_handle_array_t flat_shrink_handle_array_t;

typedef struct flat_function_info_s {
	task_action_t   function;	/* Plugin function number */
	u_int32_t	name_offset;	/* Short, unique (within the plug-in) name for the function */
					/* e.g., "addspare" */
	u_int32_t	title_offset;	/* Short title for the function */
					/* e.g. "Add a spare" */
					/* Example usage:  A UI might put this in */
					/* a menu of functions to select. */
	u_int32_t	verb_offset;	/* One or two action words for the function */
					/* e.g. "Add" */
					/* Example usage:  A GUI may use this on an */
					/* action button for the function. */
	u_int32_t	help_offset;	/* Full help text */
					/* e.g. "Use this function to add a spare blah blah blah..." */
	u_int32_t	flags;
} flat_function_info_t;

typedef struct flat_function_info_array_s {
	uint                 count;
	flat_function_info_t info[0];
} flat_function_info_array_t;

/*
 * A flat_declined_handle_array_t looks just like a declined_handle_array_t except
 * with the fields set to network byte order.
 */
typedef declined_handle_array_t flat_declined_handle_array_t;

/*
 * A flat_value_t looks just like a value_t except
 * with the fields set to network byte order.
 */
typedef value_t      flat_value_t;
typedef value_list_t flat_value_list_t;

/*
 * A flat_value_range_t looks just like a flat_value_range_t except
 * with the fields set to network byte order.
 */
typedef value_range_t flat_value_range_t;

typedef struct flat_key_value_pair_s {
	u_int32_t       name_offset;	/* Key if name-based */
	u_int16_t       number;		/* Key if number-based */
	boolean         is_number_based;/* TRUE if number-based */
	u_int32_t       type;		/* Value type */
	value_t         value;		/* Union of basic data types */
} flat_key_value_pair_t;

typedef struct flat_option_array_s {
	u_int			count;
	flat_key_value_pair_t	option[0];
} flat_option_array_t;

/*
 * A flat_value_collection_t looks just like a value_collection_t except
 * with the fields set to network byte order.
 */
typedef value_collection_t flat_value_collection_t;

typedef struct flat_group_info_s {
	u_int32_t	group_number;		/* group number, 0 if not grouped */
	u_int32_t	group_level;		/* possibly used for indenting, or sub fields */
	u_int32_t	group_name_offset;	/* name of group                              */
} flat_group_info_t;

typedef struct flat_option_descriptor_s {
	u_int32_t                 name_offset;		/* Option name/key */
	u_int32_t                 title_offset;		/* One or two word description of option */
	u_int32_t                 tip_offset;		/* Multi-sentence description of option for tip */
	u_int32_t                 help_offset;		/* Multi-paragraph detailed option help */
	value_type_t              type;			/* Defines option data type */
	u_int32_t                 unit;			/* Defines unit value */
	value_format_t            format;		/* Suggested format for display of values */
	u_int32_t                 min_len;		/* Minimum length for string types */
	u_int32_t                 max_len;		/* Maximum length for string types */
	u_int64_t                 flags;		/* Option flags (defined below) */
	u_int32_t                 constraint_type;	/* Constraint type (none, range, list) */
	u_int32_t                 constraint_offset;	/* Either a list or range of valid input values */
	value_t                   value;		/* Initial/current value */
	flat_group_info_t         group;		/* Group information for display purposes     */
} flat_option_descriptor_t;

typedef struct flat_option_desc_array_s {
	u_int32_t                count;			/* Number of option descriptors in the following array */
	flat_option_descriptor_t option[1];		/* option_descriptor_t is defined in option.h */
} flat_option_desc_array_t;

typedef struct flat_extended_info_s {
	u_int32_t               name_offset;	/* Info field name */
	u_int32_t               title_offset;	/* One or two word description of info field */
	u_int32_t               desc_offset;	/* Multi-sentence description of info field */
	u_int32_t               type;		/* Defines info data type */
	u_int32_t               unit;		/* Defines info unit value */
	u_int32_t               format;		/* Suggested format for display of values */
	value_t                 value;		/* Single value if not a collection */
	u_int32_t               collection_type;/* Defines if either a list or range of values */
	u_int32_t               collection_offset;	/* Either a list or range of values of value_type_t */
	flat_group_info_t       group;		/* Group information for display purposes */
	u_int16_t               flags;		/* Extended info flags (defined below) */
} flat_extended_info_t;

typedef struct flat_extended_info_array_s {
	u_int32_t		count;		/* Count of extended_info_t structs in array */
	flat_extended_info_t	info[0];	/* Info descriptors */
} flat_extended_info_array_t;

typedef struct flat_node_info_s {
	node_flags_t    flags;
	u_int32_t       node_name_offset;
} flat_node_info_t;


typedef struct flat_node_list_s {
	u_int32_t        count;
	flat_node_info_t node_info[0];
} flat_node_list_t;

typedef struct flat_progress_s {
	u_int32_t       id;
	u_int32_t       title_offset;
	u_int32_t       description_offset;
	u_int32_t       type;
	u_int64_t       count;
	u_int64_t       total_count;
	u_int32_t       remaining_seconds;

	/*
	 * The following fields are pointers in the normal progress_t structure.
	 * Since we have no idea what kind of data they point to, w cannot
	 * flatten the private data.  However, the fields are only accessed at
	 * one end of the progress call -- the plugin_private_data is only
	 * set/accessed by the plug-in, the ui_private_data is only set/accessed
	 * by the UI.  Therefore it is safe to just send the pointers as
	 * integers  since niether side will derefernce the other side's pointer.
	 */
	u_int64_t       plugin_private_data;
	u_int64_t       ui_private_data;
} flat_progress_t;

typedef struct flat_change_record_s {
	u_int32_t       name_offset;
	u_int32_t       type;
	u_int32_t       changes;
} flat_change_record_t;

typedef struct flat_change_record_array_s {
	u_int32_t            count;
	flat_change_record_t changes_pending[0];
} flat_change_record_array_t;


/*
 * Definitions for flattend argument structures for API parameters.
 */
typedef struct flat_open_engine_args_s {
	u_int32_t     mode;
	u_int32_t     debug_level;
	u_int32_t     log_name_offset;
} flat_open_engine_args_t;

typedef struct flat_get_object_list_args_s {
	u_int32_t object_type;
	u_int32_t data_type;
	u_int32_t plugin;
	u_int32_t disk_group;
	u_int32_t flags;
} flat_get_object_list_args_t;

typedef struct flat_get_container_list_args_s {
	u_int32_t plugin_handle;
	u_int32_t disk_group;
	u_int32_t flags;
} flat_get_container_list_args_t;

typedef struct flat_get_volume_list_args_s {
	u_int32_t fsim_handle;
	u_int32_t disk_group;
	u_int32_t flags;
} flat_get_volume_list_args_t;

typedef struct flat_get_plugin_list_args_s {
	u_int32_t type;
	u_int32_t flags;
} flat_get_plugin_list_args_t;

/*
 * Definitions for flattend returned data.
 */
//typedef struct flat_int_ret_s {
//	u_int32_t     x;
//} flat_int_ret_t;
typedef u_int32_t flat_int_ret_t;


typedef struct flat_user_message_s {
	u_int32_t message_text_offset;
	u_int32_t answer;
	u_int32_t choices_count;
	u_int32_t choices_offset;
} flat_user_message_t;

typedef struct flat_get_object_list_ret_s {
	u_int32_t           rc;
	flat_handle_array_t objects;
} flat_get_object_list_ret_t;

/*
 * Flat list return info for containers, volumes, and plug-ins looks
 * exactly like flat list return info for objects.
 */
typedef flat_get_object_list_ret_t flat_get_container_list_ret_t;
typedef flat_get_object_list_ret_t flat_get_volume_list_ret_t;
typedef flat_get_object_list_ret_t flat_get_plugin_list_ret_t;

typedef struct flat_get_info_ret_s {
	u_int32_t rc;
	flat_handle_object_info_t info;
} flat_get_info_ret_t;

typedef struct flat_changes_pending_ret_s {
	u_int32_t                  rc;
	u_int32_t                  result;
	flat_change_record_array_t change_records;
} flat_changes_pending_ret_t;


int evms_mark_for_rediscover(char * name);

#define evms_mark_for_rediscover_args_f string_f

#define evms_mark_for_rediscover_rets_f int_f

int evms_rediscover(void);

#define evms_rediscover_args_f ""

#define evms_rediscover_rets_f int_f

#endif
