/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: memman.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "fullengine.h"
#include "memman.h"
#include "engine.h"


/*
 * engine_alloc is a service provided so that plug-ins use the same memory
 * management library that the Engine does.  This is especially needed if
 * the Engine is ever expected to free the memory.
 */
#ifdef EVMS_DEBUG
#define MEM_SIG     "EVMSdbug"
#define MEM_SIG_LEN (sizeof(MEM_SIG) - 1)


struct mem_hdr {
	char sig[MEM_SIG_LEN];
	uint len;
};
#endif

void * engine_alloc(u_int32_t size) {

	void * pMem = NULL;

	LOG_PROC_EXTRA_ENTRY();

	LOG_EXTRA("Request to allocate %u bytes.\n", size);

	if (size > 0) {
#ifndef EVMS_DEBUG
		pMem = calloc(1, size);
#else
		struct mem_hdr * p_hdr;
		u_int32_t new_size = size + sizeof(struct mem_hdr) + MEM_SIG_LEN;

		LOG_EXTRA("Allocate %u bytes.\n", new_size);

		p_hdr = calloc(1, new_size);
		if (p_hdr != NULL) {
			LOG_EXTRA("Real buffer is at %p.\n", p_hdr);
			memcpy(p_hdr->sig, MEM_SIG, MEM_SIG_LEN);
			p_hdr->len = size;
			pMem = p_hdr + 1;
			memcpy(pMem + size, MEM_SIG, MEM_SIG_LEN);
		} else {
			pMem = NULL;
		}
#endif
	}

	LOG_PROC_EXTRA_EXIT_PTR(pMem);
	return pMem;
}


void * engine_realloc(void * pBuf, u_int32_t size) {

	void * pMem = NULL;

	LOG_PROC_EXTRA_ENTRY();

	LOG_EXTRA("Request to reallocate buffer at %p to %u bytes.\n", pBuf, size);

#ifndef EVMS_DEBUG
	pMem = realloc(pBuf, size);
#else
	if (pBuf != NULL) {
		struct mem_hdr * p_hdr = pBuf -= sizeof(struct mem_hdr);
		u_int32_t new_size = size + sizeof(struct mem_hdr) + MEM_SIG_LEN;

		LOG_EXTRA("Reallocate real buffer at %p to %u bytes.\n", p_hdr, new_size);

		p_hdr = realloc(p_hdr, new_size);
		if (p_hdr != NULL) {
			LOG_EXTRA("Real buffer is at %p.\n", p_hdr);
			p_hdr->len = size;
			pMem = p_hdr + 1;
			memcpy(pMem + size, MEM_SIG, MEM_SIG_LEN);
		}
	} else {
		pMem = engine_alloc(size);
	}
#endif

	LOG_PROC_EXTRA_EXIT_PTR(pMem);
	return pMem;
}


char * engine_strdup(const char * str) {

	char * new_str;

	LOG_PROC_EXTRA_ENTRY();

	LOG_EXTRA("Request to duplicate string \"%s\".\n", str);

	if (str == NULL) {
		LOG_PROC_EXTRA_EXIT_PTR(NULL);
		return NULL;
	}

#ifndef EVMS_DEBUG
	new_str = strdup(str);
#else
	new_str = engine_alloc(strlen(str) + 1);
	if (new_str != NULL) {
		strcpy(new_str, str);
	}
#endif

	LOG_PROC_EXTRA_EXIT_PTR(new_str);
	return new_str;
}


/*
 * engine_free is the compliment to engine_alloc.
 */
void engine_free(void * thing) {

	LOG_PROC_EXTRA_ENTRY();

	LOG_EXTRA("Request to free memory at %p.\n", thing);

	if (thing != NULL) {
#ifndef EVMS_DEBUG
		free(thing);
#else
		struct mem_hdr * p_hdr;

		p_hdr = (struct mem_hdr *) (thing - sizeof(struct mem_hdr));
		LOG_EXTRA("Buffer size is %u.\n", p_hdr->len);

		if (memcmp(p_hdr->sig, MEM_SIG, MEM_SIG_LEN) == 0) {
			memset(p_hdr->sig, '\0', MEM_SIG_LEN);

                        if (memcmp(thing + p_hdr->len, MEM_SIG, MEM_SIG_LEN) != 0) {
				LOG_SERIOUS("Memory overrun detected!  Signature at offset %u in buffer %p has been trashed.\n", p_hdr->len, thing);
			} else {
				memset(thing + p_hdr->len, '\0', MEM_SIG_LEN);
			}

			LOG_EXTRA("Free real buffer at %p.\n", p_hdr);
			free(p_hdr);

		} else {
			LOG_SERIOUS("Memory at address %p is not Engine allocated memory!  I'm not freeing it.\n", thing);
		}
#endif
	}

	LOG_PROC_EXTRA_EXIT_VOID();
}


#define MEM_OBJ_HEAD_SIGNATURE  0x54455448  /* "MEMH" */

typedef struct mem_object_header_s {
	u_int32_t   signature;
	void        (*free_function)(void *);
} mem_object_header_t;

/*
 * Allocate a structure for returning to an application.
 * A header, which contains a signature and a pointer to a function for
 * freeing the contents of the user structure, is prepended to the buffer
 * that is allocated.
 */
void * alloc_app_struct(uint size, void (*free_function)(void *)) {

	void * pMem;

	mem_object_header_t * mem_obj = engine_alloc(sizeof(mem_object_header_t) + size);

	LOG_PROC_ENTRY();

	if (mem_obj != NULL) {
		mem_obj->signature = MEM_OBJ_HEAD_SIGNATURE;
		mem_obj->free_function = free_function;

		pMem = mem_obj + 1;

	} else {
		pMem = NULL;
	}

	LOG_PROC_EXIT_PTR(pMem);
	return pMem;
}


/*
 * Free an application buffer.
 * Look for a header before the beginning of the buffer.  If one is there and
 * is has a function for freeing the contents of the buffer, then call the
 * function.  If we don't find a header before the buffer, ignore this call.
 * In that case we don't know where the memory came from so we don't know how
 * to free it.  It will be cleaned up when the process exits.
 */
void evms_free(void * buffer) {

	mem_object_header_t * mem_obj = buffer - sizeof(mem_object_header_t);

	LOG_PROC_ENTRY();

	LOG_EXTRA("Request to free application buffer at %p.\n", buffer);

	if (buffer != NULL) {
		if (mem_obj->signature == MEM_OBJ_HEAD_SIGNATURE) {
			if (mem_obj->free_function != NULL) {
				mem_obj->free_function(buffer);

			} else {
				LOG_EXTRA("Application buffer has no supplementary free_function().\n");
			}

			engine_free(mem_obj);

		} else {
			LOG_WARNING("Application buffer does not have our memory object header.  Request ignored.\n");
		}
	}

	LOG_PROC_EXIT_VOID();
}
