/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: options.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "bbr_seg.h"
#include "options.h"

/*
 *  Returns count of options for specified task
 */
int BBR_GetOptionCount(task_context_t * task)
{
    int count;

    LOG_ENTRY();

    switch (task->action) {
	
    case EVMS_Task_Assign_Plugin:
    case EVMS_Task_Create:
    case EVMS_Task_Expand:
    case EVMS_Task_Shrink:
        count = BBR_OPTION_COUNT;
        break;

    default:
        count = 0;
        break;
    }

    LOG_EXIT_INT(count);
    return count;
}

/*
 * Validate the objects in the selected_objects list in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects list.
 * Modify the accepatble_objects list in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int BBR_SetObjects( task_context_t * context,
                    list_anchor_t    declined_objects,
                    task_effect_t  * effect )
{

    int rc = EINVAL;

    LOG_ENTRY();

    if (context) {

        switch (context->action) {

	case EVMS_Task_Assign_Plugin:
        case EVMS_Task_Create:
        case EVMS_Task_Expand:
        case EVMS_Task_Shrink:
	    rc = 0;
        default:

            LOG_ERROR("context->action is unknown or unsupported\n");
            break;
        }
    }

    LOG_EXIT_INT(rc);
    return rc;
}

int BBR_SetOption( task_context_t * context,
                   u_int32_t        index,
                   value_t        * value,
                   u_int32_t      * info )
{
    int rc=EINVAL;

    LOG_ENTRY();

    // a measure of protection ...
    if (context == NULL) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    switch (context->action) {

    case EVMS_Task_Assign_Plugin:
    case EVMS_Task_Create:
    case EVMS_Task_Expand:
    case EVMS_Task_Shrink:
	rc = 0;
	break;
    default:
        break;
    }


    LOG_EXIT_INT(rc);

    return rc;
}

/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_create_objects( task_context_t * context )
{
    storage_object_t * object;
    list_element_t     itr1, itr2;
    int                rc = EINVAL;
    int                count;

    LOG_ENTRY();

    count = EngFncs->list_count(context->acceptable_objects);

    if (count == 0) {

        rc = EngFncs->get_object_list( SEGMENT | DISK, DATA_TYPE, NULL,
                                       NULL, VALID_INPUT_OBJECT,
                                       &context->acceptable_objects );

        if (!rc) {
            LIST_FOR_EACH_SAFE(context->acceptable_objects, itr1, itr2, object) {
                if (object->plugin == my_plugin_record) {
                   EngFncs->delete_element(itr1);
                }
            }
        }
    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }

    LOG_EXIT_INT(rc);
    return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_shrink_objects( task_context_t * context  )
{
    storage_object_t * object, * child;
    list_anchor_t      shrink_points;
    list_element_t     itr1, itr2;
    int                count, rc = EINVAL;

    LOG_ENTRY();

    count = EngFncs->list_count(context->acceptable_objects);
    if (count == 0) {

        shrink_points = EngFncs->allocate_list();
        if (shrink_points) {

            rc = EngFncs->get_object_list( SEGMENT, DATA_TYPE, my_plugin_record,
                                           NULL, VALID_INPUT_OBJECT,
                                           &context->acceptable_objects );
            if (!rc) {

                LIST_FOR_EACH_SAFE(context->acceptable_objects, itr1, itr2, object) {
                    child = GET_BBR_CHILD( object );
                    if ( child ) {
                        rc = child->plugin->functions.plugin->can_shrink(child, NULL,
                                                                         shrink_points);
                        if (rc) {
                            EngFncs->delete_element(itr1);
                        }
                        EngFncs->delete_all_elements(shrink_points);
                    }
                }
            }

            EngFncs->destroy_list(shrink_points);
        }
	else {
           LOG_ERROR("Cannot allocate shrink-points list.\n");
        }
    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }

    LOG_EXIT_INT(rc);
    return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_expand_objects( task_context_t * context  )
{
    storage_object_t * object, * child;
    list_anchor_t    expand_points;
    list_element_t   itr1, itr2;
    int              count, rc = EINVAL;

    LOG_ENTRY();

    count = EngFncs->list_count(context->acceptable_objects);
    if (count == 0) {

        expand_points = EngFncs->allocate_list();
        if (expand_points) {

            rc = EngFncs->get_object_list( SEGMENT, DATA_TYPE, my_plugin_record,
                                           NULL, VALID_INPUT_OBJECT,
                                           &context->acceptable_objects );
            if (!rc) {

                LIST_FOR_EACH_SAFE(context->acceptable_objects, itr1, itr2, object) {
                    child = GET_BBR_CHILD( object );
                    if ( child ) {
                        rc = child->plugin->functions.plugin->can_expand(child, NULL,
                                                                         expand_points);
                        if (rc) {
                            EngFncs->delete_element(itr1);
                        }
                        EngFncs->delete_all_elements(expand_points);
                    }
                }
            }
            EngFncs->destroy_list(expand_points);
        }
	else {
           LOG_ERROR("Cannot allocate expand-points list.\n");
        }
    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }

    LOG_EXIT_INT(rc);
    return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
int BBR_InitTask( task_context_t * context )
{
    int rc = EINVAL;

    LOG_ENTRY();


    if (context) {

        switch (context->action) {
	    
	case EVMS_Task_Assign_Plugin:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

	    rc = get_acceptable_create_objects( context );
	    break;

        case EVMS_Task_Create:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

	    rc = get_acceptable_create_objects( context );
	    break;

        case EVMS_Task_Expand:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

	    rc = get_acceptable_expand_objects( context );
            break;

        case EVMS_Task_Shrink:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

	    rc = get_acceptable_shrink_objects( context );
	    break;
	case BBR_FUNCTION_ENABLE:
	case BBR_FUNCTION_DISABLE:
            context->min_selected_objects = 0;
            context->max_selected_objects = 0;
	    context->option_descriptors->count = 0;
	    rc = 0;
	    break;
	    
        default:
               LOG_ERROR("context->action is unknown or unsupported\n");
               break;
        }
    }


    LOG_EXIT_INT(rc);
    return rc;
}


/*
 * Returns information about a BBR object
 */
int BBR_GetInfo( storage_object_t  * object, char *name,  
		 extended_info_array_t * * info)
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    BBR_Private_Data        *pdata = (BBR_Private_Data *) object->private_data;


    LOG_ENTRY();

    // a measure of protection ...
    if ((info == NULL)||(pdata->signature != EVMS_BBR_SIGNATURE)) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    if ( object->object_type == SEGMENT ) {

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + 
					 (BBR_INFO_COUNT * 
					  sizeof(extended_info_t) ) );
        if (Info) {

            Info->count = BBR_INFO_COUNT;

            SET_STRING( Info->info[BBR_INFO_NAME_INDEX].name, "Name" );
            SET_STRING( Info->info[BBR_INFO_NAME_INDEX].title, "Name" );
            SET_STRING( Info->info[BBR_INFO_NAME_INDEX].desc, 
			      "This is the unique name the system gave the segment." );
            Info->info[BBR_INFO_NAME_INDEX].type               = EVMS_Type_String;
            Info->info[BBR_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
            SET_STRING( Info->info[BBR_INFO_NAME_INDEX].value.s, object->name );
            Info->info[BBR_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING( Info->info[BBR_INFO_UUID_INDEX].name, "Uuid" );
            SET_STRING( Info->info[BBR_INFO_UUID_INDEX].title, "Uuid" );
            SET_STRING( Info->info[BBR_INFO_UUID_INDEX].desc, 
			      "This is the unique id the system gave the segment." );
            Info->info[BBR_INFO_UUID_INDEX].type               = EVMS_Type_String;
            Info->info[BBR_INFO_UUID_INDEX].unit               = EVMS_Unit_None;
            SET_STRING( Info->info[BBR_INFO_UUID_INDEX].value.s, object->uuid );
            Info->info[BBR_INFO_UUID_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_UUID_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING( Info->info[BBR_INFO_SIZE_INDEX].name, "Size" );
            SET_STRING( Info->info[BBR_INFO_SIZE_INDEX].title, "Size" );
            SET_STRING( Info->info[BBR_INFO_SIZE_INDEX].desc, "This is the size in sectors of the storage object after reserving space for metadata.");
            Info->info[BBR_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
            Info->info[BBR_INFO_SIZE_INDEX].value.ui64         = object->size;
            Info->info[BBR_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));
            Info->info[BBR_INFO_SIZE_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

            SET_STRING( Info->info[BBR_INFO_BLKS_INDEX].name, "Reserve Blocks" );
            SET_STRING( Info->info[BBR_INFO_BLKS_INDEX].title, "Blocks" );
            SET_STRING( Info->info[BBR_INFO_BLKS_INDEX].desc, "This is the number of replacement blocks BBR is reserving for this storage object.");
            Info->info[BBR_INFO_BLKS_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_BLKS_INDEX].unit               = EVMS_Unit_None;
            Info->info[BBR_INFO_BLKS_INDEX].value.ui64         = pdata->replacement_blocks_needed;
            Info->info[BBR_INFO_BLKS_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_BLKS_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING( Info->info[BBR_INFO_BLKSZ_INDEX].name, "Block Size" );
            SET_STRING( Info->info[BBR_INFO_BLKSZ_INDEX].title, "Block Size" );
            SET_STRING( Info->info[BBR_INFO_BLKSZ_INDEX].desc, "This value tells you the size of a replacement block.");
            Info->info[BBR_INFO_BLKSZ_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_BLKSZ_INDEX].unit               = EVMS_Unit_None;
            Info->info[BBR_INFO_BLKSZ_INDEX].value.ui64         = pdata->block_size;
            Info->info[BBR_INFO_BLKSZ_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_BLKSZ_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING( Info->info[BBR_INFO_TBLSZ_INDEX].name, "Table Size" );
            SET_STRING( Info->info[BBR_INFO_TBLSZ_INDEX].title, "Size of BBR table" );
            SET_STRING( Info->info[BBR_INFO_TBLSZ_INDEX].desc, "This is the number of sectors being used by the BBR remapping table.");
            Info->info[BBR_INFO_TBLSZ_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[BBR_INFO_TBLSZ_INDEX].unit               = EVMS_Unit_Sectors;
            Info->info[BBR_INFO_TBLSZ_INDEX].value.ui64         = pdata->bbr_table_size_in_sectors;
            Info->info[BBR_INFO_TBLSZ_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[BBR_INFO_TBLSZ_INDEX].group, 0, sizeof(group_info_t));
            Info->info[BBR_INFO_TBLSZ_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

            *info = Info;

            rc = 0;
        }
        else {
            LOG_ERROR("unable to malloc memory for extended info array\n");
        }
    }

    LOG_EXIT_INT(rc);
    return rc;
}


/*
 *
 */
int BBR_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    char                     version_string[64];
    char                     required_engine_api_version_string[64];
    char                     required_plugin_api_version_string[64];


    LOG_ENTRY();

    // a measure of protection ...
    if (info == NULL) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + 
				     (BBR_PLUGIN_INFO_COUNT * 
				      sizeof(extended_info_t))  );
    if (Info) {

        Info->count = BBR_PLUGIN_INFO_COUNT;

        sprintf(version_string, "%d.%d.%d",
                MAJOR_VERSION,
                MINOR_VERSION,
                PATCH_LEVEL );

        sprintf(required_engine_api_version_string, "%d.%d.%d",
                my_plugin_record->required_engine_api_version.major,
                my_plugin_record->required_engine_api_version.minor,
                my_plugin_record->required_engine_api_version.patchlevel );

        sprintf(required_plugin_api_version_string, "%d.%d.%d",
                my_plugin_record->required_plugin_api_version.plugin.major,
                my_plugin_record->required_plugin_api_version.plugin.minor,
                my_plugin_record->required_plugin_api_version.plugin.patchlevel );

        SET_STRING( Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].name, "Short Name" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].title, "Short Name" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].desc, "A short name given to this plugin.");
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
        SET_STRING( Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].value.s, my_plugin_record->short_name );
        Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING( Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].name, "Long Name" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].title, "Long Name" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].desc, "A long name given to this plugin.");
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
        SET_STRING( Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].value.s, my_plugin_record->long_name );
        Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING( Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].name, "Type" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].title, "Plugin Type" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].desc, "There are various types of plugins; each responsible for some kind of storage object.");
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
        SET_STRING( Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].value.s, "Segment Manager");
        Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING( Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].name, "Version" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].title, "Plugin Version" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].desc, "This is the version number of the plugin.");
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
        SET_STRING( Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].value.s, version_string );
        Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING( Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].name, "Required Engine Services Version" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].title, "Required Engine Services Version" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].desc, "This is the version of the Engine services that this plug-in requires. It will not run on older versions of the Engine services.");
        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].unit               = EVMS_Unit_None;
        SET_STRING( Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].value.s, required_engine_api_version_string );
        Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING( Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].name, "Required Engine Plug-in API Version" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].title, "Required Engine Plug-in API Version" );
        SET_STRING( Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].desc, "This is the version of the Engine plug-in API that this plug-in requires. It will not run on older versions of the Engine plug-in API.");
        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].type               = EVMS_Type_String;
        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].unit               = EVMS_Unit_None;
        SET_STRING( Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].value.s, required_plugin_api_version_string );
        Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].group, 0, sizeof(group_info_t));

        *info = Info;

        rc = 0;
    }


    LOG_EXIT_INT(rc);
    return rc;
}
