/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libbsd.so
 *
 *   File: commit.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "bsd_plugin.h"


static int do_bsd_commit( LOGICALDISK *ld, DISKSEG *md )
{
	struct bsd_partition       *p;
	DISKSEG                    *seg;
	char                        buffer[EVMS_VSECTOR_SIZE];
	bsd_disklabel_t            *disk_label=(bsd_disklabel_t *)buffer;
	int                         rc = 0;
	seg_private_data_t         *pdata;
	list_element_t		    iter;

	LOG_ENTRY();

	// read bsd info & verify we have valid disk label
	rc = READ( ld, 0, 1, buffer );

	if (!rc) {
		if ( DISK_TO_CPU32(disk_label->d_magic) != BSD_DISKMAGIC) {
			rc = EINVAL;
		}
	}

	if (rc) {
		LOG_EXIT_INT(rc);
		return rc;
	}



	// walk through the partition table, removing all linux partition records        
	for ( p = disk_label->d_partitions; 
	    p - disk_label->d_partitions < DISK_TO_CPU16(disk_label->d_npartitions); 
	    p++) {

		// only delete the record if it has an file system type. this 
		// will prevent us from deleting those bsd partition records that
		// are used to reserve space at the front of the disk. These 
		// records seem to have a size and offset but no file system type.
		if (p->p_fstype != BSD_TAG_UNUSED) {

			memset( (void *) p, 0, sizeof(struct bsd_partition) );  

		}

	}


	// now walk through the segment list on this drive and add each 
	// data segment the BSD partition table.
	LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

		if (seg->data_type == DATA_TYPE) {

			// find an unused entry in the bsd partition table
			for ( p = disk_label->d_partitions, rc=ENOMEM; 
			    p - disk_label->d_partitions < DISK_TO_CPU16(disk_label->d_npartitions); 
			    p++) {

				if ( (p->p_fstype == BSD_TAG_UNUSED) &&
				     (p->p_size   == 0 ) && 
				     (p->p_offset == 0 )) {

					rc = 0;
					break;

				}

			}

			// create entry in bsd partition table for this segment
			if (!rc) {

				pdata = (seg_private_data_t *) seg->private_data;

				memcpy( (void *) p, (void *) &pdata->p_record, sizeof(bsd_partition_t) );

				p->p_offset  = CPU_TO_DISK32( seg->start );
				p->p_size    = CPU_TO_DISK32( seg->size );
			}

		}

	}

	// update BSD info
	rc = WRITE( ld, 0, 1, buffer );

	LOG_EXIT_INT(rc);
	return rc;
}



/*
 *  Function:  commit_bsd_metadata
 *
 *  Called to commit the BSD disk label information.
 */
int  commit_bsd_segments( DISKSEG *seg, LOGICALDISK *ld,  uint commit_phase )
{
	int rc=0;

	LOG_ENTRY();

	if (seg->data_type == META_DATA_TYPE) {

		rc = do_bsd_commit( ld, seg );
		if (!rc) {
			seg->flags &= ~SOFLAG_DIRTY;
		}

	}
	else {
		seg->flags &= ~SOFLAG_DIRTY;
	}


	LOG_EXIT_INT(rc);
	return rc;
}

