/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvm_move.c
 *
 * Description: This file contains all functions related to moving extents.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"

/**
 * lvm_move_extent_setup
 * @le:		Logical extent to move.
 * @dst_pe:	Physical extent to move the LE to.
 *
 * Setup the specified LE to be moved from its current PE to the new PE. Make
 * sure the destination PE is available before calling this function.
 **/
static void lvm_move_extent_setup(lvm_logical_extent_t * le,
				  lvm_physical_extent_t * dst_pe)
{
	lvm_logical_volume_t * volume = le->volume;
	lvm_volume_group_t * group = volume->group;

	LOG_ENTRY();

	le->new_pe = dst_pe;
	dst_pe->new_le = le;
	dst_pe->pv->move_extents++;
	group->move_extents++;
	volume->flags |= LVM_LV_FLAG_MOVE_PENDING;

	lvm_append_region_to_segment(volume->region, dst_pe->pv->segment);

	LOG_EXIT_VOID();
}

/**
 * move_extent_init_copy_job
 **/
static int move_extent_init_copy_job(lvm_logical_extent_t * le,
				     copy_job_t * copy_job)
{
	lvm_physical_extent_t * src_pe = le->pe;
	lvm_physical_extent_t * dst_pe = le->new_pe;
	u_int32_t pe_size = src_pe->pv->pv->pe_size;
	int title_size = (EVMS_NAME_SIZE+5)*3 + 30;
	int rc = 0;

	LOG_ENTRY();

	copy_job->src.obj = src_pe->pv->segment;
	copy_job->src.start = src_pe->sector;
	copy_job->src.len = pe_size;
	copy_job->trg.obj = dst_pe->pv->segment;
	copy_job->trg.start = dst_pe->sector;
	copy_job->trg.len = pe_size;
	copy_job->description = NULL;

	copy_job->title = EngFncs->engine_alloc(title_size);
	if (!copy_job->title) {
		rc = ENOMEM;
		goto out;
	}
	snprintf(copy_job->title, title_size,
		 "LVM: Copying LE %s:%d from %s:%d to %s:%d",
		 le->volume->region->name, le->number,
		 src_pe->pv->segment->name, src_pe->number,
		 dst_pe->pv->segment->name, dst_pe->number);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * move_extent_copy_extent
 *
 * Copy the data from the source PE to the destination PE.
 *
 * Offline implementation:
 * - Setup an engine copy job and submit to the engine's copy-service.
 *
 * Online implementation:
 * - Call the engine to setup the mirror device.
 * - Load new mapping for the region.
 * - Suspend the region.
 * - Call engine to activate the mirror device.
 * - Resume the region.
 * - Wait for the engine's copy to finish.
 **/
static int move_extent_copy_extent(lvm_logical_extent_t * le,
				   copy_job_t * copy_job)
{
	lvm_logical_volume_t *volume = le->volume;
	dm_target_t *target_list = NULL;
	int rc;

	LOG_ENTRY();

	if (EngFncs->can_online_copy()) {
		/* Set up the mirror. */
		rc = EngFncs->copy_setup(copy_job);
		if (rc) {
			goto out;
		}

		/* Load the new mapping and suspend. */
		le->copy_job = copy_job;
		target_list = lvm_build_volume_targets(volume);
		if (!target_list) {
			rc = ENOMEM;
			goto out;
		}

		rc = EngFncs->dm_load_targets(volume->region, target_list);
		if (rc) {
			goto out;
		}

		EngFncs->dm_set_suspended_flag(TRUE);

		rc = EngFncs->dm_suspend(volume->region, TRUE);
		if (rc) {
			EngFncs->dm_set_suspended_flag(FALSE);
			EngFncs->dm_clear_targets(volume->region);
			goto out;
		}

		/* Activate the mirror. */
		rc = EngFncs->copy_start(copy_job);
		if (rc) {
			EngFncs->dm_clear_targets(volume->region);
			EngFncs->dm_suspend(volume->region, FALSE);
			EngFncs->dm_set_suspended_flag(FALSE);
			goto out;
		}

		/* Resume and activate the new mapping. */
		EngFncs->dm_suspend(volume->region, FALSE);
		EngFncs->dm_set_suspended_flag(FALSE);

		/* Wait for the copy to complete. */
		rc = EngFncs->copy_wait(copy_job);

	} else {
		rc = EngFncs->offline_copy(copy_job);
	}

out:
	EngFncs->dm_deallocate_targets(target_list);
	le->copy_job = NULL;
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * move_extent_update_metadata
 *
 * Update the metadata for the source and target PEs after the copy has
 * completed and write the new metadata to disk.
 **/
static int move_extent_update_metadata(lvm_logical_extent_t * le)
{
	lvm_physical_extent_t * src_pe = le->pe;
	lvm_physical_extent_t * dst_pe = le->new_pe;

	LOG_ENTRY();

	/* Update the source's metadata. */
	if (src_pe) {
		src_pe->pe.lv_num = 0;
		src_pe->pe.le_num = 0;
		src_pe->pv->pv->pe_allocated--;
		if (!lvm_volume_is_on_pv(le->volume, src_pe->pv)) {
			src_pe->pv->pv->lv_cur--;
			lvm_remove_region_from_segment(le->volume->region,
						       src_pe->pv->segment);
		}
		src_pe->le = NULL;
	}

	/* Update the destination's metadata. */
	if (!lvm_volume_is_on_pv(le->volume, dst_pe->pv)) {
		dst_pe->pv->pv->lv_cur++;
	}
	dst_pe->pe.lv_num = le->volume->number;
	dst_pe->pe.le_num = le->number;
	dst_pe->pv->pv->pe_allocated++;
	dst_pe->new_le = NULL;
	dst_pe->le = le;
	dst_pe->pv->move_extents--;
	dst_pe->pv->group->move_extents--;

	/* Update volume's map. */
	le->pe = dst_pe;
	le->new_pe = NULL;

	/* Write the metadata to disk. */
	/* FIXME: How to handle I/O errors? */
	lvm_write_pv(dst_pe->pv);
	lvm_write_pe_map(dst_pe->pv);
	if (src_pe) {
		lvm_write_pv(src_pe->pv);
		lvm_write_pe_map(src_pe->pv);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * move_extent_cleanup_copy_job
 *
 * Free the title memory. If we did an online move, call the engine to
 * cleanup the mirror device.
 **/
static void move_extent_cleanup_copy_job(copy_job_t * copy_job)
{
	LOG_ENTRY();

	if (copy_job->mirror && EngFncs->can_online_copy()) {
		EngFncs->copy_cleanup(copy_job);
	}
	if (copy_job->title) {
		EngFncs->engine_free(copy_job->title);
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_commit_move_extent
 *
 * Move the specified LE from its current PE to its new PE. Copy the data
 * from the old PE to the new PE, update the mappings for both PEs, write
 * out the new metadata, and reactivate the volume with the new mapping.
 **/
static int lvm_commit_move_extent(lvm_logical_extent_t * le)
{
	copy_job_t copy_job;
	int rc = 0;

	LOG_ENTRY();

	memset(&copy_job, 0, sizeof(copy_job));

	/* Copy the data from the source to the destination. */
	if (le->pe) {
		rc = move_extent_init_copy_job(le, &copy_job);
		if (rc) {
			goto out;
		}

		rc = move_extent_copy_extent(le, &copy_job);
		if (rc) {
			goto out;
		}
	}

	/* Update the metadata */
	move_extent_update_metadata(le);

out:
	/* Re-activate the volume with the new mapping. */
	my_plugin_record->functions.plugin->activate(le->volume->region);

	move_extent_cleanup_copy_job(&copy_job);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_volume_is_busy
 *
 * Check if the specified volume is open/in-use/mounted. If so, ask the user
 * if they'd like to unmount and retry.
 *
 * Return: TRUE: skip processing this volume.
 *         FALSE: proceed with this volume.
 **/
static int lvm_volume_is_busy(lvm_logical_volume_t * volume,
			      int prompt_user)
{
	char * choices[] = {"Skip", "Retry", NULL};
	logical_volume_t * evms_vol;
	int offline, answer;
	int rc = FALSE;

	LOG_ENTRY();

	/* If online move is enabled, we can skip
	 * checking if the volume is busy.
	 */
	if (EngFncs->can_online_copy()) {
		goto out;
	}

	/* This loop will not exit unless the volume is unmounted, or the
	 * user decides to skip this volume.
	 */
	while (!(offline = EngFncs->is_offline(volume->region, &evms_vol))) {
		answer = 0;
		if (prompt_user) {
			QUESTION(&answer, choices,
				"Region \"%s\" has extents scheduled to be "
				"moved. However, this region is part of volume "
				"\"%s\", which is mounted at %s. Please "
				"unmount the volume and choose \"%s\" to "
				"continue with the move, or choose \"%s\" to "
				"skip the move at this time (the move will be "
				"attempted again the next time changes are "
				"saved).",
				volume->region->name, evms_vol->name,
				evms_vol->mount_point, choices[1], choices[0]);
		}

		if (answer == 0) {
			break;
		}
	}

	rc = (!offline && answer == 0);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_commit_move_extents
 *
 * Move all extents that are waiting to be moved in this container.
 **/
int lvm_commit_move_extents(lvm_volume_group_t * group)
{
	lvm_logical_volume_t * volume;
	int i, j, rc = 0;

	LOG_ENTRY();

	for (i = 1; i < MAX_LV; i++) {
		volume = group->volume_list[i];
		if (volume && (volume->flags & LVM_LV_FLAG_MOVE_PENDING)) {

			rc = lvm_volume_is_busy(volume, TRUE);
			if (rc) {
				continue;
			}

			for (j = 0; j < volume->lv->lv_allocated_le; j++) {
				if (volume->le_map[j].new_pe) {
					rc |= lvm_commit_move_extent(&volume->le_map[j]);
				}
			}

			if (!rc) {
				volume->flags &= ~LVM_LV_FLAG_MOVE_PENDING;
			}
		}
	}

	/* Need to rebuild the freespace, since the number and
	 * location of unused extents has probably changed.
	 */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace for container %s\n",
			    group->container->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/********** Move-Extent Task Functions **********/


/**
 * lvm_move_extent_init_task
 *
 * Initialize the option descriptor for a MOVE_EXTENT task.
 * Option 0: Logical Extent
 * Option 1: Physical Volume
 * Option 2: Physical Extent
 **/
int lvm_move_extent_init_task(task_context_t * context)
{
	option_desc_array_t * od = context->option_descriptors;
	lvm_logical_volume_t * volume = context->object->private_data;
	lvm_volume_group_t * group = volume->group;
	lvm_physical_volume_t * pv_entry;
	u_int32_t i, index, count = 0;

	LOG_ENTRY();

	/* Option 0: logical extent number.
	 * Generate a list of all the extents in this
	 * volume that are not already being moved.
	 */
	index = LVM_OPTION_MOVE_EXTENT_LE_INDEX;
	SET_STRING(od->option[index].name, LVM_OPTION_MOVE_EXTENT_LE_STR);
	SET_STRING(od->option[index].title, "Logical Extent");
	SET_STRING(od->option[index].tip, "Logical extent to move");
	od->option[index].type = EVMS_Type_Unsigned_Int32;
	od->option[index].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[index].constraint_type = EVMS_Collection_List;
	od->option[index].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) +
								  sizeof(value_t) *
								  volume->lv->lv_allocated_le);
	for (i = 0; i < volume->lv->lv_allocated_le; i++) {
		if (!lvm_le_is_scheduled_for_move(&volume->le_map[i])) {
			od->option[index].constraint.list->value[count].ui32 = i;
			count++;
		}
	}
	od->option[index].constraint.list->count = count;

	/* Option 1: new physical volume.
	 * Generate a list of all PVs that have unused extents.
	 */
	count = 0;
	index = LVM_OPTION_MOVE_EXTENT_PV_INDEX;
	SET_STRING(od->option[index].name, LVM_OPTION_MOVE_EXTENT_PV_STR);
	SET_STRING(od->option[index].title, "Physical Volume");
	SET_STRING(od->option[index].tip, "Physical volume to move this logical extent to.");
	od->option[index].type = EVMS_Type_String;
	od->option[index].max_len = EVMS_NAME_SIZE;
	od->option[index].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[index].constraint_type = EVMS_Collection_List;
	od->option[index].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) +
								  sizeof(value_t) * MAX_PV);
	for (i = 0; i < MAX_PV; i++) {
		pv_entry = group->pv_list[i];
		if (pv_entry && lvm_pv_has_available_extents(pv_entry)) {
			SET_STRING(od->option[index].constraint.list->value[count].s,
				   pv_entry->segment->name);
			count++;
		}
	}
	od->option[index].constraint.list->count = count;
	od->option[index].value.s = EngFncs->engine_alloc(EVMS_NAME_SIZE+1);

	/* Option 2: new physical extent.
	 * Set up a list option, but can't fill it in yet. Have to wait
	 * until the user chooses which PV to move this extent to.
	 */
	index = LVM_OPTION_MOVE_EXTENT_PE_INDEX;
	SET_STRING(od->option[index].name, LVM_OPTION_MOVE_EXTENT_PE_STR);
	SET_STRING(od->option[index].title, "Physical Extent");
	SET_STRING(od->option[index].tip, "Physical extent to move this logical extent to.");
	od->option[index].type = EVMS_Type_Unsigned_Int32;
	od->option[index].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE |
				  EVMS_OPTION_FLAGS_INACTIVE;
	od->option[index].constraint_type = EVMS_Collection_List;

	od->count = index + 1;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_move_extent_set_option
 *
 * Verify and set an option during a MOVE_EXTENT task.
 **/
int lvm_move_extent_set_option(task_context_t * context,
			       u_int32_t index,
			       value_t * value,
			       task_effect_t * effect)
{
	option_desc_array_t * od = context->option_descriptors;
	lvm_logical_volume_t * volume = context->object->private_data;
	lvm_physical_volume_t * pv_entry;
	u_int32_t i, count = 0;
	int rc = EINVAL;

	LOG_ENTRY();
	LOG_EXTRA("Setting option %d\n", index);

	switch (index) {

	case LVM_OPTION_MOVE_EXTENT_LE_INDEX:
		/* Make sure the LE is in range. */
		if (lvm_le_is_valid(volume, value->ui32) &&
		    !lvm_le_is_scheduled_for_move(&volume->le_map[value->ui32])) {
			od->option[index].value.ui32 = value->ui32;
			rc = 0;
		}
		break;

	case LVM_OPTION_MOVE_EXTENT_PV_INDEX:
		/* Make sure a valid PV was selected, and
		 * that the PV has unused PEs available.
		 */
		pv_entry = lvm_get_pv_for_name(value->s, volume->group);
		if (pv_entry && lvm_pv_has_available_extents(pv_entry)) {
			strncpy(od->option[index].value.s, value->s, EVMS_NAME_SIZE);

			/* Set up the list of possible PEs on this PV. */
			index = LVM_OPTION_MOVE_EXTENT_PE_INDEX;
			EngFncs->engine_free(od->option[index].constraint.list);
			od->option[index].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) +
										  sizeof(value_t) *
										  pv_entry->pv->pe_total);
			for (i = 0; i < pv_entry->pv->pe_total; i++) {
				if (lvm_pe_is_available(&pv_entry->pe_map[i])) {
					od->option[index].constraint.list->value[count].ui32 = i;
					count++;
				}
			}
			od->option[index].constraint.list->count = count;
			od->option[index].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
			od->option[index].flags |= EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

			*effect |= EVMS_Effect_Reload_Options;
			rc = 0;
		}
		break;

	case LVM_OPTION_MOVE_EXTENT_PE_INDEX:
		/* Get the selected PV from the option descriptor,
		 * and make sure the selected PE is available.
		 */
		pv_entry = lvm_get_pv_for_name(od->option[LVM_OPTION_MOVE_EXTENT_PV_INDEX].value.s,
					       volume->group);
		if (pv_entry &&
		    lvm_pe_is_valid(pv_entry, value->ui32) &&
		    lvm_pe_is_available(&pv_entry->pe_map[value->ui32])) {
			od->option[index].value.ui32 = value->ui32;
			rc = 0;
		}
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_extent_parse_options
 **/
static void lvm_move_extent_parse_options(option_array_t * options,
					  lvm_logical_volume_t * volume,
					  lvm_physical_volume_t ** pv_entry,
					  u_int32_t * le,
					  u_int32_t * pe)
{
	int i;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name, LVM_OPTION_MOVE_EXTENT_LE_STR)) {
				options->option[i].number = LVM_OPTION_MOVE_EXTENT_LE_INDEX;
			} else if (!strcmp(options->option[i].name, LVM_OPTION_MOVE_EXTENT_PV_STR)) {
				options->option[i].number = LVM_OPTION_MOVE_EXTENT_PV_INDEX;
			} else if (!strcmp(options->option[i].name, LVM_OPTION_MOVE_EXTENT_PE_STR)) {
				options->option[i].number = LVM_OPTION_MOVE_EXTENT_PE_INDEX;
			} else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {
		case LVM_OPTION_MOVE_EXTENT_LE_INDEX:
			*le = options->option[i].value.ui32;
			break;
		case LVM_OPTION_MOVE_EXTENT_PV_INDEX:
			*pv_entry = lvm_get_pv_for_name(options->option[i].value.s,
							volume->group);
			break;
		case LVM_OPTION_MOVE_EXTENT_PE_INDEX:
			*pe = options->option[i].value.ui32;
			break;
		default:
			break;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_move_extent_verify_options
 *
 * Verify that the specified options are all valid for moving an extent.
 **/
static int lvm_move_extent_verify_options(lvm_logical_volume_t * volume,
					  lvm_physical_volume_t * pv_entry,
					  u_int32_t le,
					  u_int32_t pe)
{
	int rc = EINVAL;

	LOG_ENTRY();

	if (!lvm_le_is_valid(volume, le)) {
		LOG_ERROR("LE %d is out of range for region %s.\n",
			  le, volume->region->name);
	} else if (lvm_le_is_scheduled_for_move(&volume->le_map[le])) {
		LOG_ERROR("LE %d on region %s is already scheduled for move.\n",
			  le, volume->region->name);
	} else if (!pv_entry) {
		LOG_ERROR("Invalid PV specified.\n");
	} else if (!lvm_pe_is_valid(pv_entry, pe)) {
		LOG_ERROR("PE %d is out of range for PV %s.\n",
			  pe, pv_entry->segment->name);
	} else if (!lvm_pe_is_available(&pv_entry->pe_map[pe])) {
		LOG_ERROR("PE %d on PV %s is not available for move.\n",
			  pe, pv_entry->segment->name);
	} else {
		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_extent
 **/
int lvm_move_extent(lvm_logical_volume_t * volume,
		    option_array_t * options)
{
	lvm_physical_volume_t * pv_entry = NULL;
	u_int32_t le_num = 0, pe_num = 0;
	int rc;

	LOG_ENTRY();

	/* Parse and verify the options for move. */
	lvm_move_extent_parse_options(options, volume,
				      &pv_entry, &le_num, &pe_num);
	rc = lvm_move_extent_verify_options(volume, pv_entry,
					    le_num, pe_num);
	if (rc) {
		LOG_ERROR("Error verifying move-extent options.\n");
		goto out;
	}

	/* Prepare the extents to be moved. */
	lvm_move_extent_setup(&volume->le_map[le_num], &pv_entry->pe_map[pe_num]);

	/* Rebuild the freespace. */
	rc = lvm_update_freespace_volume(volume->group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n",
			  volume->group->container->name);
	}

	volume->group->container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/********** Move-PV Task Functions **********/

/**
 * lvm_can_move_pv_stripes
 *
 * Helper-function for lvm_can_move_pv for determining whether striped-regions
 * on the specified PV can be moved.
 **/
static int lvm_can_move_pv_stripes(lvm_physical_volume_t * source_pv,
				   u_int32_t * free_extents,
				   int maintain_stripes)
{
	storage_object_t * region, * segment;
	lvm_logical_volume_t * volume;
	lvm_physical_volume_t * pv_entry;
	list_element_t itr, itr2;
	u_int32_t free_extents_striped[MAX_PV+1] = {0};
	u_int32_t extents_per_pv;
	int i, j, rc;

	LOG_ENTRY();

	LIST_FOR_EACH(source_pv->segment->parent_objects, itr, region) {
		if (region->data_type != DATA_TYPE) {
			/* Don't care about freespace regions. */
			continue;
		}

		volume = region->private_data;
		if (volume->lv->lv_stripes <= 1) {
			/* Only concerned about striped regions. */
			continue;
		}

		/* Make sure this region hasn't already been moved. */
		if (volume->flags & LVM_LV_FLAG_MOVE_PENDING) {
			LOG_WARNING("Region %s has extents waiting to be "
				    "moved.\n", region->name);
			LOG_WARNING("Please save pending moves before "
				    "performing \"Move-PV\"\n");
			rc = EINVAL;
			goto out;
		}

		/* Make a local copy of the free-extents array, since
		 * we're going to mangle it.
		 */
		memcpy(free_extents_striped, free_extents,
		       sizeof(free_extents_striped));

		if (maintain_stripes != MAINTAIN_STRIPES_OFF) {
			/* If we are trying to maintain striping, zero the
			 * entries for the PVs that this region already maps
			 * to, so we don't consider them.
			 */
			LIST_FOR_EACH(region->child_objects, itr2, segment) {
				pv_entry = lvm_get_pv_for_segment(segment);
				free_extents_striped[pv_entry->number] = 0;
			}
		}

		if (maintain_stripes == MAINTAIN_STRIPES_STRICT) {
			/* Count the number of extents on this
			 * region that map to this PV.
			 */
			extents_per_pv = 0;
			for (i = 0; i < volume->lv->lv_allocated_le; i++) {
				if (volume->le_map[i].pe &&
				    volume->le_map[i].pe->pv == source_pv) {
					extents_per_pv++;
				}
			}

			/* Search for a target PV with this many free PEs. */
			for (i = 0; i <= MAX_PV; i++) {
				if (free_extents_striped[i] >= extents_per_pv) {
					free_extents_striped[i] -= extents_per_pv;
					free_extents[i] -= extents_per_pv;
					break;
				}
			}
			if (i > MAX_PV) {
				/* Couldn't find a free PE for this LE. */
				LOG_WARNING("Not enough free PEs to move "
					    "region %s.\n", region->name);
				rc = EINVAL;
				goto out;
			}
		} else {
			for (i = 0, j = 0; i < volume->lv->lv_allocated_le; i++) {
				if (volume->le_map[i].pe &&
				    volume->le_map[i].pe->pv == source_pv) {
					/* This LE needs to be moved.
					 * Find a free PE.
					 */
					for ( ; j <= MAX_PV; j++) {
						if (free_extents_striped[j]) {
							free_extents_striped[j]--;
							free_extents[j]--;
							break;
						}
					}
					if (j > MAX_PV) {
						/* No free PE for this LE. */
						LOG_WARNING("Not enough free PEs to "
							    "move region %s.\n",
							    region->name);
						rc = EINVAL;
						goto out;
					}
				}
			}
		}
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_move_pv_linears
 *
 * Helper-function for lvm_can_move_pv for determining whether linear-regions
 * on the specified PV can be moved.
 **/
static int lvm_can_move_pv_linears(lvm_physical_volume_t * source_pv,
				   u_int32_t * free_extents)
{
	storage_object_t * region;
	lvm_logical_volume_t * volume;
	list_element_t itr;
	int rc, i, j = 0;

	LOG_ENTRY();

	LIST_FOR_EACH(source_pv->segment->parent_objects, itr, region) {
		if (region->data_type != DATA_TYPE) {
			/* Don't care about freespace regions. */
			continue;
		}

		volume = region->private_data;
		if (volume->lv->lv_stripes > 1) {
			/* Only concerned about linear regions. */
			continue;
		}

		/* Make sure this region hasn't already been moved. */
		if (volume->flags & LVM_LV_FLAG_MOVE_PENDING) {
			LOG_WARNING("Region %s has extents waiting to be "
				    "moved.\n", region->name);
			LOG_WARNING("Please save pending moves before "
				    "performing \"Move-PV\"\n");
			rc = EINVAL;
			goto out;
		}

		for (i = 0; i < volume->lv->lv_allocated_le; i++) {
			if (volume->le_map[i].pe &&
			    volume->le_map[i].pe->pv == source_pv) {
				/* This LE needs to be moved. Find a free PE. */
				for ( ; j <= MAX_PV; j++) {
					if (free_extents[j]) {
						free_extents[j]--;
						break;
					}
				}
				if (j > MAX_PV) {
					/* No free PEs for this LE. */
					LOG_WARNING("Not enough free PEs to "
						    "move region %s.\n",
						    region->name);
					rc = EINVAL;
					goto out;
				}
			}
		}
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_move_pv
 * @source_pv:		The PV from which all PEs should be moved.
 * @target_pvs:		List of PVs to use in considering how to move source_pv.
 *			If the first entry is NULL, use all PVs in the group.
 *			If the first entry is non-NULL, then this argument must
 *			point to an array of MAX_PV+1 entries (not all non-NULL).
 * @maintain_stripes:	If 0, don't bother maintaining striped regions. This
 *			means a striped region may wind up with multiple stripes
 *			on the same PV, but allows the most flexibility in
 *			moving the desired PV.
 *			If 1, maintain stripes "loosely". This will ensure that
 *			moved extents don't wind up on PVs that the striped
 *			region already uses. However, the moved extents may not
 *			all end up on the same PV.
 *			If 2, maintain stripes "strictly". This will ensure that
 *			all moved extents end up on the same PV, thus
 *			maintaining the same stripe configuration. This option
 *			is the most restrictive, and may not always allow the PV
 *			to be moved.
 *
 * Can all of the extents on this PV be moved to other PVs in the container?
 * This is only a check to see if the move is possible. No extents are actually
 * setup for a real move.
 **/
int lvm_can_move_pv(lvm_physical_volume_t * source_pv,
		    lvm_physical_volume_t * target_pvs[],
		    int maintain_stripes)
{
	lvm_volume_group_t * group = source_pv->group;
	u_int32_t free_extents[MAX_PV+1] = {0};
	int i, rc;

	LOG_ENTRY();

	/* There is nothing to move if the source PV has no allocated PEs. */
	if (source_pv->pv->pe_allocated == 0) {
		LOG_DETAILS("No extents allocated on PV %s.\n",
			    source_pv->segment->name);
		rc = EINVAL;
		goto out;
	}

	/* If no target PVs were specified, use all PVs in the group. */
	if (!target_pvs[0]) {
		target_pvs = group->pv_list;
	}

	/* Set up an array with the number of free-extents per target-PV. */
	for (i = 0; i <= MAX_PV; i++) {
		if (target_pvs[i] && target_pvs[i] != source_pv) {
			free_extents[target_pvs[i]->number] = lvm_pv_num_available_extents(target_pvs[i]);
		}
	}

	/* Check striped regions first. */
	rc = lvm_can_move_pv_stripes(source_pv, free_extents, maintain_stripes);
	if (rc) {
		goto out;
	}

	/* Now check linear regions. */
	rc = lvm_can_move_pv_linears(source_pv, free_extents);
	if (rc) {
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_parse_maintain_stripes
 **/
static int lvm_parse_maintain_stripes(char * string)
{
	int rc = MAINTAIN_STRIPES_OFF;

	LOG_ENTRY();

	if (strcasecmp(string, "strict") == 0) {
		rc = MAINTAIN_STRIPES_STRICT;
	} else if (strcasecmp(string, "loose") == 0) {
		rc = MAINTAIN_STRIPES_LOOSE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_pv_init_task
 *
 * Initialize the option descriptor for a MOVE_PV task.
 * Option 0: List of Target PVs
 * Option 1: Flag to maintain striped regions
 **/
int lvm_move_pv_init_task(task_context_t * context)
{
	option_desc_array_t * od = context->option_descriptors;
	lvm_volume_group_t * group = context->container->private_data;
	lvm_physical_volume_t * dummy_pv = NULL;
	int i, rc = 0, index;

	LOG_ENTRY();

	/* Populate the acceptable-objects list with all PVs that could
	 * be moved.
	 */
	for (i = 0; i <= MAX_PV; i++) {
		if (group->pv_list[i]) {
			rc = lvm_can_move_pv(group->pv_list[i], &dummy_pv,
					     MAINTAIN_STRIPES_OFF);
			if (!rc) {
				EngFncs->insert_thing(context->acceptable_objects,
						      group->pv_list[i]->segment,
						      INSERT_AFTER, NULL);
			}
		}
	}

	if (EngFncs->list_empty(context->acceptable_objects)) {
		/* No PVs can be moved. */
		rc = EINVAL;
		goto out;
	}

	rc = 0;
	context->min_selected_objects = 1;
	context->max_selected_objects = 1;

	/* Setup the option descriptor. */

	/* Option 0: List of Target PVs
	 * We don't know which PVs can be targets until we have selected a
	 * source PV. Just allocate the empty list right now.
	 */
	index = LVM_OPTION_MOVE_PV_TARGET_LIST_IDX;
	SET_STRING(od->option[index].name, LVM_OPTION_MOVE_PV_TARGET_LIST_STR);
	SET_STRING(od->option[index].title, "Target PVs");
	SET_STRING(od->option[index].tip, "List of PVs that should be used as "
					  "targets for moving the desired source "
					  "PV. Defaults to all remaining PVs in "
					  "the container.");
	od->option[index].type = EVMS_Type_String;
	od->option[index].min_len = 1;
	od->option[index].max_len = EVMS_NAME_SIZE;
	od->option[index].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
				  EVMS_OPTION_FLAGS_AUTOMATIC |
				  EVMS_OPTION_FLAGS_VALUE_IS_LIST;
	od->option[index].constraint_type = EVMS_Collection_List;
	od->option[index].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) +
								  sizeof(value_t) * MAX_PV);
	od->option[index].value.list = EngFncs->engine_alloc(sizeof(value_list_t) +
							     sizeof(value_t) * MAX_PV);
	/* Don't need to allocate space for the strings yet. That will
	 * happen when one of the acceptable-objects is selected.
	 */


	/* Option 1: Maintain Striping
	 * No, Loose, or Strict
	 */
	index = LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_IDX;
	SET_STRING(od->option[index].name, LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_STR);
	SET_STRING(od->option[index].title, "Maintain Stripes");
	SET_STRING(od->option[index].tip, "No: Don't bother to maintain true striping.\n"
					  "Loose: Don't move extents to PVs that are already "
					  "used by a striped region.\n"
					  "Strict: Maintain true striping.");
	od->option[index].type = EVMS_Type_String;
	od->option[index].min_len = 1;
	od->option[index].max_len = EVMS_NAME_SIZE;
	od->option[index].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
				  EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[index].constraint_type = EVMS_Collection_List;
	od->option[index].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) +
								  sizeof(value_t) * 2);
	SET_STRING(od->option[index].constraint.list->value[0].s, "no");
	SET_STRING(od->option[index].constraint.list->value[1].s, "loose");
	SET_STRING(od->option[index].constraint.list->value[2].s, "strict");
	od->option[index].constraint.list->count = 3;
	SET_STRING(od->option[index].value.s, "no");

	index++;
	od->count = index;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_pv_set_objects
 *
 * Verify that the selected PV is valid for Move-PV. If it is, finish
 * initializing the option-descriptor.
 **/
int lvm_move_pv_set_objects(task_context_t * context,
			    list_anchor_t declined_objects,
			    task_effect_t * effect)
{
	option_desc_array_t * od = context->option_descriptors;
	lvm_volume_group_t * group = context->container->private_data;
	lvm_physical_volume_t * source_pv, * dummy_pv = NULL;
	int rc, i, j = 0;

	LOG_ENTRY();

	/* Get the PV from the selected-objects list. */
	source_pv = lvm_get_selected_segment(context->selected_objects);
	if (!source_pv) {
		rc = EINVAL;
		goto out;
	}

	/* Make sure this PV can be moved. */
	rc = lvm_can_move_pv(source_pv, &dummy_pv, MAINTAIN_STRIPES_OFF);
	if (rc) {
		goto out;
	}

	/* Fill in the initial Target-PVs list. */
	od->option[LVM_OPTION_MOVE_PV_TARGET_LIST_IDX].constraint.list->count = 0;
	od->option[LVM_OPTION_MOVE_PV_TARGET_LIST_IDX].value.list->count = 0;
	for (i = 0; i <= MAX_PV; i++) {
		if (group->pv_list[i] &&
		    group->pv_list[i] != source_pv) {
			SET_STRING(od->option[LVM_OPTION_MOVE_PV_TARGET_LIST_IDX].constraint.list->value[j].s,
				   group->pv_list[i]->segment->name);
			SET_STRING(od->option[LVM_OPTION_MOVE_PV_TARGET_LIST_IDX].value.list->value[j].s,
				   group->pv_list[i]->segment->name);
			od->option[LVM_OPTION_MOVE_PV_TARGET_LIST_IDX].constraint.list->count++;
			od->option[LVM_OPTION_MOVE_PV_TARGET_LIST_IDX].value.list->count++;
			j++;
		}
	}

	/* Reset the Maintain-Stripes option to the default. */
	SET_STRING(od->option[LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_IDX].value.s, "no");

	*effect |= EVMS_Effect_Reload_Options;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_pv_set_option
 *
 * Verify that the specified option, along with the currently-selected
 * object and the remaining options present in the option-descriptor, are
 * valid for performing a PV-Move.
 **/
int lvm_move_pv_set_option(task_context_t * context,
			   u_int32_t index,
			   value_t * value,
			   task_effect_t * effect)
{
	option_desc_array_t * od = context->option_descriptors;
	lvm_volume_group_t * group = context->container->private_data;
	lvm_physical_volume_t * source_pv;
	lvm_physical_volume_t * target_pvs[MAX_PV+1] = {NULL};
	value_list_t * pv_list;
	char * maintain_stripes_string;
	int maintain_stripes;
	int i, rc;

	LOG_ENTRY();

	LOG_EXTRA("Setting option %d\n", index);

	/* Get the PV from the selected-objects list. */
	source_pv = lvm_get_selected_segment(context->selected_objects);
	if (!source_pv) {
		rc = EINVAL;
		goto out;
	}

	switch (index) {

	case LVM_OPTION_MOVE_PV_TARGET_LIST_IDX:
		pv_list = value->list;
		maintain_stripes_string = od->option[LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_IDX].value.s;
		break;

	case LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_IDX:
		pv_list = od->option[LVM_OPTION_MOVE_PV_TARGET_LIST_IDX].value.list;
		maintain_stripes_string = value->s;
		break;

	default:
		rc = EINVAL;
		goto out;
	}

	rc = lvm_parse_pv_list_option(pv_list, target_pvs, group);
	if (rc) {
		goto out;
	}

	maintain_stripes = lvm_parse_maintain_stripes(maintain_stripes_string);

	rc = lvm_can_move_pv(source_pv, target_pvs, maintain_stripes);
	if (rc) {
		*effect |= EVMS_Effect_Reload_Options;
		goto out;
	}

	switch (index) {

	case LVM_OPTION_MOVE_PV_TARGET_LIST_IDX:
		for (i = 0; i < pv_list->count; i++) {
			SET_STRING(od->option[index].value.list->value[i].s, pv_list->value[i].s);
		}
		for (; i <= MAX_PV; i++) {
			EngFncs->engine_free(od->option[index].value.list->value[i].s);
			od->option[index].value.list->value[i].s = NULL;
		}
		od->option[index].value.list->count = pv_list->count;
		break;

	case LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_IDX:
		SET_STRING(od->option[index].value.s, maintain_stripes_string);
		break;

	default:
		break;
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_pv_parse_options
 **/
static int lvm_move_pv_parse_options(option_array_t * options,
				     lvm_physical_volume_t * pv_entries[],
				     lvm_volume_group_t * group,
				     int * maintain_stripes)
{
	int i, rc = 0;

	LOG_ENTRY();

	/* Default value. */
	*maintain_stripes = MAINTAIN_STRIPES_OFF;

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name, LVM_OPTION_MOVE_PV_TARGET_LIST_STR)) {
				options->option[i].number = LVM_OPTION_MOVE_PV_TARGET_LIST_IDX;
			}
			else if (!strcmp(options->option[i].name, LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_STR)) {
				options->option[i].number = LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_IDX;
			}
			else {
				continue;
			}
		}

		LOG_EXTRA("Parsing option %d\n", options->option[i].number);

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {
		case LVM_OPTION_MOVE_PV_TARGET_LIST_IDX:
			rc = lvm_parse_pv_list_option(options->option[i].value.list,
						      pv_entries, group);
			if (rc) {
				goto out;
			}
			break;
		case LVM_OPTION_MOVE_PV_MAINTAIN_STRIPES_IDX:
			*maintain_stripes = lvm_parse_maintain_stripes(options->option[i].value.s);
			break;
		default:
			break;
		}
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_pv_display_warning
 **/
static int lvm_move_pv_display_warning(lvm_physical_volume_t * source_pv)
{
	storage_object_t * region;
	lvm_logical_volume_t * volume;
	list_element_t itr;
	char * volume_string;
	char * choices[] = {"Continue",
			    "Don't Continue",
			    NULL};
	int rc, count, answer = 0;

	LOG_ENTRY();

	count = EngFncs->list_count(source_pv->segment->parent_objects);

	volume_string = EngFncs->engine_alloc(count*(EVMS_NAME_SIZE+1) + 1);
	if (!volume_string) {
		rc = ENOMEM;
		goto out;
	}

	count = 0;

	LIST_FOR_EACH(source_pv->segment->parent_objects, itr, region) {
		if (region->data_type == DATA_TYPE) {
			volume = region->private_data;
			rc = lvm_volume_is_busy(volume, FALSE);
			if (rc) {
				count++;
				if (region->volume) {
					strncat(volume_string, region->volume->name, EVMS_NAME_SIZE);
				} else {
					strncat(volume_string, region->name, EVMS_NAME_SIZE);
				}
				strncat(volume_string, "\n", 1);
			}
		}
	}

	if (count) {
		QUESTION(&answer, choices,
			"You have chosen to move PV %s. Currently, all move "
			"operations must be performed off-line. The following "
			"volumes and/or regions must be unmounted before "
			"saving these changes:\n\n%s",
			source_pv->segment->name, volume_string);
	}

	rc = answer ? E_CANCELED : 0;

	EngFncs->engine_free(volume_string);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_pv_stripes
 *
 * Move all extents (belonging to striped regions) from the source PV to
 * PEs on PVs in the target list.
 **/
static int lvm_move_pv_stripes(lvm_physical_volume_t * source_pv,
			       lvm_physical_volume_t * target_pvs[],
			       int maintain_stripes)
{
	storage_object_t * region, * segment;
	lvm_logical_volume_t * volume;
	lvm_physical_volume_t * target_pvs_striped[MAX_PV+1] = {NULL};
	lvm_physical_volume_t * pv_entry;
	list_element_t itr, itr2;
	int extents_per_pv;
	int i, j = 0, k = 0;
	int rc;

	LOG_ENTRY();

	/* If no target PVs were specified, use the group's whole list. */
	if (!target_pvs[0]) {
		target_pvs = source_pv->group->pv_list;
	}

	/* Check every region that maps to the source PV. */
	LIST_FOR_EACH(source_pv->segment->parent_objects, itr, region) {
		if (region->data_type != DATA_TYPE) {
			/* Don't care about freespace regions. */
			continue;
		}

		volume = region->private_data;
		if (volume->lv->lv_stripes <= 1) {
			/* Only move LEs on striped regions. */
			continue;
		}

		/* Need a local copy of the target PV array, since we're
		 * going to mangle it.
		 */
		memcpy(target_pvs_striped, target_pvs,
		       sizeof(target_pvs_striped));

		if (maintain_stripes != MAINTAIN_STRIPES_OFF) {
			/* It we are maintaining any form of striping, we need
			 * to prune the list of target PVs.
			 */
			LIST_FOR_EACH(region->child_objects, itr2, segment) {
				pv_entry = lvm_get_pv_for_segment(segment);
				for (i = 0; i <= MAX_PV; i++) {
					if (target_pvs_striped[i] == pv_entry) {
						target_pvs_striped[i] = NULL;
						break;
					}
				}
			}
		}

		if (maintain_stripes == MAINTAIN_STRIPES_STRICT) {
			/* Count the number of extents on this region
			 * that map to this PV.
			 */
			extents_per_pv = 0;
			for (i = 0; i < volume->lv->lv_allocated_le; i++) {
				if (volume->le_map[i].pe &&
				    volume->le_map[i].pe->pv == source_pv) {
					extents_per_pv++;
				}
			}

			/* Search for a target PV with this many free
			 * extents.
			 */
			for (i = 0; i <= MAX_PV; i++) {
				if (!target_pvs_striped[i] ||
				    target_pvs_striped[i] == source_pv ||
				    lvm_pv_num_available_extents(target_pvs_striped[i]) < extents_per_pv) {
					continue;
				}

				/* This PV has enough free extents. Search its
				 * PE map and setup the moves.
				 */
				for (j = 0, k = 0; j < volume->lv->lv_allocated_le; j++) {
					if (!volume->le_map[j].pe ||
					    volume->le_map[j].pe->pv != source_pv) {
						continue;
					}

					/* Find an available PE on this target PV. */
					for ( ; k < target_pvs_striped[i]->pv->pe_total; k++) {
						if (lvm_pe_is_available(&(target_pvs_striped[i]->pe_map[k]))) {
							break;
						}
					}

					if (k < target_pvs_striped[i]->pv->pe_total) {
						lvm_move_extent_setup(&(volume->le_map[j]),
								      &(target_pvs_striped[i]->pe_map[k]));
					} else {
						/* Wasn't able to find a free PE! Based
						 * on the above checks, this should
						 * really never happen.
						 */
						LOG_SERIOUS("BUG! Error finding free PEs "
							    "for region %s.\n", region->name);
						rc = EINVAL;
						goto out;
					}
				}

				break;
			}

			if (i > MAX_PV) {
				/* Wasn't able to find a PV with free PEs! If we
				 * checked everything ahead of time, this should
				 * not happen.
				 */
				LOG_SERIOUS("BUG! Error finding PV with free PEs "
					    "for region %s.\n", region->name);
				rc = EINVAL;
				goto out;
			}

		} else {
			/* MAINTAIN_STRIPES_LOOSE or MAINTAIN_STRIPES_OFF */

			/* Check every LE on this region. */
			for (i = 0; i < volume->lv->lv_allocated_le; i++) {
				if (!volume->le_map[i].pe ||
				    volume->le_map[i].pe->pv != source_pv) {
					/* Only move LEs that map to the source PV. */
					continue;
				}

				/* This LE needs to be moved. */
				for ( ; j <= MAX_PV; j++, k = 0) {
					if (!target_pvs_striped[j] ||
					    target_pvs_striped[j] == source_pv ||
					    !lvm_pv_has_available_extents(target_pvs_striped[j])) {
						/* Can't use this PV as the target. */
						continue;
					}

					/* Find an available PE on this target PV. */
					for ( ; k < target_pvs_striped[j]->pv->pe_total; k++) {
						if (lvm_pe_is_available(&(target_pvs_striped[j]->pe_map[k]))) {
							break;
						}
					}

					if (k < target_pvs_striped[j]->pv->pe_total) {
						lvm_move_extent_setup(&(volume->le_map[i]),
								      &(target_pvs_striped[j]->pe_map[k]));
						break;
					} else {
						/* Wasn't able to find a free PE! Based
						 * on the above checks, this should
						 * really never happen.
						 */
						LOG_SERIOUS("BUG! Error finding free PEs "
							    "for region %s.\n", region->name);
						rc = EINVAL;
						goto out;
					}
				}

				if (j > MAX_PV) {
					/* Wasn't able to find a PV with free PEs! If we
					 * checked everything ahead of time, this should
					 * not happen.
					 */
					LOG_SERIOUS("BUG! Error finding PV with free PEs "
						    "for region %s.\n", region->name);
					rc = EINVAL;
					goto out;
				}
			}
		}
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_pv_linears
 *
 * Move all extents (belonging to linear regions) from the source PV to
 * PEs on PVs in the target list.
 **/
static int lvm_move_pv_linears(lvm_physical_volume_t * source_pv,
			       lvm_physical_volume_t * target_pvs[])
{
	storage_object_t * region;
	lvm_logical_volume_t * volume;
	list_element_t itr;
	int i, j = 0, k = 0, rc;

	LOG_ENTRY();

	/* If no target PVs were specified, use the group's whole list. */
	if (!target_pvs[0]) {
		target_pvs = source_pv->group->pv_list;
	}

	/* Check every region that maps to the source PV. */
	LIST_FOR_EACH(source_pv->segment->parent_objects, itr, region) {
		if (region->data_type != DATA_TYPE) {
			/* Don't care about freespace regions. */
			continue;
		}

		volume = region->private_data;
		if (volume->lv->lv_stripes > 1) {
			/* Only move LEs on linear regions. */
			continue;
		}

		/* Check every LE on this region. */
		for (i = 0; i < volume->lv->lv_allocated_le; i++) {
			if (!volume->le_map[i].pe ||
			    volume->le_map[i].pe->pv != source_pv) {
				/* Only move LEs that map to the source PV. */
				continue;
			}

			/* This LE needs to be moved. */
			for ( ; j <= MAX_PV; j++, k = 0) {
				if (!target_pvs[j] ||
				    target_pvs[j] == source_pv ||
				    !lvm_pv_has_available_extents(target_pvs[j])) {
					/* Can't use this PV as the target. */
					continue;
				}

				/* Find an available PE on this target PV. */
				for ( ; k < target_pvs[j]->pv->pe_total; k++) {
					if (lvm_pe_is_available(&(target_pvs[j]->pe_map[k]))) {
						break;
					}
				}

				if (k < target_pvs[j]->pv->pe_total) {
					lvm_move_extent_setup(&(volume->le_map[i]),
							      &(target_pvs[j]->pe_map[k]));
					break;
				} else {
					/* Wasn't able to find a free PE! Based
					 * on the above checks, this should
					 * really never happen.
					 */
					LOG_SERIOUS("BUG! Error finding free PEs "
						    "for region %s.\n", region->name);
					rc = EINVAL;
					goto out;
				}
			}

			if (j > MAX_PV) {
				/* Wasn't able to find a PV with free PEs! If we
				 * checked everything ahead of time, this should
				 * not happen.
				 */
				LOG_SERIOUS("BUG! Error finding PV with free PEs "
					    "for region %s.\n", region->name);
				rc = EINVAL;
				goto out;
			}
		}
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_move_pv
 *
 * Move all extents from the selected PV to remaining PVs in the container.
 **/
int lvm_move_pv(lvm_volume_group_t * group,
		list_anchor_t objects,
		option_array_t * options)
{
	lvm_physical_volume_t * source_pv;
	lvm_physical_volume_t * target_pvs[MAX_PV+1] = {NULL};
	int maintain_stripes, rc;

	LOG_ENTRY();

	/* Get the PV from the selected-objects list. */
	source_pv = lvm_get_selected_segment(objects);
	if (!source_pv) {
		rc = EINVAL;
		goto out;
	}

	/* Parse the option array. */
	rc = lvm_move_pv_parse_options(options, target_pvs,
				       group, &maintain_stripes);
	if (rc) {
		goto out;
	}

	/* Test that the move is possible. */
	rc = lvm_can_move_pv(source_pv, target_pvs, maintain_stripes);
	if (rc) {
		goto out;
	}

	/* Display warning message */
	rc = lvm_move_pv_display_warning(source_pv);
	if (rc) {
		goto out;
	}

	/* Move striped regions first. */
	rc = lvm_move_pv_stripes(source_pv, target_pvs, maintain_stripes);
	if (rc) {
		goto out;
	}

	/* Now move linear regions. */
	rc = lvm_move_pv_linears(source_pv, target_pvs);
	if (rc) {
		goto out;
	}

	/* Rebuild the freespace. */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_ERROR("Error updating freespace for container %s\n",
			  group->container->name);
	}

	group->container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

