/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvm_names.c
 *
 * Description: This file contains all functions related to translating names
 *		between the LVM style and the EVMS style. This applies to
 *		VGs/containers and LVs/regions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"

#define DEV_DIRECTORY		DEV_PATH "/"
#define EVMS_DEV_DIRECTORY	EVMS_DIR_NAME "/"

/**
 * lvm_translate_vg_name_to_container_name
 *
 * In LVM, volume groups have simple names, like "SystemGroup". In EVMS,
 * the container representing that VG should appear as "lvm/SystemGroup".
 * Thus, the name from the VG needs to be translated before copying to the
 * associated container. The container_name parameter should point to a
 * string buffer with sufficient space for copying the name.
 *
 * Also, if the container is consuming objects from a CSM disk-group, that
 * disk-group name must be prepended to the normal container-name.
 **/
void lvm_translate_vg_name_to_container_name(char * vg_name,
					     char * container_name,
					     storage_container_t * disk_group)
{
	LOG_ENTRY();

	container_name[0] = '\0';

	if (disk_group) {
		strncat(container_name, disk_group->name, EVMS_NAME_SIZE);
		strncat(container_name, "/", EVMS_NAME_SIZE-strlen(container_name));
	}
	strncat(container_name, LVM_DEV_DIRECTORY, EVMS_NAME_SIZE-strlen(container_name));
	strncat(container_name, vg_name, EVMS_NAME_SIZE-strlen(container_name));

	LOG_EXIT_VOID();
}

/**
 * lvm_translate_container_name_to_vg_name
 *
 * In LVM, volume groups have simple names, like "SystemGroup". In EVMS,
 * the container representing that VG will have the name "lvm/SystemGroup".
 * The container name needs to be translated to the VG name before the
 * name can be copied to the group's metadata. The vg_name parameter
 * should point to a string buffer with sufficient space for copying the
 * name.
 *
 * Also, container_name can now start with something besides "lvm/". If the
 * container consumes objects from a CSM disk-group, that disk-group's name
 * will be at the start of the container-name, followed by "lvm/" and then
 * the VG name.
 **/
int lvm_translate_container_name_to_vg_name(lvm_volume_group_t * group,
					    char * vg_name)
{
	char * ptr = group->container->name;
	int rc = 0;

	LOG_ENTRY();

	/* Skip over the disk-group name (if it exists). */
	if (group->container->disk_group) {
		ptr = strstr(ptr, group->container->disk_group->name);
		if (!ptr) {
			LOG_ERROR("Invalid container name: %s\n",
				  group->container->name);
			rc = EINVAL;
			goto out;
		}
		ptr += strlen(group->container->disk_group->name);
	}

	/* Find the "lvm/" and skip over it. */
	ptr = strstr(ptr, LVM_DEV_DIRECTORY);
	if (!ptr) {
		LOG_ERROR("Invalid container name: %s\n",
			  group->container->name);
		rc = EINVAL;
		goto out;
	}
	ptr += strlen(LVM_DEV_DIRECTORY);

	/* ptr should now be pointing at "group_name".
	 * Use this to create the name for the LV.	
	 */
	strncpy(vg_name, ptr, EVMS_NAME_SIZE);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_translate_lv_name_to_region_name
 *
 * In LVM, volumes have names based on their dev-node, which follow the
 * pattern /dev/group_name/volume_name. In EVMS, the region representing
 * that volume should appear as lvm/group_name/volume_name. Thus, the name
 * from the lv_disk_t needs to be translated before copying to the
 * associated region. The region_name parameter should point to a string
 * buffer with sufficient space for copying the name.
 **/
int lvm_translate_lv_name_to_region_name(char * lv_name,
					 char * disk_group_name,
					 char * region_name)
{
	char * ptr;
	int rc = 0;

	LOG_ENTRY();

	/* Make sure the string starts with /dev/, and skip over it. */
	ptr = strstr(lv_name, DEV_DIRECTORY);
	if (ptr != lv_name) {
		LOG_ERROR("Invalid LV name: %s\n", lv_name);
		rc = EINVAL;
		goto out;
	}
	ptr += strlen(DEV_DIRECTORY);

	/* The ptr should now be point to "group_name/volume_name".
	 * Use this to create the name for the region.
	 */
	region_name[0] = '\0';
	if (disk_group_name) {
		strncat(region_name, disk_group_name, EVMS_NAME_SIZE-strlen(region_name));
		strncat(region_name, "/", EVMS_NAME_SIZE-strlen(region_name));
	}
	strncat(region_name, LVM_DEV_DIRECTORY, EVMS_NAME_SIZE-strlen(region_name));
	strncat(region_name, ptr, EVMS_NAME_SIZE-strlen(region_name));

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_make_lv_name
 *
 * This function will be used during creation of a new volume. The user
 * will specify a simple name for the volume, and we need to generate
 * an LV name, using the specified group.
 **/
void lvm_make_lv_name(char * simple_name,
		      lvm_volume_group_t * group,
		      char * lv_name)
{
	char vg_name[EVMS_NAME_SIZE+1] = {0};

	LOG_ENTRY();

	lvm_translate_container_name_to_vg_name(group, vg_name);

	strncpy(lv_name, DEV_DIRECTORY, EVMS_NAME_SIZE);
	strncat(lv_name, vg_name, EVMS_NAME_SIZE-strlen(lv_name));
	strncat(lv_name, "/", EVMS_NAME_SIZE-strlen(lv_name));
	strncat(lv_name, simple_name, EVMS_NAME_SIZE-strlen(lv_name));

	LOG_EXIT_VOID();
}

