/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libs390.so
 *
 *   File: dm.c
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "390segmgr.h"
#include "helpers.h"
#include "dm.h"



/*
 *  Function: dos_schedule_dm_rename
 *
 *  Called to schedule the renaming of an active kernel DM 
 *  segment to a new name.  This is done when we discover that
 *  the name of the segment (e.g. hd1) doesn't match the 
 *  name that is should have ... because names are based on 
 *  metadata rules ... hd1 == name of partition found in the 
 *  1st partition table entry of the MBR ptable.  If the name
 *  is not in agreement with the actual partition record ...
 *  but we have matching metadata (starting lba and size ) then
 *  the kernel has the wrong name for the DM object and we 
 *  schedule a rename ... that will occur if the user commits
 *  changes ... see SEG_activate in dm.c
 */
int s390_schedule_dm_rename( char *dm_name, DISKSEG *seg)
{
        SEG_PRIVATE_DATA *pdata = (SEG_PRIVATE_DATA *)seg->private_data;
        int rc=EINVAL;

        LOG_ENTRY();

        if (dm_name && seg && pdata) {
                LOG_DEBUG("scheduling rename of kernel segment %s to %s\n", dm_name, seg->name ); 
                
                if (pdata->dm_name==NULL) {
                        pdata->dm_name = malloc(EVMS_NAME_SIZE+1);                      
                }
                if (pdata->dm_name) {           
                        strncpy(pdata->dm_name, dm_name, EVMS_NAME_SIZE);                               
                        pdata->cflags |= SEG_NEEDS_DM_RENAME;
                        seg->flags    |= SOFLAG_NEEDS_ACTIVATE;
                        rc = 0;
                }
                else {
                        rc = ENOMEM;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  get_DM_info
 *
 *  Called when a DM activate fails.  The job here is to see if the 
 *  segment is active in the kernel.  If so ... compare the target 
 *  mapping against the engine segment object.  If the segments starting
 *  LBA and SIZE match ... then we dont need to activate the segment
 *  because it is already active and has a correct mapping ... return 0.
 */
static int get_DM_info( DISKSEG *seg )
{       
        dm_target_t *targets=NULL;
        dm_device_t *dev = NULL;
        int rc;

        LOG_ENTRY();
        LOG_DEBUG("seg= %s\n", seg->name);

        rc = EngFncs->dm_update_status(seg);
        if (!rc) {
                if (seg->flags & SOFLAG_ACTIVE) {
                        LOG_DEBUG("segment IS active in the kernel\n");
                        rc = EngFncs->dm_get_targets(seg, &targets);
                        if ( !rc && targets != NULL) {
                                dev = (dm_device_t *) targets->data.linear;
                                if ( seg->start == dev->start &&
                                     seg->size  == targets->length) {
                                        LOG_DEBUG("kernel object matches ... marking segment active\n");
                                }
                                else {
                                        LOG_ERROR("error, got a DM object using our segment name but the metadata differs. dont know what to do!\n");
                                        rc = ENODEV; // dont know how to handle this.
                                }
                        }
                        if (targets) EngFncs->dm_deallocate_targets(targets);
                }
                else {
                        LOG_DEBUG("segment is NOT active in the kernel\n");
                        rc=ENODEV;
                }
        }

        if (rc) {
                seg->flags |= SOFLAG_NEEDS_ACTIVATE;
        }

        LOG_EXIT_INT(rc);
        return rc;
}                         

/*
 *  Function: s390_activate
 *
 *  Called in order to create a mapping of a disk partition
 *  in the kernel.
 */
int s390_activate( DISKSEG *seg )
{
        int rc=EINVAL;
        LOGICALDISK *ld;
        dm_target_t target;
        dm_device_t linear;
        SEG_PRIVATE_DATA *pdata=NULL;

        LOG_ENTRY();

        ld = get_logical_disk(seg);
        if (ld) {

                pdata = (SEG_PRIVATE_DATA *)seg->private_data;

                // we are either going to rename an existing dm
                // object or else create a new dm mapping ...
                if ( pdata->cflags & SEG_NEEDS_DM_RENAME) {                     
                        LOG_DEBUG("private dm rename flag is on for this object\n");
                        rc = EngFncs->dm_rename(seg, pdata->dm_name, seg->name);
                        if (!rc) {                                                      
                                if (pdata->dm_name) free(pdata->dm_name);
                                pdata->dm_name = NULL;
                                pdata->cflags &= ~SEG_NEEDS_DM_RENAME;
                                seg->flags   &= ~SOFLAG_NEEDS_ACTIVATE;
                        }
                }
                else {
                        target.start = 0;
                        target.length = seg->size;
                        target.params = NULL;
                        target.next = NULL;
                        target.type = DM_TARGET_LINEAR;
                        target.data.linear = &linear;
                        linear.major = ld->dev_major;
                        linear.minor = ld->dev_minor;
                        linear.start = seg->start;

                        rc = EngFncs->dm_activate(seg, &target);

                        if (rc) {
                                rc = get_DM_info(seg);  // FAILED ... test if already active
                        }

                        if (!rc) {
                                seg->flags &= ~SOFLAG_NEEDS_ACTIVATE;
                        }
                }
        }
        else {
                LOG_ERROR("error, can't get logical disk\n");
                seg->flags &= ~SOFLAG_NEEDS_ACTIVATE;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: s390_deactivate
 *
 *  Called in order to delete an existing mapping of a disk
 *  partition in the kernel.
 */
int s390_deactivate( DISKSEG *seg )
{
        int rc;

        LOG_ENTRY();

        rc = EngFncs->dm_deactivate(seg);

        LOG_EXIT_INT(rc);
        return rc;
}

