/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: reopen.c
 */

#include <unistd.h>
#include <sys/types.h>
#include <pthread.h>
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "commit.h"
#include "progress.h"
#include "logging.h"
#include "pixmap.h"
#include "views.h"
#include "thing.h"
#include "message.h"
#include "widget.h"
#include "task.h"
#include "main.h"

/*
 *
 *   void on_remote_node_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine handles switching to the new remote node. We first check to see if there
 *      are pending changed before closing the engine. If so, we prompt the user to see if they
 *      want to changes saved. After getting the response, we save changes if necessary then
 *      close the engine and reopen the engine with the new node. Once the engine has reopened,
 *      we schedule to have the views refreshed.
 *
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - not used
 *
 *   Exit:
 *      See description.
 *
 */
void on_remote_node_button_clicked(GtkButton * button, gpointer user_data)
{
	gint rc;
	gchar *node_name;
	GtkCList *clist;
	boolean changes_pending;

	clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button), "selection_window_clist"));
	node_name = g_strdup(get_single_select_current_row_data(clist));
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));

	rc = evms_changes_pending(&changes_pending, NULL);
	if (rc == 0) {
		if (changes_pending) {
			display_save_before_reopen_confirmation(node_name);
			gtk_main();
		}

		while (gtk_events_pending())
			gtk_main_iteration_do(FALSE);

		rc = evms_set_current_node(node_name);
		if (rc == 0) {
			gchar *message;

			message = g_strdup_printf(_("Now administering node %s"), node_name);
			set_status_bar_message(message);
			g_free(message);

			set_main_window_title(GTK_WINDOW(get_main_window_id()), node_name);
			refresh_main_window_views();
		} else {
			display_results_and_exit(rc, _("Unable to open remote engine"),
						 _
						 ("Unable to open the engine on the remote node. The application will now shutdown."),
						 NULL, FALSE, get_main_window_id());
		}

	} else {
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
		display_results_window(rc, NULL, _("Unable to determine if changes are pending!"),
				       NULL, FALSE, get_main_window_id());
	}

	g_free(node_name);
}

/*
 *  Free the node name string associated with a row.
 */
void on_node_row_destroyed(gpointer data)
{
	g_free(data);
}

/*
 *   void add_node_to_selection_list (GtkCList *, node_info_t *)
 *
 *   Description:
 *      This routine appends a node name to the selection list and
 *      associates the node name string with the new row data.
 *
 *   Entry:
 *      clist - the GtkCList to contain the node names
 *      node  - the node to append to the clist
 *
 *   Exit:
 *      See description.
 *
 */
void add_node_to_selection_list(GtkCList * clist, node_info_t * node)
{
	gint row = -1;
	gchar *node_name;
	gchar *text[] = { "", "", "", "" };

	node_name = g_strdup(node->node_name);
	text[SL_DESC_COLUMN] = node_name;

	row = clist_append_row(clist, text);

	if (row != -1) {
		gtk_clist_set_row_data_full(clist, row, node_name,
					    (GtkDestroyNotify) on_node_row_destroyed);
	}
}

/*
 *
 *   void on_remote_node_clist_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of active nodes.
 *
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - not used
 *
 *   Exit:
 *      Selection list populated with active nodes.
 *
 */
void on_remote_node_clist_realize(GtkWidget * widget, gpointer user_data)
{
	gint rc = 0;
	GtkCList *clist = GTK_CLIST(widget);
	node_list_t *nodes;

	gtk_clist_set_column_visibility(clist, SL_ICON_COLUMN, FALSE);
	gtk_clist_set_column_visibility(clist, SL_SIZE_COLUMN, FALSE);
	set_selection_window_clist_column_titles(clist, NULL, _("Nodename"), NULL);

	rc = evms_get_node_list(ACTIVE_NODES_ONLY, &nodes);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_node_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		guint i;

		for (i = 0; i < nodes->count; i++) {
			add_node_to_selection_list(clist, &(nodes->node_info[i]));
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(nodes);
	}
}
