/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <string.h>
#include <frontend.h>
#include <glib.h>

#include "common.h"
#include "logging.h"
#include "enum.h"

/**
 *	enum_volumes - enumerates all logical volumes
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the current logical volumes.
 **/
int enum_volumes(void *user_data, handle_array_t **handles)
{
	return evms_get_volume_list(0, 0, 0, handles);
}

/**
 *	enum_topmost_objects - enumerates all topmost storage objects
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the topmost storage objects.
 **/
int enum_topmost_objects(void *user_data, handle_array_t **handles)
{
	return evms_get_object_list(0, DATA_TYPE, 0, 0, TOPMOST, handles);
}

/**
 *	enum_data_objects - enumerates all data storage objects
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all data type storage objects.
 */
int enum_data_objects(void *user_data, handle_array_t **handles)
{
	return evms_get_object_list(0, DATA_TYPE, 0, 0, 0, handles);
}

/**
 *	enum_feature_objects - enumerates all feature objects
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all feature objects.
 **/
int enum_feature_objects(void *user_data, handle_array_t **handles)
{
	return evms_get_object_list(EVMS_OBJECT, DATA_TYPE, 0, 0, 0, handles);
}

/**
 *	enum_regions - enumerates all storage regions
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the storage regions.
 **/
int enum_regions(void *user_data, handle_array_t **handles)
{
	return evms_get_object_list(REGION, 0, 0, 0, 0, handles);
}

/**
 *	enum_containers - enumerates all storage containers
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all storage containers.
 **/
int enum_containers(void *user_data, handle_array_t **handles)
{
	return evms_get_container_list(0, 0, 0, handles);
}

/**
 *	enum_segments - enumerates all disk segments
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all disk segments.
 **/
int enum_segments(void *user_data, handle_array_t **handles)
{
	return evms_get_object_list(SEGMENT, 0, 0, 0, 0, handles);
}

/**
 *	enum_disks - enumerates all logical disks
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all logical disks.
 **/
int enum_disks(void *user_data, handle_array_t **handles)
{
	return evms_get_object_list(DISK, 0, 0, 0, 0, handles);
}

/**
 *	enum_all_plugins - enumerates all user-space plugins
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the loaded
 *	user-space plugins.
 **/
int enum_all_plugins(void *user_data, handle_array_t **handles)
{
	return evms_get_plugin_list(0, 0, handles);
}

/**
 *	duplicate_handle_array - create a dynamically allocated copy of a given handle array
 *	@old: the original/source handle array
 *
 *	This routine make a copy of given handle array. The copy should be freed
 *	with g_free() when no longer needed.
 */
handle_array_t *duplicate_handle_array(handle_array_t *old)
{
	int count;
	handle_array_t *new;

	count = old->count > 0 ? old->count - 1 : 0;
	new = g_memdup(old, sizeof(handle_array_t) + (sizeof(object_handle_t) * count));
	
	return new;
}

/**
 *	merge_handle_array - combine two handles areas to produce a new one
 *	@array1: the first handle array
 *	@array2: the second handle array
 *
 *	This routine takes the address of two handle arrays, allocates space
 *	to copy the handles in both into one array and returns the new array. 
 */
handle_array_t *merge_handle_array(handle_array_t *array1, handle_array_t *array2)
{
	int array1_size, array2_size, merged_size;
	handle_array_t *merged_array = NULL;

	if (array1 && array2) {
		array1_size = sizeof(object_handle_t) * array1->count;
		array2_size = sizeof(object_handle_t) * array2->count;
		merged_size = sizeof(handle_array_t) + array1_size + array2_size;

		merged_array = g_malloc(merged_size);

		if (merged_array) {
			merged_array->count = array1->count + array2->count;
			memcpy(&(merged_array->handle[0]), &(array1->handle[0]), array1_size);
			memcpy(&(merged_array->handle[array1->count]), &(array2->handle[0]), array2_size);
		}
	}
	return merged_array;
}

/**
 *	enum_feature_plugins - enumerates all feature plugins
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a single, merged handle array of all
 *	the different feature type plugins.
 **/
int enum_feature_plugins(void *user_data, handle_array_t **handles)
{
	int rc;
	handle_array_t *features;
	handle_array_t *assoc_features;

	rc = evms_get_plugin_list(EVMS_FEATURE, 0, &features);
	if (rc == 0) {
		rc = evms_get_plugin_list(EVMS_ASSOCIATIVE_FEATURE, 0, &assoc_features);
		if (rc == 0) {
			*handles = merge_handle_array(features, assoc_features);            
			evms_free(features);
			evms_free(assoc_features);
		} else {
			evms_free(features);
			log_error("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
		}
	} else {    
		log_error ("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
	}
	return rc;
}

/**
 *	enum_container_plugins - enumerates all user-space plugins that support containers
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the user-space plugins that
 *	support containers.
 **/
int enum_container_plugins(void *user_data, handle_array_t **handles)
{
	return evms_get_plugin_list(0, SUPPORTS_CONTAINERS, handles);
}

/**
 *	enum_region_manager_plugins - enumerates all region manager plugins
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the region manager
 *	user-space plugins.
 **/
int enum_region_manager_plugins(void *user_data, handle_array_t **handles)
{
	return evms_get_plugin_list(EVMS_REGION_MANAGER, 0, handles);
}

/**
 *	enum_device_manager_plugins - enumerates all device manager plugins
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the device manager
 *	user-space plugins.
 **/
int enum_device_manager_plugins(void *user_data, handle_array_t **handles)
{
	return evms_get_plugin_list(EVMS_DEVICE_MANAGER, 0, handles);
}

/**
 *	enum_segment_manager_plugins - enumerates all segment manager plugins
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the segment manager
 *	user-space plugins.
 **/
int enum_segment_manager_plugins(void *user_data, handle_array_t **handles)
{
	return evms_get_plugin_list(EVMS_SEGMENT_MANAGER, 0, handles);
}

/**
 *	enum_fsim_plugins - enumerates all File System Interface Module plugins
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the FSIM plugins.
 **/
int enum_fsim_plugins(void *user_data, handle_array_t **handles)
{
	return evms_get_plugin_list(EVMS_FILESYSTEM_INTERFACE_MODULE, 0, handles);
}

/**
 *	enum_acceptable_objects - enumerates acceptable objects for a task
 *	@user_data: contains task handle
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array of all the acceptable objects for
 *	a task.
 **/
int enum_acceptable_objects(void *user_data, handle_array_t **handles)
{
	task_handle_t task = GPOINTER_TO_UINT(user_data);

	return evms_get_acceptable_objects(task, handles);
}

/**
 *	enum_changed_objects - enumerates volumes, storage objects, and containers that have changes
 *	@user_data: not used
 *	@handles: the address of where to store the handle array
 *
 *	This routine returns a handle array containing volumes, storage objects, and
 *	containers that may need to be activated, re-activated, deleted, need to re-write
 *	meta-data or whatever.
 */
 int enum_changed_objects(void *user_data, handle_array_t **handles)
 {
 	int rc = 0;
	handle_array_t *volumes = NULL;
	handle_array_t *objects = NULL;
	handle_array_t *containers = NULL;

	*handles = NULL;
	
	evms_get_volume_list(0, 0, VOL_CHANGES_PENDING, &volumes);
	evms_get_container_list(0, 0, CON_CHANGES_PENDING, &containers);
	
	if (volumes != NULL && containers != NULL) {
		*handles = merge_handle_array(volumes, containers);
		evms_free(volumes);
		evms_free(containers);
	} else if (volumes != NULL) {
		*handles = duplicate_handle_array(volumes);
		evms_free(volumes);
	} else if (containers != NULL) {
		*handles = duplicate_handle_array(containers);
		evms_free(containers);
	}
	
	evms_get_object_list(0, 0, 0, 0, OBJ_CHANGES_PENDING, &objects);
	
	if (objects != NULL && *handles != NULL) {
		handle_array_t *tmp = *handles;
		
		*handles = merge_handle_array(tmp, objects);
		g_free(tmp);
		evms_free(objects);
	} else if (objects != NULL) {
		*handles = objects;
	} else if (*handles == NULL) {
		rc = ENODATA;
	}
	return rc;
 }
