/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_io.c
 *
 * Description: This file contains all functions pertaining to disk I/O within
 *              the MD region manager plugin. This file should be the only
 *              location of calls to READ and WRITE (except the md_read and
 *              md_write API calls).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <plugin.h>

#include "md.h"

#define my_plugin_record my_plugin

/* Function: md_region_rw
 *
 * If the region is active in the kernel,
 * perform IO on the region device node.
 * Returns 0 for success, else error.
 *
 */
int md_region_rw(storage_object_t *region,
		 lsn_t lsn,
		 sector_count_t count,
		 void *buffer,
		 int rw)
{
	int rc=0;
	int fd;

	LOG_ENTRY();

	if (!count) {
		LOG_EXIT_INT(0); /* nothing to do */
		return 0;
	}

	if ((region->flags & SOFLAG_ACTIVE) == 0) {
		return ENODEV;
	}

	fd = EngFncs->open_object(region, O_SYNC | O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Error opening md object %s.\n",region->name);
		rc = ENODEV;
	} else {
		if (rw == 1)
			rc = EngFncs->write_object(region, fd, buffer,
						   count<<EVMS_VSECTOR_SIZE_SHIFT,
						   lsn<<EVMS_VSECTOR_SIZE_SHIFT);
		else
			rc = EngFncs->read_object(region, fd, buffer,
						  count<<EVMS_VSECTOR_SIZE_SHIFT,
						  lsn<<EVMS_VSECTOR_SIZE_SHIFT);
		if (rc > 0) {
			rc = 0;
		} else {
			rc = EIO;
			LOG_WARNING("%s failed, md object %s rc=%d.\n",
				    (rw == 1) ? "WRITE" : "READ",
				    region->name,
				    rc);
		}
		EngFncs->close_object(region, fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

#define BLKFLSBUF _IO(0x12,97)
int md_ioctl_flush_buffer_cache(storage_object_t *obj)
{
	int rc = 0, fd;
	LOG_ENTRY();

	fd = EngFncs->open_object(obj, O_RDONLY);
	if (fd <= 0) {
		LOG_DEBUG("Unable to open object %s to send ioctl\n", obj->name);
		rc = -fd;
	} else {
		rc = EngFncs->ioctl_object(obj, fd, BLKFLSBUF, NULL);
		if (rc) {
			LOG_ERROR("Unable to flush buffer cache on %s (major=%d, minor=%d)\n",
				obj->name, obj->dev_major, obj->dev_minor);
		}
		EngFncs->close_object(obj, fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_start_array(storage_object_t *region, int dev_major, int dev_minor)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = START_ARRAY;
		arg = makedev(dev_major, dev_minor);
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("Error starting MD array %s (major=%d, minor=%d), rc=%d\n", region->name, dev_major, dev_minor, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;

}

int md_ioctl_stop_array(storage_object_t *region)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = STOP_ARRAY;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_run_array(storage_object_t *region)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = RUN_ARRAY;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_get_disk_info(storage_object_t *region, mdu_disk_info_t *info)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = GET_DISK_INFO;
		arg = (ulong)info;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_get_array_info(storage_object_t *region, mdu_array_info_t *md_info)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = GET_ARRAY_INFO;
		arg = (ulong)md_info;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_EXTRA("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_set_array_info(storage_object_t *region, mdu_array_info_t *md_info)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = SET_ARRAY_INFO;
		arg = (ulong)md_info;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s) rc(%d)\n", __FUNCTION__, region->name, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_add_new_disk(storage_object_t *region, mdu_disk_info_t *disk_info)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open region %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = ADD_NEW_DISK;
		arg = (ulong)disk_info;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("%s: IOCTL failed, region(%s), disk(%d:%d) rc(%d)\n", __FUNCTION__, region->name, disk_info->major, disk_info->minor, rc);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

#define BLKGETSIZE64 _IOR(0x12,114,size_t)
int md_ioctl_get_blk_size(storage_object_t *region, u_int64_t *size)
{
	int rc = 0, fd, ioctl_cmd;
	ulong arg = 0;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open md object %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		ioctl_cmd = BLKGETSIZE64;
		arg = (ulong)size;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("Error getting size of md region %s, rc=%d size=%"PRIu64".\n", region->name, rc, *size);
		}
		EngFncs->close_object(region, fd);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int md_ioctl_hot_add_remove(storage_object_t *region, int ioctl_cmd, int kdev)
{
	int rc = 0, fd;
	ulong arg;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open md object %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		arg = (ulong)kdev;
		rc = EngFncs->ioctl_object(region, fd, ioctl_cmd, arg);
		if (rc) {
			LOG_ERROR("Unable to hot %s the spare (major=%d, minor=%d) to [%s]\n",
				  (ioctl_cmd == HOT_ADD_DISK) ? "add" : "remove",
                                  major(kdev), minor(kdev), region->name);
		}
		EngFncs->close_object(region, fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_hot_add_disk(storage_object_t *region, int kdev)
{
	int rc;
	rc = md_ioctl_hot_add_remove(region, HOT_ADD_DISK, kdev);
	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_hot_remove_disk(storage_object_t *region, int kdev)
{
	int rc;
	rc = md_ioctl_hot_add_remove(region, HOT_REMOVE_DISK, kdev);
	LOG_EXIT_INT(rc);
	return rc;
}

int md_ioctl_set_disk_faulty(storage_object_t *region, int kdev)
{
	int rc = 0, fd;
	ulong arg;
	LOG_ENTRY();

	fd = EngFncs->open_object(region, O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Unable to open md object %s to send ioctl\n", region->name);
		rc = -fd;
	} else {
		arg = (ulong)kdev;
		rc = EngFncs->ioctl_object(region, fd, SET_DISK_FAULTY, arg);
		if (rc) {
			LOG_ERROR("Unable to set disk faulty (major=%d, minor=%d) to [%s]\n",
                                  major(kdev), minor(kdev), region->name);
		}
		EngFncs->close_object(region, fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}


