# (C) Copyright IBM Corp. 2004
#
# This program is free software;  you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY;  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program;  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Perl module with common subroutines for managing DOS segments.

package Evms::Dos;

use strict;
use warnings;

use Evms::Common;
use Evms::Object;
use Evms::Log;

BEGIN {
	use Exporter();
	our (@ISA, @EXPORT);
        @ISA = qw(Exporter);
        @EXPORT = qw(&assign_dos_plugin
		     &unassign_dos_plugin
		     &create_dos_segment
		     &create_dos_primary_segment
		     &create_dos_logical_segment
		     &get_dos_data_segments
		     &expand_dos_segment
		     &shrink_dos_segment
		     &move_dos_segment
		     &get_dos_segment_details
		     &create_dos_segments
		    );
}

# assign_dos_plugin
#
# Arguments:
#    disk: Disk to assign the DOS plugin to.
sub assign_dos_plugin($)
{
        my $disk = $_[0];
        my $command = "assign:DosSegMgr={},$disk";
        my $rc;

        $rc = run_evms_command($command);

        return $rc;
}

# unassign_dos_plugin
#
# Arguments:
#    disk: Disk to unassign the DOS plugin from.
sub unassign_dos_plugin($)
{
	my $disk = $_[0];
	my $command = "remove:$disk";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# create_dos_segment
#
# Arguments:
#    freespace: Name of freespace segment to create the new segment from.
#    options: Reference of hash containing options for the create. Allowable
#             hash keys are values are:
#               Size: Size for new segment. Must include appropriate suffix.
#               Offset: Starting offset within the freespace segment. Must
#                       include size suffix.
#               TypeByName: Type-identifier for new segment. Must be "Linux",
#                           "Linux Swap", "Linux LVM", "NTFS", "HPFS",
#                           or "FAT16".
#               TypeByNumber: Type-identifier for new segment. Must be in
#                             decimal (no hex values).
#               Bootable: Create bootable segment ("TRUE" or "FALSE").
#               Primary: Create primary or logical segment ("TRUE" or "FALSE").
sub create_dos_segment($ $)
{
	my $freespace = $_[0];
	my $options_ref = $_[1];
	my %options = %{$options_ref};
	my ($command, $key, @keys, $rc);

	@keys = ("Size", "Offset", "TypeByName",
		 "TypeByNumber", "Bootable", "Primary");
	$command = "create:segment,$freespace";

	# Ideally, we'd just loop foreach kyes(%options), but with DOS, we
	# must ensure the ordering of the options on the command-line. In
	# particular, "Size" must come before "Offset", or "Offset" will be
	# ignored. Yuck!
	foreach $key (@keys) {
		if (defined($options{$key})) {
			$command .= ",$key=$options{$key}";
		}
	}

	$rc = run_evms_command($command);

	return $rc;
}

# create_dos_primary_segment
#
# Arguments:
#    freespace: Name of freespace segment to create the new segment from.
#    size: Size of the new segment. Must include "KB", "MB", or "GB".
sub create_dos_primary_segment($ $)
{
        my $freespace = $_[0];
        my $size = $_[1];
	my %options;
        my $rc;

	$options{"Size"} = $size;
	$options{"Primary"} = "TRUE";

	$rc = create_dos_segment($freespace, \%options);

	return $rc;
}

# create_dos_logical_segment
#
# Arguments:
#    freespace: Name of freespace segment to create the new segment from.
#    size: Size of the new segment. Must include "KB", "MB", or "GB".
sub create_dos_logical_segment($ $)
{
        my $freespace = $_[0];
        my $size = $_[1];
	my %options;
        my $rc;

	$options{"Size"} = $size;
	$options{"Primary"} = "FALSE";

	$rc = create_dos_segment($freespace, \%options);

	return $rc;
}

# get_dos_data_segments
#	Get name of data segments.
#
# Arguments:
#	disk: Name of disk to query
#
sub get_dos_data_segments($$)
{
        my $disk = $_[0];
	my $segments = $_[1];
        my $command = "Query:Segments,Plugin=DosSegMgr,Disk=$disk";
	my @output;
        my $rc;
	my $i=0;

        $rc = run_evms_command($command, \@output);

	# evms returns many lines of output...
	foreach (@output) {
		if ($_ =~ /Segment Name/) {
			my @list;

			next if (($_ =~ /_mbr/) || ($_ =~ /_ebr/) || ($_ =~ /_freespace/));

			# This line contains "Segment Name",
			# split it into tokens
			# and we're only interested in the last token.

			@list = split /\s* \s*/, $_;
			# remove trailing newline if any.
			chomp $list[-1];
			push @{$segments}, $list[-1];
		}
	}

out:
        return $rc;
}

# expand_dos_segment
#
# Arguments:
#    segment: Name of segment to expand.
#    freespace: Name of freespace segment to use for the expand.
#    size: Amount to expand by. Must include appropriate size suffix.
sub expand_dos_segment($ $ $)
{
	my $segment = $_[0];
	my $freespace = $_[1];
	my $size = $_[2];
	my $command = "expand:$segment,Size=$size,$freespace";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# shrink_dos_segment
#
# Arguments:
#    segment: Name of segment to shrink.
#    size: Amount to shrink by. Must include appropriate size suffix.
sub shrink_dos_segment($ $)
{
	my $segment = $_[0];
	my $size = $_[1];
	my $command = "shrink:$segment,Size=$size,$segment";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# move_dos_segment
#
# Arguments:
#    segment: Name of segment to move.
#    freespace: Freespace segment to move to.
sub move_dos_segment($ $)
{
	my $segment = $_[0];
	my $freespace = $_[1];
	my $command = "task:Move,$segment,$freespace";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# get_dos_segment_details
# Return a hash with extended info details about the specified segment.
#
# Arguments:
#    segment: Name of segment to get details for
sub get_dos_segment_details($)
{
	my $segment = $_[0];
	my @query_output;
	my %details;

	# Standard info.
	%details = get_object_details($segment);

	# Extended info.
	@query_output = extended_query_object($segment);
	$details{'Primary'} = get_extended_info_value("Flag", \@query_output);

	return %details;
}

# create_dos_segments
# Create one or more DOS logical segments on a blank disk.
#
# Arguments
#    disk: Name of disk to create the segments on.
#    count: Number of logical segments to create.
#    size: Size of each segment. All segments will be the same size. Must
#          include appropriate size suffix.
sub create_dos_segments($ $ $)
{
	my $disk = $_[0];
	my $count = $_[1];
	my $size = $_[2];
	my ($freespace, $rc, $i);

	$rc = assign_dos_plugin($disk);
	if ($rc) {
		log_error("Error assigning DOS to disk $disk.\n");
		return $rc;
	}

	$freespace = $disk . "_freespace1";
	for ($i = 0; $i < $count; $i++) {
		$rc = create_dos_logical_segment($freespace, $size);
		if ($rc) {
			log_error("Error creating $size logical segment " .
				  "from $freespace.\n");
			return $rc;
		}
		$freespace = $disk . "_freespace2";
	}

	return $rc;
}


1;

