/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: support.c
 */

#include <unistd.h>
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "results.h"
#include "selection.h"
#include "selection_cb.h"
#include "views.h"
#include "thing.h"
#include "message.h"
#include "logging.h"
#include "help.h"
#include "pixmap.h"

#include <string.h>

static GtkWidget *main_window = NULL;
static GtkWidget *status_bar = NULL;
static GtkWidget *menu_bar = NULL;
static GtkWidget *progress_bar = NULL;
static GtkWidget *button_toolbar = NULL;

/*
 *
 *   GtkWidget* lookup_widget (GtkWidget *, const gchar *)
 *
 *   Description:
 *      This is a utility function that came from Glade. It
 *      allows finding a GtkWidget by name by searching
 *      through the children of a higher-level widget such
 *      as the top-level window or any parent of the widget
 *      we want.
 * 
 *   Entry:
 *      widget      - address of an ancestor of the widget we want
 *      widget_name - pointer to the string with the name of the
 *                    widget to lookup.
 *
 *   Exit:
 *      Returns the GtkWidget * of the widget we want.
 *
 */
GtkWidget *lookup_widget(GtkWidget * widget, const gchar * widget_name)
{
	GtkWidget *parent, *found_widget;

	for (;;) {
		if (GTK_IS_MENU(widget))
			parent = gtk_menu_get_attach_widget(GTK_MENU(widget));
		else
			parent = widget->parent;
		if (parent == NULL)
			break;
		widget = parent;
	}

	found_widget = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(widget), widget_name);
	if (!found_widget)
		log_warning("Widget not found: %s\n", widget_name);

	return found_widget;
}

/*
 *
 *   void on_button_clicked_destroy_window (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine dismisses the parent containing the button
 *      that was clicked.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Dismisses/destroys parent window of button
 *
 */
void on_button_clicked_destroy_window(GtkButton * button, gpointer user_data)
{
	GtkWidget *window;

	/*
	 * Get the handle the toplevel parent window to destroy it.
	 */

	window = gtk_widget_get_toplevel(GTK_WIDGET(button));

	gtk_widget_destroy(window);
}

/*
 *
 *   void display_not_implemented_yet_popup (void)
 *   
 *   Description:
 *      This routine display a small popup that indicates
 *      that the function selected has not been implemented
 *      yet. Eventually this function will become dormant
 *      when we reach a fully functional implementation.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      Popup is displayed awaiting dismissal.
 *
 */
void display_not_implemented_yet_popup(void)
{
	display_popup_window("Function not yet implemented.", "Function not yet implemented");
}

/*
 *
 *   void display_popup_window (gchar *, gchar *)
 *   
 *   Description:
 *      This routine display a small popup that suitable
 *      for short informational messages.
 * 
 *   Entry:
 *      title   - address of string containing window title
 *      message - address of string containing window text 
 *
 *   Exit:
 *      Popup is displayed awaiting dismissal.
 *
 */
void display_popup_window(gchar * title, gchar * message)
{
	GtkWidget *popup;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *separator;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkAccelGroup *accel_group;

	popup = gtk_window_new(GTK_WINDOW_DIALOG);
	vbox = gtk_vbox_new(FALSE, 10);
	label = gtk_label_new(message);
	button = gtk_button_new();
	separator = gtk_hseparator_new();
	accel_group = gtk_accel_group_new();

	gtk_accel_group_attach(accel_group, GTK_OBJECT(popup));

	get_ok_pixmap(&pixmap, &mask);
	add_pixmap_label_to_button(button, pixmap, mask, _("_OK"), accel_group);

	gtk_window_set_title(GTK_WINDOW(popup), title);
	gtk_window_set_position(GTK_WINDOW(popup), GTK_WIN_POS_CENTER);
	gtk_container_set_border_width(GTK_CONTAINER(popup), 15);

	gtk_container_add(GTK_CONTAINER(popup), vbox);
	gtk_box_pack_start_defaults(GTK_BOX(vbox), label);
	gtk_box_pack_start_defaults(GTK_BOX(vbox), separator);
	gtk_box_pack_start_defaults(GTK_BOX(vbox), button);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  gtk_widget_destroy, GTK_OBJECT(popup));

	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);

	gtk_widget_show_all(popup);
}

/*
 *
 *   void display_help_window (GtkWindow *, gchar *)
 *   
 *   Description:
 *      This routine display a small popup that suitable
 *      for displaying help.
 * 
 *   Entry:
 *      parent - window that we should become a transient window of
 *      text   - string containing text for help text box
 *
 *   Exit:
 *      Popup is displayed awaiting dismissal.
 *
 */
void display_help_window(GtkWindow * parent, gchar * text)
{
	GtkWidget *help_window;
	GtkWidget *top_vbox;
	GtkWidget *hbox;
	GtkWidget *frame;
	GtkWidget *scrolledwindow;
	GtkWidget *help_text;
	GtkWidget *hseparator;
	GtkWidget *hbuttonbox;
	GtkWidget *button;
	GtkWidget *pixmap;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();

	get_dialog_pixmap(INFO_PIXMAP, &gdk_pixmap, &mask);

	help_window = gtk_window_new(GTK_WINDOW_DIALOG);
	top_vbox = gtk_vbox_new(FALSE, 0);
	hbox = gtk_hbox_new(FALSE, 10);
	frame = gtk_frame_new(NULL);
	scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
	help_text = gtk_text_new(NULL, NULL);
	hseparator = gtk_hseparator_new();
	hbuttonbox = gtk_hbutton_box_new();
	button = gtk_button_new();
	pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	gtk_accel_group_attach(accel_group, GTK_OBJECT(help_window));

	get_ok_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(button, gdk_pixmap, mask, _("_OK"), accel_group);

	gtk_window_set_title(GTK_WINDOW(help_window), _("Help"));
	gtk_widget_set_usize(help_window, 550, 360);
	gtk_window_set_policy(GTK_WINDOW(help_window), FALSE, FALSE, FALSE);

	gtk_widget_show(top_vbox);
	gtk_widget_show(frame);
	gtk_widget_show(scrolledwindow);
	gtk_widget_show(help_text);
	gtk_widget_show(hseparator);
	gtk_widget_show(hbuttonbox);
	gtk_widget_show(button);
	gtk_widget_show(pixmap);
	gtk_widget_show(hbox);

	gtk_container_add(GTK_CONTAINER(help_window), top_vbox);

	gtk_box_pack_start(GTK_BOX(hbox), pixmap, FALSE, FALSE, 0);
	gtk_misc_set_alignment(GTK_MISC(pixmap), 0.10, 0.10);

	gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);

	gtk_box_pack_start(GTK_BOX(top_vbox), hbox, TRUE, TRUE, 0);

	gtk_container_set_border_width(GTK_CONTAINER(frame), 6);
	gtk_container_add(GTK_CONTAINER(frame), scrolledwindow);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow), GTK_POLICY_NEVER,
				       GTK_POLICY_ALWAYS);
	gtk_container_add(GTK_CONTAINER(scrolledwindow), help_text);
	gtk_box_pack_start(GTK_BOX(top_vbox), hseparator, FALSE, TRUE, 12);
	gtk_box_pack_start(GTK_BOX(top_vbox), hbuttonbox, FALSE, TRUE, 0);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 9);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), 118, 68);
	gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
	gtk_container_set_border_width(GTK_CONTAINER(button), 10);

	gtk_text_insert(GTK_TEXT(help_text), NULL, NULL, NULL, text, -1);
	gtk_text_insert(GTK_TEXT(help_text), NULL, NULL, NULL, get_key_navigation_help_text(), -1);

	gtk_tooltips_set_tip(tooltips, button, _("Close the help window"), NULL);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  gtk_widget_destroy, GTK_OBJECT(help_window));

	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);

	gtk_widget_show(help_window);

	if (parent)
		gtk_window_set_transient_for(GTK_WINDOW(help_window), parent);
}

/*
 * The following inline functions retrieve and set the window list
 * that keeps track of the windows a user progresses through. The
 * lists is used to hide or unhide previous windows. It is also
 * ultimetly handed to the display_results_window() function to
 * destroy all of the windows is the list.
 */

inline GList *get_window_list(GtkWidget * widget)
{
	return gtk_object_get_data(GTK_OBJECT(gtk_widget_get_toplevel(widget)), "window_list");
}

inline void set_window_list(GtkWidget * widget, GList * window_list)
{
	gtk_object_set_data(GTK_OBJECT(gtk_widget_get_toplevel(widget)), "window_list",
			    window_list);
}

/*
 *
 *   inline void destroy_window_list (GList *window_list)
 *   
 *   Description:
 *      This routine destroys the windows in a list starting
 *      at the given link It moves on to the next link as it
 *      removes the current link.
 * 
 *   Entry:
 *      window_list - list of windows to destroy
 *
 *   Exit:
 *      Windows in the list are destroyed and the 
 *      corresponding links are removed.
 *
 */
inline void destroy_window_list(GList * window_list)
{
	GList *link = window_list;
	GList *next_link;

	while (link != NULL) {
		if (link->data != NULL)
			gtk_widget_destroy(GTK_WIDGET(link->data));

		next_link = link->next;
		g_list_remove(link, link->data);
		link = next_link;
	}
}

/*
 *
 *   void complete_display_results_window_work (GtkWidget *, gchar *, GSList *, gboolean)
 *   
 *   Description:
 *      This routine completes the work resulting from displaying a results
 *      window. This includes making sure that the message window is stacked
 *      above this window.
 * 
 *   Entry:
 *      window        - results window id
 *      message       - status bar message string
 *      window_list   - optional list of windows to destroy
 *      refresh_views - TRUE if we should refresh the main window views
 *
 *   Exit:
 *      message window is hopefully stacked above results window,
 *      status bar message is updated, views are refreshed and
 *      window list is destroyed
 *
 */
void complete_display_results_window_work(GtkWidget * window, gchar * message, GList * window_list,
					  gboolean refresh_views)
{
	set_message_window_transient_for(GTK_WINDOW(window));

	set_status_bar_message(message);

	if (refresh_views)
		refresh_main_window_views();

	if (window_list != NULL)
		destroy_window_list(window_list);
}

/*
 *
 *   GtkWidget* display_results_window (gint, gchar *, gchar *, GSList *, gboolean, GtkWidget *)
 *   
 *   Description:
 *      This routine display a dialog that provides feedback
 *      on the results of a certain operation. The caller
 *      can provide the window title,an optional message, 
 *      a list of windows that should be destroyed
 *      after showing the results window as well as the 
 *      actual error code to be used in pulling a descriptive
 *      message for the return code.
 * 
 *   Entry:
 *      rc            - the return code (the results of an operation)
 *      title         - optional string to use as the title
 *      message       - optional supplemental message
 *      window_list   - optional list of windows to destroy
 *      refresh_views - TRUE if we should refresh the main window views
 *      parent_window - if not NULL, the results window is made a transient of
 *                      this window
 *
 *   Exit:
 *      Popup is displayed awaiting dismissal.
 *
 */
GtkWidget *display_results_window(gint rc, gchar * title, gchar * message, GList * window_list,
				  gboolean refresh_views, GtkWidget * parent_window)
{
	gchar *status;
	GtkWidget *window;
	GtkWidget *pixmap;
	GtkWidget *vbox;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;

	if (rc == E_CANCELED) {
		if (window_list != NULL)
			destroy_window_list(window_list);

		set_status_bar_message(_("The operation was cancelled by the user."));
		return NULL;
	}

	/*
	 * Create the results window and make any modifications 
	 * wanted by caller before showing it.
	 */

	window = create_results_window();

	if (title != NULL)
		gtk_window_set_title(GTK_WINDOW(window), title);

	if (message != NULL) {
		GtkWidget *label;

		label =
		    gtk_object_get_data(GTK_OBJECT(window), "results_window_user_message_label");
		gtk_label_set_text(GTK_LABEL(label), message);
	}

	vbox = gtk_object_get_data(GTK_OBJECT(window), "results_window_vbox");

	if (rc != SUCCESS) {
		GtkWidget *label;

		label = gtk_object_get_data(GTK_OBJECT(window), "results_window_return_code_label");
		gtk_label_set_text(GTK_LABEL(label), evms_strerror(ABS(rc)));

		gdk_beep();
		status = _("The operation completed with an error.");
		get_dialog_pixmap(ERROR_PIXMAP, &gdk_pixmap, &mask);
	} else {
		status = _("The operation completed successfully.");
		get_dialog_pixmap(INFO_PIXMAP, &gdk_pixmap, &mask);
	}

	pixmap = gtk_pixmap_new(gdk_pixmap, mask);
	gtk_widget_show(pixmap);

	gtk_box_pack_start(GTK_BOX(vbox), pixmap, TRUE, TRUE, 0);
	gtk_misc_set_alignment(GTK_MISC(pixmap), 0.17, 1);
	gtk_misc_set_padding(GTK_MISC(pixmap), 0, 32);

	complete_display_results_window_work(window, status, window_list, refresh_views);

	if (parent_window)
		gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(parent_window));

	gtk_widget_show(window);

	return window;
}

/*
 *
 *   gboolean editable_field_is_blank (GtkEditable *editable)
 *   
 *   Description:
 *      This routine determines if the given editable widget
 *      has any non-whitespace characters in it.
 * 
 *   Entry:
 *      editable - address of GtkEditable widget to check
 *
 *   Exit:
 *      Return TRUE if widget has no viable characters 
 *      or FALSE if contains non-whitespace characters
 *
 */
gboolean editable_field_is_blank(GtkEditable * editable)
{
	gboolean answer = TRUE;
	gchar *text = gtk_editable_get_chars(editable, 0, -1);

	if (text != NULL) {
		if ((strlen(g_strstrip(text)) > 0))
			answer = FALSE;

		g_free(text);
	}

	return answer;
}

/*
 *
 *   gboolean check_for_empty_list_when_exposed (GtkCList *, GdkEventExpose *, gpointer)
 *   
 *   Description:
 *      This routine determines if a list to be exposed (drawn)
 *      contains any rows. If empty, we append a blank row
 *      and set the some text indicating the list was not
 *      populated.
 * 
 *   Entry:
 *      clist     - the list we want to check to see if empty
 *      event     - pointer to the expose event structure
 *      user_data - column as to where to place empty list message
 *
 *   Exit:
 *      Return FALSE so the default handler and other expose
 *      event handlers execute.
 *
 */
gboolean check_for_empty_list_when_exposed(GtkCList * clist, GdkEventExpose * event,
					   gpointer user_data)
{
	/*
	 * If the list is empty, append a blank row and set the text at the 
	 * column where we were instructed to place the empty list text at. Also,
	 * make the list insensitive to avoid the row selection handler from
	 * kicking in on a bogus list.
	 */

	if (clist->rows == 0) {
		gint column = GPOINTER_TO_UINT(user_data);
		gchar *blank_row[] = { "", "", "", "", "", "", "", "", "", "", "", "", "", "", "" };

		gtk_clist_append(clist, blank_row);
		gtk_clist_set_text(clist, 0, column,
				   _("No items were found matching the search criteria."));
		gtk_widget_set_sensitive(GTK_WIDGET(clist), FALSE);
	}

	return FALSE;
}

/*
 *
 *   void set_selection_window_instructions_label (GtkWidget *, gchar *)
 *
 *   Description:
 *      This routine sets the text in the selection dialog window that 
 *      instructs the user what to do.
 * 
 *   Entry:
 *      window           - the selection dialog window id
 *      next_button_text - the text to be displayed on the "Next" button
 *
 *   Exit:
 *      The instructions label text is set.
 *
 */
void set_selection_window_instructions_label(GtkWidget * window, gchar * next_button_text)
{
	gchar *instruction_label_text;
	GtkLabel *instruction_label;

	instruction_label = gtk_object_get_data(GTK_OBJECT(window),
						"selection_window_instruction_label");

	instruction_label_text =
	    g_strdup_printf(_
			    ("Make a selection from the list shown above and then choose %s to continue"),
			    next_button_text);

	gtk_label_set_text(instruction_label, instruction_label_text);

	g_free(instruction_label_text);
}

/*
 *
 *   void replace_next_button_hbox_with_label (GtkWidget *, const gchar *)
 *
 *   Description:
 *      This routine replaces whatever is in the next button with a label
 *      containing the given text.
 * 
 *   Entry:
 *      next_button - the button id
 *      text        - text for the label to be placed in the button
 *
 *   Exit:
 *      The next button contains a new label
 *
 */
void replace_next_button_hbox_with_label(GtkWidget * next_button, const gchar * text)
{
	GtkWidget *label;
	GtkWidget *window;
	GtkWidget *tooltips;

	window = gtk_widget_get_toplevel(next_button);
	tooltips = gtk_object_get_data(GTK_OBJECT(window), "tooltips");

	label = gtk_label_new(text);
	gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);

	gtk_container_remove(GTK_CONTAINER(next_button), GTK_BIN(next_button)->child);
	gtk_container_add(GTK_CONTAINER(next_button), label);

	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), next_button,
			     _("Click this button to complete the operation"), NULL);

	gtk_object_set_data(GTK_OBJECT(window), "selection_window_next_button_label", label);

	gtk_widget_show(label);
}

/*
 *
 *   GtkWidget* create_standard_selection_window (gchar *, gchar *, gchar *,
 *                                                GtkSignalFunc, GtkSignalFunc,
 *                                                GtkSignalFunc, GtkSignalFunc,
 *                                                GtkSignalFunc, GtkSignalFunc,
 *                                                gpointer)
 *   
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      initiate the operation actions.
 * 
 *   Entry:
 *      window_title                  - title for the selection window
 *      next_button_text              - label text for next button (default is "Next")
 *      help_text                     - text to be displayed when help button is clicked
 *      clist_realize_sighandler      - function that will populate list when realized
 *      next_button_sighandler        - function to call when next button clicked
 *      prev_button_sighandler        - function to call when prev button clicked
 *      cancel_button_sighandler      - function to call when cancel button clicked
 *      clist_row_select_sighandler   - function to call when row is selected (optional)
 *      clist_row_unselect_sighandler - function to call when row is unselected (optional)
 *      user_data                     - address of user data to bind with signal handlers
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
GtkWidget *create_standard_selection_window(gchar * window_title,
					    gchar * next_button_text,
					    gchar * help_text,
					    GtkSignalFunc clist_realize_sighandler,
					    GtkSignalFunc next_button_sighandler,
					    GtkSignalFunc prev_button_sighandler,
					    GtkSignalFunc cancel_button_sighandler,
					    GtkSignalFunc clist_row_select_sighandler,
					    GtkSignalFunc clist_row_unselect_sighandler,
					    gpointer user_data)
{
	GtkWidget *window;

	window = create_selection_window();

	if (window != NULL) {
		GtkWidget *clist;
		GtkWidget *help_button;
		GtkWidget *cancel_button;
		GtkWidget *prev_button;
		GtkWidget *next_button;

		clist = gtk_object_get_data(GTK_OBJECT(window), "selection_window_clist");
		help_button = gtk_object_get_data(GTK_OBJECT(window), "help_button");
		prev_button =
		    gtk_object_get_data(GTK_OBJECT(window), "selection_window_prev_button");
		next_button =
		    gtk_object_get_data(GTK_OBJECT(window), "selection_window_next_button");
		cancel_button =
		    gtk_object_get_data(GTK_OBJECT(window), "selection_window_cancel_button");

		/*
		 * By default we hide the min/max object size column. Only the resize
		 * code fills it in a makes use of it.
		 */

		gtk_clist_set_column_visibility(GTK_CLIST(clist), SL_MINMAX_SIZE_COLUMN, FALSE);
		gtk_clist_column_titles_passive(GTK_CLIST(clist));

		if (window_title != NULL)
			gtk_window_set_title(GTK_WINDOW(window), window_title);

		if (next_button_text != NULL) {
			set_selection_window_instructions_label(window, next_button_text);
			replace_next_button_hbox_with_label(next_button, next_button_text);
		}

		/*
		 * Before we show the window, setup the signal handlers
		 * that will populate the clist when the clist is realized. 
		 * Also initialize the "Cancel" button to dismiss the window
		 * when no signal handler is provided. Setup the "Next" 
		 * button with the signal handler given. The "Next"
		 * button's sensitivity is reliant on a row selection.
		 * There is a default set of signal handlers that change
		 * the "Next" button sensitivity for "select_row" and
		 * "unselect_row" signals. If no signal handler is
		 * provided for the "Previous" button then assume this
		 * is the first selection panel and disable the button.
		 */

		if (next_button_sighandler != NULL)
			gtk_signal_connect(GTK_OBJECT(next_button), "clicked",
					   next_button_sighandler, user_data);

		if (prev_button_sighandler != NULL)
			gtk_signal_connect(GTK_OBJECT(prev_button), "clicked",
					   prev_button_sighandler, user_data);
		else
			gtk_widget_hide(prev_button);

		if (cancel_button_sighandler != NULL)
			gtk_signal_connect(GTK_OBJECT(cancel_button), "clicked",
					   cancel_button_sighandler, user_data);
		else
			gtk_signal_connect(GTK_OBJECT(cancel_button), "clicked",
					   on_button_clicked_destroy_window_list, NULL);

		if (clist_row_select_sighandler != NULL)
			gtk_signal_connect(GTK_OBJECT(clist), "select_row",
					   clist_row_select_sighandler, user_data);
		else
			gtk_signal_connect(GTK_OBJECT(clist), "select_row",
					   GTK_SIGNAL_FUNC(on_selection_window_clist_select_row),
					   NULL);

		if (clist_row_unselect_sighandler != NULL)
			gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
					   clist_row_unselect_sighandler, user_data);
		else
			gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
					   GTK_SIGNAL_FUNC(on_selection_window_clist_unselect_row),
					   NULL);

		gtk_signal_connect(GTK_OBJECT(help_button), "clicked",
				   on_selection_window_help_button_clicked, help_text);

		if (clist_realize_sighandler != NULL)
			gtk_signal_connect(GTK_OBJECT(clist), "realize",
					   clist_realize_sighandler, user_data);

		gtk_signal_connect(GTK_OBJECT(clist), "expose_event",
				   GTK_SIGNAL_FUNC(check_for_empty_list_when_exposed),
				   GUINT_TO_POINTER(SL_DESC_COLUMN));

		gtk_widget_set_sensitive(next_button, FALSE);
		gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(get_main_window_id()));
	}

	return window;
}

/*
 *
 *  void display_selection_window_results (GtkWidget *, gint, gchar *, gchar *)
 *   
 *   Description:
 *      This routine displays the results window and destroys the selection
 *      window if the operation was successful.
 * 
 *   Entry:
 *      widget      - a widget to help us lookup the selection window
 *      rc          - the return code for the selection window operation
 *      error_msg   - if an error, the message to use
 *      success_msg - if success, the message to use
 *
 *   Exit:
 *      The results window is displayed and if a successful operation, the
 *      the selection window is destroyed.
 *
 */
void display_selection_window_results(GtkWidget * widget, gint rc, gchar * error_msg,
				      gchar * success_msg)
{
	gchar *message;
	GList *window_list = NULL;
	GtkWidget *parent_window;
	GtkWidget *selection_window;

	/*
	 * Call the function that displays the window with everything
	 * we want to say about the outcome of the operation. Provide
	 * the selection_window in the window_list to have it destroyed
	 * it after popping up the results_window. 
	 */

	selection_window = gtk_widget_get_toplevel(widget);

	if (rc != SUCCESS && rc != E_CANCELED) {
		message = error_msg;
		parent_window = selection_window;
	} else {
		message = success_msg;
		parent_window = get_main_window_id();
		window_list = g_list_prepend(window_list, selection_window);
	}

	display_results_window(rc, NULL, message, window_list, (rc == SUCCESS), parent_window);
}

/*
 * Since I personally dislike public globals, they are sometimes
 * useful. As a compromise, I've added some getter and setter
 * functions to some common application wide variables that are
 * actually private to this source file.
 */

inline void set_main_window_id(GtkWidget * widget)
{
	main_window = widget;
}

inline GtkWidget *get_main_window_id(void)
{
	return main_window;
}

inline void set_status_bar_id(GtkWidget * widget)
{
	status_bar = widget;
}

inline GtkWidget *get_status_bar_id(void)
{
	return status_bar;
}

inline void set_menu_bar_id(GtkWidget * widget)
{
	menu_bar = widget;
}

inline GtkWidget *get_menu_bar_id(void)
{
	return menu_bar;
}

inline void set_progress_bar_id(GtkWidget * widget)
{
	progress_bar = widget;
}

inline GtkWidget *get_progress_bar_id(void)
{
	return progress_bar;
}

inline void set_button_toolbar_id(GtkWidget * widget)
{
	button_toolbar = widget;
}

inline GtkWidget *get_button_toolbar_id(void)
{
	return button_toolbar;
}

/*
 *
 *   inline void set_status_bar_message (const gchar *)
 *   
 *   Description:
 *      This routine sets the message in the main window status bar.
 * 
 *   Entry:
 *      message - the string to push onto the statusbar
 *
 *   Exit:
 *      Returns nothing
 *
 */
inline void set_status_bar_message(const gchar * message)
{
	if (status_bar != NULL)
		gtk_statusbar_push(GTK_STATUSBAR(status_bar),
				   gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), ""),
				   message);
}

/*
 *
 *   inline void set_status_bar_message (const gchar *)
 *   
 *   Description:
 *      This routine removes the current message from the
 *      main window status bar.
 * 
 *   Entry:
 *      None
 *
 *   Exit:
 *      Returns nothing
 *
 */
inline void remove_status_bar_message(void)
{
	if (status_bar != NULL)
		gtk_statusbar_pop(GTK_STATUSBAR(status_bar),
				  gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), ""));

}

/*
 *
 *   gpointer get_single_select_current_row_data (GtkCList *)
 *   
 *   Description:
 *      This routine is a wrapper call to get_clist_selection_data()
 *      to allow retrieving the user data for single selected row.
 * 
 *   Entry:
 *      clist - address of GtkCList 
 
 *   Exit:
 *      The address of the user data associated with the currently
 *      selected row in the list, otherwise NULL is returned.
 *
 */
gpointer get_single_select_current_row_data(GtkCList * clist)
{
	GSList *selection;
	gpointer data = NULL;

	selection = get_clist_selection_data(clist);

	if (selection != NULL) {
		if (g_slist_length(selection) > 0)
			data = selection->data;

		g_slist_free(selection);
	}

	return data;
}

/*
 *
 *   GSList* get_clist_selection_data (GtkCList *) 
 *   
 *   Description:
 *      This routine builds a singly linked list containing the
 *      user data corresponding to each row selected. When 
 *      building the list, we walk the selection list in reverse
 *      (last to first) in order to allow optimization in 
 *      prepending items to the head of the list rather than
 *      appending to the end.
 *
 *      NOTE: The caller should issue a g_slist_free() on the
 *            GSList when it is no longer needed. The user
 *            data in the list should be considered read-only
 *            items. 
 * 
 *   Entry:
 *      clist - address of GtkCList
 *
 *   Exit:
 *      See description.
 *
 */
GSList *get_clist_selection_data(GtkCList * clist)
{
	GSList *selection = NULL;

	if (clist != NULL) {
		if (clist->selection != NULL) {
			if (g_list_length(clist->selection) > 0) {
				gint row;
				GList *item;

				item = clist->selection_end;

				while (item != NULL) {
					row = GPOINTER_TO_INT(item->data);

					if (row != -1)
						selection = g_slist_prepend(selection,
									    gtk_clist_get_row_data
									    (clist, row));
					else
						log_error("%s: No row selected!\n", __FUNCTION__);

					item = g_list_previous(item);
				}
			} else {
				log_error("%s: No row selected?!?\n", __FUNCTION__);
			}
		} else {
			log_error("%s: clist->selection list is NULL.\n", __FUNCTION__);
		}
	} else {
		log_error("%s: clist is NULL!\n", __FUNCTION__);
	}

	return selection;
}

/*
 *
 *   void on_button_clicked_display_prev_window (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine searches the window_list for the current window
 *      then walks the list back from there to find the id of a 
 *      previous window in the list to redisplay. The current window
 *      is hidden prior to showing the previous window.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - user data (not used)
 *
 *   Exit:
 *      The current window is hidden and the previous one is displayed.
 *
 */
void on_button_clicked_display_prev_window(GtkButton * button, gpointer user_data)
{
	GList *window_list;

	window_list = get_window_list(GTK_WIDGET(button));

	if (window_list != NULL) {
		GList *link;
		GtkWidget *current_window;

		current_window = gtk_widget_get_toplevel(GTK_WIDGET(button));

		link = g_list_find(window_list, current_window);

		if (link != NULL && link->prev != NULL) {
			while (link->prev != NULL) {
				if (link->prev->data != NULL) {
					gtk_widget_show(GTK_WIDGET(link->prev->data));
					gtk_widget_hide(current_window);

					break;
				} else {
					link = link->prev;
				}
			}
		} else {
			log_warning
			    ("%s: Attempt to display window prior to first one or current window not in list.\n",
			     __FUNCTION__);
		}
	} else {
		log_error("%s: window_list was not found.\n", __FUNCTION__);
	}
}

/*
 *
 *   void on_button_clicked_destroy_window_list (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine extracts the window_list from the current window
 *      object data and proceeds to destroy all the windows in the list.
 *
 *      If there is no window list, this routine simply calls the
 *      on_button_clicked_destroy_window () routine to destroy the
 *      button's parent window only.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - user data (not used)
 *
 *   Exit:
 *      All the windows in the window_list are destroyed along with the
 *      window_list.
 *
 */
void on_button_clicked_destroy_window_list(GtkButton * button, gpointer user_data)
{
	GList *window_list = get_window_list(GTK_WIDGET(button));

	if (window_list)
		destroy_window_list(window_list);
	else
		on_button_clicked_destroy_window(button, user_data);
}

/*
 *
 *   void on_version_label_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine sets the version string in the about
 *      box to the compiled version information.
 * 
 *   Entry:
 *      widget    - the version label widget
 *      user_data - user data (not used)
 *
 *   Exit:
 *      The about label is updated with the version string.
 *
 */
void on_version_label_realize(GtkWidget * widget, gpointer user_data)
{
	gchar *version;

	version = g_strdup_printf(_("Version %s"), VERSION);

	gtk_label_set_text(GTK_LABEL(widget), version);

	g_free(version);
}

/*
 *
 *   void on_readonly_clist_select_row (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is selected on
 *      the read-only clist so that the selection gets undone.
 * 
 *   Entry:
 *      clist     - address of GtkCList object that received select-row signal
 *      row       - index of row selected
 *      column    - index of column selected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - address of user data bound to clist (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_readonly_clist_select_row(GtkCList * clist, gint row, gint column,
				  GdkEventButton * event, gpointer user_data)
{
	gtk_clist_unselect_row(clist, row, column);
}

/*
 *
 *   inline void destroy_slist (GSList *list)
 *
 *   Description:
 *      This routine calls g_free() for each data item
 *      in the list and then removes the GSList element.
 *      It continues this until the entire list is 
 *      destroyed.
 * 
 *   Entry:
 *      list - the address of head of the list
 *
 *   Exit:
 *      Returns nothing.
 *
 */
inline void destroy_slist(GSList * list)
{
	while (list) {
		g_free(list->data);
		list = g_slist_remove(list, list->data);
	}
}

/*
 *
 *   gboolean update_statusbar_message (GtkWidget *, GdkEventCrossing *, gpointer)
 *
 *   Description:
 *      This routine is called to display or remove messages
 *      on the main window status bar for menu items. 
 * 
 *   Entry:
 *      widget    - the menu item that is getting or losing the pointer focus
 *      event     - the event->type will tell whether the pointer is entering
 *                  or leaving
 *      user_data - if entering, this should be a pointer to a string with the
 *                  the status message
 *
 *   Exit:
 *      Always returns FALSE
 *
 */
gboolean update_statusbar_message(GtkWidget * widget, GdkEventCrossing * event, gpointer user_data)
{
	if (event->type == GDK_ENTER_NOTIFY)
		set_status_bar_message(user_data);
	else if (event->type == GDK_LEAVE_NOTIFY)
		remove_status_bar_message();

	return FALSE;
}

/*
 *
 *   gboolean on_menu_item_focus_event (GtkWidget *, GdkEventFocus *, gpointer)
 *
 *   Description:
 *      This routine is called to display or remove messages
 *      on the main window status bar for menu items. 
 * 
 *   Entry:
 *      widget    - the menu item that is getting or losing the keyboard focus
 *      event     - the event->in boolean will tell us whether we are getting 
 *                  or losing keyboard focus
 *      user_data - if getting focus, this should be a pointer to a string with
 *                  the the status message
 *
 *   Exit:
 *      Always returns FALSE
 *
 */
gboolean on_menu_item_focus_event(GtkWidget * widget, GdkEventFocus * event, gpointer user_data)
{
	if (event->in)
		set_status_bar_message(user_data);
	else
		remove_status_bar_message();

	return FALSE;
}

/*
 *
 *   void set_selection_window_clist_column_titles (gchar *, gchar *, gchar *)
 *
 *   Description:
 *      This routine allows setting the text for the selection window
 *      columns. Only those for a which a non-NULL string pointer is given
 *      are changed.
 * 
 *   Entry:
 *      clist           - the id of the selection window clist
 *      size_col_text   - string for size column or NULL to not change
 *      desc_col_text   - string for description/name column or NULL
 *      minmax_col_text - string for min/max size column or NULL
 *
 *   Exit:
 *      Nothing
 *
 */
void set_selection_window_clist_column_titles(GtkCList * clist, gchar * size_col_text,
					      gchar * desc_col_text, gchar * minmax_col_text)
{
	if (size_col_text)
		gtk_clist_set_column_title(clist, SL_SIZE_COLUMN, size_col_text);
	if (desc_col_text)
		gtk_clist_set_column_title(clist, SL_DESC_COLUMN, desc_col_text);
	if (minmax_col_text)
		gtk_clist_set_column_title(clist, SL_MINMAX_SIZE_COLUMN, minmax_col_text);
}

/*
 *
 *   GtkWidget* create_pixmap (GtkWidget *, const gchar *)
 *
 *   Description:
 *      This routine is a replacement for a utility routine generated by
 *      Glade. This routine, unlike the Glade routine, does not create
 *      the pixmap widget from a filename. Instead, it calls the a helper
 *      routine that creates the pixmap widget from static information
 *      already part of the program.
 * 
 *   Entry:
 *      widget - reference widget (not used)
 *      name   - pixmap filename
 *
 *   Exit:
 *      Returns a GtkPixmap widget or NULL if the name was incorrect.
 *
 */
GtkWidget *create_pixmap(GtkWidget * widget, const gchar * name)
{
	return create_pixmap_widget(name);
}

/*
 *
 *  inline void add_pixmap_label_to_button (GtkWidget *, GdkPixmap *, GdkBitmap *, 
 *                                          const gchar *, GtkAccelGroup *)
 *
 *   Description:
 *      This routine creates a horizontal box and places a pixmap in
 *      one cell and label in the other. It then places the hbox in
 *      the button provided.
 * 
 *   Entry:
 *      button - the button to contain the pixmap and label
 *      pixmap - the GDK pixmap to create the GtkPixmap from
 *      mask   - the mask parameter for gtk_pixmap_new
 *      text   - the text for the label
 *      accel_group - if not NULL, the indicates that the key
 *               from the text label should be added to the
 *               accelerator group
 *
 *   Exit:
 *      The pixmap and label are placed in the button.
 *
 */
inline void add_pixmap_label_to_button(GtkWidget * button, GdkPixmap * pixmap, GdkBitmap * mask,
				       const gchar * text, GtkAccelGroup * accel_group)
{
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *gtk_pixmap;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_pixmap = gtk_pixmap_new(pixmap, mask);

	if (accel_group) {
		guint key;

		label = gtk_label_new("");
		key = gtk_label_parse_uline(GTK_LABEL(label), text);

		gtk_widget_add_accelerator(button, "clicked", accel_group, key, GDK_NONE, GDK_NONE);
	} else
		label = gtk_label_new(text);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_pixmap, TRUE, TRUE, 0);
	gtk_misc_set_alignment(GTK_MISC(gtk_pixmap), 0.9, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);

	gtk_container_add(GTK_CONTAINER(button), hbox);

	gtk_widget_show(hbox);
	gtk_widget_show(label);
	gtk_widget_show(gtk_pixmap);
}

/*
 *
 *   inline void adjust_clist_column_width (GtkCList *, gint, gint)
 *
 *   Description:
 *      This routine allows widening or narrowing the width of a
 *      column in a GtkCList. The delta is added to the current
 *      width of the column.
 * 
 *   Entry:
 *      clist  - the address of the GtkCList containing the column
 *      column - the index of the column to effect
 *      delta  - the number of pixels to add or remove
 *
 *   Exit:
 *      The minimum width is adjust for the column based on the current width
 *      plus the delta.
 *
 */
inline void adjust_clist_column_width(GtkCList * clist, gint column, gint delta)
{
	if (column < clist->columns)
		gtk_clist_set_column_min_width(clist, column, clist->column[column].width + delta);
}

/*
 *
 *   inline void pick_clist_colors (GtkCList *, gboolean, GdkColor *, GdkColor *)
 *
 *   Description:
 *      This routine returns either the default foreground and background
 *      colors for a GtkCList row or an alternate set of colors (currently
 *      a light gray background with a black foreground).
 * 
 *   Entry:
 *      clist      - the address of the GtkCList
 *      alternate  - TRUE if the alternate color is to be returned
 *      background - address to store background color
 *      foreground - address to store foreground color
 *
 *   Exit:
 *      Either the default colors or the alternate colors are returned.
 *
 */
inline void pick_clist_colors(GtkCList * clist, gboolean alternate, GdkColor * background,
			      GdkColor * foreground)
{
	if (alternate) {
		foreground->red = 0x0000;
		foreground->green = 0x0000;
		foreground->blue = 0x0000;

		background->red = 0xf400;
		background->green = 0xf400;
		background->blue = 0xf400;
	} else {
		GtkStyle *style;

		/*
		 * Set the background and foreground for a "normal" to the same as
		 * the style as define for the clist itself.
		 */

		style = GTK_WIDGET(clist)->style;

		foreground->red = style->fg[GTK_STATE_NORMAL].red;
		foreground->green = style->fg[GTK_STATE_NORMAL].green;
		foreground->blue = style->fg[GTK_STATE_NORMAL].blue;

		background->red = style->base[GTK_STATE_NORMAL].red;
		background->green = style->base[GTK_STATE_NORMAL].green;
		background->blue = style->base[GTK_STATE_NORMAL].blue;
	}
}

/*
 *
 *   inline void set_clist_row_color (GtkCList *, gint, gboolean)
 *
 *   Description:
 *      This routine set the background and foreground color for
 *      a row in a GtkCList. The choices are default style background
 *      and foreground for normal color or light gray background
 *      with black foreground for alternate color.
 * 
 *   Entry:
 *      clist     - the address of the GtkCList
 *      row       - the row number of the row to change
 *      alternate - TRUE if the alternate color should be used
 *
 *   Exit:
 *      The row colors are changed.
 *
 */
inline void set_clist_row_color(GtkCList * clist, gint row, gboolean alternate)
{
	GdkColor foreground;
	GdkColor background;

	pick_clist_colors(clist, alternate, &background, &foreground);

	gtk_clist_set_background(clist, row, &background);
	gtk_clist_set_foreground(clist, row, &foreground);
}

/*
 *
 *   inline void set_ctree_node_color (GtkCTree *, GtkCTreeNode *, gboolean)
 *
 *   Description:
 *      This routine set the background and foreground color for
 *      a row in a GtkCList. The choices are default style background
 *      and foreground for normal color or light gray background
 *      with black foreground for alternate color.
 * 
 *   Entry:
 *      clist     - the address of the GtkCList
 *      row       - the row number of the row to change
 *      alternate - TRUE if the alternate color should be used
 *
 *   Exit:
 *      The row colors are changed.
 *
 */
inline void set_ctree_node_color(GtkCTree * ctree, GtkCTreeNode * node, gboolean alternate)
{
	GdkColor foreground;
	GdkColor background;

	pick_clist_colors(GTK_CLIST(ctree), alternate, &background, &foreground);

	gtk_ctree_node_set_background(ctree, node, &background);
	gtk_ctree_node_set_foreground(ctree, node, &foreground);
}

/*
 *
 *   inline gint clist_append_row (GtkCList *, gchar **)
 *
 *   Description:
 *      This routine is basically a wrapper function to gtk_clist_append
 *      that allows for alternating the row background depending on
 *      whether the row is even or odd.
 * 
 *   Entry:
 *      clist - the address of the GtkCList to append row to
 *      text  - the array of text strings for columns on row
 *
 *   Exit:
 *      The row is appended and the backgound is either set to a
 *      light gray with black foreground for odd numbered rows.
 *
 */
inline gint clist_append_row(GtkCList * clist, gchar ** text)
{
	gint row;

	row = gtk_clist_append(clist, text);

	if (row != -1)
		set_clist_row_color(clist, row, row % 2);

	return row;
}

/*
 *
 *   void set_all_ctree_node_colors (GtkCTree *)
 *
 *   Description:
 *      This routine walks all of the viewable nodes in a ctree
 *      and sets every other nodes color to an alternating color.
 * 
 *   Entry:
 *      ctree - the address of the GtkCTree to set alternating colors
 *
 *   Exit:
 *      All the viewable nodes in the ctree have their background colors
 *      set with alternating colors.
 *
 */
void set_all_ctree_node_colors(GtkCTree * ctree)
{
	GtkCTreeNode *node;
	gboolean alternate = FALSE;

	node = gtk_ctree_node_nth(ctree, 0);

	gtk_clist_freeze(GTK_CLIST(ctree));

	while (node != NULL) {
		set_ctree_node_color(ctree, node, alternate);
		node = GTK_CTREE_NODE_NEXT(node);
		alternate = !alternate;
	}

	gtk_clist_thaw(GTK_CLIST(ctree));
}

/*
 *
 *   inline void set_widget_foreground_color (GtkWidget *, gchar *)
 *
 *   Description:
 *      This routine sets the normal foreground color of a widget.
 *      It is typically called for label widgets.
 * 
 *   Entry:
 *      widget       - the widget to have its foreground color changed
 *      color_string - the color, e.g. "red" or "#FF0000" (see rgb.txt)
 *
 *   Exit:
 *      The foreground color for the normal state is changed
 *
 */
inline void set_widget_foreground_color(GtkWidget * widget, gchar * color_string)
{
	GdkColor color;
	GtkRcStyle *style;

	gdk_color_parse(color_string, &color);

	style = gtk_rc_style_new();

	/*
	 * Set the foreground color and indicate the color the GtkRcStyle
	 * will affect.
	 */

	style->fg[GTK_STATE_NORMAL] = color;
	style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_FG;

	gtk_widget_modify_style(widget, style);
	gtk_rc_style_unref(style);
}

/*
 *
 *  void set_main_window_title (GtkWindow *, gchar *)
 *
 *  Description:
 *     This routine sets the main window title with the system host
 *     name. If administering a remote node, we also display the
 *     node name.
 *
 *  Entry:
 *     window    - the main window id
 *     node_name - remote node name or NULL if local
 *
 *  Exit:
 *     The main window title is changed.
 *
 */
void set_main_window_title(GtkWindow * window, gchar * node_name)
{
	gchar hostname[MAX_HOSTNAME + 1];

	if (gethostname(hostname, MAX_HOSTNAME) == 0) {
		gchar *title_with_hostname;

		if (node_name != NULL)
			title_with_hostname =
			    g_strdup_printf(_("EVMS Administration Utility (console=%s) (node=%s)"),
					    hostname, node_name);
		else
			title_with_hostname =
			    g_strdup_printf(_("EVMS Administration Utility (%s)"), hostname);

		gtk_window_set_title(window, title_with_hostname);

		g_free(title_with_hostname);
	}
}

/*
 *
 *   void on_button_clicked_destroy_window_and_exit_loop (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine dismisses the parent containing the button
 *      that was clicked and calls gtk_main_quit() to exit the
 *      current event loop.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Dismisses/destroys parent window of button and exits main loop.
 *
 */
void on_button_clicked_destroy_window_and_exit_loop(GtkButton * button, gpointer user_data)
{
	on_button_clicked_destroy_window(button, user_data);
	gtk_main_quit();
}


/*
 *
 *   GtkWidget* display_results_and_exit (gint, gchar *, gchar *, GSList *, gboolean, GtkWidget *)
 *   
 *   Description:
 *      This routine display a dialog that provides feedback
 *      on the results of a certain operation. The caller
 *      can provide the window title,an optional message, 
 *      a list of windows that should be destroyed
 *      after showing the results window as well as the 
 *      actual error code to be used in pulling a descriptive
 *      message for the return code. It also exits the
 *      current event loop when the user clicks on OK button.
 * 
 *   Entry:
 *      rc            - the return code (the results of an operation)
 *      title         - optional string to use as the title
 *      message       - optional supplemental message
 *      window_list   - optional list of windows to destroy
 *      refresh_views - TRUE if we should refresh the main window views
 *      parent_window - if not NULL, the results window is made a transient of
 *                      this window
 *
 *   Exit:
 *      Popup is displayed awaiting dismissal.
 *
 */
void display_results_and_exit(gint rc, gchar * title, gchar * message, GList * window_list,
			      gboolean refresh_views, GtkWidget * parent_window)
{
	GtkWidget *window;
	GtkWidget *close_button;

	window =
	    display_results_window(rc, title, message, window_list, refresh_views, parent_window);

	close_button = gtk_object_get_data(GTK_OBJECT(window), "results_window_button");

	gtk_signal_disconnect_by_func(GTK_OBJECT(close_button),
				      GTK_SIGNAL_FUNC(on_button_clicked_destroy_window), NULL);

	gtk_signal_connect(GTK_OBJECT(close_button), "clicked",
			   on_button_clicked_destroy_window_and_exit_loop, NULL);

}
