/*
 *   (C) Copyright IBM Corp. 2001, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * evms_metadata_backup
 *
 * Simple command-line utility to open the engine and call the API to
 * backup the metadata
 */

#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <libgen.h>
#include <locale.h>
#include <frontend.h>
#include <enginestructs.h>

#define REQUIRED_ENGINE_API_MAJOR 10
#define REQUIRED_ENGINE_API_MINOR 1
#define REQUIRED_ENGINE_API_PATCH 0

#define MSG(msg, args...)         if (!quiet) { printf(msg, ## args); }
#define VERBOSE_MSG(msg, args...) if (verbose) { printf(msg, ## args); }
#define ERROR_MSG(msg, args...)   fprintf(stderr, msg, ## args)

static char * prog_name;
static debug_level_t debug_level = -1;
static char * log_file = NULL;
static char * node_name = NULL;
static boolean quiet = FALSE;
static boolean verbose = FALSE;


static boolean is_numeric(char * str)
{
	while (*str != '\0') {
		if ((*str < '0') || (*str > '9')) {
			return FALSE;
		}
		str++;
	}
	return TRUE;
}


static void show_help(void)
{
	MSG(_("\nUsage: %s [options] [directory]\n"), prog_name);
	MSG(_("Backup EVMS metadata.\n\n"

	      "options:\n"
	      "  [-d debug_level, --debug-level debug_level]\n"
	      "     debug_level = [0-9] |\n"
	      "                   [critical | serious | error | warning | default |\n"
	      "                    details | entry_exit | debug | extra | everything]\n"
	      "  [-l log_file_name, --log-file log_file_name]\n"
	      "  [-n node_name, --node-name node_name]\n"
	      "  [-q, --quiet] quiet mode\n"
	      "  [-v, --verbose] verbose mode\n"
	      "  [-h | -? | --help] display this help\n\n"

	      "directory:\n"
	      "  directory in which to save the backup file\n\n"
	
	      "EVMS metadata backup files are named \"%syyyymmddhhmmss\".\n\n"),
	       EVMS_METADATA_FILE_NAME_PREFIX);
}


static int parse_options(int argc, char ** argv)
{
	int c, rc = 0;
	char * short_opts = "d:l:n:qvh?";
	struct option long_opts[] = { { "debug-level",  required_argument, NULL, 'd'},
				      { "log-file",     required_argument, NULL, 'l'},
				      { "devnum",	required_argument, NULL, 'D'},
				      { "node-name",    required_argument, NULL, 'n'},
				      { "quiet",        no_argument,       NULL, 'q'},
				      { "verbose",      no_argument,       NULL, 'v'},
				      { "help",         no_argument,       NULL, 'h'},
				      { NULL,           0,                 NULL,   0} };

	while ((c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF) {
		switch (c) {
		case 'd':
			if (is_numeric(optarg)) {
				int level = atoi(optarg);

				if ((level < CRITICAL) ||
				    (level > EVERYTHING)) {
					ERROR_MSG(_("%s is not a valid debug level.\n"),
						  optarg);

					/* Display the help. */
					rc = EINVAL;
				}
			} else if (strcasecmp(optarg, "critical") == 0) {
				debug_level = CRITICAL;
			} else if (strcasecmp(optarg, "serious") == 0) {
				debug_level = SERIOUS;
			} else if (strcasecmp(optarg, "error") == 0) {
				debug_level = ERROR;
			} else if (strcasecmp(optarg, "warning") == 0) {
				debug_level = WARNING;
			} else if (strcasecmp(optarg, "default") == 0) {
				debug_level = DEFAULT;
			} else if (strcasecmp(optarg, "details") == 0) {
				debug_level = DETAILS;
			} else if (strcasecmp(optarg, "entry_exit") == 0) {
				debug_level = ENTRY_EXIT;
			} else if (strcasecmp(optarg, "debug") == 0) {
				debug_level = DEBUG;
			} else if (strcasecmp(optarg, "extra") == 0) {
				debug_level = EXTRA;
			} else if (strcasecmp(optarg, "everything") == 0) {
				debug_level = EVERYTHING;
			} else {
				ERROR_MSG(_("%s is not a valid debug level.\n"),
					  optarg);

				/* Display the help. */
				rc = EINVAL;
			}

			break;

		case 'l':
			log_file = strdup(optarg);
			break;

		case 'n':
			node_name = strdup(optarg);
			break;

		case 'q':
			quiet = TRUE;
			if (verbose) {
				ERROR_MSG(_("You cannot specify -v (verbose) and -q (quiet) at the same time.  "
					    "-q will override -v.\n"));
				verbose = FALSE;
			}
			break;

		case 'v':
			verbose = TRUE;
			if (quiet) {
				ERROR_MSG(_("You cannot specify -q (quiet) and -v (verbose) at the same time.  "
					    "-v will override -q.\n"));
				quiet = FALSE;
			}
			break;

		case 'h':
		case '?':
			/* Display the help. */
			rc = EINVAL;
			break;

		default:
			ERROR_MSG(_("%s -- unrecognized option \"%c\"\n\n"),
				  prog_name, c);

			/* Display the help. */
			rc = EINVAL;
			break;
		}
	}

	return rc;
}


static int message_callback(char * message_text,
			    int * answer,
			    char ** choices)
{
	MSG("%s", message_text);

	if (choices != NULL) {
		MSG(_("%s: Responding with default selection \"%s\".\n"),
		    prog_name, choices[*answer]);
	}

	return 0;
}

static void status_callback(char * message_text)
{
	VERBOSE_MSG("Engine status: %s", message_text);
}


static ui_callbacks_t callbacks = {
	user_message: message_callback,
	user_communication: NULL,
	progress: NULL,
	status: status_callback
};


int main(int argc, char **argv)
{
	evms_version_t engine_api_version;
	int rc;
	char * directory;

	setlocale(LC_MESSAGES, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	prog_name = basename(argv[0]);

	rc = parse_options(argc, argv);
	if (rc != 0) {
		show_help();
		return rc;
	}

	directory = (optind < argc) ? argv[optind] : NULL;

	evms_get_api_version(&engine_api_version);

	if ((engine_api_version.major != REQUIRED_ENGINE_API_MAJOR) ||
	    (engine_api_version.major == REQUIRED_ENGINE_API_MAJOR &&
	     engine_api_version.minor <  REQUIRED_ENGINE_API_MINOR)) {
		ERROR_MSG(_("EVMS API version mismatch.\n"));
		ERROR_MSG(_("\tEVMS provides %d.%d.%d\n"),
			  engine_api_version.major, engine_api_version.minor, engine_api_version.patchlevel);
		ERROR_MSG(_("\t%s requires %d.%d.%d\n"),
			  prog_name,
			  REQUIRED_ENGINE_API_MAJOR, REQUIRED_ENGINE_API_MINOR, REQUIRED_ENGINE_API_PATCH);
		return ELIBACC;
	}

	VERBOSE_MSG("Opening the EVMS Engine...\n");
	rc = evms_open_engine(node_name, ENGINE_READWRITE,
			      &callbacks, debug_level, log_file);
	if (rc != 0) {
		 ERROR_MSG(_("evms_open_engine() returned error code %d: %s\n"),
			   rc, evms_strerror(rc));
		return rc;
	}
	VERBOSE_MSG(_("The EVMS Engine is open.\n"));

	VERBOSE_MSG(_("Backing up the metadata...\n"));
	rc = evms_metadata_backup(directory);

	if (rc == 0) {
		MSG(_("Metadata backed up successfully.\n"));

	} else {
		ERROR_MSG(_("evms_metadata_backup() returned error code %d: %s\n"),
			  rc, evms_strerror(rc));
	}

	VERBOSE_MSG(_("Closing the EVMS Engine...\n"));
	evms_close_engine();
	VERBOSE_MSG(_("The EVMS Engine is closed.\n"));

	return rc;
}


