/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: snap_create.c
 *
 * Functions to help in the creation of snapshot volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "snapshot.h"

/* List of plugins whose volumes cannot be snapshotted. */
static char * reject_plugins[] = {
	"LocalDskMgr",
	"MDRaid1RegMgr",
	"MDRaid5RegMgr",
	NULL
};

/**
 * make_parent_and_child
 *
 * Make "parent" a parent of "child", and make "child" a child of "parent".
 **/
void make_parent_and_child(storage_object_t * parent,
			   storage_object_t * child)
{
	LOG_ENTRY();

	EngFncs->insert_thing(parent->child_objects, child, INSERT_BEFORE, NULL);
	EngFncs->insert_thing(child->parent_objects, parent, INSERT_BEFORE, NULL);

	LOG_EXIT_VOID();
}

/**
 * unmake_parent_and_child
 *
 * Opposite of make_parent_and_child().
 **/
void unmake_parent_and_child(storage_object_t * parent,
			     storage_object_t * child)
{
	LOG_ENTRY();

	if (parent && child) {
		EngFncs->remove_thing(parent->child_objects, child);
		EngFncs->remove_thing(child->parent_objects, parent);
	}

	LOG_EXIT_VOID();
}

/**
 * add_snapshot_to_origin
 *
 * Add the specified snapshot to the specified origin's list of snapshots.
 * The snapshot state should be up-to-date before calling this.
 **/
void add_snapshot_to_origin(snapshot_volume_t * snap_volume,
			    snapshot_volume_t * org_volume)
{
	LOG_ENTRY();
	LOG_DEBUG("Adding snapshot %s to origin %s.\n",
		  snap_volume->parent->name, org_volume->parent->name);

	EngFncs->insert_thing(org_volume->parent->associated_parents,
			      snap_volume->parent, INSERT_AFTER, NULL);
	EngFncs->insert_thing(snap_volume->parent->associated_children,
			      org_volume->parent, INSERT_AFTER, NULL);

	snap_volume->origin = org_volume;
	snap_volume->next = org_volume->next;
	org_volume->next = snap_volume;
	org_volume->count++;
	if (is_active(snap_volume)) {
		org_volume->active_count++;
	}

	LOG_EXIT_VOID();
}

/**
 * remove_snapshot_from_origin
 *
 * Remove the specified snapshot from its origin's list of snapshots.
 **/
void remove_snapshot_from_origin(snapshot_volume_t * snap_volume)
{
	snapshot_volume_t * org_volume = snap_volume->origin;
	snapshot_volume_t * this_volume;

	LOG_ENTRY();

	EngFncs->remove_thing(org_volume->parent->associated_parents,
			      snap_volume->parent);

	for (this_volume = org_volume;
	     this_volume; this_volume = this_volume->next) {
		if (this_volume->next == snap_volume) {
			this_volume->next = snap_volume->next;
			org_volume->count--;
			break;
		}
	}
	snap_volume->next = NULL;

	LOG_EXIT_VOID();
}

/**
 * skip_dev_path
 *
 * Given a name of a volume, return a pointer that skips over the "/dev/evms"
 * at the front of the name.
 **/
char * skip_dev_path(char * volume_name)
{
	char * name = strstr(volume_name, EVMS_DEV_NODE_PATH);
	if (name) {
		name += strlen(EVMS_DEV_NODE_PATH);
	} else {
		name = volume_name;
	}
	return name;
}

/**
 * get_origin_object_name
 *
 * Generate the name for an origin object based on the origin child object
 * name. If the object is in a disk-group, prepend the disk-group name. Then
 * simply append a "#origin#" at the end of the name.
 **/
static void get_origin_object_name(storage_object_t *child_object,
				   char * parent_name)
{
	LOG_ENTRY();

	parent_name[0] = '\0';

	if (child_object->disk_group) {
		strncat(parent_name, child_object->disk_group->name, EVMS_NAME_SIZE);
		strncat(parent_name, "/", EVMS_NAME_SIZE-strlen(parent_name));
	}

	strncat(parent_name, child_object->name, EVMS_NAME_SIZE-strlen(parent_name));
	strncat(parent_name, "#origin#", EVMS_NAME_SIZE-strlen(parent_name));

	LOG_EXIT_VOID();
}

/**
 * allocate_snapshot
 *
 * Allocate a new EVMS feature-object and all necessary private data for a
 * snapshot. The metadata must already be allocated and fully initialized,
 * and the child object must already have an initialized feature header.
 **/
snapshot_volume_t * allocate_snapshot(storage_object_t * snap_child,
				      snapshot_metadata_t * metadata)
{
	snapshot_volume_t * volume = NULL;
	storage_object_t * snap_parent = NULL;
	storage_object_t * snap_sibling = NULL;
	char name[EVMS_NAME_SIZE] = {0};
	int rc;

	LOG_ENTRY();

	/* If the snapshot is in a disk-group, prepend the disk-group name
	 * to the snapshot object name.
	 */
	if (snap_child->disk_group) {
		strncat(name, snap_child->disk_group->name, EVMS_NAME_SIZE);
		strncat(name, "/", EVMS_NAME_SIZE-strlen(name));
	}
	strncat(name, snap_child->feature_header->object_name,
		EVMS_NAME_SIZE-strlen(name));

	LOG_DEBUG("Allocating snapshot %s.\n", name);

	/* Allocate the private data and the EVMS object. */
	volume = EngFncs->engine_alloc(sizeof(*volume));
	if (!volume) {
		LOG_ERROR("Memory error allocating private data for snapshot "
			  "%s.\n", name);
		goto out;
	}

	rc = EngFncs->allocate_evms_object(name, &snap_parent);
	if (rc) {
		LOG_ERROR("Memory error allocating object for snapshot %s.\n",
			  name);
		EngFncs->engine_free(volume);
		volume = NULL;
		goto out;
	}

	snap_sibling = EngFncs->engine_alloc(sizeof(storage_object_t));
	if (!snap_sibling) {
		LOG_ERROR("Memory error allocating sibling object for snapshot "
			  "%s.\n", name);
		EngFncs->free_evms_object(snap_parent);
		EngFncs->engine_free(volume);
		volume = NULL;
		goto out;
	}

	/* Intialize the private data. */
	volume->parent = snap_parent;
	volume->child = snap_child;
	volume->sibling = snap_sibling;
	volume->metadata = metadata;
	volume->flags |= SNAPSHOT | (metadata->flags & SNAPSHOT_ROLLBACK);

	/* Initialize the object. */
	snap_parent->data_type = DATA_TYPE;
	snap_parent->plugin = my_plugin_record;
	snap_parent->flags |= SOFLAG_MUST_BE_TOP;
	snap_parent->size = metadata->origin_size;
	snap_parent->geometry = snap_child->geometry;
	snap_parent->private_data = volume;
	if (! (metadata->flags & SNAPSHOT_WRITEABLE)) {
		snap_parent->flags |= SOFLAG_READ_ONLY;
	}

	make_parent_and_child(snap_parent, snap_child);

	/* Initialize the sibling. */
	snap_sibling->object_type = EVMS_OBJECT;
	snap_sibling->data_type = DATA_TYPE;
	snap_sibling->plugin = my_plugin_record;
	snap_sibling->size = snap_child->feature_header->feature_data1_start_lsn;
	snap_sibling->disk_group = snap_parent->disk_group;
	strncpy(snap_sibling->name, name, EVMS_NAME_SIZE);
	strncat(snap_sibling->name, "#sibling#",
		EVMS_NAME_SIZE-strlen(snap_sibling->name));

out:
	LOG_EXIT_PTR(volume);
	return volume;
}

/**
 * deallocate_snapshot
 *
 * Free the memory for the specified snapshot volume. This includes the
 * metadata, the parent object, and the volume itself. The child object
 * is not freed, but the parent object is removed from its parent list.
 * The snapshot should be removed from its origin's list before calling
 * this function.
 **/
void deallocate_snapshot(snapshot_volume_t * snap_volume)
{
	LOG_ENTRY();

	if (!snap_volume) {
		goto out;
	}

	/* Free the metadata. */
	EngFncs->engine_free(snap_volume->metadata);

	EngFncs->engine_free(snap_volume->sibling);

	if (snap_volume->parent) {
		unmake_parent_and_child(snap_volume->parent,
					snap_volume->child);
		EngFncs->free_evms_object(snap_volume->parent);
	}
	
	EngFncs->engine_free(snap_volume);

out:
	LOG_EXIT_VOID();
}

/**
 * allocate_origin
 *
 * Allocate a new EVMS feature-object and all necessary private data for an
 * origin. The org_child must already point to a logical_volume.
 **/
snapshot_volume_t * allocate_origin(storage_object_t * org_child)
{
	snapshot_volume_t * org_volume = NULL;
	storage_object_t * org_parent = NULL;
	char origin_name[EVMS_NAME_SIZE + 1] = {0};
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Allocating origin %s.\n", org_child->volume->name);


	/* Allocate the private data and the EVMS object. */
	org_volume = EngFncs->engine_alloc(sizeof(*org_volume));
	if (!org_volume) {
		LOG_ERROR("Memory error allocating private data for origin "
			  "%s.\n", org_child->volume->name);
		goto out;
	}

	get_origin_object_name(org_child, origin_name);
	rc = EngFncs->allocate_evms_object(origin_name, &org_parent);
	if (rc) {
		LOG_ERROR("Memory error allocating object for origin %s.\n",
			  org_child->volume->name);
		EngFncs->engine_free(org_volume);
		org_volume = NULL;
		goto out;
	}

	/* Initialize the private data. */
	org_volume->parent = org_parent;
	org_volume->child = org_child;
	org_volume->flags |= SNAPSHOT_ORIGIN;

	/* Initialize the object. */
	org_parent->data_type = DATA_TYPE;
	org_parent->plugin = my_plugin_record;
	org_parent->flags |= SOFLAG_MUST_BE_TOP;
	org_parent->size = org_child->size;
	org_parent->volume = org_child->volume;
	org_parent->geometry = org_child->geometry;
	org_parent->private_data = org_volume;

	/* Round-down the paren't size to nearest 1k if this is a 2.4 kernel. */
	if (EngFncs->is_2_4_kernel()) {
		org_parent->size &= ~(1);
	}

	make_parent_and_child(org_parent, org_child);

	/* Add the new object as the top object in the volume stack. */
	org_parent->volume->object = org_parent;

out:
	LOG_EXIT_PTR(org_volume);
	return org_volume;
}

/**
 * deallocate_origin
 *
 * Free the memory for the specified origin volume. This includes the parent
 * object and the volume itself. The child object is not freed, but the parent
 * object is removed from its parent list, and the EVMS volume is pointed back
 * at the child object.
 **/
void deallocate_origin(snapshot_volume_t * org_volume)
{
	LOG_ENTRY();

	if (!org_volume) {
		LOG_EXIT_VOID();
		return;
	}

	/* Place the child object back as the top object in the volume stack. */
	if (org_volume->child) {
		org_volume->child->volume->object = org_volume->child;
	}

	if (org_volume->parent) {
		unmake_parent_and_child(org_volume->parent,
					org_volume->child);
		EngFncs->free_evms_object(org_volume->parent);
	}

	EngFncs->engine_free(org_volume);

	LOG_EXIT_VOID();
}

/**
 * allocate_feature_header
 *
 * Allocate and initialize a feature header for snapshotting.
 **/
evms_feature_header_t * allocate_feature_header(storage_object_t * snap_child,
						char * snap_name)
{
	evms_feature_header_t * fh;

	LOG_ENTRY();

	fh = EngFncs->engine_alloc(EVMS_FEATURE_HEADER_SECTORS *
				   EVMS_VSECTOR_SIZE);
	if (!fh) {
		LOG_ERROR("Memory error allocating feature header for "
			  "snapshot %s.\n", snap_name);
	} else {
		fh->signature = EVMS_FEATURE_HEADER_SIGNATURE;
		fh->flags |= EVMS_FEATURE_ACTIVE;
		fh->feature_id = my_plugin_record->id;
		fh->feature_data1_start_lsn = snap_child->size - 3;
		fh->feature_data1_size = 1;
		strncpy(fh->object_name, snap_name, EVMS_VOLUME_NAME_SIZE);
	}

	LOG_EXIT_PTR(fh);
	return fh;
}

/**
 * allocate_metadata
 *
 * Allocate and initialize a snapshot metadata sector.
 **/
snapshot_metadata_t * allocate_metadata(char * org_vol_name,
					u_int64_t org_size,
					u_int64_t snap_size,
					u_int32_t chunk_size,
					int writeable)
{
	snapshot_metadata_t * metadata;

	LOG_ENTRY();

	metadata = EngFncs->engine_alloc(EVMS_VSECTOR_SIZE);
	if (!metadata) {
		LOG_ERROR("Memory error allocating metadata for snapshot.\n");
		goto out;
	}

	metadata->signature = SNAPSHOT_SIGNATURE;
	metadata->version = my_plugin_record->version;
	metadata->origin_size = org_size;
	metadata->chunk_size = chunk_size;
	metadata->total_chunks = calculate_data_chunks(snap_size,
						       chunk_size);
	strncpy(metadata->origin_volume,
		org_vol_name+strlen(EVMS_DEV_NODE_PATH),
		EVMS_NAME_SIZE);
	if (writeable) {
		metadata->flags |= SNAPSHOT_WRITEABLE;
	}

out:
	LOG_EXIT_PTR(metadata);
	return metadata;
}

/**
 * calculate_data_chunks
 *
 * Calculate the number of data chunks that a snapshot-child object can hold,
 * based on the size of that object and the snapshot chunk size.
 *
 * - total_chunks is the number of full chunks on the objects, minus a chunk
 *   for the header at the beginning.
 * - exceptions_per_cow_table is the number of data chunks that each metadata
 *   chunk can address.
 * - stride is the number of chunks in one set of metadata and data.
 * - cow_tables is the number of strides that can fit on the object, plus one
 *   if part of a stride can fit at the end.
 * - data_chunks is the total number of chunks minus the number of cow tables.
 **/
u_int64_t calculate_data_chunks(u_int64_t snap_child_size,
				u_int32_t chunk_size)
{
	u_int64_t data_chunks;
	u_int64_t total_chunks;
	u_int64_t cow_tables;
	u_int32_t exceptions_per_cow_table;
	u_int32_t stride;

	LOG_ENTRY();

	total_chunks = snap_child_size / chunk_size - 1;
	exceptions_per_cow_table = EXCEPTION_TABLE_ENTRIES(chunk_size);
	stride = exceptions_per_cow_table + 1;
	cow_tables = total_chunks / stride +
		     (total_chunks % stride) ? 1 : 0;
	data_chunks = total_chunks - cow_tables;

	LOG_EXIT_INT((int)data_chunks);
	return data_chunks;
}

/**
 * verify_origin
 *
 * Only other Device-Mapper devices can be snapshotted. This is because the
 * origin device's map must be replaced with an origin map, which obviously
 * won't work for non-Device-Mapper devices. Thus, when creating a new snapshot,
 * verify that the specified origin belongs to a plugin that uses Device-Mapper.
 **/
int verify_origin(storage_object_t * org_child,
		  storage_object_t * snap_child)
{
	plugin_record_t * plugin;
	snapshot_volume_t * org_volume;
	int rc = 0, i;

	LOG_ENTRY();
	LOG_DEBUG("Verifying that %s can be used as a snapshot origin.\n",
		  org_child->name);

	/* Only create snapshots of volumes, not objects. */
	if (!org_child->volume) {
		LOG_ERROR("Only volumes can be snapshotted. Object %s is "
			  "not a volume.\n", org_child->name);
		rc = EINVAL;
		goto out;
	}

	/* Don't create snapshots of shared volumes. */
	if (org_child->flags & SOFLAG_CLUSTER_SHARED) {
		LOG_ERROR("Object %s is cluster-shared. Cannot create "
			  "snapshots of shared volumes.\n", org_child->name);
		rc = EINVAL;
		goto out;
	}

	/* The snapshot and the origin must be in the same CSM disk-group. */
	if (org_child->disk_group != snap_child->disk_group) {
		LOG_ERROR("Snapshot %s and origin %s are not in the same "
			  "disk-group.\n", snap_child->name, org_child->name);
		rc = EINVAL;
		goto out;
	}

	/* Don't allow creating snapshots of origins which already have a
	 * snapshot which is pending a rollback operation.
	 */
	if (org_child->plugin == my_plugin_record) {
		org_volume = org_child->private_data;
		if (rollback_is_pending(org_volume->next)) {
			LOG_ERROR("Origin %s has a snapshot which has a "
				  "pending roll-back operation.\n",
				  org_child->volume->name);
			rc = EINVAL;
			goto out;
		}
	}

	/* Search the list of disallowed plugins. */
	for (i = 0; reject_plugins[i]; i++) {
		rc = EngFncs->get_plugin_by_name(reject_plugins[i], &plugin);
		if (rc) {
			continue;
		}

		if (org_child->plugin == plugin) {
			LOG_ERROR("Cannot create snapshots of "
				  "volumes from the %s plugin.\n",
				  plugin->short_name);
			rc = EINVAL;
			goto out;
		}
	}
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

