/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <time.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "views.h"
#include "callbacks.h"
#include "enum.h"
#include "readable.h"
#include "logging.h"
#include "save.h"

/**
 *	append_changed_item - append a changed item to the given clist
 *	@clist: the column list
 *	@item: the pointer to a changed record structure
 *
 *	This routine appends a row pertaining to the name and reason for a
 *	storage object that has changed.
 */
inline void append_changed_item(struct clist *clist, change_record_t *item)
{
	GPtrArray *text;
	
	text = g_ptr_array_new();
	g_ptr_array_add(text, g_strdup(item->name));
	g_ptr_array_add(text, changes_flag_to_string(item->changes));
	append_item(clist, text, NULL, NULL);
	g_ptr_array_free(text, TRUE);
}

/**
 *	populate_clist_with_changed_objects - populate a clist with item that have changes pending
 *	@clist: the column list
 *
 *	This routine gets the list of storage objects that the engine indicates have changed
 *	and we place the name and reason for the change as a row in a column list.
 */
void populate_clist_with_changed_objects(struct clist *clist)
{
	int rc;
	boolean changes_pending;
	change_record_array_t *changes = NULL;
	
	rc = evms_changes_pending(&changes_pending, &changes);
	if (rc == 0) {
		if (changes_pending && changes != NULL) {
			int i;
		
			for (i = 0; i < changes->count; i++) {
				append_changed_item(clist, &(changes->changes_pending[i]));
			}
			evms_free(changes);
		}
	} else {
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
	}
}

/**
 *	details_button_activated - display a dialog window with a list of things with pending changes
 *	@item: the menu item that was activated
 *
 *	This routine displays a dialog with the list of objects that indicate they have
 *	so sort of change pending. It is presented when the user activates the "Details"
 *	button from the pending changes or save confirmation dialogs.
 */
int details_button_activated(struct menu_item *item)
{
	struct selwin *list_dialog;
	struct dialog_window *dialog;

	list_dialog = create_list_dialog(_("Objects with Pending Changes"),
				NULL,
				_("This list shows objects with changes pending"),
				2, NULL);
	dialog = (struct dialog_window *)list_dialog;

	set_clist_column_count(list_dialog->clist, 2);
	set_clist_column_info(list_dialog->clist, 0, calc_clist_column_width(list_dialog->clist, 0.60),
				0,
				CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(list_dialog->clist, 1, calc_clist_column_width(list_dialog->clist, 0.40),
				get_clist_column_end(list_dialog->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(list_dialog->clist, 0, _("Name"));
	print_clist_column_title(list_dialog->clist, 1, _("Reason"));

	populate_clist_with_changed_objects(list_dialog->clist);
	process_modal_dialog(dialog);
	
	return 0;
}

/**
 *	save_button_activated - invoked to save changes in the engine to disk
 *	@item: the save button that invoked this function
 *
 *	This routine is invoked when the user activates the Save button in 
 *	either the quit or save confirmation dialogs.
 */
int save_button_activated(struct menu_item *item)
{
	int rc, *event_rc;
	struct dialog_window *dialog = item->user_data;

	enable_message_queuing();
	rc = evms_commit_changes();
	disable_message_queuing();
	display_queued_messages();

	update_status(_("Save"), rc);
	event_rc = dialog->user_data;
	if (rc == 0) {
		/*
		 * If we have an event return code to set, set it so that the
		 * application quits the main event loop and exits nicely. 
		 * Don't bother refreshing the views as we are exiting anyways.
		 */
		if (event_rc != NULL)
			*event_rc = ESHUTDOWN;
		else
			refresh_views();
	} else if (event_rc != NULL) {
		/*
		 * Commit failed so set the error so that we don't exit the app.
		 */
		*event_rc = EIO;
	}
	dialog->status = DLG_STATUS_CLOSING;

	return 0;
}

/**
 *	cancel_save_button_activated - invoked when user does not wish to save changes
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the Cancel button (in the save confirmation dialog)
 *	or the "Quit without Saving" button (in the exit dialog) are activated.
 */
int cancel_save_button_activated(struct menu_item *item)
{
	int *event_rc;
	struct dialog_window *dialog = item->user_data;

	event_rc = dialog->user_data;
	if (event_rc != NULL)
		*event_rc = ESHUTDOWN;
	dialog->status = DLG_STATUS_CLOSING;
	return 0;
}

/**
 *	cancel_exit_button_activated - invoked when user cancels application exit
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the Cancel button in the exit dialog is activated.
 */
int cancel_exit_button_activated(struct menu_item *item)
{
	int *event_rc;
	struct dialog_window *dialog = item->user_data;

	event_rc = dialog->user_data;
	if (event_rc != NULL)
		*event_rc = 0;
	dialog->status = DLG_STATUS_CLOSING;
	return 0;
}

/**
 *	show_commit_on_exit_dialog - invoked to prompt user that changes should be saved before exiting
 *	@void
 *
 *	This routine is invoked to present the user with a dialog to allow him/her/it to save
 *	pending changes before exiting the application.
 */
int show_commit_on_exit_dialog(void)
{
	int rc;
	struct dialog_window *dialog;

	dialog = create_dialog_window(
				_("Unsaved Changes Detected"),
				NULL,
				(dialog_event_handler)NULL,
				(dialog_show_func)NULL,
				(dialog_delete_func)NULL,
				(dialog_delete_cb)NULL,
				_("_Quit without Saving"),
				(menuitem_activate_cb)cancel_save_button_activated,
				_("_Details"),
				(menuitem_activate_cb)details_button_activated,
				&rc);

	print_centered(dialog->win, getmaxy(dialog->win) / 2,
			_("Save changes before quitting?"));

	set_menu_item_sensitivity(dialog->next_button, TRUE);
	set_menu_item_activate_cb(dialog->cancel_button,
				(menuitem_activate_cb)cancel_exit_button_activated);
	dialog->next_button = pack_menu_item_at_end(dialog->menu, _("[Save]"),
				get_accelerator_char(_("[_Save]")),
				(menuitem_activate_cb)save_button_activated,
				dialog);
	set_horizontal_menu_focus(dialog->menu, dialog->next_button);
	process_modal_dialog(dialog);
	return rc;
}

/**
 *	show_save_confirmation_dialog - invoked to confirm a save operation
 *	@title: the window title
 *	@cancel_label: the label text for the cancel button
 *	@prompt: the prompt to display in the window
 *
 *	This routine is invoked to present the user a confirmation dialog for a
 *	save operation.
 */
void show_save_confirmation_dialog(char *title, char *cancel_label, char *prompt)
{
	struct dialog_window *dialog;
	dialog = create_dialog_window(
				title,
				NULL,
				(dialog_event_handler)NULL,
				(dialog_show_func)NULL,
				(dialog_delete_func)NULL,
				(dialog_delete_cb)NULL,
				cancel_label,
				(menuitem_activate_cb)cancel_save_button_activated,
				_("_Details"),
				(menuitem_activate_cb)details_button_activated,
				NULL);

	print_centered(dialog->win, getmaxy(dialog->win) / 2, prompt);

	set_menu_item_sensitivity(dialog->next_button, TRUE);
	set_menu_item_visibility(dialog->cancel_button, FALSE);
	dialog->next_button = pack_menu_item_at_end(dialog->menu, _("[Save]"),
				get_accelerator_char(_("[_Save]")),
				(menuitem_activate_cb)save_button_activated,
				dialog);
	set_horizontal_menu_focus(dialog->menu, dialog->next_button);
	process_modal_dialog(dialog);
}

/**
 *	on_quit_menuitem_activated - callback invoked when user has selected to quit app
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Actions->Quit menu item to attempt to save
 *	any pending changes.
 */
int on_quit_menuitem_activated(struct menu_item *item)
{
	int rc;
	boolean changes_pending;

	rc = evms_changes_pending(&changes_pending, NULL);
	if (rc == 0) {
		if (changes_pending) 
			rc = show_commit_on_exit_dialog();
		else
			rc = ESHUTDOWN;
	} else {
		show_message_dialog(_("Error"), _("Unable to determine if changes are pending!"));
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
		rc = 0;
	}

	return rc;
}

/**
 *	on_save_menuitem_activated - callback innvoked when user has selected to save changes
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Actions->Save menu item to attempt to save
 *	any pending changes.
 */
int on_save_menuitem_activated(struct menu_item *item)
{
	int rc;
	boolean changes_pending;

	rc = evms_changes_pending(&changes_pending, NULL);
	if (rc == 0) {
		if (changes_pending)
			show_save_confirmation_dialog(_("Save Changes Confirmation"),
							_("_Cancel"),
							_("Do you wish to save changes now or cancel?"));
		else
			print_statusbar_text("There are currently no changes to save.");
	} else {
		show_message_dialog(_("Error"), _("Unable to determine if changes are pending!"));
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
	}

	return 0;
}
