/*
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: md multipath region mgr
 * File: multipath.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <glob.h>
#include <sys/wait.h>
#include <plugin.h>

#include "md.h"
#include "multipath.h"

#define my_plugin_record mp_plugin

#define DEBUG_DAEMON		1

#define LOCK_FILE_PREFIX	"/var/lock/evms-mpathd-"
#define SLASH_REPLACEMENT	'|'
#define MP_SELECTOR		"round-robin"

static void remove_slashes(char *string)
{
	for (; *string; string++)
		if (*string == '/') *string = SLASH_REPLACEMENT;
}

static void mp_get_lock_file_name(md_volume_t *volume,
				  char *lock_file_name)
{
	char region_name[EVMS_NAME_SIZE];

	LOG_ENTRY();

	strncpy(region_name, volume->region->name, EVMS_NAME_SIZE);
	remove_slashes(region_name);
	snprintf(lock_file_name, 256, "%s%s", LOCK_FILE_PREFIX, region_name);

	LOG_EXIT_VOID();
}

/**
 * mp_stop_daemon
 *
 * Stop the evms_mpathd instance that's monitoring this region.
 **/
static int mp_stop_daemon(md_volume_t *volume)
{
	char lock_file_name[256];
	int rc = 0;

	LOG_ENTRY();

	if (volume->daemon_pid > 0) {
		LOG_DEBUG("Sending SIGTERM to process %d\n", volume->daemon_pid);
		rc = kill(volume->daemon_pid, SIGTERM);
		if (!rc) {
			LOG_DEBUG("Waiting for process %d to terminate.\n",
				  volume->daemon_pid);
			volume->daemon_pid = 0;

			mp_get_lock_file_name(volume, lock_file_name);
			unlink(lock_file_name);
		} else {
			rc = errno;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * mp_start_daemon
 *
 * Start an instance of evms_mpathd to monitor this region.
 **/
static int mp_start_daemon(md_volume_t *volume)
{
	storage_object_t *child;
	list_element_t itr;
	char **argv = NULL;
	char object_size[20];
	char command[256];
	pid_t pid;
	int status;
	int argc = 0, sz = 0;
	int i = 0, rc = 0;

	LOG_ENTRY();

	if (volume->daemon_pid > 0) {
		goto out;
	}

	argc = 5 + EngFncs->list_count(volume->region->child_objects);
	argv = EngFncs->engine_alloc(argc * sizeof(char *));
	if (!argv) {
		rc = ENOMEM;
		goto out;
	}

	snprintf(object_size, 20, "%"PRIu64, volume->region->size);
	argv[i++] = "evms_mpathd";
#if DEBUG_DAEMON
	argv[i++] = "-d";
#endif
	argv[i++] = volume->region->name;
	argv[i++] = object_size;
	
	LIST_FOR_EACH(volume->region->child_objects, itr, child) {
		argv[i++] = child->name;
	}

	for (i = 0; i < argc; i++)
		if (argv[i])
			sz += snprintf(command + sz, 256 - sz, "%s ", argv[i]);
	LOG_DEBUG("Starting daemon process: %s\n", command);

	pid = EngFncs->fork_and_execvp(NULL, argv, NULL, NULL, NULL);
	if (pid < 0) {
		rc = errno;
		goto out;
	}

	LOG_DEBUG("Started daemon as process %d\n", pid);
	waitpid(pid, &status, WNOHANG);
	volume->daemon_pid = pid;

out:
	EngFncs->engine_free(argv);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: mp_check_daemon
 *
 * Check to see if an instance of evms_mpathd is monitoring this region. We
 * do this by trying to lock the appropriate lock-file. If we get the lock,
 * the daemon is not running. If we can't get the lock, then we can get the
 * pid of the daemon that has the lock. We can then use this pid to stop the
 * daemon if necessary.
 */
static int mp_check_daemon(md_volume_t *volume)
{
	struct flock lockinfo;
	char lock_file_name[256];
	int lock_fd, rc;

	LOG_ENTRY();

	mp_get_lock_file_name(volume, lock_file_name);

	lock_fd = open(lock_file_name, O_RDWR | O_CREAT, 0660);
	if (lock_fd < 0) {
		rc = errno;
		goto out;
	}

	lockinfo.l_whence = SEEK_SET;
	lockinfo.l_start = 0;
	lockinfo.l_len = 0;
	lockinfo.l_type = F_WRLCK;

	LOG_DEBUG("Attempting to lock file %s\n", lock_file_name);
	rc = fcntl(lock_fd, F_SETLK, &lockinfo);
	if (rc) {
		/* File is locked. Get the lock-holder's pid. */
		rc = fcntl(lock_fd, F_GETLK, &lockinfo);
		if (rc) {
			rc = errno;
			goto out;
		}

		if (lockinfo.l_type != F_UNLCK) {
			LOG_DEBUG("File %s is locked by process %d\n",
				  lock_file_name, lockinfo.l_pid);
			volume->daemon_pid = lockinfo.l_pid;
		}
	} else {
		/* File was not locked. Unlock it so the daemon can start. */
		LOG_DEBUG("File %s is not locked. Need to start daemon.\n",
			  lock_file_name);
		lockinfo.l_type = F_UNLCK;
		rc = fcntl(lock_fd, F_SETLK, &lockinfo);
		if (rc) {
			rc = errno;
			goto out;
		}

		volume->region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

out:
	if (lock_fd > 0) {
		close(lock_fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * filter_discovered_regions
 *
 * Search the list of evms_mpathd lock files, and filter out any files for
 * which we discovered a multipath region. All remaining files represent
 * stale daemons.
 */
static void filter_discovered_regions(glob_t *result)
{
	md_volume_t *volume;
	char lock_file_name[256];
	int i, rc;

	LOG_ENTRY();

	for (volume = volume_list_head; volume; volume = volume->next) {
		if (volume->personality != MULTIPATH) {
			continue;
		}

		mp_get_lock_file_name(volume, lock_file_name);

		for (i = 0; i < result->gl_pathc; i++) {
			rc = strncmp(lock_file_name, result->gl_pathv[i], 256);
			if (!rc) {
				*(result->gl_pathv[i]) = '\0';
				break;
			}
		}
	}

	LOG_EXIT_VOID();
}

/*
 * cleanup_stale_daemon
 *
 * Open and attempt to lock the specified lock file. If the file is locked,
 * get the PID of that daemon instance and kill the daemon.
 */
static void cleanup_stale_daemon(char *lock_file_name)
{
	struct flock lockinfo;
	int lock_fd, rc;

	LOG_ENTRY();

	lock_fd = open(lock_file_name, O_RDWR | O_CREAT, 0660);
	if (lock_fd < 0) {
		goto out;
	}

	lockinfo.l_whence = SEEK_SET;
	lockinfo.l_start = 0;
	lockinfo.l_len = 0;
	lockinfo.l_type = F_WRLCK;

	LOG_DEBUG("Attempting to lock file %s\n", lock_file_name);
	rc = fcntl(lock_fd, F_SETLK, &lockinfo);
	if (rc) {
		/* File is locked. Get the lock-holder's pid. */
		rc = fcntl(lock_fd, F_GETLK, &lockinfo);
		if (rc) {
			close(lock_fd);
			goto out;
		}

		if (lockinfo.l_type != F_UNLCK) {
			LOG_DEBUG("File %s is locked by process %d\n",
				  lock_file_name, lockinfo.l_pid);

			kill(lockinfo.l_pid, SIGTERM);
		}
	} else {
		/* File was not locked. */
		LOG_DEBUG("File %s is not locked.\n", lock_file_name);
		lockinfo.l_type = F_UNLCK;
		fcntl(lock_fd, F_SETLK, &lockinfo);
	}

	close(lock_fd);
	unlink(lock_file_name);

out:
	LOG_EXIT_VOID();
}

static int globerror(const char *path, int error)
{
	return 0;
}

/*
 * mp_cleanup_stale_daemons
 *
 * Look for daemon lock-files for which we didn't discover a corresponding
 * multipath region. Use the lock-file to get the PID and stop the daemons
 * to force them to release any devices they have open.
 */
static void mp_cleanup_stale_daemons(void)
{
	glob_t result;
	int i, rc;

	LOG_ENTRY();

	/* Get a list of all evms_mpathd lock-files. */
	rc = glob(LOCK_FILE_PREFIX"md|*", 0, globerror, &result);
	if (rc) {
		goto out;
	}

	filter_discovered_regions(&result);

	/* Cleanup all daemons that don't have a corresponding region. */
	for (i = 0; i < result.gl_pathc; i++) {
		if (*(result.gl_pathv[i]) != '\0') {
			cleanup_stale_daemon(result.gl_pathv[i]);
		}
	}

	globfree(&result);

out:
	LOG_EXIT_VOID();
}

/*
 * Function: mp_compare_targets
 *
 * Compare the MD-multipath volume with the target list from the kernel.
 *
 *  Returns:  0  ... if they match
 *            EINVAL if they dont match ... results in the NEEDS_ACTIVATE flag being set
 */
static int mp_compare_targets(md_volume_t *vol,
			      dm_target_t *kernel_target)
{
	dm_target_multipath_t *kernel_mp = kernel_target->data.multipath;
	dm_priority_group_t *kernel_pg = &kernel_mp->group[0];
	int found, matched_paths = 0, active_paths = 0;
	int j, rc = EINVAL;
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();

	/* All basic target information must match. */
	if (kernel_target->start != 0 ||
	    kernel_target->length != vol->region->size ||
	    kernel_target->next != NULL) {
		goto out;
	}

	/* Number of groups must match. */
	if (kernel_mp->num_groups != 1) {
		goto out;
	}

	/* Priority-group information must match. The number of paths in the
	 * kernel must be equal to *or greater than* the number found by the
	 * MD plugin. If a path is currently down, EVMS won't discover it, but
	 * that doesn't mean it doesn't exist. :)
	 */
	if (strncmp(kernel_pg->selector, MP_SELECTOR, DM_SELECTOR_NAME_SIZE) != 0 ||
	    kernel_pg->num_paths < vol->nr_disks) {
		goto out;
	}

	/* Make sure each path found by MD is present and active in the
	 * kernel mapping.
	 */
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj) {
			found = FALSE;
			for (j = 0; j < kernel_pg->num_paths; j++) {
				if (kernel_pg->path[j].device.major == member->obj->dev_major &&
				    kernel_pg->path[j].device.minor == member->obj->dev_minor) {
					found = TRUE;
					matched_paths++;
					if (!kernel_pg->path[j].has_failed) {
						active_paths++;
					}
				}
			}
			if (!found) {
				goto out;
			}
		}
	}

	/* Comparison passes if kernel has same number of
	 * (or more) active paths as MD found.
	 */
	if (active_paths >= vol->nr_disks) {
		rc = 0;
	}

out:					
	LOG_DEBUG("there are %d active paths and %d matched path structs\n", active_paths, matched_paths);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: mp_update_targets_info
 *
 * Run through the paths in the MP target and update with status from the
 * info string.
 */
static void mp_update_targets_info(dm_target_t *targets, char *info)
{
	dm_target_multipath_t *mp = targets->data.multipath;
	dm_priority_group_t *pg = &mp->group[0];
	char device[25];
	char *dev, active;
	int i, fail_count;

	LOG_ENTRY();

	for (i = 0; i < pg->num_paths; i++) {
		/* Find major:minor in info string. */
		snprintf(device, 25, "%u:%u",
			 pg->path[i].device.major,
			 pg->path[i].device.minor);
		dev = strstr(info, device);
		if (dev) {
			sscanf(dev, "%*u:%*u %c %u", &active, &fail_count);
			if (!(active == 'A' || active == 'a')) {
				pg->path[i].has_failed = TRUE;
			}
			pg->path[i].fail_count = fail_count;
  		}
	}

	LOG_EXIT_VOID();
}

/*
 * mp_identify_backup_paths
 *
 * Check the EVMS config file to see which (if any) of this volume's child
 * objects should be treated as a backup path instead of an active path.
 */
static int mp_identify_backup_paths(md_volume_t *vol)
{
	const char * const * backups = NULL;
	int *backup_array = vol->private_data;
	int count = 0, rc, j;
	char key[256];
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();

	snprintf(key, 256, "multipath.%s", vol->name);

	rc = EngFncs->get_config_string_array(key, &count, &backups);
	if (rc || !count) {
		goto out;
	}

	/* Check each MD child to see if it's in the "backups" list. */
	LIST_FOR_EACH(vol->members, iter, member) {
		if (!member->obj) {
			continue;
		}
		for (j = 0; j < count; j++) {
			rc = strncmp(member->obj->name,
				     backups[j], EVMS_NAME_SIZE);
			if (!rc) {
				backup_array[member->dev_number] = TRUE;
				break;
			}
		}
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Function: mp_update_status
 *
 *  Called to test if the region has an active device mapper
 *  node in the kernel and set the object info accordingly.
 */
static int mp_update_status(md_volume_t *volume)
{
	dm_target_t *kernel_targets = NULL;
	char *info = NULL;
	int rc;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	rc = EngFncs->dm_update_status(volume->region);
	if (rc) {
		goto out;
	}

	if (!(volume->region->flags & SOFLAG_ACTIVE)) {
		rc = ENODEV;
		goto out;
	}

	rc = EngFncs->dm_get_targets(volume->region, &kernel_targets);
	if (rc) {
		goto out;
	}

	rc = EngFncs->dm_get_info(volume->region, &info);
	if (rc) {
		goto out;
	}

	mp_update_targets_info(kernel_targets, info);

	rc = mp_compare_targets(volume, kernel_targets);

out:
	if (rc) {
		volume->region->flags |= SOFLAG_NEEDS_ACTIVATE;
		volume->flags |= MD_MP_ACTIVATE_REGION;
	}
	EngFncs->dm_deallocate_targets(kernel_targets);
	EngFncs->engine_free(info);
	LOG_EXIT_INT(rc);
	return rc;
}

static void display_volume_info( md_volume_t * vol )
{
	int count=0;
	md_member_t *member;
	list_element_t iter;
	md_super_info_t info;
	mdu_disk_info_t d;

	my_plugin = mp_plugin;

	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj) {
			count++;
		}
	}

	md_volume_get_super_info(vol, &info);
	LOG_DEBUG("Volume ... %s\n", vol->name );
	if (vol->region)
		LOG_DEBUG("              region name: %s\n", vol->region->name);
	else
		LOG_DEBUG("              region name: n/a\n");
	LOG_DEBUG("                  nr disks: %d\n", vol->nr_disks);
	LOG_DEBUG("        child object count: %d\n", count);
	LOG_DEBUG("      o               flags: 0x%X\n", vol->flags);
	LOG_DEBUG("SuperBlock ...\n");
	LOG_DEBUG("                  nr disks: %d\n", info.nr_disks);
	LOG_DEBUG("                     state: 0x%X\n", info.state_flags);
	LOG_DEBUG("              active disks: %d\n", info.active_disks);
	LOG_DEBUG("             working disks: %d\n", info.working_disks);
	LOG_DEBUG("              failed disks: %d\n", info.failed_disks);
	LOG_DEBUG("               spare disks: %d\n", info.spare_disks);

	LIST_FOR_EACH(vol->members, iter, member) {
		vol->sb_func->get_sb_disk_info(member, &d);
		LOG_DEBUG("                  disk[%02d]: maj= %d  min= %d  number= %d  raid_number= %d\n",
			  member->dev_number, d.major, d.minor, d.number, d.raid_disk);
	}
}


/*
 *  Function:  Verify a MULTIPATH Volume as described by the MD superblock
 *
 *  Perform validation ...displays error messages if do_msg==TRUE
 *
 *  returns:   0 ... if Ok
 *             EINVAL ... if multipath superblock does not match the current
 *                    	  multipath configuration seen by the engine
 */
static int  multipath_verify_sb_info(md_volume_t * vol, boolean do_msg)
{
	int  i, nr_disks, rc = 0;
	int  raid_disks = 0, spare_disks = 0, working_disks=0, active_disks=0, failed_disks=0;
	md_super_info_t info;
	md_member_t *member;
	mdu_disk_info_t d;
	list_element_t iter;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	// display debug info
	display_volume_info(vol);

	md_volume_get_super_info(vol, &info);

	// see if the superblock and volume agree on the number of disks
	if (vol->nr_disks != info.nr_disks) {
		rc = EINVAL;
		if (do_msg) {
			MESSAGE( _("Region %s says there should be %d disks but the number actually found is %d."),
				 vol->name, info.nr_disks, vol->nr_disks );
		}
	}

	// get maximum number of disks described by volume
	nr_disks = VOL_MAX_NR_DISKS(vol, (&info));

	// walk through the superblock disk descriptors, looking for
	// missing child storage objects or missing superblock disk
	// descriptors.
	for (i=0; i < nr_disks; i++ ) {

		member = md_volume_find_member(vol, i);

		if ( member && member->obj ) {
			if (i >= info.nr_disks) {
				rc = EINVAL;
				if (do_msg) {
					MESSAGE( _("Region %s has too few disk discriptors.  "
						   "Could not find a disk descriptor at index %d for object %s.\n"),
						 vol->name, i, member->obj->name );
				}
			} else {
				vol->sb_func->get_sb_disk_info(member, &d);
				if ( d.number != i || d.raid_disk != i ) {
					rc = EINVAL;
					if (do_msg) {
						MESSAGE( _("Region %s has a disk descriptor at index %d"
							   " with a raid number that is not valid. "
							   "Superblock says %d but it should be %d\n"),
							 vol->name, i, info.raid_disks, i );
					}
				}
			}

		} else {
			rc = EINVAL;
			if (do_msg) {
				MESSAGE( _("Region %s has a missing child object at index %d.\n"),
					 vol->name, i );
			}
		}

	}

	// calculate the superblock disk count fields
	LIST_FOR_EACH(vol->members, iter, member) {
		vol->sb_func->get_sb_disk_info(member, &d);

		switch (d.state) {
		
		case (1<<MD_DISK_ACTIVE | 1<<MD_DISK_SYNC):
			active_disks++;
			raid_disks++;
			working_disks++;
			break;

			// active, but not sync, kernel just kind of ignores these drives
			// so make him a spare so that the kernel will re-sync if needed.
			// or sync, but not active, do the same.
		case (1<<MD_DISK_ACTIVE):
			rc = EINVAL;
			if (do_msg) {
				MESSAGE( _("Region %s says the disk found at index %d is ACTIVE but NOT in SYNC.  "
					   "The object with this state that is not valid is %s.\n"),
					 vol->name, i, member->obj ? member->obj->name : "NOT KNOWN");
			}
		case (1<<MD_DISK_SYNC):
		case 0:	// 0 = spare
			spare_disks++;
			working_disks++;
			break;

		case (1<<MD_DISK_REMOVED):
		case (1<<MD_DISK_FAULTY):
		case (1<<MD_DISK_FAULTY | 1<<MD_DISK_REMOVED):
		default:
			rc = EINVAL;
			if (do_msg) {
				if (member->obj) {
					MESSAGE( _("Region %s says that a faulty disk is found at index %d.  "
						   "The faulty object is %s.\n"),
						 vol->name, member->dev_number, member->obj->name );
				} else {
					MESSAGE( _("Region %s says that a faulty disk is found at index %d.  "
						   "The faulty object is unknown.\n"),
						 vol->name, member->dev_number );
				}
			}
			failed_disks++;
			break;
		}
	}

	// validate the superblock disk counts
	if ( info.active_disks   != active_disks  ||
	     info.working_disks != working_disks ||
	     info.failed_disks  != failed_disks  ||
	     info.spare_disks   != spare_disks ) {
		rc = EINVAL;
		if (do_msg) {
			MESSAGE( _("Region %s has incorrect (actual, working, faulty, or stale) disk counts.\n"),
				 vol->name);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: multipath_create_region
 *
 *  Create an MD region that consumes the storage objects
 *  collected by the volume struct.
 */
static int multipath_create_region(md_volume_t * vol, list_anchor_t output_list, boolean final_call)
{
	int rc=0;
	storage_object_t * region;
	storage_object_t * child;
	list_element_t iter;
	int *backup_array;
	md_super_info_t info;
	md_member_t *member;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(vol != NULL);

	LOG_DEBUG("Checking volume %s.\n", vol->name);

	if (!vol->sb) {
		LOG_MD_BUG();
		LOG_EXIT_INT(0);
		return 0;
	}

	md_volume_get_super_info(vol, &info);

	if ((vol->nr_disks != info.nr_disks) && !final_call) {
		LOG_DETAILS("Region is still missing members, delaying discovery\n");
		LOG_EXIT_INT(0);
		return 0;
	}

	LOG_DEBUG("Allocating region storage object\n");

	rc = EngFncs->allocate_region(vol->name, &region);
	if (rc) {
		LOG_WARNING("Region %s is already created, try new name.\n", vol->name);
		rc = md_volume_get_alternative_name(vol, 255);
		if (!rc) {
			LOG_WARNING("Trying tnew region name: %s...\n", vol->name);
			rc = EngFncs->allocate_region(vol->name, &region);
			if (!rc) {
				LOG_WARNING("OK. got it.\n");
			} else {
				LOG_ERROR("Give up.\n");
			}
		}
	}

	if (rc) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Array to keep track of which child objects are backup paths. */
	backup_array = EngFncs->engine_alloc(MAX_DISKS(vol) * sizeof(int));
	if (!backup_array) {
		LOG_ERROR("error, engine_alloc failed.\n");
		EngFncs->free_region(region);
		LOG_EXIT_INT(rc);
		return rc;
	}
	vol->private_data = backup_array;

	LOG_DEBUG("Region consuming child objects\n");
	LIST_FOR_EACH(vol->members, iter, member) {
		LOG_DEBUG("Adding object %s to %s region\n",
			  member->obj->name, vol->name);
		md_append_region_to_object(region, member->obj);
	}

	region->data_type    = DATA_TYPE;
	region->plugin       = mp_plugin;
	region->private_data = (void *)vol;
	region->dev_major    = MD_MAJOR;
	region->dev_minor    = vol->md_minor;
	region->size         = info.size;
	vol->region       = region;

	LOG_DEBUG("Add region to discovery output list\n");

	rc = md_add_object_to_list(region, output_list);

	if (rc) {
		LIST_FOR_EACH( region->child_objects, iter, child ) {
			md_remove_region_from_object(region,child);
		}
		EngFncs->free_region(region);
	} else {
		vol->flags  |= MD_DISCOVERED;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: discover_regions
 *
 *  Run the global list of MD volume structs and find those
 *  that are owned by the MULTIPATH personality that have
 *  not been through discovery yet.
 */
static int multipath_discover_regions( list_anchor_t output_list, int *count, boolean final_call )
{
	int rc = 0;
	md_volume_t * volume = volume_list_head;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	while (volume != NULL) {

		LOG_DEBUG("MD VOLUME ...\n");
		LOG_DEBUG("        name: %s\n", volume->name);
		LOG_DEBUG("  discovered: %d (Y/N)\n", (volume->flags & MD_DISCOVERED));
		LOG_DEBUG(" personality: %d \n", volume->personality);
		LOG_DEBUG("  final call: %d (Y/N)\n", final_call);

		if ( !(volume->flags & MD_DISCOVERED) ) {

			if ( volume->personality == MULTIPATH ) {

				rc = multipath_create_region(volume, output_list, final_call);

				if (volume->flags & MD_DISCOVERED) {

					multipath_verify_sb_info(volume, TRUE);

					mp_identify_backup_paths(volume);
					mp_update_status(volume);
					mp_check_daemon(volume);

					LOG_DEBUG("Success ... discovered MP volume\n");

					*count += 1;
				}

			}

		}

		volume = volume->next;
	}

	if (final_call) {
		md_display_corrupt_messages(MULTIPATH);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: multipath_setup_evms_plugin
 *
 *  This function gets called shortly after the plugin is loaded by the
 *  Engine. It performs all tasks that are necessary before the initial
 *  discovery pass.
 */
static int multipath_setup_evms_plugin(engine_functions_t  *functions)
{
	int rc = 0;

	// Parameter check
	REQUIRE(functions);

	EngFncs = functions;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	rc = md_register_name_space();
	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function:  free_multipath_region
 *
 *  Only called by cleanup() to walk the multipath
 *  region dlist and free storage for each region.
 */
static int free_region ( storage_object_t *region )
{
	md_volume_t *vol;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(region->private_data != NULL);

	vol = (md_volume_t *)region->private_data;

	md_free_volume(vol);
	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Function: multipath_plugin_cleanup
 *
 *  This is our EVMS cleanup API which receives
 *  notification when the engine is going to close.
 */
static void multipath_plugin_cleanup(void)
{
	int rc;
	list_anchor_t regions_list;
	storage_object_t *region;
	list_element_t iter;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	rc = EngFncs->get_object_list(REGION, DATA_TYPE, mp_plugin, NULL, 0, &regions_list);
	if (rc==0) {

		LIST_FOR_EACH( regions_list, iter, region ){
			free_region(region);
		}

		EngFncs->destroy_list( regions_list );
	}

	LOG_EXIT_VOID();
}



/* Function: multipath_can_delete
 *
 *	Can we remove the specified MD logical volume, and consolidate the
 *	space with the freespace volume?
 */
static int multipath_can_delete( storage_object_t  *region )
{
	my_plugin = mp_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: multipath_discover
 *
 *	Examine all disk segments and find MD PVs. Assemble volume groups
 *	and export all MD logical volumes as EVMS regions.
 *
 *	All newly created regions must be added to the output list, and all
 *	segments from the input list must either be claimed or moved to the
 *	output list.
 */
static int multipath_discover( list_anchor_t  input_list,
			       list_anchor_t  output_list,
			       boolean  final_call )
{
	int count = 0;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	if ( input_list != NULL && output_list != NULL ) {
		md_discover_volumes(input_list, output_list);
		multipath_discover_regions(output_list, &count, final_call);
	}

	if (final_call) {
		mp_cleanup_stale_daemons();
	}

	LOG_EXIT_INT(count);
	return count;
}


static int multipath_create_new_region(md_volume_t * vol, list_anchor_t output_list)
{
	int rc=0;
	md_member_t *member;
	list_element_t iter;
	storage_object_t *region;
	int *backup_array = NULL;

	LOG_ENTRY();

	/* Array to keep track of which child objects are backup paths. */
	backup_array = EngFncs->engine_alloc(MAX_DISKS(vol) * sizeof(int));
	if (!backup_array) {
		rc = ENOMEM;
		goto out;
	}
	vol->private_data = backup_array;

	rc = EngFncs->allocate_region(vol->name, &region);
	if (rc) {
		LOG_ERROR("Region %s is already created (rc=%d).\n",
			  vol->name, rc);
		goto error_out;
	}

	if (!rc) {
		LOG_DEBUG("Creating new region %s: nr_disks=%d, raid_disks=%d,"
			  " spares=%d, actives=%d, working=%d\n",
			  vol->name, vol->nr_disks, vol->raid_disks,
			  vol->spare_disks, vol->active_disks, vol->working_disks);
	

		LIST_FOR_EACH(vol->members, iter, member) {
			md_append_region_to_object(region, member->obj);
		}
		region->size = md_volume_calc_size(vol);
		region->data_type = DATA_TYPE;
		region->plugin = mp_plugin;
		region->private_data = (void *)vol;
		region->dev_major = MD_MAJOR;
		region->dev_minor = vol->md_minor;
		vol->region = region;
		region->flags |= SOFLAG_DIRTY;
		md_add_object_to_list(region, output_list);
		
		mp_identify_backup_paths(vol);
	}
error_out:
	if (rc && backup_array) {
		EngFncs->engine_free(backup_array);
	}
out:	
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Function: multipath_create
 *
 *  Create a new MD volume using the DLIST of acceptable
 *  storage objects, placing the new region in the new region
 *  DLIST.
 */
static int multipath_create( list_anchor_t       objects,
			     option_array_t     *options,
			     list_anchor_t       new_region_list )
{
	int rc = 0;
	md_volume_t * vol;
	storage_object_t * object;
	md_member_t *member;
	list_element_t iter1;
	list_element_t iter2;
	md_sb_ver_t sb_ver = {MD_SB_VER_0, 90, 0};
	u_int64_t size = -1;
	int raid_disk;


	my_plugin = mp_plugin;
	LOG_ENTRY();

	if (!(vol = md_allocate_volume())) {
		LOG_CRITICAL("Memory error new volume structure.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	rc = md_volume_get_available_name(vol, 256);
	if (rc) {
		goto error_free;
	}

	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		if (size == -1) {
			size = md_object_usable_size(object, &sb_ver, 0);
		} else {
			if (size != md_object_usable_size(object, &sb_ver, 0)) {
				LOG_ERROR("Current size: %"PRIu64", data size of %s: %"PRIu64".\n",
					  size, object->name, md_object_usable_size(object, &sb_ver, 0));
				goto error_free;
			}
		}
	}

	rc = md_init_sb(vol, &sb_ver, -4, 0, size, 0);
	if (rc) {
		goto error_free;
	}

	// Add raid members
	raid_disk = 0;
	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		member = md_allocate_member(object);
		if (member) {
			// This will add the member and update the MD superblock.
			member->dev_number = raid_disk;
			member->raid_disk = raid_disk;
			member->data_offset = 0;
			member->data_size = size;
			member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC);
			raid_disk++;
			rc = md_volume_add_new_member(vol, member);
			if (rc) {
				md_free_member(member);
				goto error_free;
			}
		} else {
			rc = ENOMEM;
		}
		if (rc) {
			goto error_free;
		}
		EngFncs->delete_element(iter1);
	}


	rc = multipath_create_new_region(vol, new_region_list);
	if (rc) {
		goto error_free;
	}

	vol->flags |= MD_MP_ACTIVATE_REGION;

	rc = mp_check_daemon(vol);
	if (rc) {
		goto error_free;
	}

	LOG_EXIT_INT(rc);
	return rc;

error_free:
	md_free_volume(vol);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: w_delete
 *
 *  Worker function for multipath_delete and multipath_discard
 */
static int w_delete(storage_object_t *region, list_anchor_t children, boolean tear_down)
{
	md_volume_t     * volume;
	int             rc = 0;

	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(region->private_data != NULL);

	volume = region->private_data;

	// Revove the "backups" array.
	if (volume->private_data) {
		EngFncs->engine_free(volume->private_data);
	}

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	// Delete the volume.
	md_delete_volume(volume, tear_down);

	// Free region itself
	EngFncs->free_region(region);

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Function: multipath_delete
 *
 *  Called to delete a MULTIPATH MD region.
 */
static int multipath_delete(storage_object_t *region, list_anchor_t children)
{
	int rc;

	LOG_ENTRY();
	rc = w_delete(region, children, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: multipath_discard
 *
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the regions.
 */
static int multipath_discard(list_anchor_t regions)
{
	storage_object_t * region;
	list_element_t le;

	LOG_ENTRY();

	LIST_FOR_EACH(regions, le, region) {
		w_delete(region, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Function: multipath_add_sectors_to_kill_list
 *
 *  Called by higher EVMS layers to write kill sectors.  All we
 *  need to do is to verify that the region is Ok and then pass
 *  the request any working i/o path.  Continue trying paths till
 *  we succeed.
 */
static int multipath_add_sectors_to_kill_list( storage_object_t  *region,
					       lsn_t              lsn,
					       sector_count_t     count )
{
	md_volume_t *vol;
	md_member_t *member;
	list_element_t iter;
	int rc=0;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(region->private_data != NULL);

	vol = (md_volume_t *)region->private_data;

	if (vol->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, data is suspect \n ",vol->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	// walk the paths ... 1st success completes the write_kill_sectors
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj && md_member_is_raid_disk(member)) {
			rc = KILL_SECTORS(member->obj, lsn, count);
			if (!rc) {
				break;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: multipath_commit_changes
 *
 *  This is our commit API. We only have work during the metadata
 *  commit phases.  If our region is dirty just call the generic
 *  MD api that will write the MULTIPATH super blocks to disk.
 */
static int multipath_commit_changes( storage_object_t * region, uint phase )
{
	md_volume_t  *volume;
	int rc = 0;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(region->private_data != NULL);

	volume = (md_volume_t *)region->private_data;

	if ( phase == FIRST_METADATA_WRITE  || phase == SECOND_METADATA_WRITE ) {

		if (region->flags & SOFLAG_DIRTY) {

			rc = md_write_sbs_to_disk(volume);
			if (!rc) {
				region->flags &= ~SOFLAG_DIRTY;
				volume->flags &= ~MD_NEW_REGION;
			}

		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function:  mp_build_target_list
 *
 *  Called to build a device mapper target list for the
 *  specified multipath storage object.
 */
static int mp_build_target_list(md_volume_t *vol, dm_target_t **target_list)
{
	int j, rc = ENOMEM;
	dm_target_t *target;
	dm_target_multipath_t *mp;
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();

	target = EngFncs->dm_allocate_target(DM_TARGET_MULTIPATH, 0,
					     vol->region->size,
					     vol->nr_disks, 1);
	if (target != NULL) {
		mp = target->data.multipath;
		mp->num_groups = 1;
		strncpy(mp->group[0].selector, MP_SELECTOR, DM_SELECTOR_NAME_SIZE);
		mp->group[0].num_paths = vol->nr_disks;
		mp->group[0].num_path_args = 0;

		j = 0;
		LIST_FOR_EACH(vol->members, iter, member) {
			if (member->obj) {
				mp->group[0].path[j].device.major = member->obj->dev_major;
				mp->group[0].path[j].device.minor = member->obj->dev_minor;
				j++;
			}
		}

		*target_list = target;
		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int mp_activate_region(md_volume_t *volume)
{
	dm_target_t *target_list = NULL;
	int rc = 0;

	LOG_ENTRY();

	if (!(volume->flags & MD_MP_ACTIVATE_REGION)) {
		goto out;
	}

	rc = mp_build_target_list(volume, &target_list);
	if (rc) {
		goto out;
	}

	rc = EngFncs->dm_activate(volume->region, target_list);
	if (rc) {
		goto out;
	}

	volume->flags &= ~MD_MP_ACTIVATE_REGION;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

static int multipath_can_activate(storage_object_t *region)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Function: multipath_activate_region
 *
 *  This routine is called by the engine to activate a MULTIPATH
 *  MD region.
 */
static int multipath_activate_region(storage_object_t *region)
{
	md_volume_t *volume;
	int rc;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(region->private_data != NULL);

	volume = (md_volume_t *)region->private_data;

	rc = mp_stop_daemon(volume);
	if (rc) {
		goto out;
	}

	rc = mp_activate_region(volume);
	if (rc) {
		goto out;
	}

	rc = mp_start_daemon(volume);
	if (rc) {
		goto out;
	}

	volume->region->flags &= ~SOFLAG_NEEDS_ACTIVATE;

out:
	LOG_EXIT_INT(rc);
	return rc;
}


static int multipath_can_deactivate(storage_object_t *region)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Function: multipath_deactivate_region
 *
 * Called to deactivate a MULTIPATH MD region. We only deactivate when a
 * region is deleted or discarded. Since we need the "volume" private-data
 * to do the deactivate, we have to free that private-data once this is
 * completed.
 */
static int multipath_deactivate_region(storage_object_t *region)
{
	md_volume_t *volume = region->private_data;
	int rc;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);

	mp_stop_daemon(volume);

	rc = EngFncs->dm_deactivate(region);
	if (!rc) {
		region->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
		if (volume->flags & MD_MP_DELETE_VOLUME) {
			/* If this deactivate is due to deleting or discarding
			 * the segment, we need to delete the private data.
			 */
			EngFncs->engine_free(volume);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: multipath_get_option_count
 *
 *  Determine the type of Task that is being performed, and return
 *  the number of options that are available for that Task. Since
 *  we have no options we always return Zero if parm is ok.
 */
static int multipath_get_option_count( task_context_t * context )
{
	int rc=0;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	if (context) {
		switch (context->action) {
		case EVMS_Task_Create:
			rc = MP_CREATE_OPTION_COUNT;
			break;
		case EVMS_Task_Rewrite_Superblock:
			rc = MP_REWRITE_SB_OPTION_COUNT;
			break;
		default:
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: multipath_init_task
 *
 *  Determine the type of Task that is being performed, and set up the
 *  context structure with the appropriate initial values.  We only
 *  support CREATE right now.
 */
static int multipath_init_task( task_context_t * context )
{
	int rc = 0;
	storage_object_t *region = NULL;
	md_volume_t *vol = NULL;
	list_element_t e;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(context != NULL);


	switch (context->action) {
	
	case EVMS_Task_Create:

		context->min_selected_objects = 1;
		context->max_selected_objects = 27;

		rc = EngFncs->get_object_list( DISK | SEGMENT,
					       DATA_TYPE,
					       NULL,
					       NULL,
					       VALID_INPUT_OBJECT,
					       &context->acceptable_objects );
		break;

	case EVMS_Task_Rewrite_Superblock:
		
		region = context->object;
		vol = (md_volume_t *)region->private_data;
		e=EngFncs->insert_thing( context->acceptable_objects,
					 region,
					 INSERT_AFTER,
					 NULL );
		if (e!=NULL) {
			rc = 0;
			context->option_descriptors->count = 0;
			context->min_selected_objects = 1;
			context->max_selected_objects = 1;
		} else {
			rc = EPERM;
		}

		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: multipath_set_option
 */
static int multipath_set_option(task_context_t  *context,
				u_int32_t        index,
				value_t         *value,
				task_effect_t   *effect )
{
	int rc=ENOSYS;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(context != NULL);
	REQUIRE(value != NULL);
	REQUIRE(effect != NULL);

	LOG_ERROR("error, unknown task ... context->action = %d\n", context->action);

	*effect = 0;

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: run multipath test
 *
 *  Determine if all the objects appear to be the same physical
 *  device.  This is just a sanity check to help the EVMS user.
 *  They should really know if they are configuring multipath
 *  devices or not.  Called by set_object code to verify selected
 *  storage objects are OK for a region create request.
 */
static int run_multipath_test(storage_object_t *obj, storage_object_t *obj2)
{
	int rc=0;
	char *sample=NULL;
	char *buffer=NULL;
	lba_t lba=0;
	sector_count_t count;

	my_plugin = mp_plugin;
	LOG_ENTRY();
	REQUIRE(obj != NULL);

	if (obj2==NULL) {
		LOG_DEBUG("only 1 path\n");
		LOG_EXIT_INT(0);
		return 0;
	}

	if (obj2->size != obj->size) {
		LOG_DEBUG("size mismatch\n");
		LOG_EXIT_INT(EMEDIUMTYPE);
		return EMEDIUMTYPE;
	}

	if (obj2->geometry.cylinders != obj->geometry.cylinders) {
		LOG_DEBUG("cylinder count mismatch\n");
		LOG_EXIT_INT(EMEDIUMTYPE);
		return EMEDIUMTYPE;
	}

	if (obj2->geometry.block_size != obj->geometry.block_size) {
		LOG_DEBUG("block size mismatch\n");
		LOG_EXIT_INT(EMEDIUMTYPE);
		return EMEDIUMTYPE;
	}

	// compare some blocks
	sample = malloc(obj->geometry.block_size);
	buffer = malloc(obj->geometry.block_size);

	if (sample && buffer) {

		count = obj->geometry.block_size>>EVMS_VSECTOR_SIZE_SHIFT;

		// Test 1 ... MBR sector
		LOG_DEBUG("Test 1 start ...\n");
		lba = 0;
		rc  = READ( obj, lba, count, sample);
		if (!rc) {
			rc = READ( obj2, lba, count, buffer);
			if (!rc) {
				if (memcmp(sample,buffer,EVMS_VSECTOR_SIZE)!=0) {
					rc = EMEDIUMTYPE;
				}
			}
		}
		if (rc) {
			LOG_DEBUG("MBR sector compare failed: rc= %d\n", rc);
		}

		// Test 2 ... 1st sector of 2nd cylinder
		if (!rc) {
			lba = (obj->size & ~(count - 1)) - count;

			LOG_DEBUG("Test 2 start ... lba= %"PRIu64"\n", lba);
			rc  = READ( obj, lba, count, sample);
			if (!rc) {
				rc = READ( obj2, lba, count, buffer);
				if (!rc) {
					if (memcmp(sample,buffer,EVMS_VSECTOR_SIZE)!=0) {
						rc = EMEDIUMTYPE;
					}
				}
			}
			if (rc) {
				LOG_DEBUG("cyl 2 sector compare failed: rc= %d\n", rc);
			}
		}

		// Test 3 ... MD superblock location		
		if (!rc) {
			lba = MD_NEW_SIZE_SECTORS(obj->size);
			LOG_DEBUG("Test 3 start ... lba= %"PRIu64"\n", lba);
			rc  = READ( obj, lba, count, sample);
			if (!rc) {
				rc = READ( obj2, lba, count, buffer);
				if (!rc) {
					if (memcmp(sample,buffer,obj->geometry.block_size)!=0) {
						rc = EMEDIUMTYPE;
					}
				}
			}
			if (rc) {
				LOG_DEBUG("MD superblock sector compare failed\n");
			}
		}

	} else {
		rc = ENOMEM;
	}

	if (sample) free(sample);
	if (buffer) free(buffer);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: multipath_set_create_object
 *
 *  Determine the type of task, and then validate that the objects on the
 *  "selected" list are valid for a MULTIPATH region create task.
 */
static int multipath_set_create_object( task_context_t *context,
					list_anchor_t declined_objects,
					task_effect_t *effect )
{
	int                 rc=0;
	storage_object_t   *obj;
	storage_object_t   *obj2=NULL;
	declined_object_t  *declined_object=NULL;
	list_element_t      iter, e;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(context != NULL);
	REQUIRE(context->selected_objects != NULL);
	REQUIRE(EngFncs->list_count(context->selected_objects) > 0);
	//REQUIRE(EngFncs->list_count(context->selected_objects) <= MD_SB_DISKS);

	LIST_FOR_EACH( context->selected_objects, iter, obj ) {

		if ( ( obj->object_type  == DISK || obj->object_type == SEGMENT ) &&
		     ( obj->volume       == NULL ) &&
		     ( EngFncs->list_count(obj->parent_objects)== 0 ) ) {

			rc = run_multipath_test(obj, obj2);

		} else {
			rc = EINVAL;
		}

		if (!rc) {
			if (obj2==NULL)	obj2 = obj;
			*effect |=  EVMS_Effect_Reload_Options;
		} else {

			declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

			if (declined_object) {

				declined_object->object = obj;
				declined_object->reason = rc;

				e=EngFncs->insert_thing( declined_objects,
							 declined_object,
							 INSERT_AFTER,
							 NULL );
				if (e!=NULL) {
					rc = 0;
  					*effect |=  EVMS_Effect_Reload_Objects;
				}
				else {
					rc = EPERM;
					EngFncs->engine_free(declined_object);
				}

			}
			else {
				LOG_ERROR("error, unable to malloc a declined object struct\n");
				rc = ENOMEM;
			}

		}

		if (rc) {
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Function: set multipath rewrite object
 *
 *  Called by set_object() to validate the task selected
 *  object.
 */
static int multipath_set_rewrite_object( task_context_t * context,
					 list_anchor_t          declined_objects,
					 task_effect_t  * effect )
{
	int  rc=0;
	storage_object_t  *object;
	storage_object_t  *region;
	list_element_t iter;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(context != NULL);

	region = (storage_object_t *) context->object;

	REQUIRE(region != NULL);
	REQUIRE(context->selected_objects != NULL);
	REQUIRE(EngFncs->list_count(context->selected_objects) == 1);

	LIST_FOR_EACH( context->selected_objects, iter, object ) {
		REQUIRE(object == region);
	}

	*effect =  EVMS_Effect_Reload_Options;

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Validate the objects in the selected_objects dlist in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects dlist.
 * Modify the accepatble_objects dlist in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
static int multipath_set_objects( task_context_t *context,
				  list_anchor_t   declined_objects,
				  task_effect_t  *effect )
{

	int rc = EINVAL;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(context != NULL);
	REQUIRE(effect  != NULL);

	switch (context->action) {
	
	case EVMS_Task_Create:
		rc = multipath_set_create_object( context, declined_objects, effect );
		break;

	case EVMS_Task_Rewrite_Superblock:
		rc = multipath_set_rewrite_object( context, declined_objects, effect );
		break;

	default:
		LOG_ERROR("context->action is unknown or unsupported\n");
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: multipath_get_info
 *
 *  Return MD-specific information about the specified region. If the
 *  name field is set, only return the "extra" information pertaining
 *  to that name.
 */
static int multipath_get_info(  storage_object_t        * region,
				char                    * name,
				extended_info_array_t   ** info_array )
{
	md_volume_t * volume = NULL;
	int  rc;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(info_array != NULL);

	volume = region->private_data;

	REQUIRE(volume != NULL);

	rc = md_get_info(volume, name, info_array);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: multipath_get_plugin_info
 *
 *  Return information about the MD plugin. There is no "extra"
 *  information about MD, so "name" should always be NULL.
 */
static int multipath_get_plugin_info( char                      * name,
				      extended_info_array_t     ** info_array )
{
	extended_info_array_t   * info = NULL;
	char                    buffer[50] = {0};
	int                     i = 0;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(info_array != NULL);

	if ( ! name ) {
		// Get memory for the info array
		if ( ! (info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*6)) ) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		// Short Name
		info->info[i].name = EngFncs->engine_strdup("ShortName");
		info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(mp_plugin->short_name);
		i++;

		// Long Name
		info->info[i].name = EngFncs->engine_strdup("LongName");
		info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(mp_plugin->long_name);
		i++;

		// Plugin Type
		info->info[i].name = EngFncs->engine_strdup("Type");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Type"));
		info->info[i].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("Region Manager"));
		i++;

		// Plugin Version
		info->info[i].name = EngFncs->engine_strdup("Version");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		// Required Engine Services Version
		info->info[i].name = EngFncs->engine_strdup("Required_Engine_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
							      "It will not run on older versions of the Engine services."));
		info->info[i].type = EVMS_Type_String;
		snprintf( buffer, 50, "%d.%d.%d",
			  mp_plugin->required_engine_api_version.major,
			  mp_plugin->required_engine_api_version.minor,
			  mp_plugin->required_engine_api_version.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		// Required Plug-in API Version
		info->info[i].name = EngFncs->engine_strdup("Required_Plugin_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Plug-in API Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine plug-in API that this plug-in requires.  "
							      "It will not run on older versions of the Engine plug-in API."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d",
			 mp_plugin->required_plugin_api_version.plugin.major,
			 mp_plugin->required_plugin_api_version.plugin.minor,
			 mp_plugin->required_plugin_api_version.plugin.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	info->count = i;
	*info_array = info;
	LOG_EXIT_INT(0);
	return 0;
}


/*
 *  Function: multipath_read
 *
 *  Perform a logical-to-physical remapping, and send the read down to
 *  the next plugin. We just need to walk the i/o paths ... sending the
 *  read down to the owner plug-in ... quiting with the first succeess.
 */
static int multipath_read( storage_object_t     *region,
			   lsn_t                 lsn,
			   sector_count_t        count,
			   void                 *buffer )
{
	md_volume_t *vol;
	int rc=0;
	md_member_t *member;
	list_element_t iter;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(region->private_data != NULL);
	REQUIRE(buffer != NULL);

	vol = (md_volume_t *) region->private_data;

	if (vol->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, returning EIO.\n ",vol->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	// walk the paths ... 1st success completes the read
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj && md_member_is_raid_disk(member)) {
			rc = READ(member->obj, lsn+member->data_offset, count, buffer);
			if (!rc) {
				break;
			} else {
				char number_buffer[64];
				sprintf(number_buffer, "%"PRIu64, lsn+count);
				MESSAGE(_("Error reading from mirror %s"
					  " of region %s sector=%s,"
					  " Mirror disabled.\n"),
					member->obj->name,
					vol->name, number_buffer);
				//BUGBUG disable mirror
				member->raid_disk = -1;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: multipath_write
 *
 *  Perform a logical-to-physical remapping, and send the write down to
 *  the next plugin. We just need to walk the i/o paths ... sending the
 *  write down to the owner plug-in ... quiting with the first succeess.
 */
static int multipath_write( storage_object_t    *region,
			    lsn_t                lsn,
			    sector_count_t       count,
			    void                *buffer )
{
	md_volume_t *vol;
	int rc=0;
	md_member_t *member;
	list_element_t iter;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(region->private_data != NULL);
	REQUIRE(buffer != NULL);

	vol = (md_volume_t *) region->private_data;

	if (vol->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, returning EIO.\n ",vol->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	// walk the paths ... 1st success completes the write
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj && md_member_is_raid_disk(member)) {
			rc = WRITE(member->obj, lsn+member->data_offset, count, buffer);
			if (!rc) {
				break;
			} else {
				char number_buffer[64];
				sprintf(number_buffer, "%"PRIu64, lsn+count);
				MESSAGE(_("Error reading from mirror %s"
					  " of region %s sector=%s,"
					  " Mirror disabled.\n"),
					member->obj->name,
					vol->name, number_buffer);
				//BUGBUG disable mirror
				member->raid_disk = -1;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static char rewrite_help_msg[] = N_("Use this function to rewrite the MD superblock. The superblock will be "
                                 "updated with the current multipath configuration before it is written "
                                 "to disk.  This is usually done after adding or removing paths to a "
				 "device.  By rewriting the superblock, the correct multipath configuration "
				 "is saved on disk so that it can be obtained and used when activating "
				 "a kernel mapping of the multipath device.");

/*
 *  Function:  multipath_get_plugin_functions
 */
static int multipath_get_plugin_functions( storage_object_t        *region,
					   function_info_array_t  **functions )
{
	int                    rc = EINVAL;
	function_info_array_t *func_info=NULL;
	char                   title[EVMS_VOLUME_NAME_SIZE+1];
	int                    index;
	md_volume_t           *volume=NULL;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(region->private_data != NULL);
	REQUIRE(functions != NULL);

	volume = (md_volume_t *) region->private_data;

	// there must be something wrong, i.e. the superblock doesn't agree with
	// the volume information, for us to provide a superblock rewrite function.
	if (multipath_verify_sb_info(volume, FALSE)==0) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	func_info = EngFncs->engine_alloc( sizeof(function_info_array_t) + sizeof(function_info_t) );
	if (func_info) {

		func_info->count = 0;
		index = 0;

		func_info->info[index].function = EVMS_Task_Rewrite_Superblock;

		sprintf(title,"Rewrite MD superblock");
		func_info->info[index].title = EngFncs->engine_strdup( title );
		func_info->info[index].verb = EngFncs->engine_strdup( _("Rewrite") );
		func_info->info[index].name = EngFncs->engine_strdup( _("Rewrite") );
		func_info->info[index].help = EngFncs->engine_strdup( _(rewrite_help_msg) );

		++func_info->count;
		++index;

		rc = 0;

	} else {
		rc = ENOMEM;
	}

	*functions = func_info;

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function:  rewrite_multipath_superblock
 *
 *  Called to reconfigure a multipath region by updating the MD
 *  superblock so that it reflects the current configuration.
 */
static int rewrite_multipath_superblock( storage_object_t * region,
					 task_action_t      action,
					 list_anchor_t      objects,
					 option_array_t   * options)
{
	int rc=EINVAL;
	md_volume_t *vol;
	list_anchor_t my_objects;
	storage_object_t *object;
	md_member_t *member;
	list_element_t iter1;
	list_element_t iter2;
	list_element_t li;
	u_int64_t size = -1;
	md_sb_ver_t sb_ver;
	int raid_disk;
	int md_minor;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region!=NULL);
	REQUIRE(region->private_data!=NULL);
	REQUIRE(objects!=NULL);

	my_objects = EngFncs->allocate_list();
	if (!my_objects) {
		LOG_CRITICAL("Can't allocate temp list.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	vol = (md_volume_t *) region->private_data;
	LIST_FOR_EACH(vol->members, iter1, member) {
		li = EngFncs->insert_thing(my_objects, member->obj, INSERT_AFTER, NULL);
		if (!li) {
			EngFncs->destroy_list(my_objects);
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}
	}
	sb_ver = vol->sb_ver;
	md_minor = vol->md_minor;
	md_free_volume(vol);

	if (!(vol = md_allocate_volume())) {
		vol->flags |= MD_CORRUPT;
		region->flags |= SOFLAG_CORRUPT;		
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}
	vol->personality = MULTIPATH;
	vol->md_minor = md_minor;
	md_volume_set_name(vol, NULL);

	LIST_FOR_EACH_SAFE(my_objects, iter1, iter2, object) {
		if (size == -1) {
			size = md_object_usable_size(object, &sb_ver, 0);
		} else {
			if (size != md_object_usable_size(object, &sb_ver, 0)) {
				LOG_ERROR("Current size: %"PRIu64", data size of %s: %"PRIu64".\n",
					  size, object->name, md_object_usable_size(object, &sb_ver, 0));
				goto error_free;
			}
		}
	}

	rc = md_init_sb(vol, &sb_ver, -4, 0, size, 0);
	if (rc) {
		goto error_free;
	}

	// Add raid members
	raid_disk = 0;
	LIST_FOR_EACH_SAFE(my_objects, iter1, iter2, object) {
		member = md_allocate_member(object);
		if (member) {
			// This will add the member and update the MD superblock.
			member->dev_number = raid_disk;
			member->raid_disk = raid_disk;
			member->data_offset = 0;
			member->data_size = size;
			member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC);
			raid_disk++;
			rc = md_volume_add_new_member(vol, member);
			if (rc) {
				md_free_member(member);
				goto error_free;
			}
		} else {
			rc = ENOMEM;
		}
		if (rc) {
			break;
		}
		EngFncs->delete_element(iter1);
	}

	if (!rc) {
		vol->flags |= MD_DIRTY;
		region->private_data = vol;
	} else {
		vol->flags |= MD_CORRUPT;
		region->flags |= SOFLAG_CORRUPT;
		goto error_free;
	}
	EngFncs->destroy_list(my_objects);
	LOG_EXIT_INT(rc);
	return rc;

error_free:
	md_free_volume(vol);
	EngFncs->destroy_list(my_objects);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function:  multipath_plugin_function
 */
static int multipath_plugin_function( storage_object_t * region,
				      task_action_t      action,
				      list_anchor_t            objects,
				      option_array_t   * options)
{
	int rc;

	my_plugin = mp_plugin;
	LOG_ENTRY();

	REQUIRE(region != NULL);
	REQUIRE(region->private_data != NULL);
	REQUIRE(objects != NULL);

	switch (action) {
	case EVMS_Task_Rewrite_Superblock:
		rc = rewrite_multipath_superblock(region,action,objects,options);
		break;
	default:
		rc = EINVAL;
		break;
	}

	if (rc==0) {
		region->flags |= SOFLAG_DIRTY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function tables for the MD Multipath Region Manager */
static plugin_functions_t multipath_functions = {
	setup_evms_plugin               : multipath_setup_evms_plugin,
	cleanup_evms_plugin             : multipath_plugin_cleanup,
	can_delete                      : multipath_can_delete,
	discover                        : multipath_discover,
	create                          : multipath_create,
	delete                          : multipath_delete,
	discard                         : multipath_discard,
	add_sectors_to_kill_list        : multipath_add_sectors_to_kill_list,
	commit_changes                  : multipath_commit_changes,
	can_activate			: multipath_can_activate,
	activate                        : multipath_activate_region,
	can_deactivate			: multipath_can_deactivate,
	deactivate                      : multipath_deactivate_region,
	get_option_count                : multipath_get_option_count,
	init_task                       : multipath_init_task,
	set_option                      : multipath_set_option,
	set_objects                     : multipath_set_objects,
	get_info                        : multipath_get_info,
	get_plugin_info                 : multipath_get_plugin_info,
	read                            : multipath_read,
	write                           : multipath_write,
	get_plugin_functions            : multipath_get_plugin_functions,
	plugin_function                 : multipath_plugin_function
};

plugin_record_t   multipath_plugin_record = {
	id:                    		SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 9),

	version:              		{ major:      MAJOR_VERSION,
					  minor:      MINOR_VERSION,
					  patchlevel: PATCH_LEVEL},

	required_engine_api_version: 	{ major:      15,
					  minor:      0,
					  patchlevel: 0},
	
	required_plugin_api_version: 	{ plugin: {major:      13,
					  minor:      0,
					  patchlevel: 0}},

	short_name:            		"MD Multipath",
	long_name:             		"MD Multipath Region Manager",
	oem_name:              		"IBM",

	functions:             		{plugin: &multipath_functions},

	container_functions:   		NULL
};

