/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "views.h"
#include "enum.h"
#include "task.h"
#include "plugin.h"
#include "readable.h"
#include "resize.h"
#include "container.h"
#include "logging.h"

/**
 *	filter_expand - filter things that can expanded
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be expanded.
 */
int filter_expand(engine_handle_t handle, void *user_data)
{
	return evms_can_expand(handle);
}

/**
 *	filter_shrink - filter things that can expanded
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be shrunk.
 */
int filter_shrink(engine_handle_t handle, void *user_data)
{
	return evms_can_shrink(handle);
}

/**
 *	format_resize_item - return column strings for a row in the resize selection clist
 *	@handle: the thing handle
 *	@max_size: the maximum shrink/expand size
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a thing placed
 *	in the resize clist containing the size and name of an object along with
 *	maximum size that the item can shrink or expand by.
 */
int format_resize_item(object_handle_t handle, sector_count_t max_size, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(" "));
		switch (object->type) {
		case CONTAINER:
			g_ptr_array_add(text, g_strdup(object->info.container.name));
			g_ptr_array_add(text, make_sectors_readable_string(object->info.container.size));
			break;
		case VOLUME:
			g_ptr_array_add(text, g_strdup(object->info.volume.name));
			g_ptr_array_add(text, make_sectors_readable_string(object->info.volume.vol_size));
			break;
		default:
			g_ptr_array_add(text, g_strdup(object->info.object.name));
			g_ptr_array_add(text, make_sectors_readable_string(object->info.object.size));
			break;
		}
		g_ptr_array_add(text, make_sectors_readable_string(max_size));
		evms_free(object);
	}
	return rc;
}

/**
 *	append_resize_point_to_clist - add shrink/expand point to the resize clist
 *	@clist: the resize clist
 *	@handle: the object handle
 *	@max_size: the maximum expand/shrink size in sectors
 *
 *	This routine appends a clist item to the resize clist.
 */
void append_resize_point_to_clist(struct clist *clist, object_handle_t handle, sector_count_t max_size)
{
	GPtrArray *text;

	text = g_ptr_array_new();
	if (format_resize_item(handle, max_size, text) == 0) {
		append_item(clist, text, GUINT_TO_POINTER(handle), NULL);
	}
	g_ptr_array_free(text, TRUE);
}

/**
 *	populate_shrink_points - populate the resize clist with shrink points
 *	@clist: the resize selection clist
 *	@handle: the handle of the thing to get shrink points for
 *
 *	This routine populates the resize clist with shrink points for a given
 *	thing.
 */
void populate_shrink_points(struct clist *clist, object_handle_t handle)
{
	int rc;
	shrink_handle_array_t *shrink_points;

	rc = evms_get_shrink_points(handle, &shrink_points);
	if (rc == 0) {
		int i;

		for (i = 0; i < shrink_points->count; i++) {
			append_resize_point_to_clist(clist, shrink_points->shrink_point[i].object,
						shrink_points->shrink_point[i].max_shrink_size);
		}
		evms_free(shrink_points);
	} else {
		log_error("%s: evms_get_shrink_points() returned error code %d.\n", __FUNCTION__, rc);
	}
}

/**
 *	populate_expand_points - populate the resize clist with expand points
 *	@clist: the resize selection clist
 *	@handle: the handle of the thing to get expand points for
 *
 *	This routine populates the resize clist with expand points for a given
 *	thing.
 */
void populate_expand_points(struct clist *clist, object_handle_t handle)
{
	int rc;
	expand_handle_array_t *expand_points;

	rc = evms_get_expand_points(handle, &expand_points);
	if (rc == 0) {
		int i;

		for (i = 0; i < expand_points->count; i++) {
			append_resize_point_to_clist(clist, expand_points->expand_point[i].object,
						expand_points->expand_point[i].max_expand_size);
		}
		evms_free(expand_points);
	} else {
		log_error("%s: evms_get_expand_points() returned error code %d.\n", __FUNCTION__, rc);
	}
}

/**
 *	on_delete_resize_dialog - callback invoked when dialog is to be deleted
 *	@dialog: the plugin selection dialog
 *
 *	This routine is invoked when the resize dialog is deleted. We take care
 *	of freeing any dynamically allocated entries in the hash table
 *	before freeing the hash table associated with the dialog.
 */
void on_delete_resize_dialog(struct dialog_window *dialog)
{
	GHashTable *hash_table = dialog->user_data;

	if (hash_table != NULL) {
		g_free(g_hash_table_lookup(hash_table, "title"));
		g_hash_table_destroy(hash_table);
	}
}

/**
 *	on_initiate_resize_button_activated - initiate the resize task now that we have resize point
 *	@item: the button that was activated to invoke this callback
 *
 *	This routine is invoked once the user has made a selection of a resize point
 *	and we are to display the task dialog with either acceptable objects or
 *	the configuration options for the resize task.
 */
int on_initiate_resize_button_activated(struct menu_item *item)
{
	GHashTable *hash_table;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;
	engine_handle_t curr_selection, prev_selection;

	hash_table = dialog->user_data;
	curr_selection = get_selected_handle(selwin->clist);
	prev_selection = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "prev_selection"));

	if (curr_selection != prev_selection) {
		char *title;
		GList *next_dialog;
		task_action_t action;
		struct dialog_window *new_dialog;
		/*
		 * Since the selected feature has changed we need to delete any
		 * dialogs that we may have built for the add feature task.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		title = g_hash_table_lookup(hash_table, "title");
		action = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "action"));

		new_dialog = create_task_dialog(dialog->list, curr_selection, action,
						title, get_task_action_string(action),
						NULL);
		if (new_dialog != NULL) {
			g_hash_table_insert(hash_table, "selection", GUINT_TO_POINTER(curr_selection));
			dialog->list->current = get_next_dialog(dialog);
		}
	} else {
		dialog->list->current = get_next_dialog(dialog);
	}

	return 0;
}

/**
 *	create_resize_points_dialog - create and populate a dialog with the shrink/expand points for handle
 *	@handle: the handle of the thing to get expand/shrink points for
 *	@action: whether an expand or shrink task
 *	@title: the window title
 *
 *	This routine creates a selection dialog containing the shrink/expand points for a given object.
 */
struct dialog_window *create_resize_points_dialog(engine_handle_t handle, task_action_t action, char *title)
{
	struct selwin *selwin;
	GHashTable *hash_table;
	struct dialog_window *dialog;
	char *window_title, *max_min, *column_title, *task_title;

	/*
	 * Allocate a hash table for storing multiple pieces of data needed by the
	 * activation callback. Save off the task action and title for starters.
	 */
	hash_table = g_hash_table_new(g_str_hash, g_str_equal);
	g_hash_table_insert(hash_table, "title", g_strdup(title));
	g_hash_table_insert(hash_table, "action", GUINT_TO_POINTER(action));

	if (action == EVMS_Task_Expand) {
		max_min = _("Expandable by");
		column_title = _("Expand Points");
		task_title = _("Expansion Point Selection");
	} else {
		max_min = _("Shrinkable by");
		column_title = _("Shrink Points");
		task_title = _("Shrink Point Selection");
	}

	window_title = g_strdup_printf("%s - %s", title, task_title);
	selwin = create_selection_window(window_title,
					NULL, NULL,
					NULL,
					(menuitem_activate_cb)on_initiate_resize_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					hash_table);

	dialog = (struct dialog_window *)selwin;

	set_clist_column_count(selwin->clist, 4);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.50),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(selwin->clist, 2, calc_clist_column_width(selwin->clist, 0.12),
				get_clist_column_end(selwin->clist, 1),
				CLIST_TEXT_JUSTIFY_RIGHT);
	set_clist_column_info(selwin->clist, 3, calc_clist_column_width(selwin->clist, 0.25),
				get_clist_column_end(selwin->clist, 2),
				CLIST_TEXT_JUSTIFY_RIGHT);

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, column_title);
	print_clist_column_title(selwin->clist, 2, _("Size"));
	print_clist_column_title(selwin->clist, 3, max_min);

	if (action == EVMS_Task_Expand)
		populate_expand_points(selwin->clist, handle);
	else
		populate_shrink_points(selwin->clist, handle);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_resize_dialog);
	g_free(window_title);

	return dialog;
}

/**
 *	resize_point_is_selected_object - determine if single resize point is same as object to resize
 *	@handle: the volume or object handle
 *	@action: EVMS_Task_Expand or EVMS_Task_Shrink
 *
 *	This routine determines if the object has one and only one resize
 *	point and whether it is the same object as the resize selection object.
 */
gboolean resize_point_is_selected_object(object_handle_t object, task_action_t action)
{
	int rc;
	gboolean result=FALSE;
	
	if (action == EVMS_Task_Expand){
		expand_handle_array_t *expand_points;
		
		rc = evms_get_expand_points(object, &expand_points);
    		if (rc == 0) {
			result = (expand_points->count == 1 && object == expand_points->expand_point[0].object);
			evms_free(expand_points);
		} else {
			log_error("%s: evms_get_expand_points() returned error code %d.\n", __FUNCTION__, rc);
		}
	} else {
		shrink_handle_array_t *shrink_points;
		
		rc = evms_get_shrink_points(object, &shrink_points);
		if (rc == 0) {
			result = (shrink_points->count == 1 && object == shrink_points->shrink_point[0].object);
			evms_free (shrink_points);
		} else {
			log_error("%s: evms_get_shrink_points() returned error code %d.\n", __FUNCTION__, rc);
		}
	}
	return result;
}

/**
 *	on_show_resize_points_button_activated - dialog shrink/expand points for the thing to resize
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the initial volume or object is selected and we are
 *	proceeding to display the expand/shrink points for the selection.
 */
int on_show_resize_points_button_activated(struct menu_item *item)
{
	GHashTable *hash_table;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;
	engine_handle_t curr_selection, prev_selection;

	hash_table = dialog->user_data;
	curr_selection = get_selected_handle(selwin->clist);
	prev_selection = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "prev_selection"));

	if (curr_selection != prev_selection) {
		char *title;
		GList *next_dialog;
		task_action_t action;
		struct dialog_window *new_dialog;
		/*
		 * Since the selected feature has changed we need to delete any
		 * dialogs that we may have built for the add feature task.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		title = g_hash_table_lookup(hash_table, "title");
		action = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "action"));

		if (resize_point_is_selected_object(curr_selection, action)) {
			new_dialog = create_task_dialog(dialog->list, curr_selection, action,
							title, get_task_action_string(action),
							NULL);
		} else {
			new_dialog = create_resize_points_dialog(curr_selection, action, title);
			append_dialog_to_list(new_dialog, dialog->list);
		}

		if (new_dialog != NULL) {
			g_hash_table_insert(hash_table, "selection", GUINT_TO_POINTER(curr_selection));
			dialog->list->current = get_next_dialog(dialog);
		}
	} else {
		dialog->list->current = get_next_dialog(dialog);
	}

	return 0;
}

/**
 *	show_resize_dialog - display one or more objects to start and expand or shrink task
 *	@handle: non-zero to display just that thing
 *	@action: the task action
 *	@window_title: the dialog title
 *	@column_title: the clist type column title
 *	@enum_func: the enumeration to use in populating the selection clist
 *
 *	This routine creates the initial selection dialog to handle processing an expand
 *	or shrink task based on a selection from this dialog.
 */
void show_resize_dialog(object_handle_t handle, task_action_t action,
			char *window_title, char *column_title, clist_enum_func enum_func)
{
	struct selwin *selwin;
	GHashTable *hash_table;
	struct dialog_list dialogs;
	struct dialog_window *dialog;
	clist_filter_func filter_func;
	handle_object_info_t *object = NULL;

	evms_get_info(handle, &object);

	/*
	 * Allocate a hash table for storing multiple pieces of data needed by the
	 * activation callback. Save off the task action and title for starters.
	 */
	hash_table = g_hash_table_new(g_str_hash, g_str_equal);
	g_hash_table_insert(hash_table, "title", g_strdup(window_title));
	g_hash_table_insert(hash_table, "action", GUINT_TO_POINTER(action));

	selwin = create_selection_window(window_title,
					NULL, NULL,
					NULL,
					(menuitem_activate_cb)on_show_resize_points_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					hash_table);

	dialog = (struct dialog_window *)selwin;

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, column_title);
	print_clist_column_title(selwin->clist, 2, _("Size"));

	switch (action) {
	default:
	case EVMS_Task_Expand:
		filter_func = filter_expand;
		break;
	case EVMS_Task_Shrink:
		filter_func = filter_shrink;
		break;
	}

	if (handle == 0)
		clist_populate(selwin->clist, enum_func, filter_func,
			format_standard_item, NULL, NULL);
	else
		clist_populate_handle(selwin->clist, handle,
			format_standard_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);
	set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_resize_dialog);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);
}

/**
 *	show_resize_volumes_dialog - display one or more volumes to start an expand or shrink task
 *	@handle: non-zero when called from context menu to display only that volume
 *	@action: either EVMS_Task_Shrink or EVMS_Task_Expand
 *
 *	This routine calls the common show_resize_dialog to display the initial object(s) to select
 *	to initiate a shrink or expand task from.
 */
inline void show_resize_volumes_dialog(object_handle_t handle, task_action_t action)
{
	show_resize_dialog(handle, action,
			action == EVMS_Task_Expand ? _("Expand Logical Volume") :
						_("Shrink Logical Volume"),
			_("Logical Volume"),
			enum_volumes);
}

/**
 *	context_expand_volume_menuitem_activated - expand a logical volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to expand some point in a volume and is
 *	initiated from the context popup menu.
 */
int context_expand_volume_menuitem_activated(struct menu_item *item)
{
	show_resize_volumes_dialog(GPOINTER_TO_UINT(item->user_data), EVMS_Task_Expand);
	return 0;
}

/**
 *	actions_expand_volume_menuitem_activated - expand a logical volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the File System->Expand->Volume Actions pulldown
 *	menu item.
 */
int actions_expand_volume_menuitem_activated(struct menu_item *item)
{
	show_resize_volumes_dialog(0, EVMS_Task_Expand);
	return 0;
}

/**
 *	context_shrink_volume_menuitem_activated - shrink a logical volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to shrink some point in a volume and is
 *	initiated from the context popup menu.
 */
int context_shrink_volume_menuitem_activated(struct menu_item *item)
{
	show_resize_volumes_dialog(GPOINTER_TO_UINT(item->user_data), EVMS_Task_Shrink);
	return 0;
}

/**
 *	actions_shrink_volume_menuitem_activated - shrink a logical volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the File System->Shrink->Volume Actions pulldown
 *	menu item.
 */
int actions_shrink_volume_menuitem_activated(struct menu_item *item)
{
	show_resize_volumes_dialog(0, EVMS_Task_Shrink);
	return 0;
}

/**
 *	show_resize_objects_dialog - display one or more objects to start an expand or shrink task
 *	@handle: non-zero when called from context menu to display only that storage object
 *	@action: either EVMS_Task_Shrink or EVMS_Task_Expand
 *
 *	This routine calls the common show_resize_dialog to display the initial object(s) to select
 *	to initiate a shrink or expand task from.
 */
inline void show_resize_objects_dialog(object_handle_t handle, task_action_t action)
{
	show_resize_dialog(handle, action,
			action == EVMS_Task_Expand ? _("Expand Storage Object") :
						_("Shrink Storage Object"),
			_("Storage Object"),
			enum_topmost_objects);
}

/**
 *	context_expand_object_menuitem_activated - expand a storage object
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to expand some point in a storage object hiearchy and is
 *	initiated from the context popup menu.
 */
int context_expand_object_menuitem_activated(struct menu_item *item)
{
	show_resize_objects_dialog(GPOINTER_TO_UINT(item->user_data), EVMS_Task_Expand);
	return 0;
}

/**
 *	actions_expand_object_menuitem_activated - expand a storage object
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the File System->Expand->Object Actions pulldown
 *	menu item.
 */
int actions_expand_object_menuitem_activated(struct menu_item *item)
{
	show_resize_objects_dialog(0, EVMS_Task_Expand);
	return 0;
}

/**
 *	context_shrink_object_menuitem_activated - shrink a storage object
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to shrink some point in a storage object hiearchy and is
 *	initiated from the context popup menu.
 */
int context_shrink_object_menuitem_activated(struct menu_item *item)
{
	show_resize_objects_dialog(GPOINTER_TO_UINT(item->user_data), EVMS_Task_Shrink);
	return 0;
}

/**
 *	actions_shrink_object_menuitem_activated - shrink a storage object
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the File System->Shrink->Object Actions pulldown
 *	menu item.
 */
int actions_shrink_object_menuitem_activated(struct menu_item *item)
{
	show_resize_objects_dialog(0, EVMS_Task_Shrink);
	return 0;
}

/**
 *	show_resize_containers_dialog - display one or more containers to start an expand or shrink task
 *	@handle: non-zero when called from context menu to display only that storage container
 *	@action: either EVMS_Task_Shrink or EVMS_Task_Expand
 *
 *	This routine calls the common show_resize_dialog to display the initial container(s) to select
 *	to initiate a shrink or expand task from.
 */
inline void show_resize_containers_dialog(object_handle_t handle, task_action_t action)
{
	show_resize_dialog(handle, action,
			action == EVMS_Task_Expand ? _("Expand Storage Container") :
						_("Shrink Storage Container"),
			_("Storage Container"),
			enum_containers);
}

/**
 *	context_expand_container_menuitem_activated - expand a storage container
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to expand a storage container and is
 *	initiated from the context popup menu.
 */
int context_expand_container_menuitem_activated(struct menu_item *item)
{
	show_resize_containers_dialog(GPOINTER_TO_UINT(item->user_data),
				EVMS_Task_Expand);
	return 0;
}

/**
 *	actions_expand_container_menuitem_activated - expand a storage container
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the File System->Expand->Container Actions pulldown
 *	menu item.
 */
int actions_expand_container_menuitem_activated(struct menu_item *item)
{
	show_resize_containers_dialog(0, EVMS_Task_Expand);
	return 0;
}

/**
 *	context_shrink_container_menuitem_activated - shrink a storage container
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to shrink a storage container and is
 *	initiated from the context popup menu.
 */
int context_shrink_container_menuitem_activated(struct menu_item *item)
{
	show_resize_containers_dialog(GPOINTER_TO_UINT(item->user_data),
				EVMS_Task_Shrink);
	return 0;
}

/**
 *	actions_shrink_container_menuitem_activated - shrink a storage container
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the File System->Shrink->Container Actions pulldown
 *	menu item.
 */
int actions_shrink_container_menuitem_activated(struct menu_item *item)
{
	show_resize_containers_dialog(0, EVMS_Task_Shrink);
	return 0;
}
