/*
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * File: replace.c
 *
 * Description: This file contains the code for the plug-in
 *              which handles replace.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <pthread.h>

#include <plugin.h>

static plugin_record_t   replace_plugin_record;
static plugin_record_t * my_plugin_record = &replace_plugin_record;

typedef struct rep_private_data_s {
	storage_object_t * source;
	storage_object_t * target;
	copy_job_t         copy_job;
} rep_private_data_t;


static engine_functions_t * EngFncs;

typedef struct replace_child_info_s {
	u_int32_t dev_major;
	u_int32_t dev_minor;
	char      name[EVMS_NAME_SIZE+1];
} replace_child_info_t;

typedef struct replace_device_info_s {
	replace_child_info_t replace_info;
	replace_child_info_t mirror_info;
	replace_child_info_t source_info;
	replace_child_info_t target_info;
	storage_object_t *   replace_obj;
	storage_object_t *   source_obj;
	storage_object_t *   target_obj;
} replace_device_info_t;

static boolean find_replace_devices = FALSE;
static list_anchor_t replace_devices;



#define REPLACE_NAME_PREFIX	"Replace_"
#define REPLACE_NAME_PREFIX_LEN	(sizeof(REPLACE_NAME_PREFIX) - 1)
#define REPLACE_NAME_INFIX	"_with_"
#define REPLACE_NAME_INFIX_LEN	(sizeof(REPLACE_NAME_INFIX) - 1)

/*
 * rep_setup_evms_plugin
 */
static int rep_setup_evms_plugin(engine_functions_t * functions) {

	EngFncs = functions;

	LOG_ENTRY();

	/* Look for leftover replace devices on the first call to discover(). */
	replace_devices = EngFncs->allocate_list();
	if (replace_devices != NULL) {
		find_replace_devices = TRUE;
	} else {
		find_replace_devices = FALSE;
	}

	LOG_EXIT_INT(0);
	return(0);
}


/*
 * rep_cleanup_evms_plugin
 */
static void rep_cleanup_evms_plugin() {

	LOG_ENTRY();

	/* destroy_list() handles NULL list pointers. */
	EngFncs->destroy_list(replace_devices);
	find_replace_devices = FALSE;

	LOG_EXIT_VOID();
	return;
}


/*
 * rep_can_delete
 */
static int rep_can_delete(storage_object_t * obj) {

	int rc = 0;

	LOG_ENTRY();

	if (obj->plugin != my_plugin_record) {
		LOG_ERROR("%s is not a replace object.  I can't delete it.\n", obj->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_can_replace_child
 *
 * Function not supported yet.
 */
static int rep_can_replace_child(storage_object_t * obj,
				 storage_object_t * child,
				 storage_object_t * new_child) {

	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return(ENOSYS);
}


static int make_replace_object(storage_object_t   * source,
			       storage_object_t   * target,
			       char               * name,
			       storage_object_t * * preplace_object) {
	int rc = 0;
	storage_object_t * replace_obj = NULL;
	char replace_obj_name[EVMS_NAME_SIZE+1] = {0};

	LOG_ENTRY();

	if (name != NULL) {
		strcpy(replace_obj_name, name);

	} else {
		snprintf(replace_obj_name, EVMS_NAME_SIZE, REPLACE_NAME_PREFIX "%s" REPLACE_NAME_INFIX "%s",
			 source->name, target->name);
	}

	switch (source->object_type) {
		case DISK:
			rc = EngFncs->allocate_logical_disk(replace_obj_name, &replace_obj);
			break;
		case SEGMENT:
			rc = EngFncs->allocate_segment(replace_obj_name, &replace_obj);
			break;
		case REGION:
			rc = EngFncs->allocate_region(replace_obj_name, &replace_obj);
			break;
		case EVMS_OBJECT:
			rc = EngFncs->allocate_evms_object(replace_obj_name, &replace_obj);
			break;
		default:
			LOG_ERROR("Object %s has an unsupported object type of %#x.\n",
				  source->name, source->object_type);
			rc = EINVAL;
	}

	if (rc == 0) {
		rep_private_data_t * pd;

		pd = EngFncs->engine_alloc(sizeof(rep_private_data_t));

		if (pd != NULL) {
			list_element_t el;

			replace_obj->private_data = pd;

			pd->source = source;
			pd->target = target;

			/*
			 * Make the source and target objects children of the replace
			 * object.  Source is first in the list.
			 */
			el = EngFncs->insert_thing(replace_obj->child_objects,
						   source,
						   INSERT_BEFORE,
						   NULL);
			if (el == NULL) {
				LOG_CRITICAL("Error when inserting source object %s into the child_list of replace object %s.\n",
					     source->name, replace_obj->name);
				EngFncs->free_evms_object(replace_obj);
				replace_obj = NULL;
				rc = ENOMEM;
			}

			if (rc == 0) {
				if (target != NULL) {
					el = EngFncs->insert_thing(replace_obj->child_objects,
								   target,
								   INSERT_AFTER,
								   NULL);
					if (el == NULL) {
						LOG_CRITICAL("Error when inserting target object %s into the child_list of replace object %s.\n",
							     target->name, replace_obj->name);
						EngFncs->free_evms_object(replace_obj);
						replace_obj = NULL;
						rc = ENOMEM;
					}
				}

				if (rc == 0) {

					replace_obj->plugin = my_plugin_record;
					if (target != NULL) {
						replace_obj->size = min(source->size, target->size);
					} else {
						replace_obj->size = source->size;
					}
					replace_obj->flags |= source->flags & (SOFLAG_READ_ONLY | SOFLAG_MUST_BE_TOP);
					replace_obj->geometry = source->geometry;
					replace_obj->volume = source->volume;
					EngFncs->destroy_list(replace_obj->associated_parents);
					replace_obj->associated_parents =  EngFncs->copy_list(source->associated_parents);
					EngFncs->destroy_list(replace_obj->associated_children);
					replace_obj->associated_children =  EngFncs->copy_list(source->associated_children);
					replace_obj->disk_group = source->disk_group;

					if (source->feature_header != NULL) {
						replace_obj->feature_header = EngFncs->engine_alloc(sizeof(evms_feature_header_t));
						if (replace_obj->feature_header != NULL) {
							memcpy(replace_obj->feature_header, source->feature_header, sizeof(evms_feature_header_t));

						} else {
							LOG_CRITICAL("Error getting memory for the replace object's feature header.\n");
							rc = ENOMEM;
						}
					}

					/*
					 * If the source is [going to be] active,
					 * the replace object and target should
					 * be active.
					 */
					if ((source->flags & (SOFLAG_ACTIVE | SOFLAG_NEEDS_ACTIVATE)) &&
					     !(source->flags & SOFLAG_NEEDS_DEACTIVATE)) {
						replace_obj->flags |= SOFLAG_NEEDS_ACTIVATE;
						if (target != NULL) {
							if (!(target->flags & SOFLAG_ACTIVE)) {
								target->flags |= SOFLAG_NEEDS_ACTIVATE;
							} else if (target->flags & SOFLAG_NEEDS_DEACTIVATE) {
								target->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
							}
						}

					} else {
						/* Leave replace object inactive. */
						if (target != NULL) {
							if (target->flags & SOFLAG_ACTIVE) {
								target->flags |= SOFLAG_NEEDS_DEACTIVATE;
							} else if (target->flags & SOFLAG_NEEDS_ACTIVATE) {
								target->flags &= ~SOFLAG_NEEDS_ACTIVATE;
							}
						}
					}
				}
			}

		} else {
			LOG_CRITICAL("Error allocating memory for a private data structure for replace object %s.\n",
				     replace_obj->name);
			rc = ENOMEM;
		}

	} else {
		LOG_CRITICAL("Error code %d when allocating a replace object.\n", rc);
	}

	if (rc == 0) {
		list_element_t el;
		/*
		 * Make the replace object the new parent of the source
		 * and target objects.
		 */

		if (target != NULL) {
			/*
			 * The target pbject was guaranteed to be a top level object.
			 * By definition it has no parents.  So just put the
			 * replace_obj in the target's parent list.
			 */
			el = EngFncs->insert_thing(target->parent_objects,
						   replace_obj,
						   INSERT_AFTER,
						   NULL);
			if (el == NULL) {
				LOG_CRITICAL("Error when inserting replace object %s into the parent_list of target object %s.\n",
					     replace_obj->name, target->name);
				rc = ENOMEM;
			}
		}

		if (rc == 0) {
			el = EngFncs->insert_thing(source->parent_objects,
						   replace_obj,
						   INSERT_AFTER,
						   NULL);

			if (el == NULL) {
				LOG_CRITICAL("Error when inserting replace object %s into the parent_list of source object %s.\n",
					     replace_obj->name, source->name);
				rc = ENOMEM;

			}
		}
	}

	*preplace_object = replace_obj;

	LOG_EXIT_INT(rc);
	return(rc);
}


static char * get_dm_device_name(dm_device_list_t * device_list,
				 u_int32_t dev_major, u_int32_t dev_minor) {

	dm_device_list_t * device;

	LOG_ENTRY();

	for (device = device_list; device != NULL; device = device->next) {
		if ((device->dev_major == dev_major) &&
		    (device->dev_minor == dev_minor)) {
			LOG_EXIT_PTR(device->name);
			return device->name;
		}
	}

	LOG_EXIT_PTR(NULL);
	return NULL;
}


static storage_object_t temp_obj;

static void get_mirror_info(dm_device_list_t * device_list, replace_device_info_t * rep_dev_info) {

	int rc;
	dm_target_t * rep_target;
	char * rep_child_name;
	dm_target_t * rep_child_target;

	LOG_ENTRY();

	strcpy(temp_obj.name, rep_dev_info->replace_info.name);
	rc = EngFncs->dm_get_targets(&temp_obj, &rep_target);

	if (rc != 0) {
		LOG_EXIT_VOID();
		return;
	}

	rep_child_name = get_dm_device_name(device_list,
					    rep_target->data.linear->major,
					    rep_target->data.linear->minor);
	if (rep_child_name != NULL) {

		strcpy(temp_obj.name, rep_child_name);
		rc = EngFncs->dm_get_targets(&temp_obj, &rep_child_target);

		if (rc == 0) {
			if (rep_child_target->type == DM_TARGET_MIRROR) {
				rep_dev_info->mirror_info.dev_major = rep_target->data.linear->major;
				rep_dev_info->mirror_info.dev_minor = rep_target->data.linear->minor;
				strcpy(rep_dev_info->mirror_info.name, rep_child_name);

				rep_dev_info->source_info.dev_major = rep_child_target->data.mirror->devices[0].major;
				rep_dev_info->source_info.dev_minor = rep_child_target->data.mirror->devices[0].minor;
				rep_dev_info->target_info.dev_major = rep_child_target->data.mirror->devices[1].major;
				rep_dev_info->target_info.dev_minor = rep_child_target->data.mirror->devices[1].minor;
			}

			EngFncs->dm_deallocate_targets(rep_child_target);
		}
	}

	EngFncs->dm_deallocate_targets(rep_target);

	LOG_EXIT_VOID();
	return;
}


static void find_replace_dm_devices(list_anchor_t result_list) {

	dm_device_list_t * device_list = NULL;
	dm_device_list_t * device;
	replace_device_info_t * rep_dev_info;
	char * source_name;
	char * infix;

	LOG_ENTRY();

	EngFncs->dm_get_devices(&device_list);

	for (device = device_list; device != NULL; device = device->next) {
		if (strncmp(device->name, REPLACE_NAME_PREFIX, REPLACE_NAME_PREFIX_LEN) == 0) {

			rep_dev_info = EngFncs->engine_alloc(sizeof(replace_device_info_t));
			if (rep_dev_info) {
				rep_dev_info->replace_info.dev_major = device->dev_major;
				rep_dev_info->replace_info.dev_minor = device->dev_minor;
				strcpy(rep_dev_info->replace_info.name, device->name);

				source_name = device->name + REPLACE_NAME_PREFIX_LEN;

				infix = strstr(source_name, REPLACE_NAME_INFIX);
				if (infix != NULL) {
					char ch = *infix;

					*infix = '\0';
					strcpy(rep_dev_info->source_info.name, source_name);
					*infix = ch;

					strcpy(rep_dev_info->target_info.name, infix + REPLACE_NAME_INFIX_LEN);
				}

				get_mirror_info(device_list, rep_dev_info);

				EngFncs->insert_thing(replace_devices, rep_dev_info, INSERT_AFTER, NULL);
			}
		}
	}

	EngFncs->dm_deallocate_device_list(device_list);

	LOG_EXIT_VOID();
	return;
}


#define COPY_TITLE_FMT "Copy %s to %s\n"

static void init_copy_job(rep_private_data_t * pd) {

	LOG_ENTRY();

	pd->copy_job.title = EngFncs->engine_alloc(sizeof(COPY_TITLE_FMT) +
						   strlen(pd->source->name) +
						   strlen(pd->target->name));

	if (pd->copy_job.title != NULL) {
		sprintf(pd->copy_job.title, COPY_TITLE_FMT, pd->source->name, pd->target->name);

	} else {
		LOG_CRITICAL("Unable to get memory for a copy progress title.\n");
	}
	pd->copy_job.description = NULL;

	pd->copy_job.src.obj = pd->source;
	pd->copy_job.src.start = 0;
	pd->copy_job.src.len = pd->source->size;

	pd->copy_job.trg.obj = pd->target;
	pd->copy_job.trg.start = 0;
	pd->copy_job.trg.len = pd->target->size;

	LOG_EXIT_VOID();
	return;
}


static void finish_replace_object_setup(replace_device_info_t * rep_dev_info) {

	LOG_ENTRY();

	MESSAGE("Found an unfinished replace of %s with %s.\n",
		rep_dev_info->source_obj->name, rep_dev_info->target_obj->name);

	if (rep_dev_info->mirror_info.dev_major != 0) {
		/*
		 * The copy is in progress.  Initialize the
		 * copy job and call the Engine to do the copy
		 * setup which connects the mirror to the copy
		 * job.
		 */
		rep_private_data_t * pd = (rep_private_data_t *) rep_dev_info->replace_obj->private_data;

		init_copy_job(pd);

		EngFncs->copy_setup(&pd->copy_job);

		MESSAGE("The copying of the data is still in progress.\n");

	} else {
		MESSAGE("The copying of the data has finished.\n");
	}

	MESSAGE("The replace operation will be completed when the changes are saved.\n");

	/*
	 * We are finished building the replace object.
	 * We can free the replace_device_info_t.
	 */
	EngFncs->remove_thing(replace_devices, rep_dev_info);
	EngFncs->engine_free(rep_dev_info);

	LOG_EXIT_VOID();
	return;
}


static int export_replace_object(storage_object_t *      obj,
				 replace_device_info_t * rep_dev_info,
				 list_anchor_t           output_list) {
	int rc;

	LOG_ENTRY();

	rep_dev_info->source_obj = obj;

	rc = make_replace_object(obj, rep_dev_info->target_obj,
				 rep_dev_info->replace_info.name,
				 &rep_dev_info->replace_obj);

	if (rc == 0) {
		/* Set this guy's major:minor and ACTIVE bit. */
		EngFncs->dm_update_status(rep_dev_info->replace_obj);

		/* This guy needs attention at commit time. */
		rep_dev_info->replace_obj->flags |= SOFLAG_DIRTY;

		EngFncs->insert_thing(output_list, rep_dev_info->replace_obj, INSERT_AFTER, NULL);

		/*
		 * If we had already found the target, we are finished
		 * with building this replace object.
		 */
		if (rep_dev_info->target_obj != NULL) {
			finish_replace_object_setup(rep_dev_info);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int add_target_to_replace_object(replace_device_info_t * rep_dev_info) {

	int rc = 0;
	rep_private_data_t * pd;
	list_element_t el;
	storage_object_t * replace_obj = rep_dev_info->replace_obj;
	storage_object_t * source = rep_dev_info->source_obj;
	storage_object_t * target = rep_dev_info->target_obj;

	LOG_ENTRY();

	pd = replace_obj->private_data;

	if (pd != NULL) {
		pd->target = target;
	}

	el = EngFncs->insert_thing(replace_obj->child_objects,
				   target,
				   INSERT_AFTER,
				   NULL);
	if (el == NULL) {
		LOG_CRITICAL("Error when inserting target object %s into the child_list of replace object %s.\n",
			     target->name, replace_obj->name);
		rc = ENOMEM;
	}

	if (rc == 0) {
		el = EngFncs->insert_thing(target->parent_objects,
					   replace_obj,
					   INSERT_AFTER,
					   NULL);
		if (el == NULL) {
			LOG_CRITICAL("Error when inserting replace object %s into the parent_list of target object %s.\n",
				     replace_obj->name, source->name);
			rc = ENOMEM;

			EngFncs->remove_thing(replace_obj->child_objects, target);
		}

		if (rc == 0) {
			replace_obj->size = min(source->size, target->size);

			finish_replace_object_setup(rep_dev_info);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int look_for_replace_sources_and_targets(list_anchor_t input_list,
						list_anchor_t output_list) {
	int rc = 0;
	list_element_t iter1;
	list_element_t iter2;
	list_element_t iter3;
	storage_object_t * obj;
	replace_device_info_t * rep_dev_info;

	LOG_ENTRY();

	LIST_FOR_EACH(input_list, iter1, obj) {
		boolean handled = FALSE;

		LIST_FOR_EACH_SAFE(replace_devices, iter2, iter3, rep_dev_info) {

			/*
			 * Look for a match by major:minor if we have it.
			 * Else, look for a match by name.
			 */
			if (rep_dev_info->source_info.dev_major != 0) {
				if ((rep_dev_info->source_info.dev_major == obj->dev_major) &&
				    (rep_dev_info->source_info.dev_minor == obj->dev_minor)) {

					/*
					 * Make sure we have the right name for
					 * the source object.
					 */
					memcpy(rep_dev_info->source_info.name,
					       obj->name,
					       sizeof(rep_dev_info->source_info.name));

					rc = export_replace_object(obj, rep_dev_info, output_list);
					handled = TRUE;
					break;
				}

			} else {
				if (strcmp(obj->name,rep_dev_info->source_info.name) == 0) {

					rc = export_replace_object(obj, rep_dev_info, output_list);
					handled = TRUE;
					break;
				}
			}

			if (rep_dev_info->target_info.dev_major != 0) {
				if ((rep_dev_info->target_info.dev_major == obj->dev_major) &&
				    (rep_dev_info->target_info.dev_minor == obj->dev_minor)) {

					/*
					 * Make sure we have the right name for
					 * the target object.
					 */
					memcpy(rep_dev_info->target_info.name,
					       obj->name,
					       sizeof(rep_dev_info->target_info.name));

					rep_dev_info->target_obj = obj;
					if (rep_dev_info->source_obj != NULL) {
						rc = add_target_to_replace_object(rep_dev_info);
					}
					handled = TRUE;
					break;
				}

			} else {
				if (strcmp(obj->name,rep_dev_info->target_info.name) == 0) {

					rep_dev_info->target_obj = obj;
					if (rep_dev_info->source_obj != NULL) {
						rc = add_target_to_replace_object(rep_dev_info);
					}
					handled = TRUE;
					break;
				}
			}
		}

		if (!handled) {
			/*
			 * This object is not involved with a replace.
			 * Put it on the output list.
			 */
			EngFncs->insert_thing(output_list, obj, INSERT_AFTER, NULL);
		}
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


static void cleanup_broken_replace_objects(void) {

	list_element_t iter1;
	list_element_t iter2;
	replace_device_info_t * rep_dev_info;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(replace_devices, iter1, iter2, rep_dev_info) {

		if (rep_dev_info->source_obj != NULL) {

			/*
			 * Free the replace_device_info_t.
			 * If the target is missing, leave it that way.
			 * The replace object will take care of cleaning
			 * up during commit.
			 */
			EngFncs->remove_thing(replace_devices, rep_dev_info);
			EngFncs->engine_free(rep_dev_info);

		} else {
			/*
			 * If the replace object has a mirror, deactivate the
			 * mirror.
			 */
			if (rep_dev_info->mirror_info.name[0] != '\0') {
				strcpy(temp_obj.name, rep_dev_info->mirror_info.name);

				EngFncs->dm_deactivate(&temp_obj);
			}

			/*
			 * Deactivate the replace device.
			 */
			strcpy(temp_obj.name, rep_dev_info->replace_info.name);

			EngFncs->dm_deactivate(&temp_obj);
		}
	}

	LOG_EXIT_VOID();
	return;
}


/*
 * rep_discover
 */
static int rep_discover(list_anchor_t input_list,
			list_anchor_t output_list,
			boolean       final_call) {
	int rc = 0;

	LOG_ENTRY();

	if (final_call) {
		cleanup_broken_replace_objects();
	}

	if (find_replace_devices) {

		find_replace_dm_devices(replace_devices);

		find_replace_devices = FALSE;
	}

	// Parameter check
	if (!input_list || !output_list) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	if (!EngFncs->list_empty(replace_devices)) {
		rc = look_for_replace_sources_and_targets(input_list, output_list);

	} else {
		/*
		 * No replace objects.  Move the input objects
		 * to the output list.
		 */
		EngFncs->merge_lists(output_list, input_list, NULL, NULL);
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_create
 *
 * Create a new replace object using the specified objects.
 */
static int rep_create(list_anchor_t    objects,
		      option_array_t * options,
		      list_anchor_t    new_obj_list) {

	int rc = 0;
	storage_object_t * source;
	storage_object_t * target;
	storage_object_t * replace_object;
	uint nr_objs;

	LOG_ENTRY();

	/* Check parameters.  Don't care about options. */
	if (!objects || !new_obj_list) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	nr_objs = EngFncs->list_count(objects);

	if (nr_objs != 2) {
		LOG_ERROR("Must specify two objects, source and target, for the replace.\n");
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	source = EngFncs->first_thing(objects, NULL);
	target = EngFncs->last_thing(objects, NULL);

	if ((source == NULL) || (target == NULL)) {
		if (source == NULL) {
			LOG_SERIOUS("Error getting source object from input list.\n");
		}
		if (target == NULL) {
			LOG_SERIOUS("Error getting target object from input list.\n");
		}
		LOG_EXIT_INT(ENOENT);
		return(ENOENT);
	}

	rc = make_replace_object(source, target, NULL, &replace_object);

	if (rc == 0) {
		list_element_t el;

		el = EngFncs->insert_thing(new_obj_list,
					   replace_object,
					   INSERT_AFTER,
					   NULL);

		if (el == NULL) {
			// BUGBUG
			// How to clean up if InsertObject() fails?
			LOG_CRITICAL("Error inserting replace object %s into the new object list.\n",
				     replace_object->name);
			rc = ENOMEM;
		}
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_discard
 *
 * We don't expect the replace plugin to ever be called for discard.
 */
static int rep_discard(list_anchor_t objects) {
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}


/*
 * rep_delete
 *
 * Delete the specified object.
 */
static int rep_delete(storage_object_t * obj,
		      list_anchor_t      children) {
	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) obj->private_data;

	LOG_ENTRY();

	rc = rep_can_delete(obj);

	if (rc == 0) {
		EngFncs->remove_thing(pd->source->parent_objects, obj);
		EngFncs->remove_thing(pd->target->parent_objects, obj);

		rc = EngFncs->concatenate_lists(children, obj->child_objects);

		if (rc == 0) {
			EngFncs->engine_free(pd->copy_job.title);

			EngFncs->copy_cleanup(&pd->copy_job);

			EngFncs->engine_free(pd);
			obj->private_data = NULL;

			rc = EngFncs->free_storage_object(obj);
		}
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_replace_child
 */
static int rep_replace_child(storage_object_t * obj,
			     storage_object_t * child,
			     storage_object_t * new_child) {
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return(ENOSYS);
}


/*
 * rep_add_sectors_to_kill_list
 *
 * Since kill sectors go straight to the disk and do not go through the call
 * stack, the kill sectors must be forwarded on to both the source and target.
 */
static int rep_add_sectors_to_kill_list(storage_object_t * obj,
					lsn_t              lsn,
					sector_count_t     count) {

	int rc = 0;
	rep_private_data_t * pd = obj->private_data;

	LOG_ENTRY();

	rc = pd->source->plugin->functions.plugin->add_sectors_to_kill_list(pd->source, lsn, count);

	if (rc == 0) {
		rc = pd->target->plugin->functions.plugin->add_sectors_to_kill_list(pd->target, lsn, count);

		if (rc != 0) {
			LOG_WARNING("Error code %d when writing kill sectors to target object %s.\n",
				    rc, pd->target->name);
		}

	} else {
		LOG_WARNING("Error code %d when writing kill sectors to source object %s.\n",
			    rc, pd->source->name);
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


static int do_online_copy(storage_object_t * replace_obj) {

	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) replace_obj->private_data;
	dm_target_t * target;

	LOG_ENTRY();

	if (pd->copy_job.mirror == NULL) {
		rc = EngFncs->copy_setup(&pd->copy_job);
	}

	if (rc != 0) {
		LOG_SERIOUS("Error code %d when setting up a copy job: %s\n", rc, EngFncs->strerror(rc));
		LOG_EXIT_INT(rc);
		return rc;
	}

	if (pd->copy_job.flags & COPY_FINISHED) {
		LOG_EXIT_INT(0);
		return 0;
	}

	if (pd->copy_job.flags & COPY_STARTED) {
		EngFncs->copy_wait(&pd->copy_job);

		LOG_EXIT_INT(0);
		return 0;
	}

	/*
	 * Remap the replace object to point to the mirror rather than the
	 * source object.
	 */
	rc = EngFncs->dm_get_targets(replace_obj, &target);

	if (rc == 0) {
		target->data.linear->major = pd->copy_job.mirror->dev_major;
		target->data.linear->minor = pd->copy_job.mirror->dev_minor;
		target->data.linear->start = 0;

		rc = EngFncs->dm_load_targets(replace_obj, target);

		EngFncs->dm_deallocate_targets(target);

		if (rc == 0) {
			/*
			 * Tell the Engine we have a supended device.  The Engine will not
			 * write to the log while a device is suspended, in order to avoid
			 * deadlock.
			 */
			EngFncs->dm_set_suspended_flag(TRUE);

			rc = EngFncs->dm_suspend(replace_obj, TRUE);
			if (rc == 0) {
				rc = EngFncs->copy_start(&pd->copy_job);
				if (rc != 0) {
					LOG_SERIOUS("Error code %d when resuming object %s: %s\n",
						    rc, replace_obj->name, EngFncs->strerror(rc));

					/*
					 * Mirror didn't start.  Don't
					 * load the new mapping for the
					 * replace object.
					 */
					EngFncs->dm_clear_targets(replace_obj);
				}
				rc = EngFncs->dm_suspend(replace_obj, FALSE);
			}

			EngFncs->dm_set_suspended_flag(FALSE);

		} else {
			LOG_SERIOUS("Error code %d when loading the new mirror target for the object %s: %s\n",
				    rc, replace_obj->name, EngFncs->strerror(rc));
		}

	} else {
		LOG_SERIOUS("Error code %d when getting the target for the object %s: %s\n",
			    rc, replace_obj->name, EngFncs->strerror(rc));
	}

	if (rc == 0) {
		rc = EngFncs->copy_wait(&pd->copy_job);

	} else {
		EngFncs->copy_cleanup(&pd->copy_job);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int do_offline_copy(storage_object_t * replace_obj) {

	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) replace_obj->private_data;

	LOG_ENTRY();

	rc = EngFncs->offline_copy(&pd->copy_job);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * rep_commit_changes
 */
static int rep_commit_changes(storage_object_t * obj,
			      uint               phase) {
	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) obj->private_data;
	logical_volume_t * vol;

	LOG_ENTRY();

	/* Sanity checks */
	if (obj->plugin->id != EVMS_REPLACE_PLUGIN_ID) {
		LOG_ERROR("Object %s is not managed by the Replace plug-in.\n", obj->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	switch (phase) {
		case MOVE:

			if (pd->target == NULL) {
				/* Can't copy if we don't have a target.
				 * Leave the DIRTY flag set so that the
				 * Engine will cleanup the replace by
				 * reinstalling the source.
				 */
				LOG_EXIT_INT(0);
				return 0;
			}

			if (pd->copy_job.src.obj == NULL) {
				init_copy_job(pd);
			}

			if (obj->flags & SOFLAG_ACTIVE &&
			    EngFncs->can_online_copy()) {
				rc = do_online_copy(obj);

			} else {
				/* Make sure volume is not mounted for off-line replace. */
				if (!EngFncs->is_offline(obj, &vol)) {
					char * choices[] = {_("Retry"), _("Cancel"), NULL};
					int answer = 0;		// Default is "Retry".

					do {
						QUESTION(&answer, choices,
							 _("Object %s is part of volume %s which is currently mounted on %s.  "
							   "The object cannot be replaced while the volume is mounted.  "
							   "Either unmount the volume and press \"Retry\" or press \"Cancel\" to cancel the replace.\n"),
							 obj->name, vol->name, vol->mount_point);
					} while ((answer == 0) && !EngFncs->is_offline(obj, &vol));

					if (answer != 0) {
						LOG_EXIT_INT(E_CANCELED);
						return E_CANCELED;
					}
				}

				rc = do_offline_copy(obj);
			}

			if (rc == 0) {
				/*
				 * Turn off the dirty bit to signal that the
				 * copy succeeded.
				 */
				obj->flags &= ~SOFLAG_DIRTY;
			}
			break;

		default:
			break;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_can_activate
 *
 * Any existing replace object will be activated.  Asking if it can be
 * activated is redundant.
 */
static int rep_can_activate(storage_object_t * obj) {

	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return(0);
}


/*
 * rep_activate
 */
static int rep_activate(storage_object_t * obj) {

	int rc = 0;
	dm_target_t target;
	dm_device_t linear;
	rep_private_data_t * pd = obj->private_data;

	LOG_ENTRY();

	/* Sanity check */
	if (obj->plugin->id != EVMS_REPLACE_PLUGIN_ID) {
		LOG_ERROR("Object %s is not managed by the Replace plug-in.\n", obj->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/*
	 * Activate with an initial mapping that is the same as the source.
	 */
	target.start = 0;
	target.length = obj->size;
	if (EngFncs->is_2_4_kernel()) {
		target.length &= ~(1);
	}
	target.type = DM_TARGET_LINEAR;
	target.data.linear = &linear;
	target.params = NULL;
	target.next = NULL;
	linear.major = pd->source->dev_major;
	linear.minor = pd->source->dev_minor;
	linear.start = obj->start;

	rc = EngFncs->dm_activate(obj, &target);

	if (rc == 0) {
		obj->flags &= ~SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_can_deactivate
 *
 * Repalce objects cannot be deactivated by the user.
 */
static int rep_can_deactivate(storage_object_t * obj) {

	LOG_ENTRY();

	LOG_EXIT_INT(EPERM);
	return(EPERM);
}


/*
 * rep_deactivate
 */
static int rep_deactivate(storage_object_t * obj) {

	int rc = 0;
	rep_private_data_t * pd = (rep_private_data_t *) obj->private_data;

	LOG_ENTRY();

	if (EngFncs->can_online_copy()) {
		rc = EngFncs->copy_cleanup(&pd->copy_job);
	}

	if (rc == 0) {
		rc = EngFncs->dm_deactivate(obj);
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_get_option_count
 *
 * Determine the type of Task that is being performed, and return
 * the number of options that are available for that Task.
 */
static int rep_get_option_count(task_context_t * task) {

	int count = 0;

	LOG_ENTRY();

	switch (task->action) {
		case EVMS_Task_Create:
			count = 0;
			break;

		default:
			count = -1;
			break;
	}

	LOG_EXIT_INT(count);
	return(count);
}


/*
 * rep_init_task
 *
 * Determine the type of Task that is being performed, and set up the
 * context structure with the appropriate initial values.
 */
static int rep_init_task(task_context_t * context) {

	int rc = 0;
	list_anchor_t tmp_list;

	LOG_ENTRY();

	// Parameter check.
	if (!context) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	switch (context->action) {
		
		case EVMS_Task_Create:
			/* No options for create. */
			context->option_descriptors->count = 0;

			/* Must select one and only one target. */
			context->min_selected_objects = 1;
			context->max_selected_objects = 1;

			/* Get a list of all valid input objects. */
			EngFncs->get_object_list(DISK | SEGMENT | REGION | EVMS_OBJECT,
						 DATA_TYPE,
						 NULL,
						 NULL,
						 VALID_INPUT_OBJECT,
						 &tmp_list);

			/* Filter out objects that are too small. */

			break;

		default:
			rc = EINVAL;
			break;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_set_option
 *
 * Determine the type of Task that is being performed. Then examine the
 * desired option (using the index), and verify that the given value is
 * appropriate. Reset the value if necessary and possible. Adjust other
 * options as appropriate.
 */
static int rep_set_option(task_context_t * context,
			  u_int32_t        index,
			  value_t        * value,
			  task_effect_t  * effect) {
	int rc = 0;

	LOG_ENTRY();

	// Parameter check.
	if (!context || !value || !effect) {
		LOG_EXIT_INT(EFAULT);
	}

	switch (context->action) {
		
		case EVMS_Task_Create:
			break;

		default:
			break;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_set_objects
 *
 * Determine the type of task, and then validate that the objects on the
 * "selected" list are valid for that task. If so, adjust the option
 * descriptor as appropriate.
 */
static int rep_set_objects(task_context_t * context,
			   list_anchor_t          declined_objects,
			   task_effect_t  * effect) {
	int rc = 0;

	LOG_ENTRY();

	// Parameter check.
	if (!context || !declined_objects || !effect) {
		LOG_EXIT_INT(EFAULT);
	}

	switch (context->action) {
		
		case EVMS_Task_Create:
			// Assume front end has selected valid objects from the
			// acceptable_objects list.  Could add safety checking.
			break;

		default:
			break;
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_get_info
 *
 * Return plug-in specific information about the specified object. If the
 * name field is set, only return the "extra" information pertaining
 * to that name.
 */
static int rep_get_info(storage_object_t        * obj,
			char                    * name,
			extended_info_array_t * * info_array) {

	int rc = 0;
	int i = 0;
	extended_info_array_t * info = NULL;
	rep_private_data_t * pd;
	storage_object_t * source;
	storage_object_t * target;

	LOG_ENTRY();

	/* Parameter check */
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	/* Make sure this is a replace object. */
	if (obj->plugin != my_plugin_record) {
		LOG_ERROR("Object %s is not owned by Replace.\n", obj->name);
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	if (name != NULL) {
		if (name[0] != '\0') {
			LOG_ERROR("There is no extra information for object %s.\n", obj->name);
			LOG_EXIT_INT(EINVAL);
			return(EINVAL);
		}
	}

	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t) * 3);
	if (info == NULL) {
		LOG_ERROR("Error allocating memory for info array\n");
		LOG_EXIT_INT(ENOMEM);
		return(ENOMEM);
	}

	pd = (rep_private_data_t *) obj->private_data;
	source = EngFncs->first_thing(obj->child_objects, NULL);
	target = EngFncs->last_thing(obj->child_objects, NULL);
	if ((source == NULL) || (target == NULL)) {
		if (source == NULL) {
			LOG_SERIOUS("Error getting source object from replace object %s.\n", obj->name);
		}
		if (target == NULL) {
			LOG_SERIOUS("Error getting target object from replace object %s.\n", obj->name);
		}
		LOG_EXIT_INT(ENOENT);
		return(ENOENT);
	}

	info->info[i].name = EngFncs->engine_strdup("source");
	info->info[i].title = EngFncs->engine_strdup(_("Source object"));
	info->info[i].desc = EngFncs->engine_strdup(_("The source object for the replace"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(source->name);
	i++;

	info->info[i].name = EngFncs->engine_strdup("target");
	info->info[i].title = EngFncs->engine_strdup(_("Target object"));
	info->info[i].desc = EngFncs->engine_strdup(_("The target object for the replace"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(target->name);
	i++;

	info->info[i].name = EngFncs->engine_strdup("copy_progress");
	info->info[i].title = EngFncs->engine_strdup(_("Copy progress"));
	info->info[i].desc = EngFncs->engine_strdup(_("How much of the copying has been completed"));

	pthread_mutex_lock(&pd->copy_job.progress_mutex);
	if (pd->copy_job.flags & COPY_STARTED) {
		if (pd->copy_job.flags & COPY_FINISHED) {
			info->info[i].type = EVMS_Type_String;
			info->info[i].value.s = EngFncs->engine_strdup(_("Finished"));

		} else {
			info->info[i].type = EVMS_Type_Real32;
			info->info[i].unit = EVMS_Unit_Percent;
			info->info[i].value.r32 = pd->copy_job.progress.count;
			info->info[i].value.r32 /= pd->copy_job.progress.total_count;
			info->info[i].value.r32 *= 100;
		}

	} else {
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("Not started"));
	}
	pthread_mutex_unlock(&pd->copy_job.progress_mutex);
	i++;

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_get_plugin_info
 *
 * Return information about the replace plug-in. There is no "extra"
 * information about MD, so "name" should always be NULL.
 */
static int rep_get_plugin_info(char                    * name,
			       extended_info_array_t * * info_array) {

	extended_info_array_t   * info = NULL;
	char                    buffer[50] = {0};
	int                     i = 0;

	LOG_ENTRY();

	/* Parameter check */
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return(EFAULT);
	}

	if (!name) {
		if (!(info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*6))) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return(ENOMEM);
		}

		info->info[i].name = EngFncs->engine_strdup("ShortName");
		info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->short_name);
		i++;

		info->info[i].name = EngFncs->engine_strdup("LongName");
		info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->long_name);
		i++;

		info->info[i].name = EngFncs->engine_strdup("Type");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Type"));
		info->info[i].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("Device Manager"));
		i++;

		info->info[i].name = EngFncs->engine_strdup("Version");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		info->info[i].name = EngFncs->engine_strdup("Required_Engine_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
							      "It will not run on older versions of the Engine services."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", my_plugin_record->required_engine_api_version.major, my_plugin_record->required_engine_api_version.minor, my_plugin_record->required_engine_api_version.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		info->info[i].name = EngFncs->engine_strdup("Required_Plugin_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Plug-in API Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine plug-in API that this plug-in requires.  "
							      "It will not run on older versions of the Engine plug-in API."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", my_plugin_record->required_plugin_api_version.plugin.major, my_plugin_record->required_plugin_api_version.plugin.minor, my_plugin_record->required_plugin_api_version.plugin.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return(0);
}


/*
 * rep_read
 *
 * Send the read to the source object.
 */
static int rep_read(storage_object_t * obj,
		    lsn_t              lsn,
		    sector_count_t     count,
		    void             * buffer) {

	int  rc = 0;
	rep_private_data_t * pd = obj->private_data;

	LOG_ENTRY();

	/* Parameter check. */
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
	}

	if ((lsn + count) > obj->size) {
		LOG_ERROR("Attempt to read past end of object %s at sector %"PRIu64"\n ", obj->name, lsn+count);
		LOG_EXIT_INT(EFAULT);
	}

	rc = pd->source->plugin->functions.plugin->read(pd->source, lsn, count, buffer);

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_write
 *
 * Send the write to the source and target objects.
 */
static int rep_write(storage_object_t * obj,
		     lsn_t              lsn,
		     sector_count_t     count,
		     void             * buffer) {

	int rc = 0;
	rep_private_data_t * pd = obj->private_data;

	LOG_ENTRY();

	/* Parameter check. */
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if ((lsn + count) > obj->size) {
		LOG_ERROR("Attempt to write past end of object %s at sector %"PRIu64"\n ", obj->name, lsn+count);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (obj->flags & SOFLAG_READ_ONLY) {
		LOG_ERROR("Object %s is read only.\n", obj->name);
		LOG_EXIT_INT(EACCES);
		return EACCES;
	}

	rc = pd->source->plugin->functions.plugin->write(pd->source, lsn, count, buffer);

	if (rc == 0) {
		/* Is this necessary? */
		pd->target->plugin->functions.plugin->write(pd->target, lsn, count, buffer);
	}

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * rep_backup_metadata
 *
 * Replace objects should never get called to backup metadata.
 * Metadata are only backed up when nothing in the system is dirty.
 * Replace objects are always dirty.  They are deleted when the
 * replace has finished, whether it succeeded or not.  So replace
 * objects are never around if nothing in the system is dirty.
 *
 * This stub is here for completeness and to document the above.
 */
static int rep_backup_metadata(storage_object_t * object) {

	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return(0);
}


/* Function table for the Replace Feature */
static plugin_functions_t replace_functions = {
	setup_evms_plugin           : rep_setup_evms_plugin,
	cleanup_evms_plugin         : rep_cleanup_evms_plugin,
	can_delete                  : rep_can_delete,
	can_replace_child           : rep_can_replace_child,
	discover                    : rep_discover,
	create                      : rep_create,
	discard                     : rep_discard,
	delete                      : rep_delete,
	replace_child               : rep_replace_child,
	add_sectors_to_kill_list    : rep_add_sectors_to_kill_list,
	commit_changes              : rep_commit_changes,
	can_activate                : rep_can_activate,
	activate                    : rep_activate,
	can_deactivate              : rep_can_deactivate,
	deactivate                  : rep_deactivate,
	get_option_count            : rep_get_option_count,
	init_task                   : rep_init_task,
	set_option                  : rep_set_option,
	set_objects                 : rep_set_objects,
	get_info                    : rep_get_info,
	get_plugin_info             : rep_get_plugin_info,
	read                        : rep_read,
	write                       : rep_write,
	backup_metadata             : rep_backup_metadata
};



/*
 *  Initialize the local plug-in record
 */

static plugin_record_t replace_plugin_record = {
	id:         EVMS_REPLACE_PLUGIN_ID,

	version:    {major:      MAJOR_VERSION,
		     minor:      MINOR_VERSION,
		     patchlevel: PATCH_LEVEL},

	required_engine_api_version: {major:      15,
		                      minor:      0,
		                      patchlevel: 0},

	required_plugin_api_version: {plugin: {major:      13,
					       minor:      1,
					       patchlevel: 0}},

	short_name: EVMS_REPLACE_PLUGIN_SHORT_NAME,
	long_name:  EVMS_REPLACE_PLUGIN_LONG_NAME,
	oem_name:   EVMS_IBM_OEM_NAME,

	functions:  {plugin: &replace_functions},

	container_functions:    NULL
};


plugin_record_t * evms_plugin_records[] = {
	&replace_plugin_record,
	NULL
};

