# (C) Copyright IBM Corp. 2004
#
# This program is free software;  you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY;  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program;  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Perl module with common subroutines for managing GPT segments.

package Evms::Gpt;

use strict;
use warnings;

use Evms::Common;
use Evms::Object;

BEGIN {
	use Exporter();
	our (@ISA, @EXPORT);
        @ISA = qw(Exporter);
        @EXPORT = qw(&assign_gpt_plugin
		     &unassign_gpt_plugin
		     &create_gpt_segment
		     &create_gpt_basic_data_segment
		     &get_gpt_data_segments
		     &expand_gpt_segment
		     &shrink_gpt_segment
		     &move_gpt_segment
		     &get_gpt_segment_details
		    );
}

# assign_gpt_plugin
#
# Arguments:
#    disk: Disk to assign the GPT plugin to.
sub assign_gpt_plugin($)
{
        my $disk = $_[0];
        my $command = "assign:GptSegMgr={},$disk";
        my $rc;

        $rc = run_evms_command($command);

        return $rc;
}

# unassign_gpt_plugin
#
# Arguments:
#    disk: Disk to unassign the GPT plugin from.
sub unassign_gpt_plugin($)
{
	my $disk = $_[0];
	my $command = "remove:$disk";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# create_gpt_segment
#
# Arguments:
#    freespace: Name of freespace segment to create the new segment from.
#    options: Reference of hash containing options for the create. Allowable
#             hash keys are values are:
#               Size: Size for new segment. Must include appropriate suffix.
#               Offset: Starting offset within the freespace segment. Must
#                       include size suffix.
#               Type: Type-identifier for new segment. Must be "basic data".
#                     "efi system partition", "legacy mbr", or "swap".
sub create_gpt_segment($ $)
{
	my $freespace = $_[0];
	my $options_ref = $_[1];
	my %options = %{$options_ref};
	my ($command, $key, @keys, $rc);

	$command = "create:segment,$freespace";

	@keys = ("Size", "Offset", "Type");
	$command = "create:segment,$freespace";

	# Ideally, we'd just loop foreach kyes(%options), but with GPT, we
	# must ensure the ordering of the options on the command-line. In
	# particular, "Size" must come before "Offset", or "Offset" will be
	# ignored. Yuck!
	foreach $key (@keys) {
		if (defined($options{$key})) {
			$command .= ",$key=$options{$key}";
		}
	}

	$rc = run_evms_command($command);

	return $rc;
}

# create_dos_logical_segment
#
# Arguments:
#    freespace: Name of freespace segment to create the new segment from.
#    size: Size of the new segment. Must include "KB", "MB", or "GB".
sub create_gpt_basic_data_segment($ $)
{
        my $freespace = $_[0];
        my $size = $_[1];
	my %options;
        my $rc;

	$options{"Size"} = $size;

	$rc = create_gpt_segment($freespace, \%options);

	return $rc;
}

# get_gpt_data_segments
#	Get name of data segments.
#
# Arguments:
#	disk: Name of disk to query
#
sub get_gpt_data_segments($$)
{
        my $disk = $_[0];
	my $segments = $_[1];
        my $command = "Query:Segments,Plugin=GptSegMgr,Disk=$disk";
	my @output;
        my $rc;
	my $i=0;

        $rc = run_evms_command($command, \@output);

	# evms returns many lines of output...
	foreach (@output) {
		if ($_ =~ /Segment Name/) {
			my @list;

			next if (($_ =~ /_mbr/) || ($_ =~ /_ebr/) || ($_ =~ /_freespace/));

			# This line contains "Segment Name",
			# split it into tokens
			# and we're only interested in the last token.

			@list = split /\s* \s*/, $_;
			# remove trailing newline if any.
			chomp $list[-1];
			push @{$segments}, $list[-1];
		}
	}

out:
        return $rc;
}

# expand_gpt_segment
#
# Arguments:
#    segment: Name of segment to expand.
#    freespace: Name of freespace segment to use for the expand.
#    size: Amount to expand by. Must include appropriate size suffix.
sub expand_gpt_segment($ $ $)
{
	my $segment = $_[0];
	my $freespace = $_[1];
	my $size = $_[2];
	my $command = "expand:$segment,Size=$size,$freespace";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# shrink_gpt_segment
#
# Arguments:
#    segment: Name of segment to shrink.
#    size: Amount to shrink by. Must include appropriate size suffix.
sub shrink_gpt_segment($ $)
{
	my $segment = $_[0];
	my $size = $_[1];
	my $command = "shrink:$segment,Size=$size,$segment";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# move_gpt_segment
#
# Arguments:
#    segment: Name of segment to move.
#    freespace: Freespace segment to move to.
sub move_gpt_segment($ $)
{
	my $segment = $_[0];
	my $freespace = $_[1];
	my $command = "task:Move,$segment,$freespace";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# get_gpt_segment_details
# Return a hash with extended info details about the specified segment.
#
# Arguments:
#    segment: Name of segment to get details for
sub get_gpt_segment_details($)
{
	my $segment = $_[0];
	my @query_output;
	my %details;

	# Standard info.
	%details = get_object_details($segment);

	return %details;
}

1;

