/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dm-ioctl3.h
 *
 * Definitions for version-3 of the Device-Mapper ioctl interface. Only
 * dm-ioctl3.c should include this header file.
 */

#ifndef _DM_IOCTL3_H_INCLUDED_
#define _DM_IOCTL3_H_INCLUDED_ 1

#define DM_MAX_TYPE_NAME 16
#define DM_NAME_LEN 128
#define DM_UUID_LEN 129

/**
 * struct dm_ioctl
 * @version:		Three parts: major - no backwards or forwards compatibility.
 *				     minor - only backwards compatibility.
 *				     patch - both backwards and forwards compatibility.
 *			This field will be filled in based on the version of the
 *			interface that the engine is compiled for. The kernel
 *			will then fill in this field before completing each
 *			ioctl to let the engine know the current version.
 * @data_size:		Total size of data passed to the ioctl (including
 *			this structure).
 * @data_start:		Offset to start of remaining data, relative to start of
 *			this structure.
 * @target_count:	Number of targets specified in this ioctl, or number
 *			of targets returned by this ioctl.
 * @open_count:		Number of openers of this device in the kernel.
 * @flags:
 * @dev:		Device major:minor.
 * @name:		Device name.
 * @uuid:		Device unique identifier.
 *
 * All ioctl arguments consist of a single chunk of memory, with this structure
 * at the start.  If a uuid is specified any lookup (e.g..  for a DM_INFO) will
 * be done on that, *not* the name.  Also, except for create ioctls, only the
 * name or the uuid may be specified, but not both.
 */
typedef struct dm_ioctl {
	u_int32_t version[3];
	u_int32_t data_size;
	u_int32_t data_start;
	int32_t target_count;
	int32_t open_count;
	u_int32_t flags;
	u_int64_t dev;
	char name[DM_NAME_LEN];
	char uuid[DM_UUID_LEN];
} dm_ioctl_t;

/**
 * struct dm_ioctl_target
 * @start:
 * @length:
 * @status:
 * @next:		Offset, in bytes from the start of this struct, to the
 *			start of the next dm_target_spec.
 * @target_type:
 *
 * Used to specify tables. These structures appear after the dm_ioctl. The
 * parameter string starts immediately after this structure. Be sure to
 * add correct padding after the string to ensure the correct alignment
 * of the next dm_target_spec.
 */
typedef struct dm_ioctl_target {
	u_int64_t start;
	u_int64_t length;
	int32_t status;
	u_int32_t next;
	char target_type[DM_MAX_TYPE_NAME];
} dm_ioctl_target_t;

/*
 * Definition of the ioctl commands.
 */

typedef enum {
	/* Top level commands */
	DM_VERSION_CMD = 0,
	DM_REMOVE_ALL_CMD,

	/* Device level commands */
	DM_DEV_CREATE_CMD,
	DM_DEV_REMOVE_CMD,
	DM_DEV_RELOAD_CMD,
	DM_DEV_RENAME_CMD,
	DM_DEV_SUSPEND_CMD,
	DM_DEV_DEPS_CMD,
	DM_DEV_STATUS_CMD,

	/* Target level commands */
	DM_TARGET_STATUS_CMD,
	DM_TARGET_WAIT_CMD
} dm_ioctl_cmd;

#define DM_IOCTL 0xfd

/* Top level commands. */
#define DM_VERSION	_IOWR(DM_IOCTL, DM_VERSION_CMD, struct dm_ioctl)
#define DM_REMOVE_ALL	_IOWR(DM_IOCTL, DM_REMOVE_ALL_CMD, struct dm_ioctl)

/* Device level commands. */
#define DM_DEV_CREATE	_IOWR(DM_IOCTL, DM_DEV_CREATE_CMD, struct dm_ioctl)
#define DM_DEV_REMOVE	_IOWR(DM_IOCTL, DM_DEV_REMOVE_CMD, struct dm_ioctl)
#define DM_DEV_RELOAD	_IOWR(DM_IOCTL, DM_DEV_RELOAD_CMD, struct dm_ioctl)
#define DM_DEV_SUSPEND	_IOWR(DM_IOCTL, DM_DEV_SUSPEND_CMD, struct dm_ioctl)
#define DM_DEV_RENAME	_IOWR(DM_IOCTL, DM_DEV_RENAME_CMD, struct dm_ioctl)
#define DM_DEV_DEPS	_IOWR(DM_IOCTL, DM_DEV_DEPS_CMD, struct dm_ioctl)
#define DM_DEV_STATUS	_IOWR(DM_IOCTL, DM_DEV_STATUS_CMD, struct dm_ioctl)

/* Target level commands. */
#define DM_TARGET_STATUS _IOWR(DM_IOCTL, DM_TARGET_STATUS_CMD, struct dm_ioctl)
#define DM_TARGET_WAIT	_IOWR(DM_IOCTL, DM_TARGET_WAIT_CMD, struct dm_ioctl)

#define DM_VERSION_MAJOR	3
#define DM_VERSION_MINOR	0
#define DM_VERSION_PATCHLEVEL	0
#define DM_VERSION_EXTRA	"-ioctl (2003-03-28)"

/* Status bits */
#define DM_READONLY_FLAG	0x00000001
#define DM_SUSPEND_FLAG		0x00000002
#define DM_EXISTS_FLAG		0x00000004
#define DM_PERSISTENT_DEV_FLAG	0x00000008

/* Flag passed into ioctl STATUS command to get table information
 * rather than current status.
 */
#define DM_STATUS_TABLE_FLAG	0x00000010

#endif

