/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * 	Module: ecellm.c
 */
#include "eceinternal.h"


typedef struct llm_info_s { 	//information about low level membership info
	int     llm_nodeCount; 	//number of nodes in the cluster
	int     llm_mynode;   	//index of mynode
	int 	llm_arrCount;  //the number of entries in llm_nodes array
	char ** llm_nodes;	//information of each node
} llm_info_t;
#define LLM_SET_MYNODE_IDX(llm,indx) llm.llm_mynode = indx
#define LLM_GET_MYNODE_IDX(llm) llm.llm_mynode
#define LLM_SET_NODECOUNT(llm, count) llm.llm_nodeCount = count
#define LLM_SET_ARRYCOUNT(llm, count) llm.llm_arrCount = count
#define LLM_INC_NODECOUNT(llm) (llm.llm_nodeCount)++
#define LLM_SET_NODEID(llm, i, name)  \
                        llm.llm_nodes[i] = name
#define LLM_GET_NODEID(llm, i)  llm.llm_nodes[i]
#define LLM_GET_MYNODEID(llm)  llm.llm_nodes[LLM_GET_MYNODE_IDX(llm)]
#define LLM_GET_NODECOUNT(llm) 	llm.llm_nodeCount
#define LLM_GET_ARRYCOUNT(llm) 	llm.llm_arrCount
#define LLM_GET_NODES(llm) 	llm.llm_nodes
#define LLM_SET_NODES(llm, ptr) llm.llm_nodes = ptr

static llm_info_t llm;

static void inline llm_ece_status_init(int , int );
static void inline llm_ece_status_cleanup(void);
static void llm_ece_str2status(const char *);
static char *llm_ece_status2str();


//
// set the context to fill in the low level membership information.
//

void
llm_cleanup(void)
{
	int i;
	LOG_ENTRY();
	for(i=0; i < LLM_GET_NODECOUNT(llm); i++){
		ECE_ASSERT(LLM_GET_NODEID(llm,i));
		g_free(LLM_GET_NODEID(llm,i));
	}
	g_free(LLM_GET_NODES(llm));
	LLM_SET_NODECOUNT(llm,0);
	LLM_SET_ARRYCOUNT(llm,0);
	LLM_SET_MYNODE_IDX(llm,-1);
	LLM_SET_NODES(llm,NULL);
	llm_ece_status_cleanup();
	LOG_EXIT_VOID();
}

void
llm_init()
{
	LOG_ENTRY();
	llm_cleanup();
	LOG_EXIT_VOID();
}

gboolean inline
llm_is_filled()
{
	LOG_ENTRY();
	LOG_EXIT_BOOL(LLM_GET_NODECOUNT(llm));
	return LLM_GET_NODECOUNT(llm);
}

static int
llmcompare(const void *value1, const void *value2)
{
	const char *t1 = *(const char * const *)value1;
	const char *t2 = *(const char * const *)value2;
	int   ret;

	LOG_ENTRY();
	ret = strcmp(t1, t2);
	LOG_EXIT_INT(ret);
	return ret;
}


int
llm_get_idx(const char *node)
{
	void *srch;
	int  indx;

	LOG_ENTRY();
	srch = bsearch((void *)&node, (void *)LLM_GET_NODES(llm),
			LLM_GET_NODECOUNT(llm),
			sizeof(char *), llmcompare);
	indx = (srch - (void *)LLM_GET_NODES(llm))/sizeof(char *);

	LOG_EXIT_INT(indx);
	return indx;
}

#ifdef ECEDEBUG
static void
llm_print()
{
	int i;
	LOG_ENTRY();
	LOG_DEBUG("nodecount = %d mynode=%s\n",
			LLM_GET_NODECOUNT(llm),
			LLM_GET_MYNODEID(llm));
	for ( i = 0 ; i < LLM_GET_NODECOUNT(llm) ; i++ ) {
		LOG_DEBUG("node %d is %s\n",i,
			LLM_GET_NODEID(llm,i));
	}
	LOG_EXIT_VOID();
}
#endif /* ECEDEBUG */

//
// done filling in the low level membership.
//
void
llm_end(const char *mynode)
{
	int  indx;

	LOG_ENTRY();
	ECE_ASSERT(LLM_GET_NODECOUNT(llm) > 0);
	qsort((void *)LLM_GET_NODES(llm), LLM_GET_NODECOUNT(llm),
		sizeof(char *), llmcompare);
	indx = llm_get_idx(mynode);
	LLM_SET_MYNODE_IDX(llm, indx);

	llm_ece_status_init(LLM_GET_NODECOUNT(llm), indx);

#ifdef ECEDEBUG
	llm_print();
#endif /* ECEDEBUG */
	LOG_EXIT_VOID();
}





gboolean
llm_is_present(const char *mynode)
{
	void *srch;
	LOG_ENTRY();
	srch = bsearch((void *)&mynode, (void *)LLM_GET_NODES(llm),
			LLM_GET_NODECOUNT(llm),
			sizeof(char *), llmcompare);
	LOG_EXIT_BOOL(srch!=NULL);
	return (srch != NULL);
}



//
// add a node to the low level membership with its
// corresponding attributes.
//
void
llm_add(const char *node, const int len)
{
	int nodecount, arr_count;
	char **ptr;
	char *ptr1;

	LOG_ENTRY();
	nodecount = LLM_GET_NODECOUNT(llm);
	arr_count = LLM_GET_ARRYCOUNT(llm);
	ECE_ASSERT(nodecount <= arr_count && nodecount >= 0);
	if(nodecount == arr_count) {
		/* increment the size of the array */
		arr_count = (arr_count == 0) ? 4 : (arr_count<<1) ;
		LLM_SET_ARRYCOUNT(llm,arr_count);
		ptr = (char **)g_realloc((void *)LLM_GET_NODES(llm),
			arr_count*sizeof(char *));
		LLM_SET_NODES(llm, ptr);
	}

	ptr1 = (char *)g_strndup((gchar*)node, len);
	LLM_SET_NODEID(llm, nodecount, ptr1);
	LLM_INC_NODECOUNT(llm);
	LOG_EXIT_VOID();
}

const char *
llm_getnodeid(int idx)
{
	LOG_ENTRY();
	ECE_ASSERT(idx>=0 && idx<LLM_GET_NODECOUNT(llm));
	LOG_EXIT_PTR(LLM_GET_NODEID(llm, idx));
	return LLM_GET_NODEID(llm, idx);
}

const char *
llm_getmynodeid()
{
	int idx = LLM_GET_MYNODE_IDX(llm);
	LOG_ENTRY();
	ECE_ASSERT(idx>=0 && idx<LLM_GET_NODECOUNT(llm));
	LOG_EXIT_PTR(LLM_GET_NODEID(llm, LLM_GET_MYNODE_IDX(llm)));
	return LLM_GET_NODEID(llm, LLM_GET_MYNODE_IDX(llm));
}

int inline
llm_getconfignodes()
{
	LOG_ENTRY();
	LOG_EXIT_INT(LLM_GET_NODECOUNT(llm));
	return LLM_GET_NODECOUNT(llm);
}

void
llm_getallnodes(ece_nodeid_t *nodes)
{
	int i;
	LOG_ENTRY();
	ECE_ASSERT(LLM_GET_NODECOUNT(llm) > 0);
	for ( i = 0 ; i < LLM_GET_NODECOUNT(llm); i++) {
		memset((nodes+i), 0, sizeof(ece_nodeid_t));
		strcpy((char *)(nodes+i), LLM_GET_NODEID(llm,i));
	}
	LOG_EXIT_VOID();
}

/* create a string of low level membership datastructure */
char *
llm_llm2str()
{
	struct ha_msg *h_msg;
	int nodestrlen,i;
	gchar *nodestr;
	gchar *nodecount_str;
	gchar *mynode_str;
	char *str, *status_str;

	LOG_ENTRY();
	if(LLM_GET_NODECOUNT(llm)==0) {
		LOG_EXIT_PTR(NULL);
		return NULL;
	}
	if ((h_msg=ha_msg_new(0)) == NULL) {
            	LOG_CRITICAL("Error creating a new heartbeat message\n");
		LOG_EXIT_PTR(NULL);
		return NULL;
	}

	nodestrlen = 0;
	for (i = 0; i < LLM_GET_NODECOUNT(llm); i++) {
		nodestrlen += strlen(LLM_GET_NODEID(llm, i))+1;
	}
	nodestr = (char *)g_malloc(nodestrlen+1);
	nodestr[0] = '\0';
	for (i = 0; i < LLM_GET_NODECOUNT(llm); i++) {
		strcat(nodestr, LLM_GET_NODEID(llm,i));
		strcat(nodestr, ",");
	}

	nodecount_str = g_strdup_printf("%d", LLM_GET_NODECOUNT(llm));
	mynode_str    = g_strdup_printf("%d", LLM_GET_MYNODE_IDX(llm));
	status_str    = llm_ece_status2str();

	/* note this message is constructed using the HA's helper functions,
	* however this message is not sent across to heartbeat daemon. It is sent
	* to ece's slave process on the ece channel */
	if ((ha_msg_add(h_msg, F_TYPE, ECEi_LLM) == HA_FAIL)
		||(ha_msg_add(h_msg, ECEi_NODECOUNT, nodecount_str) == HA_FAIL)
		||(ha_msg_add(h_msg, ECEi_MYNODEID, mynode_str) == HA_FAIL)
		||(ha_msg_add(h_msg, ECEi_NODES, nodestr) == HA_FAIL)
		||(ha_msg_add(h_msg, ECEi_ECESTAT,status_str) == HA_FAIL)) {
            	LOG_CRITICAL("Error filling a heartbeat message\n");
		str = NULL;
	} else {
		str = msg2string(h_msg);
	}

	ha_msg_del(h_msg);
	g_free(nodestr);
	g_free(nodecount_str);
	g_free(mynode_str);
	g_free(status_str);

	LOG_EXIT_PTR(str);
	return str;
}

void
llm_str2llm(const char *str, int str_size)
{
	gchar *mynodestr=NULL;
	struct ha_msg *h_msg;
	const char *nodestr, *tmpstr, *statusstr, *type,
		*nodecount_str, *comma, *mynodeid_str;
	int mynodeid, nodecount, counter;

	LOG_ENTRY();
	h_msg = string2msg(str, str_size);

	ECE_ASSERT(type = ha_msg_value(h_msg, F_TYPE));
	ECE_ASSERT(strcmp(type, ECEi_LLM)==0);
	ECE_ASSERT(nodecount_str = ha_msg_value(h_msg, ECEi_NODECOUNT));
	nodecount = atoi(nodecount_str);

	ECE_ASSERT(mynodeid_str = ha_msg_value(h_msg, ECEi_MYNODEID));
	mynodeid = atoi(mynodeid_str);

	ECE_ASSERT(nodestr = ha_msg_value(h_msg, ECEi_NODES));
	ECE_ASSERT(statusstr = ha_msg_value(h_msg, ECEi_ECESTAT));

	tmpstr = nodestr;

	counter=0;
	llm_init();
	while((comma = strchr(tmpstr,','))) {
		llm_add(tmpstr, comma-tmpstr);
		if(counter==mynodeid){
			mynodestr = g_strndup(tmpstr, comma-tmpstr);
		}
		tmpstr = comma+1;
		counter++;
	}
	ECE_ASSERT(mynodestr);
	llm_end(mynodestr);

	ECE_ASSERT(nodecount==LLM_GET_NODECOUNT(llm));
	ECE_ASSERT(mynodeid==LLM_GET_MYNODE_IDX(llm));

 	llm_ece_str2status(statusstr);

	ha_msg_del(h_msg);
	g_free(mynodestr);

#ifdef ECEDEBUG
	llm_print();
#endif /* ECEDEBUG */

	LOG_EXIT_VOID();
}


inline const char *
llm_get_clusterid()
{
	static const char str[]="linuxha";
	LOG_ENTRY();
	LOG_EXIT_PTR(str);
	return str;
}


/* BEGIN OF FUNCTIONS THAT TRACK THE STATUS OF ECE DAEMONS ON EACH NODE
 *  OF THE CLUSTER
 */

static gchar *llm_ece_status=NULL; //status of ece on each node of the cluster
static int	llm_ece_status_size = 0;
static pthread_mutex_t 	llm_ece_mutex=PTHREAD_MUTEX_INITIALIZER;

/* initialize the ece status on all nodes*/
static void inline
llm_ece_status_init(int no_nodes, int indx)
{

	LOG_ENTRY();

  	pthread_mutex_lock(&llm_ece_mutex);
	llm_ece_status_size =  no_nodes/CHAR_BIT;
	if (no_nodes%CHAR_BIT) {
		llm_ece_status_size++;
	}

	llm_ece_status = (gchar *)g_malloc0(sizeof(char)*llm_ece_status_size);
	/* set the ece status on this node as running */
	llm_ece_status[indx / CHAR_BIT] |= 1 << (indx % CHAR_BIT);
  	pthread_mutex_unlock(&llm_ece_mutex);
	LOG_EXIT_VOID();
}

static void inline
llm_ece_status_cleanup()
{
	LOG_ENTRY();
	g_free(llm_ece_status);
	llm_ece_status=NULL;
	llm_ece_status_size=0;
	LOG_EXIT_VOID();
}



void
llm_set_ece_status(const char *node, gboolean flag)
{
	int  indx;

	LOG_ENTRY();
	indx = llm_get_idx(node);
	ECE_ASSERT(llm_ece_status && indx >= 0 && indx < llm_getconfignodes());
  	pthread_mutex_lock(&llm_ece_mutex);
	if(flag){
		llm_ece_status[indx / CHAR_BIT] |= (1 << (indx % CHAR_BIT));
	}else {
		llm_ece_status[indx / CHAR_BIT] &= ~(1 << (indx % CHAR_BIT));
	}
  	pthread_mutex_unlock(&llm_ece_mutex);

	LOG_EXIT_VOID();
}


gboolean
llm_get_ece_status(const char *node)
{
	int  indx;
	gboolean status;

	LOG_ENTRY();
	LOG_DEBUG("Get status for %s.\n", node);
	indx = llm_get_idx(node);
	ECE_ASSERT(llm_ece_status && indx >= 0 && indx < llm_getconfignodes());
  	pthread_mutex_lock(&llm_ece_mutex);
	status =  ((llm_ece_status[indx/CHAR_BIT]&(1<<(indx%CHAR_BIT)))!=0);
  	pthread_mutex_unlock(&llm_ece_mutex);
	LOG_EXIT_BOOL(status);
	return status;
}

static void
llm_ece_str2status(const char *str)
{
	size_t str_len =  strlen(str);
	//int    outbytes = B64_maxbytelen(str_len);

	LOG_ENTRY();
  	pthread_mutex_lock(&llm_ece_mutex);
	base64_to_binary(str, str_len,
			llm_ece_status, llm_ece_status_size);
  	pthread_mutex_unlock(&llm_ece_mutex);
	LOG_EXIT_VOID();
}

static char *
llm_ece_status2str()
{
	int maxstrsize;
	char *str;

	LOG_ENTRY();
	maxstrsize = B64_stringlen(llm_ece_status_size)+1;
	str = (char *)g_malloc(maxstrsize);
	binary_to_base64(llm_ece_status, llm_ece_status_size, str,
			maxstrsize);
	LOG_EXIT_PTR(str);
	return str;
}
/* END OF FUNCTIONS THAT TRACK THE STATUS OF ECE DAEMONS ON EACH NODE
 *  OF THE CLUSTER
 */
