/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libs390.so
 *
 *   File: commit.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#include "vtoc.h"
#include "390segmgr.h"
#include "helpers.h"
#include "format.h"

/*
 *  Called to walk a segment storage object list and return
 *  the metadata segment that corresponds to the commit phase.
 *
 *  Returns: metadata segment ptr if successful
 *           null ptr if not successful
 */
static int  build_vtoc( LOGICALDISK       *ld,
                        DISK_PRIVATE_DATA *disk_pdata,
                        void             **vtoc_records )
{
        int                   rc = ENOSYS;
        DISKSEG              *seg=NULL;
        char                 *vtoc_buffer=NULL;
        char                 *vtoc_record=NULL;
        format1_label_t      *f1 = NULL;
        format4_label_t      *f4 = NULL;
        list_anchor_t               seglist=ld->parent_objects;
        SEG_PRIVATE_DATA     *pdata;
        int                   data_seg_count=0;
        int                   blocksize = disk_pdata->vsectors_per_block;
        list_element_t        iter;

        LOG_ENTRY();


        // get vtoc i/o buffer
        vtoc_buffer = (char *) calloc( 1, disk_pdata->vtoc_sector_count * EVMS_VSECTOR_SIZE );
        if (vtoc_buffer==NULL) {
                *vtoc_records = NULL;
                LOG_ERROR("error, unable to malloc %d blocks of memory for vtoc\n", disk_pdata->max_partitions_allowed+1);
                rc = ENOMEM;
                LOG_EXIT_INT(rc);
                return rc;
        }
        else {
                vtoc_record = vtoc_buffer;
        }

        // build freespace dscb for disk
        rc = init_format5_dscb( ld );
        if (rc) {
                *vtoc_records = NULL;
                free(vtoc_buffer);
                LOG_ERROR("error, unable to build format 5 dscb\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        // add format 4 - vtoc descriptor
        memcpy(vtoc_record, &disk_pdata->f4, sizeof(format4_label_t) );
        vtoc_record += blocksize * EVMS_VSECTOR_SIZE;

        // add format 5 - freespace descriptor
        memcpy(vtoc_record, &disk_pdata->f5, sizeof(format5_label_t) );
        vtoc_record += blocksize * EVMS_VSECTOR_SIZE;

        // add data segments to the vtoc
        LIST_FOR_EACH( seglist, iter, seg ) {

                if ( seg->data_type == DATA_TYPE ) {

                        LOG_DEBUG("building dscb for segment %s, start= %"PRIu64"  size= %"PRIu64"\n", seg->name, seg->start, seg->size );

                        pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                        ++data_seg_count;

                        f1    = (format1_label_t *) vtoc_record;

                        // copy dscb from seg private data area to vtoc buffer
                        memcpy(f1, &pdata->f1, sizeof(format1_label_t) );

                        // update size and offset fields in case we did resize
                        rc = blk2cchh( (seg->start/blocksize), &f1->DS1EXT1.llimit, &disk_pdata->geometry);
                        if (rc == 0) {

                                rc = blk2cchh( ((seg->start+seg->size-1)/blocksize), &f1->DS1EXT1.ulimit, &disk_pdata->geometry);
                                if (rc == 0) {

                                        vtoc_record += blocksize * EVMS_VSECTOR_SIZE;

                                }
                        }

                }
                
                if (rc) break;          
        }
 
        // return results
        if (rc == 0) {

                // update available dscb count
                f4 = (format4_label_t *) vtoc_buffer;
                f4->DS4DSREC = disk_pdata->vtoc_record_count - data_seg_count - 2;

                *vtoc_records  = (void *)vtoc_buffer;  // return the vtoc buffer
        }
        else {
                *vtoc_records  = NULL;
                free(vtoc_buffer);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to walk a segment storage object list and update the
 *  DSCBs out on the 390 disk.
 *
 *  Returns: 0 if partition information is written to disk
 *           error codes otherwise
 *
 */
static int write_vtoc( LOGICALDISK *ld, DISK_PRIVATE_DATA *disk_pdata,
                       DISKSEG *seg, boolean backup )
{
        int     rc=0;
        void   *vtoc_records=NULL;


        LOG_ENTRY();

        if (disk_pdata->vtoc_record_count) {

                rc = build_vtoc( ld, disk_pdata, &vtoc_records );

                if (rc == 0) {

                        if (backup) {
                                rc = EngFncs->save_metadata( seg->name, ld->name, disk_pdata->vtoc_lba,
                                                             disk_pdata->vtoc_sector_count, vtoc_records );
                        } else {
                                rc = WRITE( ld, disk_pdata->vtoc_lba, disk_pdata->vtoc_sector_count, vtoc_records );
                        }

                }
                else {
                        LOG_ERROR("error, build_partition_table() failed to build DSCBs\n");
                }

                if (vtoc_records) free(vtoc_records);
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *   Called to commit the volume label record on the specified disk.
 *
 *   Returns:  0 if successful
 *             error codes otherwise
 *
static int  write_volume_label( LOGICALDISK *ld, DISK_PRIVATE_DATA *disk_pdata )
{
    char               volume_label_sector1[EVMS_VSECTOR_SIZE];
    volume_label_t    *vlabel = (volume_label_t *) volume_label_sector1;
    int                rc;
    lba_t              lba;


    LOG_ENTRY();

    // read the volume label record off disk
    lba  = VOLUME_LABEL_BLOCK_NUMBER * disk_pdata->blocksize;

    rc = READ( ld, lba, 1, (void *) vlabel);

    if (rc == 0) {

        //  update volume label info
        memcpy( vlabel,  &disk_pdata->vlabel, sizeof(volume_label_t) );

        // write it back out to disk
        rc = WRITE( ld, lba, 1, (void *) vlabel);

    }

    LOG_EXIT_INT(rc);
    return rc;
}
****************/


/*
 *  Called by SEG_COMMIT() to write out partition tables and dlat tables for
 *  the specified disk.
 */
int  commit_390_partition_table( LOGICALDISK *ld, DISKSEG *seg, uint commit_phase, boolean backup )
{
        int rc=0;
        DISK_PRIVATE_DATA *disk_pdata;

        LOG_ENTRY();

        disk_pdata = get_s390_disk_private_data(ld);

        if (disk_pdata) {

                rc = write_vtoc(ld, disk_pdata, seg, backup);

        }

        LOG_EXIT_INT(rc);
        return rc;
}

